from flask import Flask, render_template, request, redirect, url_for

app = Flask(__name__)

# --- 多语言翻译数据字典 (Muti-language Translation Dictionary) ---

# 包含中文、英文、法语、德语、意大利语、葡萄牙语、西班牙语、匈牙利语
LANG_DATA = {
    'zh': {'name': '中文', 'flag': '🇨🇳'},
    'en': {'name': 'English', 'flag': '🇺🇸'},
    'fr': {'name': 'Français', 'flag': '🇫🇷'},
    'de': {'name': 'Deutsch', 'flag': '🇩🇪'},
    'it': {'name': 'Italiano', 'flag': '🇮🇹'},
    'pt': {'name': 'Português', 'flag': '🇵🇹'},
    'es': {'name': 'Español', 'flag': '🇪🇸'},
    'hu': {'name': 'Magyar', 'flag': '🇭🇺'},
}

# 结构: [language_code]: { [text_key]: [translated_string] }
TRANSLATIONS = {
    # 网站名称/Meta
    'title': {
        'zh': "BUGHOLD - 高端抓虫吸尘器附件",
        'en': "BUGHOLD - Premium Bug Catcher Vacuum Attachment",
        'fr': "BUGHOLD - Accessoire aspirateur anti-insectes haut de gamme",
        'de': "BUGHOLD - Premium-Vakuumaufsatz zum Insektenfangen",
        'it': "BUGHOLD - Accessorio aspirapolvere cattura insetti Premium",
        'pt': "BUGHOLD - Acessório premium para aspirador de pó para captura de insetos",
        'es': "BUGHOLD - Accesorio premium para aspiradora atrapa-insectos",
        'hu': "BUGHOLD - Prémium rovarfogó porszívó tartozék",
    },
    'meta_desc': {
        'zh': "BUGHOLD 抓虫吸尘器附件，一吸锁住虫子，干净、优雅、不脏手。适配多品牌吸尘器。",
        'en': "BUGHOLD vacuum attachment: gently catch bugs, clean, elegant, and safe. Compatible with multiple vacuum brands.",
        'fr': "Accessoire BUGHOLD : attrapez les insectes en douceur, propre, élégant et sûr. Compatible avec plusieurs marques d'aspirateurs.",
        'de': "BUGHOLD Aufsatz: fängt Insekten sanft, sauber, elegant und sicher. Kompatibel mit verschiedenen Staubsaugermarken.",
        'it': "Accessorio BUGHOLD: cattura insetti delicatamente, pulito, elegante e sicuro. Compatibile con diverse marche di aspirapolvere.",
        'pt': "Acessório BUGHOLD: captura insetos suavemente, limpo, elegante e seguro. Compatível com várias marcas de aspiradores.",
        'es': "Accesorio BUGHOLD: atrapa insectos suavemente, limpio, elegante y seguro. Compatible con varias marcas de aspiradoras.",
        'hu': "BUGHOLD tartozék: gyengéden fogja meg a rovarokat, tiszta, elegáns és biztonságos. Több porszívó márkával kompatibilis.",
    },

    # Header/Footer/Nav
    'logo_sub': {'zh': "虫迹不留", 'en': "No Trace", 'fr': "Sans Trace", 'de': "Keine Spur", 'it': "Nessuna traccia", 'pt': "Sem Vestígios", 'es': "Sin Rastro", 'hu': "Nyom nélkül"},
    'nav_features': {'zh': "产品亮点", 'en': "Features", 'fr': "Points forts", 'de': "Highlights", 'it': "Caratteristiche", 'pt': "Destaques", 'es': "Características", 'hu': "Jellemzők"},
    'nav_how_it_works': {'zh': "工作原理", 'en': "How it works", 'fr': "Fonctionnement", 'de': "Funktionsweise", 'it': "Come funziona", 'pt': "Como funciona", 'es': "Cómo funciona", 'hu': "Működés"},
    'nav_scenes': {'zh': "使用场景", 'en': "Use Cases", 'fr': "Scénarios", 'de': "Einsatzszenarien", 'it': "Casi d'uso", 'pt': "Cenários de Uso", 'es': "Escenarios", 'hu': "Használati esetek"},
    'nav_contact': {'zh': "联系合作", 'en': "Contact", 'fr': "Contact", 'de': "Kontakt", 'it': "Contatti", 'pt': "Contato", 'es': "Contacto", 'hu': "Kapcsolat"},
    'footer_contact': {'zh': "商务合作", 'en': "Partnerships", 'fr': "Partenariats", 'de': "Geschäftskontakt", 'it': "Collaborazioni", 'pt': "Parcerias", 'es': "Colaboraciones", 'hu': "Együttműködések"},
    'footer_privacy': {'zh': "隐私政策", 'en': "Privacy Policy", 'fr': "Politique de Confidentialité", 'de': "Datenschutz", 'it': "Politica sulla Privacy", 'pt': "Política de Privacidade", 'es': "Política de Privacidad", 'hu': "Adatvédelmi irányelvek"},
    'footer_copy': {'zh': "保留所有权利", 'en': "All rights reserved", 'fr': "Tous droits réservés", 'de': "Alle Rechte vorbehalten", 'it': "Tutti i diritti riservati", 'pt': "Todos os direitos reservados", 'es': "Todos los derechos reservados", 'hu': "Minden jog fenntartva"},

    # Hero Section
    'hero_title': {'zh': "一吸锁住虫子，也留住好奇心", 'en': "One gentle vacuum holds the bugs – and keeps curiosity alive.", 'fr': "Un aspirateur doux retient les insectes et maintient la curiosité en éveil.", 'de': "Ein sanfter Sauger fängt Insekten und hält die Neugier lebendig.", 'it': "Un'aspirazione delicata cattura gli insetti e mantiene viva la curiosità.", 'pt': "Um aspirador suave captura insetos e mantém a curiosidade viva.", 'es': "Una succión suave atrapa insectos y mantiene viva la curiosidad.", 'hu': "Egy gyengéd porszívó megtartja a rovarokat, és életben tartja a kíváncsiságot."},
    'hero_subtitle': {'zh': "BUGHOLD 是一款连接吸尘器使用的高端抓虫附件。为怕虫子的人、热爱观察昆虫的孩子与兴趣爱好者、以及需要野外采集样本的科研人员提供一种干净、优雅、可观察又安全的抓虫方式。", 'en': "BUGHOLD is a high-end attachment for your vacuum cleaner, designed for people who are afraid of bugs, for kids and hobbyists who love watching them, and for researchers who need to collect insect samples in the field.", 'fr': "BUGHOLD est un accessoire haut de gamme pour votre aspirateur, conçu pour les personnes qui ont peur des insectes, pour les enfants et les amateurs qui aiment les observer, et pour les chercheurs qui ont besoin de collecter des échantillons d'insectes sur le terrain.", 'de': "BUGHOLD ist ein High-End-Aufsatz für Ihren Staubsauger, konzipiert für Menschen mit Insektenphobie, für Kinder und Hobbyisten, die gerne Insekten beobachten, sowie für Forscher, die Proben im Feld sammeln müssen.", 'it': "BUGHOLD è un accessorio di fascia alta per il tuo aspirapolvere, progettato per le persone che hanno paura degli insetti, per i bambini e gli hobbisti che amano osservarli e per i ricercatori che devono raccogliere campioni di insetti sul campo.", 'pt': "BUGHOLD é um acessório de alta qualidade para o seu aspirador de pó, projetado para pessoas com medo de insetos, para crianças e entusiastas que gostam de observá-los e para pesquisadores que precisam coletar amostras de insetos no campo.", 'es': "BUGHOLD es un accesorio de alta gama para su aspiradora, diseñado para personas con miedo a los insectos, para niños y aficionados que disfrutan observándolos, y para investigadores que necesitan recolectar muestras de insectos en el campo.", 'hu': "A BUGHOLD egy csúcskategóriás porszívó tartozék, amelyet rovaroktól félő embereknek, a megfigyelést kedvelő gyerekeknek és hobbistáknak, valamint terepi mintagyűjtést végző kutatóknak terveztek."},
    'hero_cta_primary': {'zh': "预约试用 / 商务合作", 'en': "Book a Demo / Business Contact", 'fr': "Réserver une démo / Contact Commercial", 'de': "Demo buchen / Geschäftskontakt", 'it': "Prenota una demo / Contatto commerciale", 'pt': "Agendar Demonstração / Contato Comercial", 'es': "Reservar demostración / Contacto comercial", 'hu': "Demó foglalása / Üzleti Kapcsolat"},
    'hero_cta_ghost': {'zh': "了解工作原理", 'en': "Learn more", 'fr': "En savoir plus", 'de': "Mehr erfahren", 'it': "Scopri di più", 'pt': "Saiba mais", 'es': "Aprender más", 'hu': "Tudjon meg többet"},
    'tag_1': {'zh': "磁吸快拆", 'en': "Magnetic quick-release", 'fr': "Détachement magnétique", 'de': "Magnetische Schnellkupplung", 'it': "Sgancio rapido magnetico", 'pt': "Liberação rápida magnética", 'es': "Liberación rápida magnética", 'hu': "Mágneses gyorskioldás"},
    'tag_2': {'zh': "透明观察虫仓", 'en': "Transparent viewing chamber", 'fr': "Chambre d'observation transparente", 'de': "Transparente Beobachtungskammer", 'it': "Camera di osservazione trasparente", 'pt': "Câmara de visualização transparente", 'es': "Cámara de visualización transparente", 'hu': "Átlátszó megfigyelő kamra"},
    'tag_3': {'zh': "科研样本采集", 'en': "Field sampling", 'fr': "Échantillonnage sur le terrain", 'de': "Probenentnahme im Feld", 'it': "Campionamento sul campo", 'pt': "Coleta de amostras de campo", 'es': "Muestreo de campo", 'hu': "Terepi mintagyűjtés"},

    # Features Section
    'feat_title': {'zh': "四种典型场景 · 一只 BUGHOLD 就够了", 'en': "One device, four core scenarios.", 'fr': "Un seul appareil, quatre scénarios principaux.", 'de': "Ein Gerät, vier Hauptszenarien.", 'it': "Un dispositivo, quattro scenari principali.", 'pt': "Um dispositivo, quatro cenários principais.", 'es': "Un dispositivo, cuatro escenarios principales.", 'hu': "Egy eszköz, négy fő felhasználási eset."},
    'feat_subtitle': {'zh': "不再只是“把虫子吸走”这么简单。BUGHOLD 从安全、观察、科研、亲子四个维度重新设计抓虫体验。", 'en': "BUGHOLD is more than a simple suction nozzle. It is re-designed around safety, observation, research and playful learning.", 'fr': "BUGHOLD est plus qu'une simple buse d'aspiration. Il est repensé autour de la sécurité, de l'observation, de la recherche et de l'apprentissage ludique.", 'de': "BUGHOLD ist mehr als eine einfache Saugdüse. Es wurde neu konzipiert für Sicherheit, Beobachtung, Forschung und spielerisches Lernen.", 'it': "BUGHOLD è più di un semplice ugello di aspirazione. È riprogettato attorno alla sicurezza, all'osservazione, alla ricerca e all'apprendimento giocoso.", 'pt': "BUGHOLD é mais do que um bico de sucção simples. Foi reprojetado com foco em segurança, observação, pesquisa e aprendizado lúdico.", 'es': "BUGHOLD es más que una simple boquilla de succión. Está rediseñado en torno a la seguridad, la observación, la investigación y el aprendizaje lúdico.", 'hu': "A BUGHOLD több, mint egy egyszerű szívófej. Újraterveztük a biztonság, a megfigyelés, a kutatás és a játékos tanulás szempontjait figyelembe véve."},
    # Feature 1
    'feat_1_title': {'zh': "① 为怕虫子的人设计", 'en': "For people afraid of bugs", 'fr': "Pour les personnes qui ont peur des insectes", 'de': "Für Menschen mit Insektenphobie", 'it': "Per chi ha paura degli insetti", 'pt': "Para pessoas com medo de insetos", 'es': "Para personas con miedo a los insectos", 'hu': "Rovaroktól félő embereknek tervezve"},
    'feat_1_body': {'zh': "不用靠近虫子、不用拍死、更不会弄脏墙面或地板。将进风口伸向墙角或天花板，一吸即可锁住虫子。", 'en': "No more smashing bugs or getting too close. Just point the intake towards a corner or ceiling and let the vacuum do the work.", 'fr': "Plus besoin d'écraser les insectes ou de s'approcher trop près. Pointez simplement l'entrée d'air vers un coin ou le plafond et laissez l'aspirateur faire le travail.", 'de': "Kein Zerquetschen von Insekten oder zu nahes Herangehen mehr. Richten Sie einfach den Einlass auf eine Ecke oder Decke und lassen Sie den Staubsauger die Arbeit machen.", 'it': "Niente più insetti schiacciati o avvicinamenti pericolosi. Basta puntare l'aspirazione verso un angolo o il soffitto e lasciare che l'aspirapolvere faccia il lavoro.", 'pt': "Chega de esmagar insetos ou se aproximar demais. Basta apontar a entrada para um canto ou teto e deixar o aspirador fazer o trabalho.", 'es': "No más aplastar insectos o acercarse demasiado. Simplemente apunte la entrada hacia una esquina o el techo y deje que la aspiradora haga el trabajo.", 'hu': "Nincs több rovar szétnyomása vagy túl közel kerülés. Csak irányítsa a szívónyílást egy sarok vagy mennyezet felé, és hagyja, hogy a porszívó végezze el a munkát."},
    # Feature 2
    'feat_2_title': {'zh': "② 为兴趣爱好者和孩子提供观察窗口", 'en': "For hobbyists and kids to observe", 'fr': "Pour les amateurs et les enfants à observer", 'de': "Für Hobbyisten und Kinder zum Beobachten", 'it': "Per hobbisti e bambini per osservare", 'pt': "Para entusiastas e crianças observarem", 'es': "Para aficionados y niños para observar", 'hu': "Hobbistáknak és gyerekeknek megfigyelési ablak"},
    'feat_2_body': {'zh': "透明兜虫容器，让孩子可以近距离安全地观察昆虫的触角、翅膀和爬行姿态，变“害怕”为“好奇心”。", 'en': "The transparent chamber turns fear into curiosity: kids can safely watch legs, wings and antennae up close.", 'fr': "La chambre transparente transforme la peur en curiosité : les enfants peuvent observer en toute sécurité les pattes, les ailes et les antennes de près.", 'de': "Die transparente Kammer verwandelt Angst in Neugier: Kinder können Beine, Flügel und Antennen sicher aus der Nähe beobachten.", 'it': "La camera trasparente trasforma la paura in curiosità: i bambini possono osservare in sicurezza zampe, ali e antenne da vicino.", 'pt': "A câmara transparente transforma o medo em curiosidade: as crianças podem observar com segurança pernas, asas e antenas de perto.", 'es': "La cámara transparente transforma el miedo en curiosidad: los niños pueden observar con seguridad patas, alas y antenas de cerca.", 'hu': "Az átlátszó kamra a félelmet kíváncsisággá változtatja: a gyerekek biztonságosan figyelhetik meg a lábakat, szárnyakat és csápokat közelről."},
    # Feature 3
    'feat_3_title': {'zh': "③ 为科研工作者提供野外样本采集工具", 'en': "For researchers in the field", 'fr': "Pour les chercheurs sur le terrain", 'de': "Für Forscher im Feld", 'it': "Per ricercatori sul campo", 'pt': "Para pesquisadores de campo", 'es': "Para investigadores en el campo", 'hu': "Terepi mintagyűjtő eszköz kutatóknak"},
    'feat_3_body': {'zh': "不锈钢结构与防虫跑弹片让虫体保持完整，便于野外快速采集、分类与转移到样品瓶中。", 'en': "Stainless steel parts and the one-way gate keep specimens intact, ideal for field collection and transfer to sample vials.", 'fr': "Les pièces en acier inoxydable et la porte anti-retour maintiennent les spécimens intacts, idéaux pour la collecte sur le terrain et le transfert dans des flacons d'échantillons.", 'de': "Edelstahlteile und das Einwegventil halten Proben intakt, ideal für die Feldsammlung und den Transfer in Probenfläschchen.", 'it': "Le parti in acciaio inossidabile e il cancello unidirezionale mantengono gli esemplari intatti, ideali per la raccolta sul campo e il trasferimento in fiale campione.", 'pt': "Peças de aço inoxidável e o portão unidirecional mantêm os espécimes intactos, ideais para coleta em campo e transferência para frascos de amostras.", 'es': "Las piezas de acero inoxidable y la compuerta unidireccional mantienen intactos los especímenes, ideales para la recolección de campo y la transferencia a viales de muestra.", 'hu': "A rozsdamentes acél alkatrészek és az egyirányú csappantyú épen tartják a mintákat, ideálisak a terepi gyűjtéshez és mintavevő fiolákba történő áthelyezéshez."},
    # Feature 4
    'feat_4_title': {'zh': "④ 为孩子春天捕捉蝴蝶与蜻蜓提供方便", 'en': "For springtime bug-catching with kids", 'fr': "Pour la chasse aux insectes printanière avec les enfants", 'de': "Für das Insektenfangen im Frühling mit Kindern", 'it': "Per la caccia agli insetti primaverile con i bambini", 'pt': "Para captura de insetos na primavera com crianças", 'es': "Para la captura de insectos en primavera con niños", 'hu': "Tavaszi rovarfogáshoz gyerekekkel"},
    'feat_4_body': {'zh': "春天里追逐蝴蝶、蜻蜓、瓢虫，只需轻轻一吸即可捕获，不伤害昆虫，也不弄脏小手。", 'en': "In spring, kids can gently capture butterflies, dragonflies and ladybugs with a soft suction instead of nets or hands.", 'fr': "Au printemps, les enfants peuvent capturer délicatement papillons, libellules et coccinelles avec une aspiration douce au lieu de filets ou de mains.", 'de': "Im Frühling können Kinder Schmetterlinge, Libellen und Marienkäfer sanft mit einem weichen Sog statt mit Netzen oder Händen fangen.", 'it': "In primavera, i bambini possono catturare delicatamente farfalle, libellule e coccinelle con una leggera aspirazione anziché con reti o mani.", 'pt': "Na primavera, as crianças podem capturar delicadamente borboletas, libélulas e joaninhas com uma sucção suave em vez de redes ou mãos.", 'es': "En primavera, los niños pueden capturar delicadamente mariposas, libélulas y mariquitas con una succión suave en lugar de redes o manos.", 'hu': "Tavasszal a gyerekek gyengéd szívással foghatják be a pillangókat, szitakötőket és katicabogarakat hálók vagy kéz helyett."},

    # How It Works Section
    'how_it_works_title': {'zh': "工作原理 · 一路气流，一路安全", 'en': "How it works – one airflow, multiple safeguards.", 'fr': "Fonctionnement – un flux d'air, de multiples sécurités.", 'de': "Funktionsweise – ein Luftstrom, mehrere Sicherheitsvorkehrungen.", 'it': "Come funziona – un flusso d'aria, molteplici misure di sicurezza.", 'pt': "Como funciona – um fluxo de ar, múltiplas salvaguardas.", 'es': "Cómo funciona – un flujo de aire, múltiples salvaguardas.", 'hu': "Működés – egy légáram, többszörös védelem."},
    'how_it_works_subtitle': {'zh': "无论你是怕虫子的人、热爱昆虫的孩子，还是专业科研人员，抓虫的过程都应该是「可控制、可观察、可清理」的。", 'en': "For every user – fearful, curious or professional – the catching process should be controllable, observable and easy to clean.", 'fr': "Pour chaque utilisateur – craintif, curieux ou professionnel – le processus de capture doit être contrôlable, observable et facile à nettoyer.", 'de': "Für jeden Benutzer – ängstlich, neugierig oder professionell – sollte der Fangprozess kontrollierbar, beobachtbar und leicht zu reinigen sein.", 'it': "Per ogni utente – timoroso, curioso o professionale – il processo di cattura dovrebbe essere controllabile, osservabile e facile da pulire.", 'pt': "Para todos os usuários – temerosos, curiosos ou profissionais – o processo de captura deve ser controlável, observável e fácil de limpar.", 'es': "Para cada usuario – temeroso, curioso o profesional – el proceso de captura debe ser controlable, observable y fácil de limpiar.", 'hu': "Minden felhasználó számára – félő, kíváncsi vagy szakember – a befogási folyamatnak ellenőrizhetőnek, megfigyelhetőnek és könnyen tisztíthatónak kell lennie."},
    # Exploded List Items
    'comp_1': {'zh': "抓虫进风管", 'en': "Intake nozzle", 'fr': "Buse d'admission", 'de': "Ansaugdüse", 'it': "Ugello di aspirazione", 'pt': "Bocal de admissão", 'es': "Boquilla de admisión", 'hu': "Szívófej"},
    'comp_1_desc': {'zh': "将长形进风管伸到角落或天花板边缘，虫子被吸入。", 'en': "Extend the nozzle to corners or ceiling edges; the bug is sucked in.", 'fr': "Étendez la buse vers les coins ou les bords du plafond ; l'insecte est aspiré.", 'de': "Führen Sie die lange Düse in Ecken oder an Deckenkanten; das Insekt wird eingesaugt.", 'it': "Estendi l'ugello negli angoli o ai bordi del soffitto; l'insetto viene aspirato.", 'pt': "Estenda o bocal para cantos ou bordas do teto; o inseto é sugado.", 'es': "Extienda la boquilla a esquinas o bordes del techo; el insecto es aspirado.", 'hu': "Hosszítsa meg a fúvókát a sarkokhoz vagy a mennyezet széléhez; a rovar beszívódik."},
    'comp_2': {'zh': "磁铁", 'en': "Magnetic ring", 'fr': "Anneau magnétique", 'de': "Magnetring", 'it': "Anello magnetico", 'pt': "Anel magnético", 'es': "Anillo magnético", 'hu': "Mágnesgyűrű"},
    'comp_2_desc': {'zh': "提供稳固又易拆的连接结构，方便拆卸虫仓。", 'en': "Provides a secure yet easy-to-detach connection for removing the chamber.", 'fr': "Fournit une connexion sécurisée mais facile à détacher pour retirer la chambre.", 'de': "Bietet eine sichere und leicht lösbare Verbindung zum Entfernen der Kammer.", 'it': "Fornisce una connessione sicura ma facile da staccare per rimuovere la camera.", 'pt': "Fornece uma conexão segura, mas fácil de soltar, para remover a câmara.", 'es': "Proporciona una conexión segura pero fácil de desmontar para quitar la cámara.", 'hu': "Biztonságos, mégis könnyen oldható csatlakozást biztosít a kamra eltávolításához."},
    'comp_3': {'zh': "不锈钢防虫跑弹片", 'en': "Stainless one-way gate", 'fr': "Clapet anti-retour en acier inoxydable", 'de': "Edelstahl-Einwegklappe", 'it': "Cancello unidirezionale in acciaio inox", 'pt': "Portão unidirecional de aço inoxidável", 'es': "Compuerta unidireccional de acero inoxidable", 'hu': "Rozsdamentes egyirányú csappantyú"},
    'comp_3_desc': {'zh': "虫子通过时弹片打开，随后立即复位，形成单向“防逃门”。", 'en': "The flap opens when the bug passes, then immediately reseats, creating a one-way 'no-escape gate'.", 'fr': "Le clapet s'ouvre lorsque l'insecte passe, puis se referme immédiatement, créant une 'porte sans évasion' unidirectionnelle.", 'de': "Die Klappe öffnet sich, wenn das Insekt passiert, und schließt sich sofort wieder, wodurch eine Einweg-'Fluchttür' entsteht.", 'it': "Il lembo si apre quando l'insetto passa, quindi si richiude immediatamente, creando un 'cancello senza fuga' unidirezionale.", 'pt': "A aba se abre quando o inseto passa e se fecha imediatamente, criando um 'portão sem fuga' unidirecional.", 'es': "La solapa se abre cuando el insecto pasa, luego se vuelve a sellar inmediatamente, creando una 'compuerta sin escape' unidireccional.", 'hu': "A csappantyú kinyílik, amikor a rovar áthalad, majd azonnal visszaáll, létrehozva egy egyirányú 'szökésgátló kaput'."},
    'comp_4': {'zh': "不锈钢兜虫容器", 'en': "Collection chamber", 'fr': "Chambre de collecte", 'de': "Sammelkammer", 'it': "Camera di raccolta", 'pt': "Câmara de coleta", 'es': "Cámara de colección", 'hu': "Gyűjtőkamra"},
    'comp_4_desc': {'zh': "所有昆虫最终被收集于此，可选择不透明或透明版本。", 'en': "All insects are finally collected here, available in opaque or transparent versions.", 'fr': "Tous les insectes sont finalement collectés ici, disponibles en versions opaque ou transparente.", 'de': "Alle Insekten werden hier gesammelt, erhältlich in opaken oder transparenten Versionen.", 'it': "Tutti gli insetti vengono raccolti qui, disponibili in versione opaca o trasparente.", 'pt': "Todos os insetos são finalmente coletados aqui, disponíveis em versões opaca ou transparente.", 'es': "Todos los insectos son finalmente recolectados aquí, disponibles en versiones opaca o transparente.", 'hu': "Minden rovar itt gyűlik össze, átlátszatlan vagy átlátszó változatban kapható."},
    'comp_5': {'zh': "吸尘器对接头", 'en': "Vacuum connector", 'fr': "Connecteur d'aspirateur", 'de': "Staubsaugeranschluss", 'it': "Connettore per aspirapolvere", 'pt': "Conector de aspirador", 'es': "Conector de aspiradora", 'hu': "Porszívó csatlakozó"},
    'comp_5_desc': {'zh': "与吸尘器主机相连，维持稳定吸力和气流路径。", 'en': "Connects to the vacuum body, maintaining stable suction and airflow path.", 'fr': "Se connecte au corps de l'aspirateur, maintenant une aspiration stable et un chemin de flux d'air.", 'de': "Wird mit dem Staubsaugerkörper verbunden, um eine stabile Saugkraft und einen konstanten Luftstromweg aufrechtzuerhalten.", 'it': "Si collega al corpo dell'aspirapolvere, mantenendo un'aspirazione stabile e un percorso del flusso d'aria.", 'pt': "Conecta-se ao corpo do aspirador, mantendo sucção estável e caminho do fluxo de ar.", 'es': "Se conecta al cuerpo de la aspiradora, manteniendo una succión estable y la trayectoria del flujo de aire.", 'hu': "Csatlakozik a porszívó testéhez, fenntartva a stabil szívást és légáramlási utat."},
    'airflow_path_title': {'zh': "气流路径", 'en': "Airflow path", 'fr': "Trajet de l'air", 'de': "Luftstrompfad", 'it': "Percorso del flusso d'aria", 'pt': "Caminho do fluxo de ar", 'es': "Trayectoria del flujo de aire", 'hu': "Légáramlási útvonal"},
    'airflow_path_desc_1': {'zh': "气流从吸虫进风口高速进入，携带虫子通过弹片进入容器，虫子被留在容器中，气流则通过底部孔洞继续流向吸尘器。", 'en': "Air enters the intake at high speed, carrying the bug through the gate into the chamber. The bug stays in the chamber, and the air continues through the perforated base into the vacuum.", 'fr': "L'air entre à grande vitesse par l'admission, transportant l'insecte à travers le clapet dans la chambre. L'insecte reste dans la chambre et l'air continue à travers la base perforée vers l'aspirateur.", 'de': "Luft tritt mit hoher Geschwindigkeit in den Einlass ein, trägt das Insekt durch das Ventil in die Kammer. Das Insekt verbleibt in der Kammer, und die Luft strömt durch den perforierten Boden weiter in den Staubsauger.", 'it': "L'aria entra nell'aspirazione ad alta velocità, trasportando l'insetto attraverso il cancello nella camera. L'insetto rimane nella camera e l'aria continua attraverso la base perforata nell'aspirapolvere.", 'pt': "O ar entra na admissão em alta velocidade, transportando o inseto através do portão para a câmara. O inseto permanece na câmara, e o ar continua através da base perfurada para o aspirador.", 'es': "El aire entra por la admisión a alta velocidad, llevando el insecto a través de la compuerta a la cámara. El insecto permanece en la cámara, y el aire continúa a través de la base perforada hacia la aspiradora.", 'hu': "A levegő nagy sebességgel jut be a szívónyílásba, átviszi a rovart a csappantyún keresztül a kamrába. A rovar a kamrában marad, és a levegő a perforált alapon keresztül tovább áramlik a porszívóba."},
    'airflow_path_desc_2': {'zh': "这种结构既避免虫子被打碎，也避免虫体进入吸尘器主机，对家庭用户、兴趣玩家和科研人员都更友好。", 'en': "This keeps insects intact and out of the vacuum body – cleaner for homes, nicer for hobbyists, and more reliable for scientific sampling.", 'fr': "Cela maintient les insectes intacts et hors du corps de l'aspirateur – plus propre pour les maisons, mieux pour les amateurs et plus fiable pour l'échantillonnage scientifique.", 'de': "Dies hält Insekten intakt und außerhalb des Staubsaugerkörpers – sauberer für Haushalte, besser für Hobbyisten und zuverlässiger für die wissenschaftliche Probenentnahme.", 'it': "Ciò mantiene gli insetti intatti e fuori dal corpo dell'aspirapolvere – più pulito per le case, migliore per gli hobbisti e più affidabile per il campionamento scientifico.", 'pt': "Isso mantém os insetos intactos e fora do corpo do aspirador – mais limpo para residências, melhor para entusiastas e mais confiável para amostragem científica.", 'es': "Esto mantiene los insectos intactos y fuera del cuerpo de la aspiradora – más limpio para hogares, mejor para aficionados y más fiable para el muestreo científico.", 'hu': "Ez épen tartja a rovarokat, és kívül tartja a porszívó testén – tisztább az otthonokban, jobb a hobbistáknak, és megbízhatóbb a tudományos mintavételhez."},

    # Scenes Section
    'scenes_title': {'zh': "典型使用场景", 'en': "Typical use cases", 'fr': "Cas d'utilisation typiques", 'de': "Typische Anwendungsfälle", 'it': "Casi d'uso tipici", 'pt': "Casos de uso típicos", 'es': "Casos de uso típicos", 'hu': "Tipikus használati esetek"},
    # Scene 1
    'scene_1_title': {'zh': "卫生间与厨房墙角", 'en': "Bathroom & kitchen corners", 'fr': "Coins de salle de bain et de cuisine", 'de': "Bad- & Küchenecken", 'it': "Angoli di bagno e cucina", 'pt': "Cantos de banheiro e cozinha", 'es': "Esquinas de baño y cocina", 'hu': "Fürdőszoba és konyhasarkok"},
    'scene_1_body': {'zh': "轻松吸走蟑螂、蜘蛛、飞蛾等，让角落保持干净整洁。", 'en': "Quickly remove roaches, spiders and moths from corners without leaving marks.", 'fr': "Enlevez rapidement les cafards, les araignées et les mites des coins sans laisser de traces.", 'de': "Entfernen Sie schnell Kakerlaken, Spinnen und Motten aus Ecken, ohne Spuren zu hinterlassen.", 'it': "Rimuovi rapidamente scarafaggi, ragni e falene dagli angoli senza lasciare segni.", 'pt': "Remova rapidamente baratas, aranhas e mariposas dos cantos sem deixar marcas.", 'es': "Elimine rápidamente cucarachas, arañas y polillas de las esquinas sin dejar marcas.", 'hu': "Gyorsan távolítsa el a csótányokat, pókokat és lepkéket a sarkokból nyomok nélkül."},
    # Scene 2
    'scene_2_title': {'zh': "卧室天花板与窗帘顶端", 'en': "Ceilings & curtain rails", 'fr': "Plafonds et tringles à rideaux", 'de': "Decken & Gardinenstangen", 'it': "Soffitti e binari per tende", 'pt': "Tetos e trilhos de cortina", 'es': "Techos y rieles de cortinas", 'hu': "Mennyezetek és függönykarnisok"},
    'scene_2_body': {'zh': "不必踩椅子挥拍子，配合吸尘器延长管即可够到高处。", 'en': "Reach high ceilings and curtain rails with the vacuum extension tube safely.", 'fr': "Atteignez les plafonds hauts et les tringles à rideaux avec le tube d'extension de l'aspirateur en toute sécurité.", 'de': "Erreichen Sie hohe Decken und Gardinenstangen sicher mit dem Staubsauger-Verlängerungsrohr.", 'it': "Raggiungi soffitti alti e binari per tende in sicurezza con il tubo di prolunga dell'aspirapolvere.", 'pt': "Alcance tetos altos e trilhos de cortina com o tubo de extensão do aspirador com segurança.", 'es': "Alcance techos altos y rieles de cortinas de forma segura con el tubo de extensión de la aspiradora.", 'hu': "Biztonságosan elérheti a magas mennyezeteket és függönykarnisokat a porszívó hosszabbítócsövével."},
    # Scene 3
    'scene_3_title': {'zh': "亲子自然教育与昆虫观察", 'en': "Nature learning with kids", 'fr': "Apprentissage de la nature avec les enfants", 'de': "Naturerziehung mit Kindern", 'it': "Apprendimento della natura con i bambini", 'pt': "Aprendizagem da natureza com crianças", 'es': "Aprendizaje de la naturaleza con niños", 'hu': "Természetismeret gyerekekkel"},
    'scene_3_body': {'zh': "透明虫仓让孩子近距离观察昆虫的生活状态，家长随时掌控安全距离。", 'en': "The transparent chamber makes close-up observation safe and supervised.", 'fr': "La chambre transparente rend l'observation de près sûre et supervisée.", 'de': "Die transparente Kammer macht Nahbeobachtungen sicher und überwacht.", 'it': "La camera trasparente rende l'osservazione ravvicinata sicura e supervisionata.", 'pt': "A câmara transparente torna a observação de perto segura e supervisionada.", 'es': "La cámara transparente hace que la observación de cerca sea segura y supervisada.", 'hu': "Az átlátszó kamra biztonságossá és felügyeltté teszi a közeli megfigyelést."},
    # Scene 4
    'scene_4_title': {'zh': "野外科研采样", 'en': "Field research sampling", 'fr': "Échantillonnage pour la recherche sur le terrain", 'de': "Feldforschungsprobenahme", 'it': "Campionamento per la ricerca sul campo", 'pt': "Amostragem para pesquisa de campo", 'es': "Muestreo para investigación de campo", 'hu': "Terepi kutatási mintavétel"},
    'scene_4_body': {'zh': "快速采集并转移到样本瓶中，适合昆虫学、生态调查和教学实习。", 'en': "Collect and transfer insects efficiently for entomology and ecology work.", 'fr': "Collectez et transférez les insectes efficacement pour les travaux d'entomologie et d'écologie.", 'de': "Sammeln und übertragen Sie Insekten effizient für entomologische und ökologische Arbeiten.", 'it': "Raccogli e trasferisci gli insetti in modo efficiente per lavori di entomologia ed ecologia.", 'pt': "Colete e transfira insetos de forma eficiente para trabalhos de entomologia e ecologia.", 'es': "Recoja y transfiera insectos de manera eficiente para trabajos de entomología y ecología.", 'hu': "Gyűjtsön és helyezzen át rovarokat hatékonyan az entomológiai és ökológiai munkához."},

    # Contact Section
    'contact_title': {'zh': "合作与订购", 'en': "Partnerships & orders", 'fr': "Partenariats et commandes", 'de': "Partnerschaften & Bestellungen", 'it': "Collaborazioni e ordini", 'pt': "Parcerias e pedidos", 'es': "Colaboraciones y pedidos", 'hu': "Partnerségek és megrendelések"},
    'contact_subtitle': {'zh': "BUGHOLD 面向家庭用户、昆虫兴趣爱好者、亲子教育机构、以及高校与科研单位开放合作。", 'en': "BUGHOLD is open for collaboration with home appliance brands, education organizations and research institutes.", 'fr': "BUGHOLD est ouvert à la collaboration avec les marques d'appareils électroménagers, les organisations éducatives et les instituts de recherche.", 'de': "BUGHOLD ist offen für die Zusammenarbeit mit Haushaltsgeräteherstellern, Bildungseinrichtungen und Forschungsinstituten.", 'it': "BUGHOLD è aperto alla collaborazione con marchi di elettrodomestici, organizzazioni educative e istituti di ricerca.", 'pt': "BUGHOLD está aberto para colaboração com marcas de eletrodomésticos, organizações educacionais e institutos de pesquisa.", 'es': "BUGHOLD está abierto a la colaboración con marcas de electrodomésticos, organizaciones educativas e institutos de investigación.", 'hu': "A BUGHOLD nyitott az együttműködésre háztartási gép márkákkal, oktatási szervezetekkel és kutatóintézetekkel."},
    'contact_list_1': {'zh': "商务合作邮箱：", 'en': "Business contact:", 'fr': "Contact commercial :", 'de': "Geschäftskontakt:", 'it': "Contatto commerciale:", 'pt': "Contato comercial:", 'es': "Contacto comercial:", 'hu': "Üzleti kapcsolat:"},
    'contact_list_2': {'zh': "支持颜色、材质、LOGO 与包装定制。", 'en': "Color, material, branding and packaging customization are supported.", 'fr': "La personnalisation des couleurs, des matériaux, de l'image de marque et de l'emballage est prise en charge.", 'de': "Farb-, Material-, Marken- und Verpackungsanpassung werden unterstützt.", 'it': "Sono supportate la personalizzazione di colore, materiale, branding e packaging.", 'pt': "São suportadas a personalização de cores, materiais, branding e embalagens.", 'es': "Se admite la personalización de color, material, marca y embalaje.", 'hu': "Szín, anyag, márkaépítés és csomagolás testreszabása támogatott."},
    'contact_highlight_1': {'zh': "想让你的吸尘器多一个“高端抓虫模式”？", 'en': "Want to add a 'premium bug-catching mode' to your vacuum?", 'fr': "Vous souhaitez ajouter un 'mode de capture d'insectes premium' à votre aspirateur?", 'de': "Möchten Sie Ihrem Staubsauger einen 'Premium-Insektenfangmodus' hinzufügen?", 'it': "Vuoi aggiungere una 'modalità premium per la cattura degli insetti' al tuo aspirapolvere?", 'pt': "Quer adicionar um 'modo premium de captura de insetos' ao seu aspirador?", 'es': "Quiere añadir un 'modo premium de captura de insectos' a su aspiradora?", 'hu': "Szeretne 'prémium rovarfogó módot' adni porszívójához?"},
    'contact_highlight_2': {'zh': "Let BUGHOLD hold the bugs for you.", 'en': "Let BUGHOLD hold the bugs for you.", 'fr': "Laissez BUGHOLD retenir les insectes pour vous.", 'de': "Lassen Sie BUGHOLD die Insekten für Sie fangen.", 'it': "Lascia che BUGHOLD catturi gli insetti per te.", 'pt': "Deixe BUGHOLD segurar os insetos para você.", 'es': "Deje que BUGHOLD atrape los insectos por usted.", 'hu': "Hagyja, hogy a BUGHOLD elkapja a rovarokat Ön helyett."},
}

DEFAULT_LANG = 'zh'

@app.route("/", defaults={'lang_code': DEFAULT_LANG})
@app.route("/<lang_code>")
def index(lang_code):
    """
    主要路由：根据 URL 中的语言代码加载翻译文本并渲染页面。
    """
    if lang_code not in LANG_DATA:
        # 如果语言代码无效，重定向到默认语言
        return redirect(url_for('index', lang_code=DEFAULT_LANG))
    
    # 提取当前语言的翻译文本
    t = {key: trans[lang_code] for key, trans in TRANSLATIONS.items()}
    
    return render_template(
        "index.html",
        t=t, # 翻译文本
        lang_code=lang_code, # 当前语言代码
        lang_data=LANG_DATA, # 所有语言信息
        current_year=2025 # 脚注年份
    )

@app.route("/change_lang", methods=['POST'])
def change_lang():
    """
    处理语言选择表单提交，重定向到新选择的语言页面。
    """
    new_lang_code = request.form.get('lang_code')
    if new_lang_code in LANG_DATA:
        return redirect(url_for('index', lang_code=new_lang_code))
    # 如果语言代码无效，返回主页
    return redirect(url_for('index', lang_code=DEFAULT_LANG))


if __name__ == "__main__":
    app.run(host="0.0.0.0", port=5000, debug=True)
