from flask import Flask, render_template, request, jsonify, redirect, url_for, session
import time
import datetime
import os
import random
import json
from DRV8825 import DRV8825

app = Flask(__name__)
app.secret_key = 'parent_config_secret'
PARENT_PASSWORD = 'rendong'  # 家长固定密码

# --------------------------
# 路径配置（动态获取，适配所有环境）
# --------------------------
current_program_dir = os.path.dirname(os.path.abspath(__file__))
RECORD_FILE = os.path.join(current_program_dir, "daily_answers.json")
CONFIG_FILE = os.path.join(current_program_dir, "parent_config.json")
print(f"程序运行目录: {current_program_dir}")
print(f"配置文件路径: {CONFIG_FILE}")
print(f"答题记录路径: {RECORD_FILE}")

# --------------------------
# 电机初始化（增加详细日志）
# --------------------------
motors = {}
try:
    # 电机1：dir=13, step=19, enable=12, mode=(16,17,20)
    Motor1 = DRV8825(dir_pin=13, step_pin=19, enable_pin=12, mode_pins=(16, 17, 20))
    # 电机2：dir=24, step=18, enable=4, mode=(21,22,27)
    Motor2 = DRV8825(dir_pin=24, step_pin=18, enable_pin=4, mode_pins=(21, 22, 27))
    motors = {"1": Motor1, "2": Motor2}
    print("✅ 电机1、2初始化成功")
except Exception as e:
    print(f"❌ 电机初始化失败: {str(e)}")
    motors = {}

# --------------------------
# 家长配置相关函数
# --------------------------
def load_parent_config():
    default_config = {
        "num_min": 1,
        "num_max": 5,
        "arithmetic_type": "add",
        "option_count": 3,
        "daily_question_count": 5
    }
    if not os.path.exists(CONFIG_FILE):
        print(f"⚠️  配置文件不存在，使用默认配置并保存: {CONFIG_FILE}")
        save_parent_config(default_config)
        return default_config
    try:
        with open(CONFIG_FILE, 'r', encoding='utf-8') as f:
            config = json.load(f)
        # 校验配置合法性（避免异常值）
        valid_config = default_config.copy()
        valid_config.update({
            "num_min": max(1, int(config.get("num_min", 1))),
            "num_max": max(valid_config["num_min"] + 1, int(config.get("num_max", 5))),
            "arithmetic_type": config.get("arithmetic_type") if config.get("arithmetic_type") in ["add", "sub", "mix"] else "add",
            "option_count": 3 if config.get("option_count") not in [3, 4] else int(config.get("option_count")),
            "daily_question_count": max(1, int(config.get("daily_question_count", 5)))
        })
        print(f"✅ 加载配置成功: {valid_config}")
        return valid_config
    except Exception as e:
        print(f"❌ 加载配置失败（{str(e)}），使用默认配置")
        save_parent_config(default_config)
        return default_config

def save_parent_config(config):
    try:
        if not os.path.exists(current_program_dir):
            os.makedirs(current_program_dir, exist_ok=True)
        with open(CONFIG_FILE, 'w', encoding='utf-8') as f:
            json.dump(config, f, indent=2, ensure_ascii=False)
        print(f"✅ 配置保存成功: {CONFIG_FILE}")
    except Exception as e:
        print(f"❌ 保存配置失败: {str(e)}")

# --------------------------
# 答题记录相关函数（核心修复：保存+加载+状态同步）
# --------------------------
def init_daily_answer_record():
    """初始化当日答题记录"""
    config = load_parent_config()
    init_record = {
        "date": datetime.datetime.now().strftime('%Y-%m-%d'),
        "total_question_count": config["daily_question_count"],
        "answered_questions": [],
        "motor_triggered": False  # 电机是否已触发
    }
    print(f"ℹ️  初始化当日答题记录: {init_record}")
    return init_record

def load_daily_answer_record():
    """修复：增加结构校验+自动修复损坏文件"""
    if not os.path.exists(RECORD_FILE):
        print(f"⚠️  未找到答题记录文件，初始化新记录: {RECORD_FILE}")
        return init_daily_answer_record()
    
    try:
        # 读取记录并校验格式
        with open(RECORD_FILE, 'r', encoding='utf-8') as f:
            record = json.load(f)
        
        # 校验日期（非今日则重新初始化）
        today = datetime.datetime.now().strftime('%Y-%m-%d')
        if record.get("date") != today:
            print(f"⚠️  记录日期[{record.get('date')}]≠今日[{today}]，重新初始化")
            return init_daily_answer_record()
        
        # 校验必要字段（避免损坏文件）
        required_fields = ["date", "total_question_count", "answered_questions", "motor_triggered"]
        if not all(field in record for field in required_fields):
            print(f"⚠️  答题记录结构不完整，删除后重新初始化")
            os.remove(RECORD_FILE)
            return init_daily_answer_record()
        
        # 同步最新的每日答题总数
        config = load_parent_config()
        if record["total_question_count"] != config["daily_question_count"]:
            old_total = record["total_question_count"]
            record["total_question_count"] = config["daily_question_count"]
            print(f"ℹ️  同步配置：每日答题总数从{old_total}更新为{config['daily_question_count']}")
        
        # 打印加载结果（便于排查）
        answered_count = len(record["answered_questions"])
        total_count = record["total_question_count"]
        print(f"✅ 加载答题记录成功：已答{answered_count}/{total_count}题，电机触发状态：{record['motor_triggered']}")
        return record
    
    # 处理JSON格式错误
    except json.JSONDecodeError:
        print(f"❌ 答题记录文件损坏（JSON错误），删除后重新初始化")
        os.remove(RECORD_FILE)
        return init_daily_answer_record()
    
    # 其他异常处理
    except Exception as e:
        print(f"❌ 加载答题记录失败（{str(e)}），重新初始化")
        return init_daily_answer_record()

def save_daily_answer_record(record):
    """修复：强制目录创建+写入验证+权限处理"""
    try:
        # 确保目录存在（解决权限问题）
        if not os.path.exists(current_program_dir):
            os.makedirs(current_program_dir, exist_ok=True)
            print(f"✅ 已创建程序目录: {current_program_dir}")
        
        # 写入记录（增加错误忽略，避免特殊字符问题）
        with open(RECORD_FILE, 'w', encoding='utf-8', errors='ignore') as f:
            json.dump(record, f, indent=2, ensure_ascii=False)
        
        # 验证写入结果（确保数据真实保存）
        with open(RECORD_FILE, 'r', encoding='utf-8') as f:
            verify_record = json.load(f)
        
        # 对比关键数据（日期+已答题数）
        write_ok = (verify_record["date"] == record["date"]) and \
                   (len(verify_record["answered_questions"]) == len(record["answered_questions"]))
        
        if write_ok:
            answered_count = len(record["answered_questions"])
            print(f"✅ 答题记录保存成功（已验证）：已答{answered_count}题，电机触发：{record['motor_triggered']}")
            return True
        else:
            print(f"❌ 答题记录保存失败：写入内容不匹配")
            return False
    
    # 处理权限错误（Raspberry Pi常见问题）
    except PermissionError:
        print(f"❌ 保存失败：目录无写入权限！请执行命令：sudo chmod 775 {current_program_dir}")
        return False
    
    # 其他异常处理
    except Exception as e:
        print(f"❌ 保存答题记录失败（{str(e)}）")
        if os.path.exists(RECORD_FILE):
            os.remove(RECORD_FILE)
            print(f"✅ 已删除损坏的记录文件")
        return False

def calculate_accuracy(record):
    """计算正确率（保留2位小数）"""
    answered_count = len(record["answered_questions"])
    if answered_count == 0:
        return 0.00
    correct_count = sum(1 for q in record["answered_questions"] if q["is_correct"])
    accuracy = round(correct_count / answered_count, 2)
    print(f"ℹ️  计算正确率：正确{correct_count}题/总{answered_count}题 = {accuracy*100:.0f}%")
    return accuracy

def is_answer_complete(record):
    """判断是否完成当日答题（修复f-string语法）"""
    complete = len(record["answered_questions"]) >= record["total_question_count"]
    # 修复点：将"题"字作为独立文本，与变量用f-string正常拼接
    print(f"ℹ️  答题完成判断：已答{len(record['answered_questions'])}题/总{record['total_question_count']}题 → {'是' if complete else '否'}")
    return complete

# --------------------------
# 题目生成函数
# --------------------------
def generate_question():
    config = load_parent_config()
    num_min = config["num_min"]
    num_max = config["num_max"]
    arithmetic_type = config["arithmetic_type"]
    option_count = config["option_count"]
    
    a = random.randint(num_min, num_max)
    b = random.randint(num_min, num_max)
    
    # 生成运算类型（加法/减法/混合）
    if arithmetic_type == "add" or (arithmetic_type == "mix" and random.choice([True, False])):
        operator = "+"
        correct_answer = a + b
    else:
        operator = "-"
        if a < b:
            a, b = b, a  # 确保减法结果非负
        correct_answer = a - b
    
    # 生成不重复选项
    options = [correct_answer]
    while len(options) < option_count:
        range_offset = num_max - num_min
        candidate = random.randint(
            max(0, correct_answer - range_offset),
            correct_answer + range_offset
        )
        if candidate not in options:
            options.append(candidate)
    random.shuffle(options)
    
    question = {
        'id': int(time.time() % 100000),  # 唯一题目ID（时间戳取模）
        'question': f"{a} {operator} {b} = ?",
        'options': options,
        'correct_index': options.index(correct_answer)
    }
    print(f"✅ 生成新题目：{question['question']}，正确答案：{correct_answer}（选项索引{question['correct_index']}）")
    return question

# --------------------------
# 电机控制函数（核心修复：增加动作日志+异常处理）
# --------------------------
def control_motor_action(motor_id, direction, steps, step_delay=0.0001, microstep='fullstep'):
    """修复：详细日志+强制停止+结果返回"""
    if motor_id not in motors:
        print(f"❌ 电机控制失败：电机编号{motor_id}无效（仅支持1/2）")
        return False, "电机编号无效（仅支持1或2）"
    
    motor = motors[motor_id]
    try:
        print(f"ℹ️  启动电机{motor_id}：方向={direction}，步数={steps}，微步={microstep}，延迟={step_delay}")
        # 配置微步模式
        motor.SetMicroStep('software', microstep)
        # 执行转动
        motor.TurnStep(Dir=direction, steps=steps, stepdelay=step_delay)
        # 强制停止（避免空转）
        motor.Stop()
        print(f"✅ 电机{motor_id}动作完成：{direction} {steps}步，已停止")
        return True, f"电机{motor_id}执行成功（{direction} {steps}步，已停止）"
    except Exception as e:
        error_msg = str(e)
        print(f"❌ 电机{motor_id}异常：{error_msg}")
        # 异常时优先强制停止电机
        try:
            motor.Stop()
            print(f"✅ 电机{motor_id}异常后已强制停止")
            return False, f"电机{motor_id}执行失败（已强制停止）：{error_msg}"
        except:
            print(f"❌ 电机{motor_id}异常后强制停止失败")
            return False, f"电机{motor_id}执行失败（强制停止也失败）：{error_msg}"

# --------------------------
# Flask路由（核心修复：电机触发逻辑+前端状态同步）
# --------------------------
@app.route('/')
def index():
    """答题首页：加载最新进度和电机状态"""
    record = load_daily_answer_record()
    answered_count = len(record["answered_questions"])
    total_count = record["total_question_count"]
    accuracy = calculate_accuracy(record)
    formatted_accuracy = int(accuracy * 100)  # 后端格式化正确率（避免前端语法问题）
    
    print(f"ℹ️  首页加载完成：进度{answered_count}/{total_count}题，正确率{formatted_accuracy}%，电机触发：{record['motor_triggered']}")
    return render_template(
        'quiz.html',
        answered_count=answered_count,
        total_count=total_count,
        accuracy=formatted_accuracy,
        motor_triggered=record["motor_triggered"]
    )

# --------------------------
# 家长密码验证路由（不变）
# --------------------------
@app.route('/parent')
def parent_redirect():
    """家长入口重定向：已登录→配置页，未登录→登录页"""
    if session.get('is_parent_logged'):
        print(f"ℹ️  家长已登录，跳转配置页")
        return redirect(url_for('parent_config_page'))
    print(f"ℹ️  家长未登录，跳转登录页")
    return redirect(url_for('parent_login'))

@app.route('/parent_login', methods=['GET', 'POST'])
def parent_login():
    """家长登录页：验证密码"""
    if request.method == 'POST':
        input_password = request.form.get('password')
        if input_password == PARENT_PASSWORD:
            session['is_parent_logged'] = True
            print(f"✅ 家长登录成功")
            return redirect(url_for('parent_config_page'))
        else:
            print(f"❌ 家长密码错误：输入{input_password}，正确为{PARENT_PASSWORD}")
            return render_template('parent_login.html', error="密码错误，请重新输入！")
    print(f"ℹ️  显示家长登录页")
    return render_template('parent_login.html')

@app.route('/parent_logout')
def parent_logout():
    """家长退出登录：清除session"""
    session.pop('is_parent_logged', None)
    print(f"✅ 家长退出登录")
    return redirect(url_for('index'))

@app.route('/parent_config')
def parent_config_page():
    """家长配置页：需登录才能访问"""
    if not session.get('is_parent_logged'):
        print(f"❌ 未登录访问配置页，跳转登录页")
        return redirect(url_for('parent_login'))
    config = load_parent_config()
    print(f"ℹ️  显示家长配置页，当前配置：{config}")
    return render_template('parent_config.html', config=config)

@app.route('/save_config', methods=['POST'])
def save_config():
    """保存家长配置：同步更新答题记录总题数"""
    if not session.get('is_parent_logged'):
        print(f"❌ 未登录保存配置，返回错误")
        return jsonify({'status': 'error', 'message': '请先登录！'})
    
    # 接收前端提交的配置
    config = {
        "num_min": int(request.form.get('num_min', 1)),
        "num_max": int(request.form.get('num_max', 5)),
        "arithmetic_type": request.form.get('arithmetic_type', 'add'),
        "option_count": int(request.form.get('option_count', 3)),
        "daily_question_count": int(request.form.get('daily_question_count', 5))
    }
    # 保存配置
    save_parent_config(config)
    
    # 同步更新当日答题记录的总题数
    record = load_daily_answer_record()
    old_total = record["total_question_count"]
    record["total_question_count"] = config["daily_question_count"]
    save_daily_answer_record(record)
    print(f"ℹ️  配置保存后同步答题记录：总题数从{old_total}更新为{config['daily_question_count']}")
    
    return redirect(url_for('parent_config_page', success="配置已保存！"))

# --------------------------
# 答题相关路由（核心修复：电机触发+状态同步）
# --------------------------
@app.route('/get_question')
def get_question():
    """获取题目：答题完成返回complete状态，否则返回新题目"""
    record = load_daily_answer_record()
    if is_answer_complete(record):
        accuracy = calculate_accuracy(record)
        motor_triggered = record["motor_triggered"]
        # 答题完成：返回完整信息（含电机状态）
        response = {
            'status': 'complete',
            'message': f'今日挑战完成！正确率{accuracy*100:.0f}%',
            'accuracy': accuracy,
            'motor_triggered': motor_triggered
        }
        print(f"ℹ️  答题已完成，返回complete状态：{response}")
        return jsonify(response)
    
    # 未完成：返回新题目
    question = generate_question()
    return jsonify({'status': 'success', 'question': question})

@app.route('/submit_answer', methods=['POST'])
def submit_answer():
    """提交答案：核心修复电机触发逻辑，确保答题完成后必检查"""
    data = request.json
    selected_index = data.get('selected_index')
    question = data.get('question')
    
    # 1. 校验题目合法性
    if not question or 'correct_index' not in question or 'id' not in question:
        print(f"❌ 提交答案失败：题目信息无效（缺失id或correct_index）")
        return jsonify({'status': 'error', 'message': '题目信息无效，请刷新页面重试'})
    
    # 2. 加载当前答题记录
    record = load_daily_answer_record()
    existing_ids = [q["question_id"] for q in record["answered_questions"]]
    
    # 3. 避免重复提交同一题目
    if question["id"] in existing_ids:
        print(f"❌ 提交答案失败：题目ID{question['id']}已提交过")
        return jsonify({'status': 'error', 'message': '该题目已提交过，请勿重复提交！'})
    
    # 4. 判断答案是否正确，添加答题记录
    is_correct = selected_index == question["correct_index"]
    current_time = datetime.datetime.now().strftime('%H:%M:%S')
    record["answered_questions"].append({
        "question_id": question["id"],
        "question_content": question["question"],
        "selected_index": selected_index,
        "correct_index": question["correct_index"],
        "is_correct": is_correct,
        "submit_time": current_time
    })
    
    # 5. 保存记录（确保成功后再继续）
    save_success = save_daily_answer_record(record)
    if not save_success:
        return jsonify({'status': 'error', 'message': '答题记录保存失败，请重试！'})
    
    # 6. 重新加载最新记录（确保数据同步）
    record = load_daily_answer_record()
    answered_count = len(record["answered_questions"])
    total_count = record["total_question_count"]
    current_accuracy = calculate_accuracy(record)
    motor_triggered = record["motor_triggered"]
    
    # 7. 核心修复：答题完成后强制检查电机触发（无论之前状态）
    motor_msg = ""
    if is_answer_complete(record):
        if current_accuracy >= 0.8:
            if not motor_triggered:
                # 正确率≥80%且未触发：启动电机2（backward 6000步）
                if '2' in motors:
                    motor_result, motor_detail = control_motor_action(
                        motor_id='2', 
                        direction='backward', 
                        steps=6000, 
                        step_delay=0.0001, 
                        microstep='fullstep'
                    )
                    if motor_result:
                        motor_msg = f'🎉 正确率{current_accuracy*100:.0f}%≥80%，电机已启动！'
                        # 立即更新电机触发状态并保存
                        record["motor_triggered"] = True
                        save_daily_answer_record(record)
                        motor_triggered = True  # 同步状态
                    else:
                        motor_msg = f'🎉 正确率达标（{current_accuracy*100:.0f}%），但电机执行失败：{motor_detail}'
                else:
                    motor_msg = f'🎉 正确率{current_accuracy*100:.0f}%≥80%，但电机未初始化，无法启动'
            else:
                # 已触发过：仅提示
                motor_msg = f'🎉 正确率{current_accuracy*100:.0f}%≥80%，电机已启动过～'
        else:
            # 正确率<80%：不启动电机
            motor_msg = f'正确率{current_accuracy*100:.0f}%<80%，电机未启动'
    
    # 8. 构建返回消息（简化语言，适配宝宝）
    if is_correct:
        result_msg = "✅ 答对啦！真厉害～"
    else:
        correct_answer = question["options"][question["correct_index"]]
        result_msg = f"❌ 再想想，正确答案是{correct_answer}"
    normal_msg = f"{result_msg} 当前进度：{answered_count}/{total_count}题"
    
    # 9. 返回结果（含最新电机状态）
    response = {
        'status': 'success',
        'is_correct': is_correct,
        'current_accuracy': current_accuracy,
        'answered_count': answered_count,
        'total_count': total_count,
        'motor_triggered': motor_triggered,
        'message': motor_msg if motor_msg else normal_msg
    }
    print(f"ℹ️  提交答案完成，返回响应：{response}")
    return jsonify(response)

@app.route('/control', methods=['POST'])
def control_motor():
    """手动控制电机：仅家长登录且答题完成后可使用"""
    if not session.get('is_parent_logged'):
        return jsonify({"status": "error", "message": "请先登录！"})
    
    record = load_daily_answer_record()
    if not is_answer_complete(record):
        return jsonify({"status": "error", "message": "请先完成今日所有答题"})
    
    if not motors:
        return jsonify({"status": "error", "message": "电机未初始化，无法控制"})
    
    data = request.json
    motor_id = data.get('motor_id')
    action = data.get('action')
    steps = int(data.get('steps', 200))
    step_delay = float(data.get('step_delay', 0.005))
    microstep = data.get('microstep', 'fullstep')
    
    # 处理停止动作
    if action == 'stop':
        if motor_id in motors:
            motors[motor_id].Stop()
            return jsonify({"status": "success", "message": f"电机{motor_id}已停止"})
        return jsonify({"status": "error", "message": "无效的电机编号（仅支持1或2）"})
    
    # 处理正转/反转动作
    if action in ['forward', 'backward']:
        success, msg = control_motor_action(motor_id, action, steps, step_delay, microstep)
        return jsonify({"status": "success" if success else "error", "message": msg})
    
    return jsonify({"status": "error", "message": "无效的动作指令（仅支持forward/backward/stop）"})

# --------------------------
# 程序入口（自动获取Raspberry Pi IP，退出时停止电机）
# --------------------------
if __name__ == '__main__':
    try:
        # 自动获取本地IP（避免硬编码错误）
        import socket
        s = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
        s.connect(("8.8.8.8", 80))
        local_ip = s.getsockname()[0]
        s.close()
        
        print(f"\n" + "="*50)
        print(f"✅ Flask服务启动成功！")
        print(f"📌 答题页面：http://{local_ip}:5000")
        print(f"📌 家长入口：http://{local_ip}:5000/parent")
        print(f"🔑 家长密码：{PARENT_PASSWORD}")
        print(f"="*50 + "\n")
        
        app.run(host='0.0.0.0', port=5000, debug=True)
    finally:
        # 程序退出时强制停止所有电机（保障硬件安全）
        print(f"\n⚠️  程序开始退出，强制停止所有电机...")
        for motor_id, motor in motors.items():
            try:
                motor.Stop()
                print(f"✅ 程序退出：电机{motor_id}已停止")
            except Exception as e:
                print(f"❌ 程序退出：电机{motor_id}停止失败：{str(e)}")
        print(f"✅ 所有电机已处理完毕，程序退出")
