from flask import Flask, render_template, request, jsonify, redirect, url_for, session  # 新增session
import time
import datetime
import os
import random
import json
from DRV8825 import DRV8825

app = Flask(__name__)
app.secret_key = 'parent_config_secret'  # 用于session加密
PARENT_PASSWORD = 'rendong'  # 家长密码（固定为rendong）

# --------------------------
# 路径配置（不变）
# --------------------------
current_program_dir = os.path.dirname(os.path.abspath(__file__))
RECORD_FILE = os.path.join(current_program_dir, "daily_answers.json")
CONFIG_FILE = os.path.join(current_program_dir, "parent_config.json")
print(f"程序运行目录: {current_program_dir}")
print(f"配置文件路径: {CONFIG_FILE}")
print(f"答题记录路径: {RECORD_FILE}")

# --------------------------
# 电机初始化（不变）
# --------------------------
motors = {}
try:
    Motor1 = DRV8825(dir_pin=13, step_pin=19, enable_pin=12, mode_pins=(16, 17, 20))
    Motor2 = DRV8825(dir_pin=24, step_pin=18, enable_pin=4, mode_pins=(21, 22, 27))
    motors = {"1": Motor1, "2": Motor2}
    print("电机初始化成功")
except Exception as e:
    print(f"电机初始化失败: {e}")
    motors = {}

# --------------------------
# 家长配置/答题记录函数（不变）
# --------------------------
def load_parent_config():
    default_config = {
        "num_min": 1,
        "num_max": 5,
        "arithmetic_type": "add",
        "option_count": 3,
        "daily_question_count": 5
    }
    if not os.path.exists(CONFIG_FILE):
        print(f"配置文件不存在，使用默认配置: {CONFIG_FILE}")
        save_parent_config(default_config)
        return default_config
    try:
        with open(CONFIG_FILE, 'r', encoding='utf-8') as f:
            config = json.load(f)
        valid_config = default_config.copy()
        valid_config.update({
            "num_min": max(1, int(config.get("num_min", 1))),
            "num_max": max(valid_config["num_min"] + 1, int(config.get("num_max", 5))),
            "arithmetic_type": config.get("arithmetic_type") if config.get("arithmetic_type") in ["add", "sub", "mix"] else "add",
            "option_count": 3 if config.get("option_count") not in [3, 4] else int(config.get("option_count")),
            "daily_question_count": max(1, int(config.get("daily_question_count", 5)))
        })
        print(f"加载配置成功: {valid_config}")
        return valid_config
    except Exception as e:
        print(f"加载配置失败（{str(e)}），使用默认配置")
        save_parent_config(default_config)
        return default_config

def save_parent_config(config):
    try:
        if not os.path.exists(current_program_dir):
            os.makedirs(current_program_dir, exist_ok=True)
        with open(CONFIG_FILE, 'w', encoding='utf-8') as f:
            json.dump(config, f, indent=2)
        print(f"配置保存成功: {CONFIG_FILE}")
    except Exception as e:
        print(f"保存配置失败: {str(e)}")

def init_daily_answer_record():
    config = load_parent_config()
    return {
        "date": datetime.datetime.now().strftime('%Y-%m-%d'),
        "total_question_count": config["daily_question_count"],
        "answered_questions": [],
        "motor_triggered": False
    }

def load_daily_answer_record():
    if not os.path.exists(RECORD_FILE):
        return init_daily_answer_record()
    try:
        with open(RECORD_FILE, 'r', encoding='utf-8') as f:
            record = json.load(f)
        today = datetime.datetime.now().strftime('%Y-%m-%d')
        if record.get("date") != today:
            return init_daily_answer_record()
        config = load_parent_config()
        if record["total_question_count"] != config["daily_question_count"]:
            record["total_question_count"] = config["daily_question_count"]
        return record
    except Exception as e:
        print(f"加载答题记录失败: {str(e)}")
        return init_daily_answer_record()

def save_daily_answer_record(record):
    try:
        if not os.path.exists(current_program_dir):
            os.makedirs(current_program_dir, exist_ok=True)
        with open(RECORD_FILE, 'w', encoding='utf-8') as f:
            json.dump(record, f, indent=2, ensure_ascii=False)
        print(f"答题记录保存成功: {RECORD_FILE}")
    except Exception as e:
        print(f"保存答题记录失败: {str(e)}")

def calculate_accuracy(record):
    answered_count = len(record["answered_questions"])
    if answered_count == 0:
        return 0.00
    correct_count = sum(1 for q in record["answered_questions"] if q["is_correct"])
    return round(correct_count / answered_count, 2)

def is_answer_complete(record):
    return len(record["answered_questions"]) >= record["total_question_count"]

def generate_question():
    config = load_parent_config()
    num_min = config["num_min"]
    num_max = config["num_max"]
    arithmetic_type = config["arithmetic_type"]
    option_count = config["option_count"]
    a = random.randint(num_min, num_max)
    b = random.randint(num_min, num_max)
    if arithmetic_type == "add" or (arithmetic_type == "mix" and random.choice([True, False])):
        operator = "+"
        correct_answer = a + b
    else:
        operator = "-"
        if a < b:
            a, b = b, a
        correct_answer = a - b
    options = [correct_answer]
    while len(options) < option_count:
        range_offset = num_max - num_min
        candidate = random.randint(
            max(0, correct_answer - range_offset),
            correct_answer + range_offset
        )
        if candidate not in options:
            options.append(candidate)
    random.shuffle(options)
    return {
        'id': int(time.time() % 100000),
        'question': f"{a} {operator} {b} = ?",
        'options': options,
        'correct_index': options.index(correct_answer)
    }

def control_motor_action(motor_id, direction, steps, step_delay=0.0001, microstep='fullstep'):
    if motor_id not in motors:
        return False, "电机编号无效"
    motor = motors[motor_id]
    try:
        motor.SetMicroStep('software', microstep)
        motor.TurnStep(Dir=direction, steps=steps, stepdelay=step_delay)
        motor.Stop()
        print(f"电机{motor_id}转动{steps}步后已停止")
        return True, "执行成功（已停止）"
    except Exception as e:
        try:
            motor.Stop()
            print(f"电机{motor_id}异常后已停止")
        except:
            pass
        return False, f"失败（已尝试停止）: {str(e)}"

# --------------------------
# 新增：家长密码验证路由
# --------------------------
@app.route('/parent')
def parent_redirect():
    # 若已登录，直接进入配置页面；否则跳转登录页
    if session.get('is_parent_logged'):
        return redirect(url_for('parent_config_page'))
    return redirect(url_for('parent_login'))

@app.route('/parent_login', methods=['GET', 'POST'])
def parent_login():
    if request.method == 'POST':
        # 接收密码并验证
        input_password = request.form.get('password')
        if input_password == PARENT_PASSWORD:
            session['is_parent_logged'] = True  # 标记登录状态
            return redirect(url_for('parent_config_page'))
        else:
            # 密码错误，返回登录页并提示
            return render_template('parent_login.html', error="密码错误，请重新输入！")
    # GET请求：显示登录页
    return render_template('parent_login.html')

@app.route('/parent_logout')
def parent_logout():
    # 退出登录：清除session
    session.pop('is_parent_logged', None)
    return redirect(url_for('index'))

# --------------------------
# 原有路由（仅修改index路由的正确率传递）
# --------------------------
@app.route('/')
def index():
    record = load_daily_answer_record()
    accuracy = calculate_accuracy(record)
    answered_count = len(record["answered_questions"])
    total_count = record["total_question_count"]
    # 后端格式化正确率（避免前端语法问题）
    formatted_accuracy = int(accuracy * 100)
    return render_template(
        'quiz.html',
        answered_count=answered_count,
        total_count=total_count,
        accuracy=formatted_accuracy,
        motor_triggered=record["motor_triggered"]
    )

@app.route('/parent_config')
def parent_config_page():
    # 验证登录状态：未登录则跳转登录页
    if not session.get('is_parent_logged'):
        return redirect(url_for('parent_login'))
    config = load_parent_config()
    return render_template('parent_config.html', config=config)

@app.route('/save_config', methods=['POST'])
def save_config():
    if not session.get('is_parent_logged'):
        return jsonify({'status': 'error', 'message': '请先登录！'})
    config = {
        "num_min": int(request.form.get('num_min', 1)),
        "num_max": int(request.form.get('num_max', 5)),
        "arithmetic_type": request.form.get('arithmetic_type', 'add'),
        "option_count": int(request.form.get('option_count', 3)),
        "daily_question_count": int(request.form.get('daily_question_count', 5))
    }
    save_parent_config(config)
    record = load_daily_answer_record()
    record["total_question_count"] = config["daily_question_count"]
    save_daily_answer_record(record)
    return redirect(url_for('parent_config_page', success="配置已保存！"))

@app.route('/get_question')
def get_question():
    record = load_daily_answer_record()
    if is_answer_complete(record):
        accuracy = calculate_accuracy(record)
        return jsonify({
            'status': 'complete',
            'message': f'今日答题已完成！正确率：{accuracy*100:.0f}%',
            'accuracy': accuracy,
            'motor_triggered': record["motor_triggered"]
        })
    question = generate_question()
    return jsonify({'status': 'success', 'question': question})

@app.route('/submit_answer', methods=['POST'])
def submit_answer():
    data = request.json
    selected_index = data.get('selected_index')
    question = data.get('question')
    if not question or 'correct_index' not in question or 'id' not in question:
        return jsonify({'status': 'error', 'message': '题目信息无效，请刷新页面重试'})
    record = load_daily_answer_record()
    existing_question_ids = [q["question_id"] for q in record["answered_questions"]]
    if question["id"] in existing_question_ids:
        return jsonify({'status': 'error', 'message': '该题目已提交过，请勿重复提交！'})
    is_correct = selected_index == question["correct_index"]
    current_time = datetime.datetime.now().strftime('%H:%M:%S')
    record["answered_questions"].append({
        "question_id": question["id"],
        "question_content": question["question"],
        "selected_index": selected_index,
        "correct_index": question["correct_index"],
        "is_correct": is_correct,
        "submit_time": current_time
    })
    motor_msg = ""
    if is_answer_complete(record) and not record["motor_triggered"]:
        accuracy = calculate_accuracy(record)
        if accuracy >= 0.8:
            if '2' in motors:
                motor_result, motor_detail = control_motor_action('2', 'backward', 6000, 0.0001)
                if motor_result:
                    motor_msg = f'正确率{accuracy*100:.0f}%≥80%，电机已启动并停止！'
                else:
                    motor_msg = f'正确率达标（{accuracy*100:.0f}%），但电机执行失败：{motor_detail}'
            else:
                motor_msg = f'正确率{accuracy*100:.0f}%≥80%，但电机未初始化，无法启动'
            record["motor_triggered"] = True
        else:
            motor_msg = f'正确率{accuracy*100:.0f}%<80%，电机未启动'
    save_daily_answer_record(record)
    current_accuracy = calculate_accuracy(record)
    answered_count = len(record["answered_questions"])
    total_count = record["total_question_count"]
    if is_correct:
        result_msg = "回答正确！"
    else:
        correct_answer = question["options"][question["correct_index"]]
        result_msg = f"回答错误，正确答案是{correct_answer}"
    normal_msg = f"{result_msg} 当前进度：{answered_count}/{total_count}题，正确率：{current_accuracy*100:.0f}%"
    return jsonify({
        'status': 'success',
        'is_correct': is_correct,
        'current_accuracy': current_accuracy,
        'answered_count': answered_count,
        'total_count': total_count,
        'message': motor_msg if motor_msg else normal_msg
    })

@app.route('/control', methods=['POST'])
def control_motor():
    if not session.get('is_parent_logged'):
        return jsonify({"status": "error", "message": "请先登录！"})
    record = load_daily_answer_record()
    if not is_answer_complete(record):
        return jsonify({"status": "error", "message": "请先完成今日所有答题，才能手动控制电机"})
    if not motors:
        return jsonify({"status": "error", "message": "电机未初始化，无法执行控制"})
    data = request.json
    motor_id = data.get('motor_id')
    action = data.get('action')
    steps = int(data.get('steps', 200))
    step_delay = float(data.get('step_delay', 0.005))
    microstep = data.get('microstep', 'fullstep')
    if action == 'stop':
        if motor_id in motors:
            motors[motor_id].Stop()
            return jsonify({"status": "success", "message": f"电机{motor_id}已停止"})
        return jsonify({"status": "error", "message": "无效的电机编号（仅支持1或2）"})
    if action in ['forward', 'backward']:
        success, msg = control_motor_action(motor_id, action, steps, step_delay, microstep)
        return jsonify({"status": "success" if success else "error", "message": msg})
    return jsonify({"status": "error", "message": "无效的动作指令（仅支持forward/backward/stop）"})

# --------------------------
# 程序入口（不变）
# --------------------------
if __name__ == '__main__':
    try:
        print("Flask服务启动，访问地址: http://192.168.10.28:5000")  # 以终端实际IP为准
        app.run(host='0.0.0.0', port=5000, debug=True)
    finally:
        for motor in motors.values():
            try:
                motor.Stop()
                print(f"程序退出，电机已停止")
            except Exception as e:
                print(f"电机停止失败: {str(e)}")
        print("所有电机已停止，程序退出")
