# backend/app.py (v1.7 - 多租户时限管理)
import sqlite3
from datetime import datetime, timedelta, date
from flask import Flask, render_template, request, jsonify, g, redirect, url_for, make_response
from flask_login import LoginManager, UserMixin, login_user, logout_user, login_required, current_user
import os, json, uuid
from flask_cors import CORS
from functools import wraps
import sys 
from collections import defaultdict

# --- Nuitka 路径逻辑 (保持不变) ---
is_frozen = getattr(sys, 'frozen', False)
is_in_onefile_temp = 'onefile_' in os.path.abspath(__file__) 

if is_frozen or is_in_onefile_temp:
    temp_base_dir = os.path.dirname(os.path.abspath(sys.executable))
    template_dir = os.path.join(temp_base_dir, 'templates')
    static_dir = os.path.join(temp_base_dir, 'static')
    exe_path = os.path.realpath(sys.argv[0])
    exe_dir = os.path.dirname(exe_path)
    DATABASE = os.path.join(exe_dir, 'fitness_manager.db')
    print("--- Detected Nuitka Onefile Path ---")
    print(f"--- **DATABASE is PERMANENT at {DATABASE}** ---")
else:
    app_file_path = os.path.abspath(__file__)
    backend_dir = os.path.dirname(app_file_path)
    project_root_dir = os.path.dirname(backend_dir)
    template_dir = os.path.join(project_root_dir, 'frontend', 'templates')
    static_dir = os.path.join(project_root_dir, 'frontend', 'static')
    DATABASE = os.path.join(backend_dir, 'fitness_manager.db')
    print("--- Detected Development Path ---")
# --- END REVISED PATH LOGIC ---


print("Template folder:", template_dir)
print("Static folder:", static_dir)
print("DB path:", DATABASE)

app = Flask(__name__, template_folder=template_dir, static_folder=static_dir)
CORS(app)
app.secret_key = str(uuid.uuid4())

# --- Flask-Login 配置 ---
login_manager = LoginManager()
login_manager.init_app(app)
login_manager.login_view = 'index' 
login_manager.login_message = "请先登录以访问此页面。"

# ★★★ 核心修改 1: User 类增加 manager_id 和 expires_at
class User(UserMixin):
    def __init__(self, id, username, is_admin, manager_id, expires_at):
        self.id = id
        self.username = username
        self.is_admin = is_admin
        self.manager_id = manager_id # 记录创建此用户的管理员ID
        self.expires_at = expires_at # 账户过期日期 (YYYY-MM-DD)
    
    # 新增：判断是否为超级管理员
    @property
    def is_super_admin(self):
        # 超级管理员的 manager_id 为 None
        return self.is_admin and (self.manager_id is None)

@login_manager.user_loader
def load_user(user_id):
    db = get_db()
    # ★★★ 核心修改 2: 加载 manager_id 和 expires_at
    user_row = db.execute("SELECT id, username, is_admin, manager_id, expires_at FROM users WHERE id = ?", (user_id,)).fetchone()
    if user_row:
        return User(
            id=user_row['id'], 
            username=user_row['username'], 
            is_admin=user_row['is_admin'], 
            manager_id=user_row['manager_id'],
            expires_at=user_row['expires_at']
        )
    return None

# -------------------- DB (保持不变) --------------------
def get_db():
    db = getattr(g, '_database', None)
    if db is None:
        db = g._database = sqlite3.connect(DATABASE, check_same_thread=False)
        db.row_factory = sqlite3.Row
    return db
def get_setting(key, default=None):
    db = get_db()
    row = db.execute("SELECT value FROM settings WHERE key = ?", (key,)).fetchone()
    if row is None:
        return default
    return row["value"]

def set_setting(key, value):
    db = get_db()
    cur = db.cursor()
    cur.execute(
        "INSERT INTO settings (key, value) VALUES (?, ?) "
        "ON CONFLICT(key) DO UPDATE SET value = excluded.value",
        (key, str(value)),
    )
    db.commit()

@app.teardown_appcontext
def close_connection(exception):
    db = getattr(g, '_database', None)
    if db is not None:
        db.close()
def init_db():
    db = get_db()
    cur = db.cursor()
    cur.execute("""
        CREATE TABLE IF NOT EXISTS todos (
            id INTEGER PRIMARY KEY AUTOINCREMENT, 
            date TEXT NOT NULL, 
            content TEXT NOT NULL,
            is_completed INTEGER DEFAULT 0, 
            user_id INTEGER, 
            recurring_todo_id INTEGER
        )
    """)
    cur.execute("""
        CREATE TABLE IF NOT EXISTS fitness_plans (
            id INTEGER PRIMARY KEY AUTOINCREMENT, 
            date TEXT NOT NULL, 
            exercise_name TEXT NOT NULL,
            sets_reps_duration TEXT, 
            is_completed INTEGER DEFAULT 0, 
            user_id INTEGER, 
            recurring_plan_id INTEGER
        )
    """)
    cur.execute("""
        CREATE TABLE IF NOT EXISTS recurring_plans (
            id INTEGER PRIMARY KEY AUTOINCREMENT, 
            user_id INTEGER, 
            plan_name TEXT NOT NULL,
            exercises_json TEXT NOT NULL, 
            start_date TEXT NOT NULL, 
            end_date TEXT,
            repeat_interval TEXT NOT NULL, 
            repeat_day INTEGER
        )
    """)
    cur.execute("""
        CREATE TABLE IF NOT EXISTS recurring_todos (
            id INTEGER PRIMARY KEY AUTOINCREMENT, 
            user_id INTEGER, 
            content TEXT NOT NULL,
            start_date TEXT NOT NULL, 
            end_date TEXT, 
            repeat_interval TEXT NOT NULL, 
            repeat_day INTEGER
        )
    """)
    cur.execute("""
        CREATE TABLE IF NOT EXISTS exercise_library (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            user_id INTEGER,
            name TEXT NOT NULL,
            details TEXT
        )
    """)
    cur.execute("""
        CREATE TABLE IF NOT EXISTS settings (
            key   TEXT PRIMARY KEY,
            value TEXT
        )
    """)
    cur.execute("""
        CREATE TABLE IF NOT EXISTS todo_library (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            user_id INTEGER,
            content TEXT NOT NULL
        )
    """)


# ★★★ 核心修改 3: 更新 users 表结构
def ensure_user_schema():
    db = get_db()
    cur = db.cursor()
    cur.execute("""
        CREATE TABLE IF NOT EXISTS users (
            id INTEGER PRIMARY KEY AUTOINCREMENT, 
            username TEXT UNIQUE NOT NULL,
            password TEXT NOT NULL, 
            is_admin INTEGER DEFAULT 0,
            manager_id INTEGER REFERENCES users(id),
            expires_at TEXT 
        )
    """)
    
    # 迁移旧表
    try: cur.execute("SELECT is_admin FROM users LIMIT 1")
    except sqlite3.OperationalError:
        print("Migrating users table: Adding is_admin column.")
        cur.execute("ALTER TABLE users ADD COLUMN is_admin INTEGER DEFAULT 0")
    
    try: cur.execute("SELECT manager_id FROM users LIMIT 1")
    except sqlite3.OperationalError:
        print("Migrating users table: Adding manager_id column.")
        cur.execute("ALTER TABLE users ADD COLUMN manager_id INTEGER REFERENCES users(id)")

    try: cur.execute("SELECT expires_at FROM users LIMIT 1")
    except sqlite3.OperationalError:
        print("Migrating users table: Adding expires_at column.")
        cur.execute("ALTER TABLE users ADD COLUMN expires_at TEXT")
    
    # 创建默认超级管理员
    cur.execute("SELECT COUNT(*) FROM users WHERE username = 'superuser'")
    if cur.fetchone()[0] == 0:
        print("Inserting default superuser: superuser/adminpassword (Admin, Never Expires)")
        cur.execute("INSERT INTO users (username, password, is_admin, manager_id, expires_at) VALUES (?, ?, ?, ?, ?)", 
                    ('superuser', 'adminpassword', 1, None, None))
        
    # 获取 superuser 的 ID
    su_row = cur.execute("SELECT id FROM users WHERE username = 'superuser'").fetchone()
    su_id = su_row['id'] if su_row else 1 # 假设 superuser ID 为 1
    
    # 创建默认普通用户
    cur.execute("SELECT COUNT(*) FROM users WHERE username = 'user1'")
    if cur.fetchone()[0] == 0:
        print("Inserting default regular user: user1/password (Regular, managed by superuser, expires in 4 weeks)")
        # 默认用户由 superuser 管理，4周后过期
        default_expiry = (datetime.now() + timedelta(weeks=4)).strftime('%Y-%m-%d')
        cur.execute("INSERT INTO users (username, password, is_admin, manager_id, expires_at) VALUES (?, ?, ?, ?, ?)", 
                    ('user1', 'password', 0, su_id, default_expiry))

    # (其他表迁移保持不变)
    try: cur.execute("SELECT repeat_interval FROM recurring_plans LIMIT 1")
    except sqlite3.OperationalError:
        print("Migrating recurring_plans table: Adding repeat_interval and repeat_day columns.")
        cur.execute("ALTER TABLE recurring_plans ADD COLUMN repeat_interval TEXT DEFAULT 'daily'")
        cur.execute("ALTER TABLE recurring_plans ADD COLUMN repeat_day INTEGER")
        cur.execute("UPDATE recurring_plans SET repeat_interval = 'daily' WHERE repeat_interval IS NULL")

    try: cur.execute("SELECT recurring_plan_id FROM fitness_plans LIMIT 1")
    except sqlite3.OperationalError:
        print("Migrating fitness_plans table: Adding recurring_plan_id column.")
        cur.execute("ALTER TABLE fitness_plans ADD COLUMN recurring_plan_id INTEGER")

    try: cur.execute("SELECT recurring_todo_id FROM todos LIMIT 1")
    except sqlite3.OperationalError:
        print("Migrating todos table: Adding recurring_todo_id column.")
        cur.execute("ALTER TABLE todos ADD COLUMN recurring_todo_id INTEGER")

    default_user_id_row = db.execute("SELECT id FROM users WHERE username = 'user1'").fetchone()
    if default_user_id_row:
        default_user_id = default_user_id_row['id']
        for table in ['fitness_plans', 'todos', 'recurring_plans', 'recurring_todos', 'exercise_library', 'todo_library']:
            if table == 'users': continue
            try: cur.execute(f"SELECT user_id FROM {table} LIMIT 1") 
            except sqlite3.OperationalError:
                try: cur.execute(f"ALTER TABLE {table} ADD COLUMN user_id INTEGER")
                except sqlite3.OperationalError: pass 
            cur.execute(f"UPDATE {table} SET user_id = ? WHERE user_id IS NULL", (default_user_id,))
    db.commit()

def ensure_todos_is_completed_column():
    db = get_db()
    cur = db.cursor()
    try: cur.execute("SELECT is_completed FROM todos LIMIT 1")
    except sqlite3.OperationalError:
        cur.execute("ALTER TABLE todos ADD COLUMN is_completed INTEGER DEFAULT 0")
        db.commit()

def ensure_fitness_plans_is_completed_column():
    db = get_db()
    cur = db.cursor()
    try: cur.execute("SELECT is_completed FROM fitness_plans LIMIT 1")
    except sqlite3.OperationalError:
        cur.execute("ALTER TABLE fitness_plans ADD COLUMN is_completed INTEGER DEFAULT 0")
        db.commit()

with app.app_context():
    init_db() 
    ensure_todos_is_completed_column()
    ensure_fitness_plans_is_completed_column()
    ensure_user_schema() 


# -------------------- Auth Routes --------------------

# ★★★ 核心修改 4: 登录时检查时限
@app.route('/login', methods=['POST'])
def login():
    username = request.form.get('username') or request.json.get('username')
    password = request.form.get('password') or request.json.get('password')
    db = get_db()
    
    user_row = db.execute("SELECT * FROM users WHERE username = ?", (username,)).fetchone()
    
    if not user_row:
        return jsonify({'status': 'error', 'message': '用户名或密码无效'}), 401
        
    if user_row['password'] != password:
        return jsonify({'status': 'error', 'message': '用户名或密码无效'}), 401
    
    # 检查过期
    today_date = date.today()
    
    # 1. 检查自己的过期时间 (超级管理员 expires_at 为 NULL，跳过此检查)
    if user_row['expires_at']:
        try:
            expires_date = datetime.strptime(user_row['expires_at'], '%Y-%m-%d').date()
            if today_date > expires_date:
                return jsonify({'status': 'expired', 'message': '您的账户已过期。'}), 401
        except ValueError:
            pass # 日期格式错误，暂时忽略

    # 2. 检查管理员的过期时间 (如果是超级管理员，manager_id 为 NULL，跳过此检查)
    if user_row['manager_id']:
        manager = db.execute("SELECT expires_at FROM users WHERE id = ?", (user_row['manager_id'],)).fetchone()
        if manager and manager['expires_at']:
            try:
                manager_expires_date = datetime.strptime(manager['expires_at'], '%Y-%m-%d').date()
                if today_date > manager_expires_date:
                    return jsonify({'status': 'expired', 'message': '您的管理员账户已过期。'}), 401
            except ValueError:
                pass # 日期格式错误

    # 登录成功
    user_obj = User(
        id=user_row['id'], 
        username=user_row['username'], 
        is_admin=user_row['is_admin'],
        manager_id=user_row['manager_id'],
        expires_at=user_row['expires_at']
    )
    login_user(user_obj, remember=True) 
    
    return jsonify({
        'status': 'success', 
        'username': user_obj.username, 
        'is_admin': user_obj.is_admin,
        'manager_id': user_obj.manager_id
    })

@app.route('/logout', methods=['POST'])
@login_required
def logout():
    logout_user() 
    return jsonify({'status': 'success'})

@app.route('/get_current_user', methods=['GET'])
def get_current_user():
    if current_user.is_authenticated:
        return jsonify({
            'status': 'logged_in', 
            'user_id': current_user.id, 
            'username': current_user.username, 
            'is_admin': current_user.is_admin,
            'manager_id': current_user.manager_id,
            'expires_at': current_user.expires_at # 传递过期日期
        })
    return jsonify({'status': 'logged_out'}), 401

# -------------------- UI Routes --------------------

# ★★★ 核心修改 5: 新增 /expired 页面路由
@app.route('/expired')
def expired_page():
    return render_template('expired.html')

# ★★★ 核心修改 6: index 路由计算剩余天数
@app.route('/')
def index():
    if not current_user.is_authenticated:
        return render_template('login.html') 
    
    view_mode = request.args.get('view')
    if view_mode == 'display':
        pass # 强制大屏模式
    elif 'Mobile' in request.headers.get('User-Agent', '') or 'Android' in request.headers.get('User-Agent', ''):
        return redirect(url_for('mobile_config'))

    today_str = datetime.now().strftime('%Y-%m-%d')
    _generate_daily_fitness_plan_if_not_exists(current_user.id, today_str)
    _generate_daily_todos_if_not_exists(current_user.id, today_str)

    # --- 计算剩余天数 ---
    remaining_days = None
    warning_target_date = current_user.expires_at
    
    # 如果用户自己没有过期日期 (例如他是普通用户)，则检查他管理员的
    if not warning_target_date and current_user.manager_id:
        db = get_db()
        manager = db.execute("SELECT expires_at FROM users WHERE id = ?", (current_user.manager_id,)).fetchone()
        if manager:
            warning_target_date = manager['expires_at']
            
    if warning_target_date:
        try:
            expires_date = datetime.strptime(warning_target_date, '%Y-%m-%d').date()
            delta = expires_date - date.today()
            remaining_days = delta.days
        except (ValueError, TypeError):
            remaining_days = None
    # --- 结束计算 ---

    return render_template(
        'display.html', 
        username=current_user.username, 
        is_admin=current_user.is_admin,
        remaining_days=remaining_days # 传递剩余天数
    ) 

@app.route('/mobile_config')
@login_required
def mobile_config():
    return render_template('mobile_config.html', username=current_user.username)

@app.route('/api/settings/scroll_speed', methods=['GET', 'POST'])
def scroll_speed_setting():
    if request.method == 'GET':
        value = get_setting('scroll_speed', '25')
        try: speed = int(value)
        except (TypeError, ValueError): speed = 25
        return jsonify({"scroll_speed": speed})

    # POST
    data = request.get_json() or {}
    try: speed = int(data.get('scroll_speed', 25))
    except (TypeError, ValueError):
        return jsonify({"status": "error", "message": "滚动速度必须是整数"}), 400
    if speed < 5: speed = 5
    if speed > 120: speed = 120
    set_setting('scroll_speed', speed)
    return jsonify({"status": "success", "scroll_speed": speed})


# ★★★ 核心修改 7: 传递 manager_id 和 is_super_admin 到 manage.html 模板
@app.route('/manage')
@login_required
def manage():
    return render_template(
        'manage.html', 
        username=current_user.username, 
        is_admin=current_user.is_admin,
        is_super_admin=current_user.is_super_admin # 传递是否为超级管理员
    )

# -------------------- User Management API (已修改) --------------------

def admin_required(f):
    @wraps(f)
    def decorated_function(*args, **kwargs):
        if not current_user.is_authenticated or not current_user.is_admin:
            return jsonify(status='error', message='需要管理员权限'), 403
        return f(*args, **kwargs)
    return decorated_function

# ★★★ 核心修改 8: list_users 只显示当前管理员创建的用户
@app.route('/api/users', methods=['GET'])
@login_required
@admin_required # 必须是管理员 (is_admin=1)
def list_users():
    db = get_db()
    # 管理员 (Super或Regular) 只能看到他们自己创建的用户
    users = db.execute(
        "SELECT id, username, is_admin, expires_at FROM users WHERE manager_id = ? ORDER BY is_admin DESC, username", 
        (current_user.id,)
    ).fetchall()
    return jsonify([dict(u) for u in users])

# ★★★ 核心修改 9: create_user 设置 manager_id 和 expires_at
@app.route('/api/users', methods=['POST'])
@login_required
@admin_required # 必须是管理员 (is_admin=1)
def create_user():
    data = request.get_json()
    username = data.get('username')
    password = data.get('password')
    is_admin_from_form = data.get('is_admin', 0)
    
    is_super_admin = current_user.is_super_admin
    
    is_admin_to_create = 0
    expires_at = None
    
    if is_super_admin:
        # 只有超级管理员才能创建其他管理员
        if is_admin_from_form:
            is_admin_to_create = 1
            # 设为管理员：计算过期时间
            try:
                weeks = int(data.get('duration_weeks', 4))
            except (ValueError, TypeError):
                weeks = 4
            expires_at = (datetime.now() + timedelta(weeks=weeks)).strftime('%Y-%m-%d')
        else:
            # 超级管理员创建普通用户：该用户永不过期
            expires_at = None 
    else:
        # 普通管理员创建普通用户
        is_admin_to_create = 0
        # 新用户继承该管理员的过期日期
        expires_at = current_user.expires_at 
    
    manager_id = current_user.id # 新用户的管理者是当前用户

    if not username or not password: return jsonify(status='error', message='缺少用户名或密码'), 400
    try:
        db = get_db()
        db.execute(
            "INSERT INTO users (username, password, is_admin, manager_id, expires_at) VALUES (?, ?, ?, ?, ?)", 
            (username, password, is_admin_to_create, manager_id, expires_at)
        )
        db.commit()
        return jsonify(status='success', message='用户创建成功')
    except sqlite3.IntegrityError: 
        return jsonify(status='error', message='用户名已存在'), 409
    except sqlite3.Error as e: 
        return jsonify(status='error', message=f'数据库错误: {e}'), 500

# ★★★ 核心修改 10: delete_user 增加 manager_id 权限检查
@app.route('/api/users/<int:target_user_id>', methods=['DELETE'])
@login_required
@admin_required # 必须是管理员 (is_admin=1)
def delete_user(target_user_id):
    if current_user.id == target_user_id: 
        return jsonify(status='error', message='不能删除自己'), 400
    
    db = get_db()
    cur = db.cursor()
    
    # 检查目标用户是否由当前管理员创建
    target_user = cur.execute(
        "SELECT id FROM users WHERE id = ? AND manager_id = ?", 
        (target_user_id, current_user.id)
    ).fetchone()

    if not target_user:
        return jsonify(status='error', message='未找到用户或您无权删除此用户'), 403

    # 执行删除 (级联删除所有相关数据)
    cur.execute("DELETE FROM users WHERE id = ?", (target_user_id,))
    cur.execute("DELETE FROM todos WHERE user_id = ?", (target_user_id,))
    cur.execute("DELETE FROM fitness_plans WHERE user_id = ?", (target_user_id,))
    cur.execute("DELETE FROM recurring_plans WHERE user_id = ?", (target_user_id,))
    cur.execute("DELETE FROM recurring_todos WHERE user_id = ?", (target_user_id,))
    cur.execute("DELETE FROM exercise_library WHERE user_id = ?", (target_user_id,))
    cur.execute("DELETE FROM todo_library WHERE user_id = ?", (target_user_id,))
    
    db.commit()
    
    if cur.rowcount > 0: 
        return jsonify(status='success', message='用户及所有数据已删除')
    
    return jsonify(status='error', message='删除失败'), 500

# ★★★ 核心修改 11: 新增：超级管理员延长管理员时限
@app.route('/api/users/<int:admin_id>/extend', methods=['POST'])
@login_required
@admin_required
def extend_admin_expiry(admin_id):
    if not current_user.is_super_admin:
        return jsonify(status='error', message='只有超级管理员才能延长时限'), 403
        
    data = request.get_json()
    try:
        weeks = int(data.get('weeks', 4))
    except (ValueError, TypeError):
        weeks = 4
        
    db = get_db()
    
    # 查找该管理员，并确保他是被当前超级管理员创建的
    admin_user = db.execute(
        "SELECT expires_at FROM users WHERE id = ? AND manager_id = ? AND is_admin = 1", 
        (admin_id, current_user.id)
    ).fetchone()
    
    if not admin_user:
        return jsonify(status='error', message='未找到该管理员或您无权修改'), 404
        
    # 计算新日期
    # 逻辑：如果已过期，从今天开始算；如果未过期，从原日期开始延长
    base_date = date.today()
    if admin_user['expires_at']:
        try:
            current_expiry = datetime.strptime(admin_user['expires_at'], '%Y-%m-%d').date()
            if current_expiry > base_date:
                base_date = current_expiry
        except ValueError:
            pass # 日期格式错误，使用今天
            
    new_expiry_date = (base_date + timedelta(weeks=weeks)).strftime('%Y-%m-%d')
    
    db.execute("UPDATE users SET expires_at = ? WHERE id = ?", (new_expiry_date, admin_id))
    db.commit()
    
    return jsonify(status='success', message=f'时限已延长至 {new_expiry_date}')


# -------------------- Profile Update API --------------------
@app.route('/api/profile', methods=['POST'])
@login_required
def update_profile():
    user_id = current_user.id
    data = request.get_json()
    new_username = data.get('username')
    new_password = data.get('password')
    db = get_db()
    cur = db.cursor()
    update_count = 0
    if new_password:
        cur.execute("UPDATE users SET password = ? WHERE id = ?", (new_password, user_id))
        update_count += 1
    if new_username:
        current_user_db = cur.execute("SELECT username FROM users WHERE id = ?", (user_id,)).fetchone()
        if current_user_db and current_user_db['username'] != new_username:
            try:
                cur.execute("UPDATE users SET username = ? WHERE id = ?", (new_username, user_id))
                update_count += 1
            except sqlite3.IntegrityError:
                db.rollback()
                return jsonify(status='error', message='用户名已存在'), 409
    db.commit()
    if update_count > 0: 
        return jsonify(status='success', message='个人资料更新成功。用户名的更改将在下次登录时生效。')
    return jsonify(status='error', message='没有要更新的字段'), 400

# -------------------- Stats API --------------------
@app.route('/api/stats/daily_completion', methods=['GET'])
@login_required
def get_daily_completion_stats():
    user_id = current_user.id
    start_date = request.args.get('start_date')
    end_date = request.args.get('end_date')
    if not start_date or not end_date: return jsonify(status='error', message='缺少开始日期或结束日期'), 400
    db = get_db()
    fitness_rows = db.execute("SELECT date, is_completed FROM fitness_plans WHERE user_id = ? AND date BETWEEN ? AND ?", (user_id, start_date, end_date)).fetchall()
    todo_rows = db.execute("SELECT date, is_completed FROM todos WHERE user_id = ? AND date BETWEEN ? AND ?", (user_id, start_date, end_date)).fetchall()
    fitness_data = defaultdict(lambda: {"total": 0, "completed": 0})
    for row in fitness_rows:
        fitness_data[row['date']]["total"] += 1
        if row['is_completed']: fitness_data[row['date']]["completed"] += 1
    todo_data = defaultdict(lambda: {"total": 0, "completed": 0})
    for row in todo_rows:
        todo_data[row['date']]["total"] += 1
        if row['is_completed']: todo_data[row['date']]["completed"] += 1
    return jsonify({"status": "success", "fitness_data": fitness_data, "todo_data": todo_data})

# -------------------- Helpers (Data Generation Logic) --------------------
def _get_daily_fitness_plans(user_id, date_str):
    db = get_db()
    plans = db.execute("SELECT * FROM fitness_plans WHERE date = ? AND user_id = ? ORDER BY id", (date_str, user_id)).fetchall()
    return [dict(plan) for plan in plans]
def _generate_daily_fitness_plan_if_not_exists(user_id, date_str):
    db = get_db()
    cur = db.cursor()
    cur.execute("SELECT COUNT(*) FROM fitness_plans WHERE date = ? AND user_id = ?", (date_str, user_id))
    if cur.fetchone()[0] > 0: return 
    today_date = datetime.strptime(date_str, '%Y-%m-%d').date()
    today_weekday = today_date.isoweekday() 
    recurring_plans = db.execute("SELECT * FROM recurring_plans WHERE user_id = ? AND (start_date <= ?) AND (end_date IS NULL OR end_date >= ?)", (user_id, date_str, date_str)).fetchall()
    new_plans_to_insert = []
    for plan in recurring_plans:
        should_add = False
        if plan['repeat_interval'] == 'daily': should_add = True
        elif plan['repeat_interval'] == 'weekly' and plan['repeat_day'] == today_weekday: should_add = True
        if should_add:
            exercises = json.loads(plan['exercises_json'])
            for ex in exercises: new_plans_to_insert.append({'name': ex['name'], 'details': ex['details'], 'recurring_id': plan['id']})
    for plan in new_plans_to_insert:
        cur.execute(
            """INSERT INTO fitness_plans (date, exercise_name, sets_reps_duration, is_completed, user_id, recurring_plan_id) VALUES (?, ?, ?, 0, ?, ?)""",
            (date_str, plan['name'], plan['details'], user_id, plan['recurring_id']))
    db.commit()
def _generate_daily_todos_if_not_exists(user_id, date_str):
    db = get_db()
    cur = db.cursor()
    cur.execute("SELECT COUNT(*) FROM todos WHERE date = ? AND user_id = ?", (date_str, user_id))
    if cur.fetchone()[0] > 0: return
    today_date = datetime.strptime(date_str, '%Y-%m-%d').date()
    today_weekday = today_date.isoweekday() 
    today_day_of_month = today_date.day 
    recurring_todos = db.execute("SELECT * FROM recurring_todos WHERE user_id = ? AND (start_date <= ?) AND (end_date IS NULL OR end_date >= ?)", (user_id, date_str, date_str)).fetchall()
    new_todos_to_insert = []
    for todo in recurring_todos:
        should_add = False
        if todo['repeat_interval'] == 'daily': should_add = True
        elif todo['repeat_interval'] == 'weekly' and todo['repeat_day'] == today_weekday: should_add = True
        elif todo['repeat_interval'] == 'monthly' and todo['repeat_day'] == today_day_of_month: should_add = True
        if should_add: new_todos_to_insert.append({'content': todo['content'], 'recurring_id': todo['id']})
    for todo in new_todos_to_insert:
        cur.execute(
            """INSERT INTO todos (date, content, is_completed, user_id, recurring_todo_id) VALUES (?, ?, 0, ?, ?)""",
            (date_str, todo['content'], user_id, todo['recurring_id']))
    db.commit()

# -------------------- ALL OTHER APIs (Data remains scoped to current_user.id) --------------------

@app.route('/fitness_plan/<date_str>', methods=['GET'])
@login_required
def get_fitness_plan(date_str):
    user_id = current_user.id
    _generate_daily_fitness_plan_if_not_exists(user_id, date_str)
    db = get_db()
    plans = db.execute("SELECT * FROM fitness_plans WHERE date = ? AND user_id = ? ORDER BY id", (date_str, user_id)).fetchall()
    return jsonify([dict(plan) for plan in plans])

@app.route('/mark_exercise_completed_mobile', methods=['POST'])
@login_required
def mark_exercise_completed_mobile():
    user_id = current_user.id
    data = request.get_json()
    plan_id = data.get('plan_id')
    is_completed = data.get('is_completed')
    if plan_id is None or is_completed is None: return jsonify(status='error', message='缺少计划ID或完成状态'), 400
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute("UPDATE fitness_plans SET is_completed = ? WHERE id = ? AND user_id = ?", (is_completed, plan_id, user_id))
        db.commit()
        if cur.rowcount > 0: return jsonify(status='success')
        else: return jsonify(status='error', message='未找到计划或您无权修改'), 404
    except sqlite3.Error as e: return jsonify(status='error', message=f'数据库错误: {e}'), 500

@app.route('/delete_fitness_plan_exercise/<int:plan_id>', methods=['POST'])
@login_required
def delete_fitness_plan_exercise(plan_id):
    user_id = current_user.id
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute("DELETE FROM fitness_plans WHERE id = ? AND user_id = ?", (plan_id, user_id))
        db.commit()
        if cur.rowcount > 0: return jsonify(status='success', message='每日健身已删除')
        else: return jsonify(status='error', message='未找到每日健身计划'), 404
    except sqlite3.Error as e: return jsonify(status='error', message=f'数据库错误: {e}'), 500

@app.route('/todos', methods=['GET'])
@login_required
def get_todos():
    user_id = current_user.id
    date_str = request.args.get('date') or datetime.now().strftime('%Y-%m-%d')
    db = get_db()
    todos = db.execute(
        "SELECT * FROM todos WHERE date = ? AND user_id = ? ORDER BY id",
        (date_str, user_id)
    ).fetchall()
    return jsonify([dict(todo) for todo in todos])

@app.route('/add_todo', methods=['POST'])
@login_required
def add_todo():
    user_id = current_user.id
    data = request.get_json()
    date_str = data.get('date')
    content = data.get('content')
    if not date_str or not content: return jsonify(status='error', message='缺少日期或内容'), 400
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute("INSERT INTO todos (date, content, is_completed, user_id) VALUES (?, ?, 0, ?)", (date_str, content, user_id))
        db.commit()
        return jsonify(status='success', todo_id=cur.lastrowid)
    except sqlite3.Error as e: return jsonify(status='error', message=f'数据库错误: {e}'), 500

@app.route('/mark_todo_completed', methods=['POST'])
@login_required
def mark_todo_completed():
    user_id = current_user.id
    data = request.get_json()
    todo_id = data.get('todo_id')
    is_completed = data.get('is_completed')
    if todo_id is None or is_completed is None: return jsonify(status='error', message='缺少待办ID或完成状态'), 400
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute("UPDATE todos SET is_completed = ? WHERE id = ? AND user_id = ?", (is_completed, todo_id, user_id))
        db.commit()
        if cur.rowcount > 0: return jsonify(status='success')
        else: return jsonify(status='error', message='未找到待办事项或您无权修改'), 404
    except sqlite3.Error as e: return jsonify(status='error', message=f'数据库错误: {e}'), 500


@app.route('/delete_todo/<int:todo_id>', methods=['POST'])
@login_required
def delete_todo(todo_id):
    user_id = current_user.id # 确保只删除自己的
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute("DELETE FROM todos WHERE id = ? AND user_id = ?", (todo_id, user_id))
        db.commit()
        if cur.rowcount > 0:
            return jsonify(status='success', message='待办事项已删除')
        else:
            return jsonify(status='error', message='未找到待办事项'), 404
    except sqlite3.Error as e:
        return jsonify(status='error', message=f'数据库错误: {e}'), 500


@app.route('/get_recurring_plans', methods=['GET'])
@login_required
def get_recurring_plans():
    user_id = current_user.id
    db = get_db()
    plans = db.execute("SELECT * FROM recurring_plans WHERE user_id = ? ORDER BY start_date DESC", (user_id,)).fetchall()
    return jsonify([dict(plan) for plan in plans])

@app.route('/add_recurring_plan', methods=['POST'])
@login_required
def add_recurring_plan():
    user_id = current_user.id
    data = request.get_json()
    if not data.get('plan_name') or not data.get('start_date') or not data.get('repeat_interval') or not data.get('exercises'):
        return jsonify(status='error', message='缺少必填字段'), 400
    if data.get('repeat_interval') == 'weekly' and not data.get('repeat_day'):
        return jsonify(status='error', message='“每周”循环缺少“星期几”设置'), 400
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute(
            """INSERT INTO recurring_plans (user_id, plan_name, exercises_json, start_date, end_date, repeat_interval, repeat_day) VALUES (?, ?, ?, ?, ?, ?, ?)""",
            (user_id, data.get('plan_name'), json.dumps(data.get('exercises')), data.get('start_date'), data.get('end_date'), data.get('repeat_interval'), data.get('repeat_day'))
        )
        db.commit()
        return jsonify(status='success', plan_id=cur.lastrowid)
    except sqlite3.Error as e: return jsonify(status='error', message=f'数据库错误: {e}'), 500

@app.route('/delete_recurring_plan/<int:plan_id>', methods=['POST'])
@login_required
def delete_recurring_plan(plan_id):
    user_id = current_user.id
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute("DELETE FROM recurring_plans WHERE id = ? AND user_id = ?", (plan_id, user_id))
        db.commit()
        if cur.rowcount > 0: return jsonify(status='success', message='循环计划已删除')
        else: return jsonify(status='error', message='未找到循环计划'), 404
    except sqlite3.Error as e: return jsonify(status='error', message=f'数据库错误: {e}'), 500

@app.route('/get_recurring_todos', methods=['GET'])
@login_required
def get_recurring_todos():
    user_id = current_user.id
    db = get_db()
    todos = db.execute("SELECT * FROM recurring_todos WHERE user_id = ? ORDER BY start_date DESC", (user_id,)).fetchall()
    return jsonify([dict(todo) for todo in todos])

@app.route('/add_recurring_todo', methods=['POST'])
@login_required
def add_recurring_todo():
    user_id = current_user.id
    data = request.get_json()
    if not data.get('content') or not data.get('start_date') or not data.get('repeat_interval'):
        return jsonify(status='error', message='缺少必填字段'), 400
    if (data.get('repeat_interval') == 'weekly' or data.get('repeat_interval') == 'monthly') and not data.get('repeat_day'):
        return jsonify(status='error', message='循环缺少“星期几”或“日期”设置'), 400
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute(
            """INSERT INTO recurring_todos (user_id, content, start_date, end_date, repeat_interval, repeat_day) VALUES (?, ?, ?, ?, ?, ?)""",
            (user_id, data.get('content'), data.get('start_date'), data.get('end_date'), data.get('repeat_interval'), data.get('repeat_day'))
        )
        db.commit()
        return jsonify(status='success', todo_id=cur.lastrowid)
    except sqlite3.Error as e: return jsonify(status='error', message=f'数据库错误: {e}'), 500

@app.route('/delete_recurring_todo/<int:todo_id>', methods=['POST'])
@login_required
def delete_recurring_todo(todo_id):
    user_id = current_user.id
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute("DELETE FROM recurring_todos WHERE id = ? AND user_id = ?", (todo_id, user_id))
        db.commit()
        if cur.rowcount > 0: return jsonify(status='success', message='循环待办已删除')
        else: return jsonify(status='error', message='未找到循环待办'), 404
    except sqlite3.Error as e: return jsonify(status='error', message=f'数据库错误: {e}'), 500

# ================== 训练项目库 API ==================

@app.route('/api/exercises', methods=['GET'])
@login_required
def list_exercises():
    user_id = current_user.id
    db = get_db()
    rows = db.execute(
        "SELECT id, name, details FROM exercise_library WHERE user_id = ? ORDER BY id DESC",
        (user_id,)
    ).fetchall()
    return jsonify([dict(r) for r in rows])


@app.route('/api/exercises', methods=['POST'])
@login_required
def create_exercise():
    user_id = current_user.id
    data = request.get_json() or {}
    name = (data.get('name') or '').strip()
    details = (data.get('details') or '').strip()
    if not name:
        return jsonify(status='error', message='缺少训练项目名称'), 400
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute(
            "INSERT INTO exercise_library (user_id, name, details) VALUES (?, ?, ?)",
            (user_id, name, details)
        )
        db.commit()
        return jsonify(status='success', exercise_id=cur.lastrowid)
    except sqlite3.Error as e:
        return jsonify(status='error', message=f'数据库错误: {e}'), 500


@app.route('/api/exercises/<int:exercise_id>', methods=['POST', 'DELETE']) # 兼容 DELETE
@login_required
def delete_exercise(exercise_id):
    user_id = current_user.id
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute(
            "DELETE FROM exercise_library WHERE id = ? AND user_id = ?",
            (exercise_id, user_id)
        )
        db.commit()
        if cur.rowcount > 0:
            return jsonify(status='success', message='训练项目已删除')
        else:
            return jsonify(status='error', message='未找到训练项目'), 404
    except sqlite3.Error as e:
        return jsonify(status='error', message=f'数据库错误: {e}'), 500


# ================== 常用待办库 API ==================

@app.route('/api/todo_library', methods=['GET'])
@login_required
def list_todo_library():
    user_id = current_user.id
    db = get_db()
    rows = db.execute(
        "SELECT id, content FROM todo_library WHERE user_id = ? ORDER BY id DESC",
        (user_id,)
    ).fetchall()
    return jsonify([dict(r) for r in rows])


@app.route('/api/todo_library', methods=['POST'])
@login_required
def create_todo_library_item():
    user_id = current_user.id
    data = request.get_json() or {}
    content = (data.get('content') or '').strip()
    if not content:
        return jsonify(status='error', message='缺少待办内容'), 400
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute(
            "INSERT INTO todo_library (user_id, content) VALUES (?, ?)",
            (user_id, content)
        )
        db.commit()
        return jsonify(status='success', item_id=cur.lastrowid)
    except sqlite3.Error as e:
        return jsonify(status='error', message=f'数据库错误: {e}'), 500


@app.route('/api/todo_library/<int:item_id>', methods=['POST', 'DELETE']) # 兼容 DELETE
@login_required
def delete_todo_library_item(item_id):
    user_id = current_user.id
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute(
            "DELETE FROM todo_library WHERE id = ? AND user_id = ?",
            (item_id, user_id)
        )
        db.commit()
        if cur.rowcount > 0:
            return jsonify(status='success', message='常用待办已删除')
        else:
            return jsonify(status='error', message='未找到该待办'), 404
    except sqlite3.Error as e:
        return jsonify(status='error', message=f'数据库错误: {e}'), 500



# -------------------- Compatibility headers & health --------------------
@app.after_request
def add_common_headers(response):
    if 'Content-Type' not in response.headers:
        response.headers['Content-Type'] = 'text/html; charset=utf-8'
    else:
        c = response.headers['Content-Type']
        if c.startswith('text/') and 'charset' not in c:
            response.headers['Content-Type'] = c + '; charset=utf-8'
    response.headers.setdefault('X-Content-Type-Options', 'nosniff')
    response.headers.setdefault('X-Frame-Options', 'SAMEORIGIN')
    path = request.path or ''
    if path.startswith('/static/') or path.endswith(('.css', '.js', '.png', '.jpg', '.svg', '.webp')):
        response.headers['Cache-Control'] = 'public, max-age=3600'
    else:
        response.headers['Cache-Control'] = 'no-cache, no-store, must-revalidate'
    return response

@app.route('/health', methods=['GET'])
def health():
    return jsonify(status='ok', time=datetime.now().isoformat())

# -------------------- Run --------------------
if __name__ == '__main__':
    app.run(host='0.0.0.0', port=8000, debug=True)
