# backend/app.py (v2.0)
import sqlite3
from datetime import datetime, timedelta, timezone
from flask import Flask, render_template, request, jsonify, g, redirect, url_for
from flask_login import LoginManager, UserMixin, login_user, logout_user, login_required, current_user
import os, json, uuid
from flask_cors import CORS
from functools import wraps
import sys 
from collections import defaultdict

CN_TZ = timezone(timedelta(hours=8))

# --- Path Setup ---
is_frozen = getattr(sys, 'frozen', False)
is_in_onefile_temp = 'onefile_' in os.path.abspath(__file__) 
if is_frozen or is_in_onefile_temp:
    base_dir = os.path.dirname(os.path.abspath(sys.executable))
    template_dir = os.path.join(base_dir, 'templates')
    static_dir = os.path.join(base_dir, 'static')
    DATABASE = os.path.join(os.path.dirname(os.path.realpath(sys.argv[0])), 'fitness_manager.db')
else:
    base_dir = os.path.dirname(os.path.abspath(__file__))
    template_dir = os.path.join(os.path.dirname(base_dir), 'frontend', 'templates')
    static_dir = os.path.join(os.path.dirname(base_dir), 'frontend', 'static')
    DATABASE = os.path.join(base_dir, 'fitness_manager.db')

app = Flask(__name__, template_folder=template_dir, static_folder=static_dir)
CORS(app)
app.secret_key = str(uuid.uuid4())

# --- Auth ---
login_manager = LoginManager()
login_manager.init_app(app)
login_manager.login_view = 'index' 

class User(UserMixin):
    def __init__(self, id, username, is_admin, manager_id):
        self.id = id; self.username = username; self.is_admin = is_admin; self.manager_id = manager_id 
    @property
    def is_super_admin(self): return self.is_admin and (self.manager_id is None or self.manager_id == 0)

@login_manager.user_loader
def load_user(uid):
    row = get_db().execute("SELECT * FROM users WHERE id=?", (uid,)).fetchone()
    return User(row['id'], row['username'], row['is_admin'], row['manager_id']) if row else None

# --- DB ---
def get_db():
    db = getattr(g, '_database', None)
    if db is None: db = g._database = sqlite3.connect(DATABASE, check_same_thread=False); db.row_factory = sqlite3.Row
    return db

@app.teardown_appcontext
def close_conn(e):
    db = getattr(g, '_database', None)
    if db is not None: db.close()

def init_db():
    db = get_db()
    cur = db.cursor()
    # Schemas
    cur.execute("CREATE TABLE IF NOT EXISTS todos (id INTEGER PRIMARY KEY AUTOINCREMENT, date TEXT NOT NULL, content TEXT NOT NULL, is_completed INTEGER DEFAULT 0, user_id INTEGER, recurring_todo_id INTEGER, reminder_at TEXT)")
    cur.execute("CREATE TABLE IF NOT EXISTS fitness_plans (id INTEGER PRIMARY KEY AUTOINCREMENT, date TEXT NOT NULL, exercise_name TEXT NOT NULL, sets_reps_duration TEXT, is_completed INTEGER DEFAULT 0, user_id INTEGER, recurring_plan_id INTEGER, reminder_at TEXT)")
    cur.execute("CREATE TABLE IF NOT EXISTS recurring_plans (id INTEGER PRIMARY KEY AUTOINCREMENT, user_id INTEGER, plan_name TEXT NOT NULL, exercises_json TEXT NOT NULL, start_date TEXT NOT NULL, end_date TEXT, repeat_interval TEXT NOT NULL, repeat_day INTEGER, reminder_time TEXT)")
    cur.execute("CREATE TABLE IF NOT EXISTS recurring_todos (id INTEGER PRIMARY KEY AUTOINCREMENT, user_id INTEGER, content TEXT NOT NULL, start_date TEXT NOT NULL, end_date TEXT, repeat_interval TEXT NOT NULL, repeat_day INTEGER, reminder_time TEXT)")
    cur.execute("CREATE TABLE IF NOT EXISTS exercise_library (id INTEGER PRIMARY KEY AUTOINCREMENT, user_id INTEGER, name TEXT, details TEXT)")
    cur.execute("CREATE TABLE IF NOT EXISTS todo_library (id INTEGER PRIMARY KEY AUTOINCREMENT, user_id INTEGER, content TEXT)")
    cur.execute("CREATE TABLE IF NOT EXISTS settings (key TEXT PRIMARY KEY, value TEXT)")
    cur.execute("CREATE TABLE IF NOT EXISTS users (id INTEGER PRIMARY KEY AUTOINCREMENT, username TEXT UNIQUE, password TEXT, is_admin INTEGER DEFAULT 0, manager_id INTEGER)")
    
    # Migrations
    try: cur.execute("SELECT reminder_at FROM todos LIMIT 1"); 
    except: cur.execute("ALTER TABLE todos ADD COLUMN reminder_at TEXT")
    try: cur.execute("SELECT reminder_at FROM fitness_plans LIMIT 1"); 
    except: cur.execute("ALTER TABLE fitness_plans ADD COLUMN reminder_at TEXT")
    try: cur.execute("SELECT reminder_time FROM recurring_plans LIMIT 1"); 
    except: cur.execute("ALTER TABLE recurring_plans ADD COLUMN reminder_time TEXT")
    try: cur.execute("SELECT reminder_time FROM recurring_todos LIMIT 1"); 
    except: cur.execute("ALTER TABLE recurring_todos ADD COLUMN reminder_time TEXT")
    
    # Seeds
    if cur.execute("SELECT COUNT(*) FROM users WHERE username='superuser'").fetchone()[0]==0:
        cur.execute("INSERT INTO users (username,password,is_admin,manager_id) VALUES (?,?,?,?)",('superuser','adminpassword',1,None))
    su_id = cur.execute("SELECT id FROM users WHERE username='superuser'").fetchone()['id']
    if cur.execute("SELECT COUNT(*) FROM users WHERE username='user1'").fetchone()[0]==0:
        cur.execute("INSERT INTO users (username,password,is_admin,manager_id) VALUES (?,?,?,?)",('user1','password',0,su_id))
    db.commit()

with app.app_context(): init_db()

# --- Routes ---
@app.route('/login', methods=['POST'])
def login():
    d = request.json or request.form
    u = get_db().execute("SELECT * FROM users WHERE username=?",(d.get('username'),)).fetchone()
    if u and u['password']==d.get('password'):
        login_user(User(u['id'],u['username'],u['is_admin'],u['manager_id']), remember=True)
        return jsonify({'status':'success'})
    return jsonify({'status':'error'}),401

@app.route('/logout', methods=['POST'])
@login_required
def logout(): logout_user(); return jsonify({'status':'success'})

@app.route('/')
def index():
    if not current_user.is_authenticated: return render_template('login.html')
    if 'Mobile' in request.headers.get('User-Agent','') and request.args.get('view')!='display': return redirect(url_for('mobile_config'))
    today = datetime.now(CN_TZ).strftime('%Y-%m-%d')
    _gen_fit(current_user.id, today)
    _gen_todo(current_user.id, today)
    return render_template('display.html', username=current_user.username)

@app.route('/manage')
@login_required
def manage(): return render_template('manage.html', username=current_user.username, is_admin=current_user.is_admin)

@app.route('/mobile_config')
@login_required
def mobile_config(): return render_template('mobile_config.html', username=current_user.username)

# --- APIs ---
@app.route('/todos', methods=['GET'])
@login_required
def get_todos():
    d = request.args.get('date') or datetime.now(CN_TZ).strftime('%Y-%m-%d')
    return jsonify([dict(r) for r in get_db().execute("SELECT * FROM todos WHERE date=? AND user_id=? ORDER BY is_completed ASC, reminder_at ASC, id DESC", (d, current_user.id)).fetchall()])

@app.route('/add_todo', methods=['POST'])
@login_required
def add_todo():
    d = request.json
    rem = d.get('reminder_at')
    if rem and len(rem)==5: rem = f"{d.get('date')} {rem}"
    get_db().execute("INSERT INTO todos (date,content,is_completed,user_id,reminder_at) VALUES (?,?,0,?,?)", (d.get('date'),d.get('content'),current_user.id,rem)).connection.commit()
    return jsonify(status='success')

@app.route('/add_recurring_todo', methods=['POST'])
@login_required
def add_rec_todo():
    d = request.json
    db = get_db()
    cur = db.cursor()
    cur.execute("INSERT INTO recurring_todos (user_id,content,start_date,end_date,repeat_interval,repeat_day,reminder_time) VALUES (?,?,?,?,?,?,?)",
                (current_user.id, d.get('content'), d.get('start_date'), d.get('end_date'), d.get('repeat_interval'), d.get('repeat_day'), d.get('reminder_time')))
    new_id = cur.lastrowid
    db.commit()
    
    # ★★★ Instant Generation Check for Today ★★★
    today = datetime.now(CN_TZ).strftime('%Y-%m-%d')
    _gen_single_rec_todo(current_user.id, today, new_id) # Check if the new rule applies to today immediately
    
    return jsonify(status='success')

@app.route('/fitness_plan/<date>', methods=['GET'])
@login_required
def get_fit(date):
    _gen_fit(current_user.id, date)
    return jsonify([dict(r) for r in get_db().execute("SELECT * FROM fitness_plans WHERE date=? AND user_id=? ORDER BY is_completed ASC, reminder_at ASC, id DESC", (date, current_user.id)).fetchall()])

@app.route('/add_recurring_plan', methods=['POST'])
@login_required
def add_rec_plan():
    d = request.json
    db = get_db()
    cur = db.cursor()
    cur.execute("INSERT INTO recurring_plans (user_id,plan_name,exercises_json,start_date,end_date,repeat_interval,repeat_day,reminder_time) VALUES (?,?,?,?,?,?,?,?)",
                (current_user.id, d.get('plan_name'), json.dumps(d.get('exercises')), d.get('start_date'), d.get('end_date'), d.get('repeat_interval'), d.get('repeat_day'), d.get('reminder_time')))
    new_id = cur.lastrowid
    db.commit()

    # ★★★ Instant Generation Check for Today ★★★
    today = datetime.now(CN_TZ).strftime('%Y-%m-%d')
    _gen_single_rec_plan(current_user.id, today, new_id)

    return jsonify(status='success')

# ... (Standard Delete/Update/Library APIs preserved for brevity but fully functional in logic) ...
@app.route('/delete_todo/<int:id>', methods=['POST'])
@login_required
def del_todo(id): get_db().execute("DELETE FROM todos WHERE id=? AND user_id=?",(id,current_user.id)).connection.commit(); return jsonify(status='success')
@app.route('/mark_todo_completed', methods=['POST'])
@login_required
def mark_todo(): get_db().execute("UPDATE todos SET is_completed=? WHERE id=? AND user_id=?",(request.json['is_completed'],request.json['todo_id'],current_user.id)).connection.commit(); return jsonify(status='success')
@app.route('/delete_fitness_plan_exercise/<int:id>', methods=['POST'])
@login_required
def del_fit(id): get_db().execute("DELETE FROM fitness_plans WHERE id=? AND user_id=?",(id,current_user.id)).connection.commit(); return jsonify(status='success')
@app.route('/mark_exercise_completed_mobile', methods=['POST'])
@login_required
def mark_fit(): get_db().execute("UPDATE fitness_plans SET is_completed=? WHERE id=? AND user_id=?",(request.json['is_completed'],request.json['plan_id'],current_user.id)).connection.commit(); return jsonify(status='success')
@app.route('/get_recurring_todos', methods=['GET'])
@login_required
def get_rec_todos_list(): return jsonify([dict(r) for r in get_db().execute("SELECT * FROM recurring_todos WHERE user_id=?",(current_user.id,)).fetchall()])
@app.route('/delete_recurring_todo/<int:id>', methods=['POST'])
@login_required
def del_rec_todo(id): get_db().execute("DELETE FROM recurring_todos WHERE id=? AND user_id=?",(id,current_user.id)).connection.commit(); return jsonify(status='success')
@app.route('/get_recurring_plans', methods=['GET'])
@login_required
def get_rec_plans_list(): return jsonify([dict(r) for r in get_db().execute("SELECT * FROM recurring_plans WHERE user_id=?",(current_user.id,)).fetchall()])
@app.route('/delete_recurring_plan/<int:id>', methods=['POST'])
@login_required
def del_rec_plan(id): get_db().execute("DELETE FROM recurring_plans WHERE id=? AND user_id=?",(id,current_user.id)).connection.commit(); return jsonify(status='success')
@app.route('/add_fitness_manual', methods=['POST'])
@login_required
def add_fit_man():
    d=request.json
    rem = d.get('reminder_at')
    if rem and len(rem)==5: rem = f"{d.get('date')} {rem}"
    get_db().execute("INSERT INTO fitness_plans (date,exercise_name,sets_reps_duration,user_id,reminder_at) VALUES (?,?,?,?,?)",(d.get('date'),d.get('exercise_name'),d.get('sets_reps_duration'),current_user.id,rem)).connection.commit()
    return jsonify(status='success')
@app.route('/api/settings/scroll_speed', methods=['GET','POST'])
def scroll_speed():
    if request.method=='POST': 
        db=get_db(); db.execute("INSERT INTO settings (key,value) VALUES ('scroll_speed',?) ON CONFLICT(key) DO UPDATE SET value=excluded.value",(str(request.json.get('scroll_speed')),)); db.commit()
        return jsonify(status='success')
    return jsonify(scroll_speed=int(get_db().execute("SELECT value FROM settings WHERE key='scroll_speed'").fetchone()['value'] or 25))
@app.route('/api/stats/daily_completion')
@login_required
def stats():
    s,e=request.args.get('start_date'),request.args.get('end_date')
    db=get_db()
    f=db.execute("SELECT date,is_completed FROM fitness_plans WHERE user_id=? AND date BETWEEN ? AND ?",(current_user.id,s,e)).fetchall()
    t=db.execute("SELECT date,is_completed FROM todos WHERE user_id=? AND date BETWEEN ? AND ?",(current_user.id,s,e)).fetchall()
    fd=defaultdict(lambda:{'total':0,'completed':0}); td=defaultdict(lambda:{'total':0,'completed':0})
    for r in f: fd[r['date']]['total']+=1; 
    for r in f: 
        if r['is_completed']: fd[r['date']]['completed']+=1
    for r in t: td[r['date']]['total']+=1; 
    for r in t:
        if r['is_completed']: td[r['date']]['completed']+=1
    return jsonify(fitness_data=fd,todo_data=td)
@app.route('/api/users', methods=['GET'])
@login_required
def list_users(): return jsonify([dict(u) for u in get_db().execute("SELECT id,username,is_admin FROM users WHERE manager_id=?",(current_user.id,)).fetchall()])
@app.route('/api/users', methods=['POST'])
@login_required
def add_user():
    d=request.json
    is_a = 1 if current_user.is_super_admin and d.get('is_admin') else 0
    try: get_db().execute("INSERT INTO users (username,password,is_admin,manager_id) VALUES (?,?,?,?)",(d.get('username'),d.get('password'),is_a,current_user.id)).connection.commit(); return jsonify(status='success')
    except: return jsonify(status='error')
@app.route('/api/users/<int:id>', methods=['DELETE'])
@login_required
def del_user(id):
    if id==current_user.id: return jsonify(status='error')
    get_db().execute("DELETE FROM users WHERE id=?",(id,)).connection.commit(); return jsonify(status='success')
@app.route('/api/profile', methods=['POST'])
@login_required
def update_profile():
    d=request.json
    if d.get('password'): get_db().execute("UPDATE users SET password=? WHERE id=?",(d.get('password'),current_user.id))
    if d.get('username'): 
        try: get_db().execute("UPDATE users SET username=? WHERE id=?",(d.get('username'),current_user.id))
        except: pass
    get_db().commit()
    return jsonify(status='success')
@app.route('/api/exercises', methods=['GET'])
@login_required
def get_ex_lib(): return jsonify([dict(r) for r in get_db().execute("SELECT * FROM exercise_library WHERE user_id=?",(current_user.id,)).fetchall()])
@app.route('/api/exercises', methods=['POST'])
@login_required
def add_ex_lib(): get_db().execute("INSERT INTO exercise_library (user_id,name,details) VALUES (?,?,?)",(current_user.id,request.json.get('name'),request.json.get('details'))).connection.commit(); return jsonify(status='success')
@app.route('/api/exercises/<int:id>', methods=['DELETE'])
@login_required
def del_ex_lib(id): get_db().execute("DELETE FROM exercise_library WHERE id=?",(id,)).connection.commit(); return jsonify(status='success')
@app.route('/api/todo_library', methods=['GET'])
@login_required
def get_td_lib(): return jsonify([dict(r) for r in get_db().execute("SELECT * FROM todo_library WHERE user_id=?",(current_user.id,)).fetchall()])
@app.route('/api/todo_library', methods=['POST'])
@login_required
def add_td_lib(): get_db().execute("INSERT INTO todo_library (user_id,content) VALUES (?,?)",(current_user.id,request.json.get('content'))).connection.commit(); return jsonify(status='success')
@app.route('/api/todo_library/<int:id>', methods=['DELETE'])
@login_required
def del_td_lib(id): get_db().execute("DELETE FROM todo_library WHERE id=?",(id,)).connection.commit(); return jsonify(status='success')

# --- Generators ---
def _gen_fit(uid, date_str):
    db = get_db()
    if db.execute("SELECT COUNT(*) FROM fitness_plans WHERE date=? AND user_id=?",(date_str,uid)).fetchone()[0]>0: return
    # If empty, generate from recurring
    plans = db.execute("SELECT * FROM recurring_plans WHERE user_id=? AND start_date<=? AND (end_date IS NULL OR end_date>=?)", (uid,date_str,date_str)).fetchall()
    dt = datetime.strptime(date_str, '%Y-%m-%d').date()
    for p in plans:
        if _is_due(p, dt):
            _insert_fit_from_rec(db, uid, date_str, p)
    db.commit()

def _gen_single_rec_plan(uid, date_str, rec_id):
    """Check ONE recurring plan and insert to today if due. Don't block if today already has items."""
    db = get_db()
    p = db.execute("SELECT * FROM recurring_plans WHERE id=?", (rec_id,)).fetchone()
    if not p: return
    dt = datetime.strptime(date_str, '%Y-%m-%d').date()
    if p['start_date'] <= date_str and (not p['end_date'] or p['end_date'] >= date_str):
        if _is_due(p, dt):
            _insert_fit_from_rec(db, uid, date_str, p)
    db.commit()

def _insert_fit_from_rec(db, uid, date_str, p):
    rem = f"{date_str} {p['reminder_time']}" if p['reminder_time'] else None
    for ex in json.loads(p['exercises_json']):
        db.execute("INSERT INTO fitness_plans (date,exercise_name,sets_reps_duration,user_id,recurring_plan_id,reminder_at) VALUES (?,?,?,?,?,?)",
                   (date_str, ex['name'], ex['details'], uid, p['id'], rem))

def _gen_todo(uid, date_str):
    db = get_db()
    if db.execute("SELECT COUNT(*) FROM todos WHERE date=? AND user_id=?",(date_str,uid)).fetchone()[0]>0: return
    recs = db.execute("SELECT * FROM recurring_todos WHERE user_id=? AND start_date<=? AND (end_date IS NULL OR end_date>=?)", (uid,date_str,date_str)).fetchall()
    dt = datetime.strptime(date_str, '%Y-%m-%d').date()
    for r in recs:
        if _is_due(r, dt):
            _insert_todo_from_rec(db, uid, date_str, r)
    db.commit()

def _gen_single_rec_todo(uid, date_str, rec_id):
    """Check ONE recurring todo and insert to today if due."""
    db = get_db()
    r = db.execute("SELECT * FROM recurring_todos WHERE id=?", (rec_id,)).fetchone()
    if not r: return
    dt = datetime.strptime(date_str, '%Y-%m-%d').date()
    if r['start_date'] <= date_str and (not r['end_date'] or r['end_date'] >= date_str):
        if _is_due(r, dt):
            _insert_todo_from_rec(db, uid, date_str, r)
    db.commit()

def _insert_todo_from_rec(db, uid, date_str, r):
    rem = f"{date_str} {r['reminder_time']}" if r['reminder_time'] else None
    db.execute("INSERT INTO todos (date,content,is_completed,user_id,recurring_todo_id,reminder_at) VALUES (?,?,0,?,?,?)",
               (date_str, r['content'], uid, r['id'], rem))

def _is_due(item, dt):
    if item['repeat_interval'] == 'daily': return True
    if item['repeat_interval'] == 'weekly' and item['repeat_day'] == dt.isoweekday(): return True
    if item['repeat_interval'] == 'monthly' and item['repeat_day'] == dt.day: return True
    return False

if __name__ == '__main__':
    app.run(host='0.0.0.0', port=8000, debug=True)
