from flask import Flask, render_template, request, jsonify, redirect, url_for, session
import time
import datetime
import os
import random
import json
import RPi.GPIO as GPIO

app = Flask(__name__)
app.secret_key = 'parent_config_secret'
PARENT_PASSWORD = 'rendong'  # 家长固定密码

# --------------------------
# 路径配置
# --------------------------
current_program_dir = os.path.dirname(os.path.abspath(__file__))
RECORD_FILE = os.path.join(current_program_dir, "daily_answers.json")
CONFIG_FILE = os.path.join(current_program_dir, "parent_config.json")
print(f"程序运行目录: {current_program_dir}")
print(f"配置文件路径: {CONFIG_FILE}")
print(f"答题记录路径: {RECORD_FILE}")

# --------------------------
# 继电器初始化（使用5个继电器对应周一至周五）
# --------------------------
# 继电器引脚定义：GPIO编号 -> 继电器编号
RELAY_PINS = {
    1: 5,   # 周一 -> GPIO5 -> 继电器1
    2: 6,   # 周二 -> GPIO6 -> 继电器2
    3: 13,  # 周三 -> GPIO13 -> 继电器3
    4: 19,  # 周四 -> GPIO19 -> 继电器4
    5: 26   # 周五 -> GPIO26 -> 继电器5
}

try:
    GPIO.setmode(GPIO.BCM)
    # 初始化所有继电器引脚
    for pin in RELAY_PINS.values():
        GPIO.setup(pin, GPIO.OUT)
        GPIO.output(pin, GPIO.HIGH)  # 初始化为关闭状态
    print("✅ 5个继电器初始化成功")
    print(f"继电器引脚映射: {RELAY_PINS}")
except Exception as e:
    print(f"❌ 继电器初始化失败: {str(e)}")

# --------------------------
# 家长配置相关函数
# --------------------------
def load_parent_config():
    default_config = {
        "num_min": 1,
        "num_max": 5,
        "arithmetic_type": "add",
        "option_count": 3,
        "daily_question_count": 5
    }
    if not os.path.exists(CONFIG_FILE):
        print(f"⚠️  配置文件不存在，使用默认配置并保存: {CONFIG_FILE}")
        save_parent_config(default_config)
        return default_config
    try:
        with open(CONFIG_FILE, 'r', encoding='utf-8') as f:
            config = json.load(f)
        valid_config = default_config.copy()
        valid_config.update({
            "num_min": max(1, int(config.get("num_min", 1))),
            "num_max": max(valid_config["num_min"] + 1, int(config.get("num_max", 5))),
            "arithmetic_type": config.get("arithmetic_type") if config.get("arithmetic_type") in ["add", "sub", "mix"] else "add",
            "option_count": 3 if config.get("option_count") not in [3, 4] else int(config.get("option_count")),
            "daily_question_count": max(1, int(config.get("daily_question_count", 5)))
        })
        print(f"✅ 加载配置成功: {valid_config}")
        return valid_config
    except Exception as e:
        print(f"❌ 加载配置失败（{str(e)}），使用默认配置")
        save_parent_config(default_config)
        return default_config

def save_parent_config(config):
    try:
        if not os.path.exists(current_program_dir):
            os.makedirs(current_program_dir, exist_ok=True)
        with open(CONFIG_FILE, 'w', encoding='utf-8') as f:
            json.dump(config, f, indent=2, ensure_ascii=False)
        print(f"✅ 配置保存成功: {CONFIG_FILE}")
    except Exception as e:
        print(f"❌ 保存配置失败: {str(e)}")

# --------------------------
# 答题记录相关函数
# --------------------------
def init_daily_answer_record():
    """初始化当日答题记录"""
    config = load_parent_config()
    init_record = {
        "date": datetime.datetime.now().strftime('%Y-%m-%d'),
        "total_question_count": config["daily_question_count"],
        "answered_questions": [],
        "relay_triggered": False,  # 当日继电器是否已触发
        "weekday_relay": None      # 触发的星期几对应的继电器编号
    }
    print(f"ℹ️  初始化当日答题记录: {init_record}")
    return init_record

def load_daily_answer_record():
    """加载答题记录"""
    if not os.path.exists(RECORD_FILE):
        print(f"⚠️  未找到答题记录文件，初始化新记录: {RECORD_FILE}")
        return init_daily_answer_record()
    
    try:
        with open(RECORD_FILE, 'r', encoding='utf-8') as f:
            record = json.load(f)
        
        # 校验日期（非今日则重新初始化）
        today = datetime.datetime.now().strftime('%Y-%m-%d')
        if record.get("date") != today:
            print(f"⚠️  记录日期[{record.get('date')}]≠今日[{today}]，重新初始化")
            return init_daily_answer_record()
        
        # 校验必要字段
        required_fields = ["date", "total_question_count", "answered_questions", "relay_triggered"]
        if not all(field in record for field in required_fields):
            print(f"⚠️  答题记录结构不完整，删除后重新初始化")
            os.remove(RECORD_FILE)
            return init_daily_answer_record()
        
        # 同步最新的每日答题总数
        config = load_parent_config()
        if record["total_question_count"] != config["daily_question_count"]:
            old_total = record["total_question_count"]
            record["total_question_count"] = config["daily_question_count"]
            print(f"ℹ️  同步配置：每日答题总数从{old_total}更新为{config['daily_question_count']}")
        
        answered_count = len(record["answered_questions"])
        total_count = record["total_question_count"]
        print(f"✅ 加载答题记录成功：已答{answered_count}/{total_count}题，继电器触发状态：{record['relay_triggered']}")
        return record
    
    except Exception as e:
        print(f"❌ 加载答题记录失败（{str(e)}），重新初始化")
        return init_daily_answer_record()

def save_daily_answer_record(record):
    """保存答题记录"""
    try:
        if not os.path.exists(current_program_dir):
            os.makedirs(current_program_dir, exist_ok=True)
        
        with open(RECORD_FILE, 'w', encoding='utf-8') as f:
            json.dump(record, f, indent=2, ensure_ascii=False)
        
        answered_count = len(record["answered_questions"])
        print(f"✅ 答题记录保存成功：已答{answered_count}题，继电器触发：{record['relay_triggered']}")
        return True
    
    except Exception as e:
        print(f"❌ 保存答题记录失败（{str(e)}）")
        return False

def calculate_accuracy(record):
    """计算正确率"""
    answered_count = len(record["answered_questions"])
    if answered_count == 0:
        return 0.00
    correct_count = sum(1 for q in record["answered_questions"] if q["is_correct"])
    accuracy = round(correct_count / answered_count, 2)
    print(f"ℹ️  计算正确率：正确{correct_count}题/总{answered_count}题 = {accuracy*100:.0f}%")
    return accuracy

def is_answer_complete(record):
    """判断是否完成当日答题"""
    complete = len(record["answered_questions"]) >= record["total_question_count"]
    print(f"ℹ️  答题完成判断：已答{len(record['answered_questions'])}题/总{record['total_question_count']}题 → {'是' if complete else '否'}")
    return complete

def get_current_weekday():
    """获取当前星期几（1-7，1=周一，7=周日）"""
    weekday = datetime.datetime.now().isoweekday()
    print(f"ℹ️  当前星期：{weekday}（1=周一，7=周日）")
    return weekday

# --------------------------
# 题目生成函数
# --------------------------
def generate_question():
    config = load_parent_config()
    num_min = config["num_min"]
    num_max = config["num_max"]
    arithmetic_type = config["arithmetic_type"]
    option_count = config["option_count"]
    
    a = random.randint(num_min, num_max)
    b = random.randint(num_min, num_max)
    
    if arithmetic_type == "add" or (arithmetic_type == "mix" and random.choice([True, False])):
        operator = "+"
        correct_answer = a + b
    else:
        operator = "-"
        if a < b:
            a, b = b, a
        correct_answer = a - b
    
    options = [correct_answer]
    while len(options) < option_count:
        range_offset = num_max - num_min
        candidate = random.randint(
            max(0, correct_answer - range_offset),
            correct_answer + range_offset
        )
        if candidate not in options:
            options.append(candidate)
    random.shuffle(options)
    
    question = {
        'id': int(time.time() % 100000),
        'question': f"{a} {operator} {b} = ?",
        'options': options,
        'correct_index': options.index(correct_answer)
    }
    print(f"✅ 生成新题目：{question['question']}，正确答案：{correct_answer}（选项索引{question['correct_index']}）")
    return question

# --------------------------
# 继电器控制函数
# --------------------------
def control_relay_by_weekday(duration=10):
    """根据星期几控制对应的继电器"""
    weekday = get_current_weekday()
    
    # 只在周一至周五触发（1-5）
    if 1 <= weekday <= 5:
        relay_num = weekday  # 周一=1，周二=2，...，周五=5
        gpio_pin = RELAY_PINS.get(relay_num)
        
        if gpio_pin is None:
            return False, f"星期{weekday}对应的继电器未配置"
        
        try:
            print(f"🔌 开启星期{weekday}对应的继电器{relay_num}（GPIO{gpio_pin}），持续{duration}秒")
            GPIO.output(gpio_pin, GPIO.LOW)  # 开启继电器
            
            # 记录开启时间
            start_time = time.time()
            return True, f"星期{weekday}的奖励已开启！", start_time, duration, relay_num
        
        except Exception as e:
            error_msg = str(e)
            print(f"❌ 继电器{relay_num}控制失败：{error_msg}")
            return False, f"奖励设备启动失败：{error_msg}", None, 0, None
    else:
        return False, "周末休息，没有奖励哦～", None, 0, None

def stop_all_relays():
    """关闭所有继电器"""
    try:
        for pin in RELAY_PINS.values():
            GPIO.output(pin, GPIO.HIGH)  # 关闭继电器
        print("🔌 所有继电器已关闭")
        return True
    except Exception as e:
        print(f"❌ 关闭继电器失败：{str(e)}")
        return False

def check_and_stop_relay(start_time, duration):
    """检查并停止继电器"""
    if start_time and time.time() - start_time >= duration:
        stop_all_relays()
        return True
    return False

# 存储继电器状态
relay_state = {
    'is_active': False,
    'start_time': None,
    'duration': 0,
    'relay_num': None
}

# --------------------------
# Flask路由
# --------------------------
@app.route('/')
def index():
    """答题首页"""
    record = load_daily_answer_record()
    answered_count = len(record["answered_questions"])
    total_count = record["total_question_count"]
    accuracy = calculate_accuracy(record)
    formatted_accuracy = int(accuracy * 100)
    
    # 检查并停止到期的继电器
    if relay_state['is_active']:
        if check_and_stop_relay(relay_state['start_time'], relay_state['duration']):
            relay_state['is_active'] = False
    
    # 获取当前星期信息
    weekday = get_current_weekday()
    weekday_names = ["", "星期一", "星期二", "星期三", "星期四", "星期五", "星期六", "星期日"]
    current_weekday_name = weekday_names[weekday] if 1 <= weekday <= 7 else "未知"
    
    print(f"ℹ️  首页加载完成：{current_weekday_name}，进度{answered_count}/{total_count}题，正确率{formatted_accuracy}%，继电器触发：{record['relay_triggered']}")
    
    return render_template(
        'quiz.html',
        answered_count=answered_count,
        total_count=total_count,
        accuracy=formatted_accuracy,
        relay_triggered=record["relay_triggered"],
        current_weekday=current_weekday_name,
        weekday_relay=record.get("weekday_relay")
    )

# 家长相关路由保持不变...
@app.route('/parent')
def parent_redirect():
    if session.get('is_parent_logged'):
        return redirect(url_for('parent_config_page'))
    return redirect(url_for('parent_login'))

@app.route('/parent_login', methods=['GET', 'POST'])
def parent_login():
    if request.method == 'POST':
        input_password = request.form.get('password')
        if input_password == PARENT_PASSWORD:
            session['is_parent_logged'] = True
            return redirect(url_for('parent_config_page'))
        else:
            return render_template('parent_login.html', error="密码错误，请重新输入！")
    return render_template('parent_login.html')

@app.route('/parent_logout')
def parent_logout():
    session.pop('is_parent_logged', None)
    return redirect(url_for('index'))

@app.route('/parent_config')
def parent_config_page():
    if not session.get('is_parent_logged'):
        return redirect(url_for('parent_login'))
    config = load_parent_config()
    
    # 显示继电器状态信息
    relay_info = {
        'monday': RELAY_PINS[1],
        'tuesday': RELAY_PINS[2],
        'wednesday': RELAY_PINS[3],
        'thursday': RELAY_PINS[4],
        'friday': RELAY_PINS[5]
    }
    
    return render_template('parent_config.html', config=config, relay_info=relay_info)

@app.route('/save_config', methods=['POST'])
def save_config():
    if not session.get('is_parent_logged'):
        return jsonify({'status': 'error', 'message': '请先登录！'})
    
    config = {
        "num_min": int(request.form.get('num_min', 1)),
        "num_max": int(request.form.get('num_max', 5)),
        "arithmetic_type": request.form.get('arithmetic_type', 'add'),
        "option_count": int(request.form.get('option_count', 3)),
        "daily_question_count": int(request.form.get('daily_question_count', 5))
    }
    save_parent_config(config)
    
    record = load_daily_answer_record()
    old_total = record["total_question_count"]
    record["total_question_count"] = config["daily_question_count"]
    save_daily_answer_record(record)
    
    return redirect(url_for('parent_config_page', success="配置已保存！"))

# --------------------------
# 答题相关路由
# --------------------------
@app.route('/get_question')
def get_question():
    """获取题目"""
    record = load_daily_answer_record()
    if is_answer_complete(record):
        accuracy = calculate_accuracy(record)
        relay_triggered = record["relay_triggered"]
        weekday_relay = record.get("weekday_relay")
        
        response = {
            'status': 'complete',
            'message': f'今日挑战完成！正确率{accuracy*100:.0f}%',
            'accuracy': accuracy,
            'relay_triggered': relay_triggered,
            'weekday_relay': weekday_relay
        }
        return jsonify(response)
    
    question = generate_question()
    return jsonify({'status': 'success', 'question': question})

@app.route('/submit_answer', methods=['POST'])
def submit_answer():
    """提交答案"""
    data = request.json
    selected_index = data.get('selected_index')
    question = data.get('question')
    
    if not question or 'correct_index' not in question or 'id' not in question:
        return jsonify({'status': 'error', 'message': '题目信息无效，请刷新页面重试'})
    
    record = load_daily_answer_record()
    existing_ids = [q["question_id"] for q in record["answered_questions"]]
    
    if question["id"] in existing_ids:
        return jsonify({'status': 'error', 'message': '该题目已提交过，请勿重复提交！'})
    
    is_correct = selected_index == question["correct_index"]
    current_time = datetime.datetime.now().strftime('%H:%M:%S')
    record["answered_questions"].append({
        "question_id": question["id"],
        "question_content": question["question"],
        "selected_index": selected_index,
        "correct_index": question["correct_index"],
        "is_correct": is_correct,
        "submit_time": current_time
    })
    
    save_success = save_daily_answer_record(record)
    if not save_success:
        return jsonify({'status': 'error', 'message': '答题记录保存失败，请重试！'})
    
    record = load_daily_answer_record()
    answered_count = len(record["answered_questions"])
    total_count = record["total_question_count"]
    current_accuracy = calculate_accuracy(record)
    relay_triggered = record["relay_triggered"]
    
    relay_msg = ""
    relay_num = None
    
    # 答题完成且正确率达标时触发对应星期的继电器
    if is_answer_complete(record) and current_accuracy >= 0.8:
        if not relay_triggered:
            success, detail, start_time, duration, triggered_relay = control_relay_by_weekday(duration=10)
            relay_num = triggered_relay
            
            if success:
                relay_msg = detail
                record["relay_triggered"] = True
                record["weekday_relay"] = triggered_relay
                save_daily_answer_record(record)
                relay_triggered = True
                
                # 更新继电器状态
                relay_state.update({
                    'is_active': True,
                    'start_time': start_time,
                    'duration': duration,
                    'relay_num': triggered_relay
                })
            else:
                relay_msg = detail
        else:
            # 已经触发过
            triggered_relay = record.get("weekday_relay")
            relay_msg = f'🎉 正确率{current_accuracy*100:.0f}%≥80%，星期{triggered_relay}的奖励已开启过～'
    elif is_answer_complete(record):
        relay_msg = f'正确率{current_accuracy*100:.0f}%<80%，继续加油哦！'
    
    if is_correct:
        result_msg = "✅ 答对啦！真厉害～"
    else:
        correct_answer = question["options"][question["correct_index"]]
        result_msg = f"❌ 再想想，正确答案是{correct_answer}"
    normal_msg = f"{result_msg} 当前进度：{answered_count}/{total_count}题"
    
    response = {
        'status': 'success',
        'is_correct': is_correct,
        'current_accuracy': current_accuracy,
        'answered_count': answered_count,
        'total_count': total_count,
        'relay_triggered': relay_triggered,
        'weekday_relay': relay_num,
        'message': relay_msg if relay_msg else normal_msg
    }
    return jsonify(response)

@app.route('/manual_control', methods=['POST'])
def manual_control():
    """手动控制继电器（家长用）"""
    if not session.get('is_parent_logged'):
        return jsonify({"status": "error", "message": "请先登录！"})
    
    data = request.json
    action = data.get('action')
    relay_num = data.get('relay_num')  # 1-5
    duration = int(data.get('duration', 10))
    
    if action == 'on' and relay_num:
        if relay_num not in RELAY_PINS:
            return jsonify({"status": "error", "message": f"无效的继电器编号{relay_num}（仅支持1-5）"})
        
        gpio_pin = RELAY_PINS[relay_num]
        try:
            # 先关闭所有继电器
            stop_all_relays()
            # 开启指定继电器
            GPIO.output(gpio_pin, GPIO.LOW)
            
            relay_state.update({
                'is_active': True,
                'start_time': time.time(),
                'duration': duration,
                'relay_num': relay_num
            })
            
            return jsonify({"status": "success", "message": f"继电器{relay_num}已开启，持续{duration}秒"})
        
        except Exception as e:
            return jsonify({"status": "error", "message": f"开启失败：{str(e)}"})
    
    elif action == 'off':
        stop_all_relays()
        relay_state['is_active'] = False
        return jsonify({"status": "success", "message": "所有继电器已关闭"})
    
    else:
        return jsonify({"status": "error", "message": "无效动作或缺少继电器编号"})

# --------------------------
# 程序入口
# --------------------------
if __name__ == '__main__':
    try:
        import socket
        s = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
        s.connect(("8.8.8.8", 80))
        local_ip = s.getsockname()[0]
        s.close()
        
        weekday = get_current_weekday()
        weekday_names = ["", "星期一", "星期二", "星期三", "星期四", "星期五", "星期六", "星期日"]
        current_weekday_name = weekday_names[weekday] if 1 <= weekday <= 7 else "未知"
        
        print(f"\n" + "="*50)
        print(f"✅ Flask服务启动成功！")
        print(f"📅 当前日期：{current_weekday_name}")
        print(f"📌 答题页面：http://{local_ip}:5000")
        print(f"📌 家长入口：http://{local_ip}:5000/parent")
        print(f"🔑 家长密码：{PARENT_PASSWORD}")
        print(f"🔌 继电器配置：")
        for day, pin in RELAY_PINS.items():
            print(f"     星期{day} -> 继电器{day} -> GPIO{pin}")
        print(f"="*50 + "\n")
        
        app.run(host='0.0.0.0', port=5000, debug=True)
    finally:
        # 程序退出时关闭所有继电器
        print(f"\n⚠️  程序退出，关闭所有继电器...")
        stop_all_relays()
        try:
            GPIO.cleanup()
            print(f"✅ GPIO已清理")
        except Exception as e:
            print(f"❌ GPIO清理失败：{str(e)}")
