from flask import Flask, render_template, request, redirect, url_for, flash, session
import json
import os
import secrets
import bcrypt # 引入 bcrypt 进行密码哈希

app = Flask(__name__)
# 强烈建议使用 secrets.token_hex(16) 或更长的字符串作为密钥
app.secret_key = secrets.token_hex(16) 

# --- 文件配置 ---
TRANSLATIONS_FILE = 'translations.json'
CREDENTIALS_FILE = 'credentials.json' # 新增凭据文件
DEFAULT_LANG = 'zh'

# 区分短文本 (Short) 和长文本 (Long)。长文本在 Admin 界面使用 contenteditable (所见即所得)
S_KEYS = [
    # 网站信息/导航
    'title', 'logo_sub', 'nav_features', 'nav_how_it_works', 'nav_scenes', 
    'nav_contact', 'footer_contact', 'footer_privacy', 'footer_copy',
    # 英雄区标签/按钮
    'hero_cta_primary', 'hero_cta_ghost', 'tag_1', 'tag_2', 'tag_3',
    # 特点标题
    'feat_title', 'feat_1_title', 'feat_2_title', 'feat_3_title', 'feat_4_title', 
    # 工作原理组件名
    'comp_1', 'comp_2', 'comp_3', 'comp_4', 'comp_5', 'airflow_path_title',
    # 场景标题
    'scenes_title', 'scene_1_title', 'scene_2_title', 'scene_3_title', 'scene_4_title',
    # 联系
    'contact_title', 'contact_list_1', 'contact_list_2', 'contact_highlight_1', 'contact_highlight_2'
]


# --- 认证文件操作函数 ---

def load_credentials():
    """从 JSON 文件加载凭据，并处理密码哈希升级"""
    if not os.path.exists(CREDENTIALS_FILE):
        return {}
    try:
        with open(CREDENTIALS_FILE, 'r', encoding='utf-8') as f:
            data = json.load(f)
            # 兼容：如果发现旧的明文密码字段，则哈希并升级
            if 'password' in data and 'password_hash' not in data:
                 print("Found old plaintext password. Hashing and saving...")
                 data['password_hash'] = bcrypt.hashpw(data['password'].encode('utf-8'), bcrypt.gensalt()).decode('utf-8')
                 del data['password']
                 save_credentials(data['username'], data['password_hash'])
            return data
    except Exception:
        return {}

def save_credentials(username, password_hash):
    """将凭据保存回 JSON 文件"""
    data = {
        'username': username,
        'password_hash': password_hash
    }
    try:
        with open(CREDENTIALS_FILE, 'w', encoding='utf-8') as f:
            json.dump(data, f, ensure_ascii=False, indent=4)
        return True
    except Exception as e:
        print(f"An error occurred while saving credentials: {e}")
        return False


# --- 内容文件操作函数 (与上次相似) ---

def load_translations():
    """从 JSON 文件加载翻译数据和语言列表"""
    if not os.path.exists(TRANSLATIONS_FILE):
        return {}, {}
    try:
        with open(TRANSLATIONS_FILE, 'r', encoding='utf-8') as f:
            data = json.load(f)
            return data.get('TRANSLATIONS', {}), data.get('LANG_DATA', {})
    except json.JSONDecodeError:
        print(f"Error decoding JSON from {TRANSLATIONS_FILE}. Returning empty data.")
        return {}, {}
    except Exception as e:
        print(f"An error occurred while loading translations: {e}")
        return {}, {}

def save_translations(translations, lang_data):
    """将翻译数据和语言列表保存回 JSON 文件"""
    data = {
        'TRANSLATIONS': translations,
        'LANG_DATA': lang_data
    }
    try:
        with open(TRANSLATIONS_FILE, 'w', encoding='utf-8') as f:
            json.dump(data, f, ensure_ascii=False, indent=4)
        return True
    except Exception as e:
        print(f"An error occurred while saving translations: {e}")
        return False
        
# --- 认证装饰器和路由 (更新登录逻辑) ---

def login_required(f):
    """自定义装饰器：检查用户是否已登录"""
    def decorated_function(*args, **kwargs):
        if 'logged_in' not in session:
            flash('请先登录才能访问此页面。', 'warning')
            return redirect(url_for('login'))
        return f(*args, **kwargs)
    decorated_function.__name__ = f.__name__
    return decorated_function

@app.route("/login", methods=['GET', 'POST'])
def login():
    if request.method == 'POST':
        username = request.form['username']
        password = request.form['password']
        
        creds = load_credentials()
        
        # 验证用户名和哈希密码
        if (creds and 
            username == creds.get('username') and 
            creds.get('password_hash') and # 确保哈希存在
            bcrypt.checkpw(password.encode('utf-8'), creds['password_hash'].encode('utf-8'))):
            
            session['logged_in'] = True
            flash('登录成功！', 'success')
            return redirect(url_for('admin_chinese_edit')) # 默认跳转到中文编辑页
        else:
            flash('用户名或密码错误。', 'danger')
            
    return render_template('login.html', title="管理员登录")

@app.route("/logout")
def logout():
    session.pop('logged_in', None)
    flash('您已登出。', 'info')
    return redirect(url_for('index'))


# --- 管理页面路由 ---

@app.route("/admin")
@login_required
def admin_panel():
    """主管理页，默认重定向到中文内容编辑"""
    return redirect(url_for('admin_chinese_edit'))

@app.route("/admin/chinese")
@login_required
def admin_chinese_edit():
    """中文内容直观编辑页 (所见即所得)"""
    TRANSLATIONS, LANG_DATA = load_translations()
    return render_template(
        "admin_chinese_edit.html",
        TRANSLATIONS=TRANSLATIONS,
        LANG_DATA=LANG_DATA,
        S_KEYS=S_KEYS,
        title="中文内容直观编辑",
        lang_code='zh' 
    )

@app.route("/admin/save", methods=['POST'])
@login_required
def save_content():
    """处理表单提交并保存内容到 JSON 文件。"""
    TRANSLATIONS, LANG_DATA = load_translations()
    new_translations = TRANSLATIONS.copy()
    
    # 仅保存中文内容 (因为该页面只提交中文)
    submit_lang_code = 'zh'
    
    for key in new_translations.keys():
        field_name = f"{key}_{submit_lang_code}"
        new_value = request.form.get(field_name)
        
        if new_value is not None:
            # 对于短文本 (非富文本)，清理空格，长文本保留 HTML 格式
            if key in S_KEYS:
                new_translations[key][submit_lang_code] = new_value.strip()
            else:
                new_translations[key][submit_lang_code] = new_value
                
    if save_translations(new_translations, LANG_DATA):
        flash('中文内容已成功保存！', 'success')
        return redirect(url_for('admin_chinese_edit'))
    else:
        flash('保存失败，请检查文件权限。', 'danger')
        return redirect(url_for('admin_chinese_edit'))


# --- 凭据管理路由：允许用户修改用户名和密码 ---

@app.route("/admin/settings", methods=['GET', 'POST'])
@login_required
def admin_settings():
    """允许用户修改用户名和密码的页面"""
    creds = load_credentials()
    
    if request.method == 'POST':
        new_username = request.form.get('username')
        current_password = request.form.get('current_password')
        new_password = request.form.get('new_password')
        
        # 1. 验证当前密码
        if not creds or not bcrypt.checkpw(current_password.encode('utf-8'), creds.get('password_hash', '').encode('utf-8')):
            flash('当前密码错误，操作失败。', 'danger')
            return redirect(url_for('admin_settings'))

        # 2. 生成新的哈希密码 (如果提供了新密码)
        new_password_hash = creds.get('password_hash')
        if new_password:
             new_password_hash = bcrypt.hashpw(new_password.encode('utf-8'), bcrypt.gensalt()).decode('utf-8')
        
        # 3. 保存新凭据
        if save_credentials(new_username, new_password_hash):
            flash('用户名和密码已更新成功！', 'success')
            return redirect(url_for('admin_settings'))
        else:
            flash('保存失败，请检查文件权限。', 'danger')
            return redirect(url_for('admin_settings'))

    return render_template(
        'settings.html',
        current_username=creds.get('username', '未知'),
        title="管理员设置"
    )

# --- 前台页面路由 (与上次相同，为节省篇幅省略) ---
@app.route("/", defaults={'lang_code': DEFAULT_LANG})
@app.route("/<lang_code>")
def index(lang_code):
    TRANSLATIONS, LANG_DATA = load_translations()
    if lang_code not in LANG_DATA:
        return redirect(url_for('index', lang_code=DEFAULT_LANG))
    t = {key: trans.get(lang_code, f"Missing Translation: {key}[{lang_code}]") 
         for key, trans in TRANSLATIONS.items()}
    return render_template(
        "index.html",
        t=t,
        lang_code=lang_code,
        lang_data=LANG_DATA,
        current_year=2025
    )
@app.route("/change_lang", methods=['POST'])
def change_lang():
    TRANSLATIONS, LANG_DATA = load_translations()
    new_lang_code = request.form.get('lang_code')
    if new_lang_code in LANG_DATA:
        return redirect(url_for('index', lang_code=new_lang_code))
    return redirect(url_for('index', lang_code=DEFAULT_LANG))


if __name__ == "__main__":
    load_credentials() # 确保在运行前检查凭据文件
    app.run(host="0.0.0.0", port=5000, debug=True)
