from flask import Flask, render_template, request, redirect, url_for, flash, session, jsonify
import json
import os
import secrets
import bcrypt
from werkzeug.utils import secure_filename

app = Flask(__name__)
app.secret_key = secrets.token_hex(16) 

# --- 配置 ---
TRANSLATIONS_FILE = 'translations.json'
CREDENTIALS_FILE = 'credentials.json'
UPLOAD_FOLDER = 'static/uploads' 
ALLOWED_EXTENSIONS = {'png', 'jpg', 'jpeg', 'gif', 'webp'}

app.config['UPLOAD_FOLDER'] = UPLOAD_FOLDER
DEFAULT_LANG = 'zh'
FALLBACK_LANG = 'en'

os.makedirs(app.config['UPLOAD_FOLDER'], exist_ok=True)

def allowed_file(filename):
    return '.' in filename and filename.rsplit('.', 1)[1].lower() in ALLOWED_EXTENSIONS

# 字段列表
S_KEYS = [
    'title', 'logo_sub', 'nav_features', 'nav_how_it_works', 'nav_scenes', 
    'nav_contact', 'footer_contact', 'footer_privacy', 'footer_copy', 'meta_desc',
    'hero_cta_primary', 'hero_cta_ghost', 'tag_1', 'tag_2', 'tag_3',
    'feat_title', 'feat_1_title', 'feat_2_title', 'feat_3_title', 'feat_4_title', 
    'comp_1', 'comp_2', 'comp_3', 'comp_4', 'comp_5', 'airflow_path_title',
    'scenes_title', 'scene_1_title', 'scene_2_title', 'scene_3_title', 'scene_4_title',
    'contact_title', 'contact_email_text', 'contact_email_link', 
    'contact_customization_text', 'contact_highlight_1', 'contact_highlight_2'
]

L_KEYS = [
    'hero_headline', 'hero_sub', 
    'feat_1_desc', 'feat_2_desc', 'feat_3_desc', 'feat_4_desc', 
    'scene_1_desc', 'scene_2_desc', 'scene_3_desc', 'scene_4_desc',
    'contact_subtitle'
]

ALL_KEYS = S_KEYS + L_KEYS 

# --- 辅助函数 ---
def load_credentials():
    if not os.path.exists(CREDENTIALS_FILE): return {}
    try:
        with open(CREDENTIALS_FILE, 'r', encoding='utf-8') as f:
            data = json.load(f)
            if 'password' in data and 'password_hash' not in data:
                 data['password_hash'] = bcrypt.hashpw(data['password'].encode('utf-8'), bcrypt.gensalt()).decode('utf-8')
                 del data['password']
                 save_credentials(data['username'], data['password_hash'])
            return data
    except Exception: return {}

def save_credentials(username, password_hash):
    data = {'username': username, 'password_hash': password_hash}
    try:
        with open(CREDENTIALS_FILE, 'w', encoding='utf-8') as f:
            json.dump(data, f, ensure_ascii=False, indent=4)
        return True
    except Exception: return False

def load_translations():
    DEFAULT_LANG_DATA = {DEFAULT_LANG: {'name': '中文', 'flag': '🇨🇳'}, FALLBACK_LANG: {'name': 'English', 'flag': '🇺🇸'}}
    if not os.path.exists(TRANSLATIONS_FILE): return {}, DEFAULT_LANG_DATA
    try:
        with open(TRANSLATIONS_FILE, 'r', encoding='utf-8') as f:
            data = json.load(f)
            return data.get('TRANSLATIONS', {}), data.get('LANG_DATA', DEFAULT_LANG_DATA)
    except Exception: return {}, DEFAULT_LANG_DATA

def save_translations(translations, lang_data):
    data = {'TRANSLATIONS': translations, 'LANG_DATA': lang_data}
    try:
        with open(TRANSLATIONS_FILE, 'w', encoding='utf-8') as f:
            json.dump(data, f, ensure_ascii=False, indent=4)
        return True
    except Exception: return False
        
def login_required(f):
    def decorated_function(*args, **kwargs):
        if 'logged_in' not in session:
            flash('请先登录才能访问此页面。', 'warning')
            return redirect(url_for('login'))
        return f(*args, **kwargs)
    decorated_function.__name__ = f.__name__
    return decorated_function

# --- 路由 ---
@app.route('/admin/upload_image', methods=['POST'])
@login_required
def upload_image():
    if 'file' not in request.files: return jsonify({'error': 'No file'}), 400
    file = request.files['file']
    if file.filename == '': return jsonify({'error': 'No file'}), 400
    if file and allowed_file(file.filename):
        filename = secure_filename(file.filename)
        name, ext = os.path.splitext(filename)
        unique_filename = f"{name}_{secrets.token_hex(4)}{ext}"
        file.save(os.path.join(app.config['UPLOAD_FOLDER'], unique_filename))
        return jsonify({'location': url_for('static', filename=f'uploads/{unique_filename}')})
    return jsonify({'error': 'Invalid file'}), 400

@app.route('/admin/images', methods=['GET'])
@login_required
def get_images():
    images = []
    try:
        files = sorted(os.listdir(app.config['UPLOAD_FOLDER']), key=lambda x: os.path.getmtime(os.path.join(app.config['UPLOAD_FOLDER'], x)), reverse=True)
        for filename in files:
            if allowed_file(filename):
                images.append({
                    'filename': filename,
                    'url': url_for('static', filename=f'uploads/{filename}')
                })
    except FileNotFoundError: pass
    return jsonify(images)

@app.route('/admin/delete_image', methods=['POST'])
@login_required
def delete_image():
    data = request.get_json()
    filename = data.get('filename')
    if not filename: return jsonify({'error': 'Missing filename'}), 400
    safe_filename = secure_filename(filename) 
    file_path = os.path.join(app.config['UPLOAD_FOLDER'], safe_filename)
    if os.path.exists(file_path):
        try:
            os.remove(file_path)
            return jsonify({'success': True})
        except Exception as e: return jsonify({'error': str(e)}), 500
    return jsonify({'error': 'File not found'}), 404

@app.route("/login", methods=['GET', 'POST'])
def login():
    if request.method == 'POST':
        username = request.form['username']
        password = request.form['password']
        creds = load_credentials()
        if not creds:
             default_hash = bcrypt.hashpw('adminpass'.encode('utf-8'), bcrypt.gensalt()).decode('utf-8')
             save_credentials('admin', default_hash)
             creds = load_credentials()
        if (creds and username == creds.get('username') and creds.get('password_hash') and bcrypt.checkpw(password.encode('utf-8'), creds['password_hash'].encode('utf-8'))):
            session['logged_in'] = True
            return redirect(url_for('admin_chinese_edit'))
        else:
            flash('用户名或密码错误。', 'danger')
    return render_template('login.html', title="管理员登录")

@app.route("/logout")
def logout():
    session.pop('logged_in', None)
    flash('您已登出。', 'info')
    return redirect(url_for('index'))

@app.route("/admin")
@login_required
def admin_panel():
    return redirect(url_for('admin_chinese_edit'))

@app.route("/admin/chinese")
@login_required
def admin_chinese_edit():
    TRANSLATIONS, LANG_DATA = load_translations()
    for key in ALL_KEYS:
        if key not in TRANSLATIONS:
            TRANSLATIONS[key] = {lang: f"[占位符] {key}" for lang in LANG_DATA.keys()}
            
    grouped_keys = {}
    grouped_keys['合作与订购 (联系方式)'] = [k for k in ALL_KEYS if k.startswith('contact_')]
    grouped_keys['首页顶部 (Hero)'] = [k for k in ALL_KEYS if k.startswith(('hero_', 'tag_'))]
    grouped_keys['核心特点与工作原理'] = [k for k in ALL_KEYS if k.startswith(('feat_', 'comp_', 'airflow_'))]
    grouped_keys['使用场景'] = [k for k in ALL_KEYS if k.startswith('scene_')]
    grouped_keys['网站通用信息'] = [k for k in ALL_KEYS if k.startswith(('title', 'logo_sub', 'meta_desc', 'nav_', 'footer_'))]
    
    return render_template("admin_chinese_edit.html", TRANSLATIONS=TRANSLATIONS, LANG_DATA=LANG_DATA, S_KEYS=S_KEYS, title="中文内容直观编辑", lang_code='zh', grouped_keys=grouped_keys)

@app.route("/admin/save", methods=['POST'])
@login_required
def save_content():
    TRANSLATIONS, LANG_DATA = load_translations()
    new_translations = TRANSLATIONS.copy()
    submit_lang_code = 'zh'
    for key in ALL_KEYS: 
        field_name = f"{key}_{submit_lang_code}"
        new_value = request.form.get(field_name)
        if new_value is not None:
            if key not in new_translations: new_translations[key] = {}
            new_translations[key][submit_lang_code] = new_value.strip() if key in S_KEYS else new_value
    if save_translations(new_translations, LANG_DATA):
        flash('保存成功！', 'success')
        return redirect(url_for('admin_chinese_edit'))
    else:
        flash('保存失败。', 'danger')
        return redirect(url_for('admin_chinese_edit'))

@app.route("/admin/settings", methods=['GET', 'POST'])
@login_required
def admin_settings():
    creds = load_credentials()
    if request.method == 'POST':
        new_username = request.form.get('username')
        current_password = request.form.get('current_password')
        new_password = request.form.get('new_password')
        if not creds or not bcrypt.checkpw(current_password.encode('utf-8'), creds.get('password_hash', '').encode('utf-8')):
            flash('当前密码错误。', 'danger')
            return redirect(url_for('admin_settings'))
        new_password_hash = creds.get('password_hash')
        if new_password:
             new_password_hash = bcrypt.hashpw(new_password.encode('utf-8'), bcrypt.gensalt()).decode('utf-8')
        if save_credentials(new_username, new_password_hash):
            flash('更新成功！', 'success')
            return redirect(url_for('admin_settings'))
    return render_template('settings.html', current_username=creds.get('username', '未知'), title="管理员设置")

@app.route("/", defaults={'lang_code': 'auto'})
@app.route("/<lang_code>")
def index(lang_code):
    TRANSLATIONS, LANG_DATA = load_translations()
    SUPPORTED_LANGS = LANG_DATA.keys()
    current_lang = DEFAULT_LANG 
    if lang_code == 'auto':
        preferred = request.accept_languages.best_match(SUPPORTED_LANGS)
        current_lang = preferred if preferred else (FALLBACK_LANG if FALLBACK_LANG in SUPPORTED_LANGS else DEFAULT_LANG)
    elif lang_code in SUPPORTED_LANGS:
        current_lang = lang_code 
    else:
        target = FALLBACK_LANG if FALLBACK_LANG in SUPPORTED_LANGS else DEFAULT_LANG
        return redirect(url_for('index', lang_code=target))
    t = {key: trans.get(current_lang, trans.get(FALLBACK_LANG, f"Missing: {key}")) for key, trans in TRANSLATIONS.items()}
    return render_template("index.html", t=t, lang_code=current_lang, lang_data=LANG_DATA, current_year=2025)

@app.route("/change_lang", methods=['POST'])
def change_lang():
    return redirect(url_for('index', lang_code=request.form.get('lang_code')))

if __name__ == "__main__":
    load_credentials()
    app.run(host="0.0.0.0", port=5000, debug=True)
