# backend/app.py (v1.6 - 修复 sqlite3.OperationalError)
import sqlite3
from datetime import datetime, timedelta
from flask import Flask, render_template, request, jsonify, g, redirect, url_for, make_response
from flask_login import LoginManager, UserMixin, login_user, logout_user, login_required, current_user
from werkzeug.security import generate_password_hash, check_password_hash
import os, json, uuid
from flask_cors import CORS
from functools import wraps
import sys 
from collections import defaultdict
import time
import requests

# --- REVISED PATH LOGIC FOR NUITHA ---
if getattr(sys, 'frozen', False):
    # === 编译 (Nuitka Onefile) 环境 ===
    base_dir = os.path.dirname(os.path.abspath(sys.executable))
    
    template_dir = os.path.join(base_dir, 'templates')
    static_dir = os.path.join(base_dir, 'static')
    DATABASE = os.path.join(base_dir, 'fitness_manager.db')
    print("--- Detected Nuitka Onefile Path ---")
    print(f"--- **DATABASE is PERMANENT at {DATABASE}** ---")
else:
    # === 开发环境 (Python 脚本运行) ===
    app_file_path = os.path.abspath(__file__)
    backend_dir = os.path.dirname(app_file_path)
    project_root_dir = os.path.dirname(backend_dir)
    
    template_dir = os.path.join(project_root_dir, 'frontend', 'templates')
    static_dir = os.path.join(project_root_dir, 'frontend', 'static')
    DATABASE = os.path.join(backend_dir, 'fitness_manager.db')
    print("--- Detected Development Path ---")

print("Template folder:", template_dir)
print("Static folder:", static_dir)
print("DB path:", DATABASE)

app = Flask(__name__, template_folder=template_dir, static_folder=static_dir)
CORS(app)

# -------------------- 配置 --------------------
# ⚠️ 确保这是一个安全且足够长的 SECRET_KEY
app.config['SECRET_KEY'] = os.environ.get('SECRET_KEY', str(uuid.uuid4())) 
app.config['JSON_AS_ASCII'] = False # 确保 JSON 响应中的中文不被转义

# -------------------- DB --------------------
def get_db():
    db = getattr(g, '_database', None)
    if db is None:
        db = g._database = sqlite3.connect(DATABASE)
        db.row_factory = sqlite3.Row
    return db

@app.teardown_appcontext
def close_connection(exception):
    db = getattr(g, '_database', None)
    if db is not None:
        db.close()

def init_db():
    with app.app_context():
        db = get_db()
        cur = db.cursor()
        
        # 用户表
        cur.execute("""
            CREATE TABLE IF NOT EXISTS users (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                username TEXT UNIQUE NOT NULL,
                password_hash TEXT NOT NULL,
                is_admin INTEGER NOT NULL DEFAULT 0
            )
        """)
        
        # 日常待办表
        cur.execute("""
            CREATE TABLE IF NOT EXISTS todos (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                user_id INTEGER NOT NULL,
                date TEXT NOT NULL,
                content TEXT NOT NULL,
                is_completed INTEGER NOT NULL DEFAULT 0,
                FOREIGN KEY (user_id) REFERENCES users (id)
            )
        """)

        # 每日健身计划表 (单个训练条目)
        cur.execute("""
            CREATE TABLE IF NOT EXISTS fitness_plan (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                user_id INTEGER NOT NULL,
                date TEXT NOT NULL,
                exercise_name TEXT NOT NULL,
                details TEXT,
                is_completed INTEGER NOT NULL DEFAULT 0,
                FOREIGN KEY (user_id) REFERENCES users (id)
            )
        """)
        
        # 循环健身计划表 (模板)
        cur.execute("""
            CREATE TABLE IF NOT EXISTS recurring_plans (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                user_id INTEGER NOT NULL,
                plan_name TEXT NOT NULL,
                exercises TEXT NOT NULL, -- JSON array of exercises
                repeat_interval TEXT NOT NULL, -- e.g., 'daily', 'weekly'
                repeat_day INTEGER, -- 1-7 for weekly, NULL for daily
                start_date TEXT NOT NULL,
                end_date TEXT,
                FOREIGN KEY (user_id) REFERENCES users (id)
            )
        """)

        # 循环待办表 (模板)
        cur.execute("""
            CREATE TABLE IF NOT EXISTS recurring_todos (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                user_id INTEGER NOT NULL,
                content TEXT NOT NULL,
                repeat_interval TEXT NOT NULL, -- e.g., 'daily', 'weekly'
                repeat_day INTEGER, -- 1-7 for weekly, NULL for daily
                start_date TEXT NOT NULL,
                end_date TEXT,
                FOREIGN KEY (user_id) REFERENCES users (id)
            )
        """)

        # === 数据库迁移/兼容性检查：确保所有表具有新版本所需的列 ===

        # 1. 修复用户表缺少 password_hash 错误
        cur.execute("PRAGMA table_info(users)")
        cols = [col[1] for col in cur.fetchall()]
        if 'password_hash' not in cols:
             # ALTER TABLE 允许添加新列，但对于已有的行，新列值将为 NULL。
             # 由于旧表可能存在用户，我们必须先添加列，再插入/更新数据。
             try:
                 cur.execute("ALTER TABLE users ADD COLUMN password_hash TEXT")
                 print("--- DB MIGRATION: users.password_hash column added. ---")
             except sqlite3.OperationalError as e:
                 # 防止因其他原因导致的错误
                 print(f"--- DB MIGRATION ERROR: Failed to add password_hash column: {e} ---")
                 pass


        # 2. 确保 todos 表有 is_completed 列
        cur.execute("PRAGMA table_info(todos)")
        cols = [col[1] for col in cur.fetchall()]
        if 'is_completed' not in cols:
             cur.execute("ALTER TABLE todos ADD COLUMN is_completed INTEGER NOT NULL DEFAULT 0")

        # 3. 确保 fitness_plan 表有 is_completed 列
        cur.execute("PRAGMA table_info(fitness_plan)")
        cols = [col[1] for col in cur.fetchall()]
        if 'is_completed' not in cols:
             cur.execute("ALTER TABLE fitness_plan ADD COLUMN is_completed INTEGER NOT NULL DEFAULT 0")
             
        # 插入默认管理员用户 (如果不存在)
        try:
            cur.execute("INSERT INTO users (username, password_hash, is_admin) VALUES (?, ?, ?)", 
                        ('superuser', generate_password_hash('adminpassword'), 1))
            print("--- Added default superuser: superuser / adminpassword ---")
        except sqlite3.IntegrityError:
            pass # 用户已存在
        except sqlite3.OperationalError as e:
            # 如果在迁移后仍然遇到此错误，说明数据库文件可能已被损坏或权限问题
            print(f"--- FATAL DB ERROR: Cannot insert default user after migration: {e} ---")
            
        # 插入默认普通用户 (如果不存在)
        try:
            cur.execute("INSERT INTO users (username, password_hash, is_admin) VALUES (?, ?, ?)", 
                        ('testuser', generate_password_hash('password'), 0))
            print("--- Added default testuser: testuser / password ---")
        except sqlite3.IntegrityError:
            pass # 用户已存在
        except sqlite3.OperationalError as e:
            pass # 忽略第二个用户的插入错误，如果在第一个用户处已打印 FATAL ERROR

        db.commit()

# -------------------- Flask-Login 配置 --------------------
class User(UserMixin):
    def __init__(self, id, username, is_admin):
        self.id = id
        self.username = username
        self.is_admin = is_admin

login_manager = LoginManager()
login_manager.init_app(app)
login_manager.login_view = 'login' 

@login_manager.user_loader
def load_user(user_id):
    db = get_db()
    user_record = db.execute('SELECT id, username, is_admin FROM users WHERE id = ?', (user_id,)).fetchone()
    if user_record:
        return User(user_record['id'], user_record['username'], user_record['is_admin'])
    return None

@login_manager.unauthorized_handler
def unauthorized():
    return redirect(url_for('login'))

# -------------------- 权限检查装饰器 --------------------
def admin_required(f):
    @wraps(f)
    def decorated_function(*args, **kwargs):
        if not current_user.is_authenticated or not current_user.is_admin:
            # 对于 API 接口，返回 JSON 错误
            if request.path.startswith('/api/') or request.path.startswith('/delete_'):
                 return jsonify(status='error', message='需要管理员权限'), 403
            # 对于页面，重定向回首页或登录页
            return redirect(url_for('index'))
        return f(*args, **kwargs)
    return decorated_function

# -------------------- 路由: 页面 --------------------

# 首页 (数字标牌)
@app.route('/', methods=['GET'])
def index():
    return render_template('display.html')

# 手机配置页面 (不强制登录，但里面的 API 调用会受限)
@app.route('/mobile_config', methods=['GET'])
def mobile_config():
    return render_template('mobile_config.html')

# 后台管理页面
@app.route('/manage', methods=['GET'])
@login_required
def manage():
    return render_template('manage.html')

@app.route('/login', methods=['GET', 'POST'])
def login():
    # GET 逻辑: 渲染登录页面
    if request.method == 'GET':
        if current_user.is_authenticated:
            # 如果已登录，重定向到首页
            return redirect(url_for('index'))
        return render_template('login.html')
    
    # POST 逻辑: 处理表单提交
    elif request.method == 'POST':
        # 尝试从 JSON 或表单数据中获取
        if request.is_json:
             data = request.get_json()
             username = data.get('username')
             password = data.get('password')
        else:
             username = request.form.get('username')
             password = request.form.get('password')

        db = get_db()
        user_record = db.execute('SELECT id, username, is_admin, password_hash FROM users WHERE username = ?', (username,)).fetchone()
        
        # 检查 user_record 是否包含 password_hash 字段（避免旧数据没有该字段导致的错误）
        if user_record and user_record['password_hash'] and check_password_hash(user_record['password_hash'], password):
            user_obj = User(user_record['id'], user_record['username'], user_record['is_admin'])
            login_user(user_obj)
            return jsonify(status='success', message='登录成功')
        else:
            # 如果 user_record 存在但 password_hash 为 None，表示是旧用户数据，需要管理员登录重设。
            if user_record and not user_record['password_hash']:
                return jsonify(status='error', message='该账户为旧版本数据，请联系管理员重设密码'), 401
                
            return jsonify(status='error', message='用户名或密码错误'), 401

@app.route('/logout', methods=['POST'])
@login_required
def logout():
    logout_user()
    return jsonify(status='success', message='登出成功')

# -------------------- API: 用户管理 --------------------

# 获取当前用户信息
@app.route('/api/current_user', methods=['GET'])
@login_required
def api_current_user():
    return jsonify({
        'status': 'success',
        'id': current_user.id,
        'username': current_user.username,
        'is_admin': current_user.is_admin
    })
    
# 管理员：获取所有用户列表
@app.route('/api/users', methods=['GET'])
@admin_required
def api_get_users():
    db = get_db()
    users = db.execute('SELECT id, username, is_admin, password_hash FROM users').fetchall()
    return jsonify({
        'status': 'success',
        'users': [dict(u) for u in users]
    })

# 管理员：删除用户及所有数据 (使用 POST 模拟 DELETE 以简化前端)
@app.route('/api/users/<int:user_id>', methods=['POST'])
@admin_required
def api_delete_user(user_id):
    if user_id == current_user.id:
        return jsonify(status='error', message='不能删除当前登录的管理员账户'), 400
    
    db = get_db()
    cur = db.cursor()
    try:
        # 删除用户相关的所有数据 (待办, 每日计划, 循环计划)
        cur.execute("DELETE FROM todos WHERE user_id = ?", (user_id,))
        cur.execute("DELETE FROM fitness_plan WHERE user_id = ?", (user_id,))
        cur.execute("DELETE FROM recurring_plans WHERE user_id = ?", (user_id,))
        cur.execute("DELETE FROM recurring_todos WHERE user_id = ?", (user_id,))
        
        # 最后删除用户本身
        cur.execute("DELETE FROM users WHERE id = ?", (user_id,))
        db.commit()

        if cur.rowcount > 0: 
            return jsonify(status='success', message='用户及其所有数据已删除')
        else: 
            return jsonify(status='error', message='未找到用户'), 404
            
    except sqlite3.Error as e:
        db.rollback()
        return jsonify(status='error', message=f'数据库错误: {e}'), 500

# -------------------- API: 待办事项 --------------------

# 获取待办事项 (特定日期)
@app.route('/todos', methods=['GET'])
# @login_required # 可以允许看板在不登录的情况下获取数据，但由权限控制决定
def get_todos():
    date_str = request.args.get('date')
    if not date_str:
        return jsonify(status='error', message='缺少日期参数'), 400

    # 看板模式下，获取所有用户的待办事项
    if not current_user.is_authenticated:
        user_id_filter = ""
        user_id_param = []
    # 后台管理模式下，只显示当前用户的待办事项
    else:
        user_id_filter = " AND user_id = ?"
        user_id_param = [current_user.id]

    db = get_db()
    
    # 1. 查询日常待办
    daily_todos = db.execute(f"""
        SELECT t.id, t.content, t.is_completed, u.username
        FROM todos t 
        JOIN users u ON t.user_id = u.id
        WHERE t.date = ? {user_id_filter}
        ORDER BY t.id DESC
    """, (date_str, *user_id_param)).fetchall()

    # 2. 查询和生成循环待办 (仅对当前用户生成，或者如果未登录则不处理)
    recurring_todos = []
    if current_user.is_authenticated:
        today_date = datetime.strptime(date_str, '%Y-%m-%d').date()
        current_day_of_week = today_date.isoweekday() # 1=Mon, 7=Sun

        recurring_records = db.execute("""
            SELECT id, content, repeat_interval, repeat_day, start_date, end_date
            FROM recurring_todos 
            WHERE user_id = ? AND start_date <= ? AND (end_date IS NULL OR end_date >= ?)
        """, (current_user.id, date_str, date_str)).fetchall()
        
        for record in recurring_records:
            is_valid = False
            if record['repeat_interval'] == 'daily':
                is_valid = True
            elif record['repeat_interval'] == 'weekly' and record['repeat_day'] == current_day_of_week:
                is_valid = True
            
            if is_valid:
                # 检查该循环待办是否已被当天手动添加/完成 (防止重复显示)
                is_already_daily = db.execute("SELECT 1 FROM todos WHERE user_id = ? AND date = ? AND content = ? LIMIT 1", 
                                                (current_user.id, date_str, record['content'])).fetchone()
                if not is_already_daily:
                    recurring_todos.append({
                        'id': f'rec-{record["id"]}', # 标记为循环待办
                        'content': f"【循环】{record['content']}",
                        'is_completed': 0,
                        'username': current_user.username
                    })

    # 合并日常待办和循环待办
    all_todos = [dict(t) for t in daily_todos] + recurring_todos
    
    return jsonify({
        'status': 'success',
        'todos': all_todos
    })


# 添加待办事项 (POST)
@app.route('/add_todo', methods=['POST'])
@login_required
def add_todo():
    data = request.get_json()
    date_str = data.get('date')
    content = data.get('content')
    
    if not date_str or not content:
        return jsonify(status='error', message='缺少日期或内容'), 400

    try:
        db = get_db()
        db.execute('INSERT INTO todos (user_id, date, content) VALUES (?, ?, ?)', 
                   (current_user.id, date_str, content))
        db.commit()
        return jsonify(status='success', message='待办已添加')
    except sqlite3.Error as e:
        return jsonify(status='error', message=f'数据库错误: {e}'), 500

# 标记待办事项完成状态 (POST)
@app.route('/mark_todo_completed', methods=['POST'])
@login_required
def mark_todo_completed():
    data = request.get_json()
    todo_id = data.get('todo_id')
    is_completed = data.get('is_completed') # 0 或 1
    
    # 检查是否是循环待办 (rec-前缀)
    if isinstance(todo_id, str) and todo_id.startswith('rec-'):
        # 循环待办，需要作为日常待办新增，并标记完成
        rec_id = int(todo_id.split('-')[1])
        db = get_db()
        rec_todo = db.execute("SELECT content FROM recurring_todos WHERE id = ? AND user_id = ?", (rec_id, current_user.id)).fetchone()
        
        if rec_todo:
            # 获取当前待办列表中的日期 (由于是从看板/手机端触发，通常是当前日期)
            date_str = datetime.now().strftime('%Y-%m-%d')
            content_text = rec_todo['content']
            
            # 检查是否已存在 (以防重复提交)
            existing_todo = db.execute("SELECT id, is_completed FROM todos WHERE user_id = ? AND date = ? AND content = ?",
                                       (current_user.id, date_str, content_text)).fetchone()

            if existing_todo:
                # 已存在，直接更新其完成状态
                db.execute('UPDATE todos SET is_completed = ? WHERE id = ?', (is_completed, existing_todo['id']))
            else:
                # 不存在，插入新的日常待办并标记完成
                db.execute('INSERT INTO todos (user_id, date, content, is_completed) VALUES (?, ?, ?, ?)', 
                           (current_user.id, date_str, content_text, is_completed))
            
            db.commit()
            return jsonify(status='success', message='循环待办已转为日常待办并标记完成')
        else:
            return jsonify(status='error', message='未找到循环待办'), 404
            
    # 日常待办
    if not isinstance(todo_id, int):
        try: todo_id = int(todo_id)
        except ValueError: return jsonify(status='error', message='待办 ID 格式错误'), 400

    try:
        db = get_db()
        cur = db.cursor()
        cur.execute('UPDATE todos SET is_completed = ? WHERE id = ? AND user_id = ?', 
                   (is_completed, todo_id, current_user.id))
        db.commit()
        if cur.rowcount > 0: return jsonify(status='success', message='待办状态已更新')
        else: return jsonify(status='error', message='未找到待办或用户不匹配'), 404
    except sqlite3.Error as e:
        return jsonify(status='error', message=f'数据库错误: {e}'), 500

# 删除待办事项 (POST)
@app.route('/delete_todo/<int:todo_id>', methods=['POST'])
@login_required
def delete_todo(todo_id):
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute('DELETE FROM todos WHERE id = ? AND user_id = ?', (todo_id, current_user.id))
        db.commit()
        if cur.rowcount > 0:
            return jsonify(status='success', message='待办已删除')
        else:
            return jsonify(status='error', message='未找到待办'), 404
    except sqlite3.Error as e:
        return jsonify(status='error', message=f'数据库错误: {e}'), 500

# -------------------- API: 健身计划 (日常) --------------------

# 获取每日健身计划 (特定日期)
@app.route('/fitness_plan/<string:date_str>', methods=['GET'])
# @login_required
def get_fitness_plan(date_str):
    # 看板模式下，获取所有用户的计划
    if not current_user.is_authenticated:
        user_id_filter = ""
        user_id_param = []
    # 后台管理模式下，只显示当前用户的计划
    else:
        user_id_filter = " AND fp.user_id = ?"
        user_id_param = [current_user.id]

    db = get_db()
    
    # 1. 查询日常健身计划
    daily_plans = db.execute(f"""
        SELECT fp.id, fp.date, fp.exercise_name, fp.details, fp.is_completed, u.username
        FROM fitness_plan fp
        JOIN users u ON fp.user_id = u.id
        WHERE fp.date = ? {user_id_filter}
        ORDER BY fp.id DESC
    """, (date_str, *user_id_param)).fetchall()

    # 2. 查询和生成循环计划 (仅对当前用户生成，或者如果未登录则不处理)
    recurring_plans = []
    if current_user.is_authenticated:
        today_date = datetime.strptime(date_str, '%Y-%m-%d').date()
        current_day_of_week = today_date.isoweekday() # 1=Mon, 7=Sun

        recurring_records = db.execute("""
            SELECT id, plan_name, exercises, repeat_interval, repeat_day, start_date, end_date
            FROM recurring_plans 
            WHERE user_id = ? AND start_date <= ? AND (end_date IS NULL OR end_date >= ?)
        """, (current_user.id, date_str, date_str)).fetchall()

        for record in recurring_records:
            is_valid = False
            if record['repeat_interval'] == 'daily':
                is_valid = True
            elif record['repeat_interval'] == 'weekly' and record['repeat_day'] == current_day_of_week:
                is_valid = True
            
            if is_valid:
                # 解析 exercises 字符串
                try:
                    exercises = json.loads(record['exercises'])
                except json.JSONDecodeError:
                    continue # 跳过无效记录

                for i, ex in enumerate(exercises):
                    exercise_name = ex.get('name', f"训练 {i+1}")
                    details = ex.get('details', '')
                    
                    # 检查该循环计划是否已被当天手动添加/完成 (防止重复显示)
                    is_already_daily = db.execute("SELECT 1 FROM fitness_plan WHERE user_id = ? AND date = ? AND exercise_name = ? AND details = ? LIMIT 1", 
                                                    (current_user.id, date_str, exercise_name, details)).fetchone()
                    if not is_already_daily:
                        recurring_plans.append({
                            'id': f'rec-{record["id"]}-{i}', # 标记为循环计划 + 索引
                            'exercise_name': f"【循环】{exercise_name} ({record['plan_name']})",
                            'details': details,
                            'is_completed': 0,
                            'username': current_user.username
                        })

    # 合并日常计划和循环计划
    all_plans = [dict(p) for p in daily_plans] + recurring_plans
    
    # 按用户分组，以支持看板多用户显示
    grouped_plans = defaultdict(list)
    for p in all_plans:
        # 将 date 字段从结果中移除，因为它已用于分组
        # p.pop('date', None) 
        grouped_plans[p.get('username', '未知用户')].append(p)
        
    final_output = []
    for username, plans in grouped_plans.items():
        # 检查是否所有计划都已完成
        all_completed = all(p['is_completed'] == 1 for p in plans if not isinstance(p['id'], str) or not p['id'].startswith('rec-'))
        final_output.append({
            'username': username,
            'is_all_completed': all_completed,
            'plans': plans
        })

    return jsonify({
        'status': 'success',
        'date': date_str,
        'plans': final_output
    })

# 添加每日健身计划 (POST)
@app.route('/add_daily_plan', methods=['POST'])
@login_required
def add_daily_plan():
    data = request.get_json()
    date_str = data.get('date')
    exercise_name = data.get('exercise_name')
    details = data.get('details')
    
    if not date_str or not exercise_name:
        return jsonify(status='error', message='缺少日期或训练名称'), 400

    try:
        db = get_db()
        db.execute('INSERT INTO fitness_plan (user_id, date, exercise_name, details, is_completed) VALUES (?, ?, ?, ?, 0)', 
                   (current_user.id, date_str, exercise_name, details))
        db.commit()
        return jsonify(status='success', message='每日计划已添加')
    except sqlite3.Error as e:
        return jsonify(status='error', message=f'数据库错误: {e}'), 500

# 标记每日健身计划条目完成状态 (POST)
@app.route('/mark_daily_plan_completed', methods=['POST'])
@login_required
def mark_daily_plan_completed():
    data = request.get_json()
    plan_id = data.get('plan_id')
    is_completed = data.get('is_completed') # 0 或 1

    # 检查是否是循环计划 (rec-前缀)
    if isinstance(plan_id, str) and plan_id.startswith('rec-'):
        # 循环计划，需要作为日常计划新增，并标记完成
        parts = plan_id.split('-')
        rec_id = int(parts[1])
        ex_index = int(parts[2])

        db = get_db()
        rec_plan = db.execute("SELECT plan_name, exercises FROM recurring_plans WHERE id = ? AND user_id = ?", (rec_id, current_user.id)).fetchone()
        
        if rec_plan:
            try:
                exercises = json.loads(rec_plan['exercises'])
                ex = exercises[ex_index]
            except (json.JSONDecodeError, IndexError):
                return jsonify(status='error', message='循环计划训练数据错误'), 400

            date_str = datetime.now().strftime('%Y-%m-%d')
            exercise_name = ex.get('name', f"训练 {ex_index+1}")
            details = ex.get('details', '')
            
            # 检查是否已存在 (以防重复提交)
            existing_plan = db.execute("SELECT id, is_completed FROM fitness_plan WHERE user_id = ? AND date = ? AND exercise_name = ? AND details = ?",
                                       (current_user.id, date_str, exercise_name, details)).fetchone()
            
            full_exercise_name = f"【循环】{exercise_name} ({rec_plan['plan_name']})"

            if existing_plan:
                # 已存在，直接更新其完成状态
                db.execute('UPDATE fitness_plan SET is_completed = ? WHERE id = ?', (is_completed, existing_plan['id']))
            else:
                # 不存在，插入新的日常计划并标记完成
                db.execute('INSERT INTO fitness_plan (user_id, date, exercise_name, details, is_completed) VALUES (?, ?, ?, ?, ?)', 
                           (current_user.id, date_str, full_exercise_name, details, is_completed))

            db.commit()
            return jsonify(status='success', message='循环计划已转为日常计划并标记完成')
        else:
            return jsonify(status='error', message='未找到循环计划'), 404
            
    # 日常计划
    if not isinstance(plan_id, int):
        try: plan_id = int(plan_id)
        except ValueError: return jsonify(status='error', message='计划 ID 格式错误'), 400

    try:
        db = get_db()
        cur = db.cursor()
        cur.execute('UPDATE fitness_plan SET is_completed = ? WHERE id = ? AND user_id = ?', 
                   (is_completed, plan_id, current_user.id))
        db.commit()
        if cur.rowcount > 0: return jsonify(status='success', message='每日计划状态已更新')
        else: return jsonify(status='error', message='未找到计划或用户不匹配'), 404
    except sqlite3.Error as e:
        return jsonify(status='error', message=f'数据库错误: {e}'), 500

# 删除每日健身计划条目 (POST)
@app.route('/delete_fitness_plan_exercise/<int:plan_id>', methods=['POST'])
@login_required
def delete_fitness_plan_exercise(plan_id):
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute('DELETE FROM fitness_plan WHERE id = ? AND user_id = ?', (plan_id, current_user.id))
        db.commit()
        if cur.rowcount > 0: return jsonify(status='success', message='每日训练已删除')
        else: return jsonify(status='error', message='未找到每日训练'), 404
    except sqlite3.Error as e: return jsonify(status='error', message=f'数据库错误: {e}'), 500

# -------------------- API: 循环计划 (管理员/用户) --------------------

# 获取所有循环健身计划
@app.route('/get_recurring_plans', methods=['GET'])
@login_required
def get_recurring_plans():
    db = get_db()
    # 非管理员只看自己的
    user_id = current_user.id if not current_user.is_admin else '%'
    
    plans = db.execute("""
        SELECT rp.id, rp.plan_name, rp.exercises, rp.repeat_interval, rp.repeat_day, rp.start_date, rp.end_date, u.username
        FROM recurring_plans rp
        JOIN users u ON rp.user_id = u.id
        WHERE u.id LIKE ?
        ORDER BY rp.id DESC
    """, (user_id,)).fetchall()
    
    results = []
    for p in plans:
        item = dict(p)
        # 解析 JSON 字符串
        try: item['exercises'] = json.loads(item['exercises'])
        except: item['exercises'] = []
        results.append(item)
        
    return jsonify(status='success', plans=results)

# 添加循环健身计划
@app.route('/add_recurring_plan', methods=['POST'])
@login_required
def add_recurring_plan():
    data = request.get_json()
    plan_name = data.get('plan_name')
    exercises = data.get('exercises') # 预期是包含 name 和 details 的列表
    repeat_interval = data.get('repeat_interval')
    repeat_day = data.get('repeat_day')
    start_date = data.get('start_date')
    end_date = data.get('end_date') # 可选

    if not all([plan_name, exercises, repeat_interval, start_date]):
        return jsonify(status='error', message='缺少必要的字段'), 400

    try:
        db = get_db()
        exercises_json = json.dumps(exercises)
        db.execute("""
            INSERT INTO recurring_plans (user_id, plan_name, exercises, repeat_interval, repeat_day, start_date, end_date)
            VALUES (?, ?, ?, ?, ?, ?, ?)
        """, (current_user.id, plan_name, exercises_json, repeat_interval, repeat_day, start_date, end_date))
        db.commit()
        return jsonify(status='success', message='循环计划已添加')
    except sqlite3.Error as e:
        return jsonify(status='error', message=f'数据库错误: {e}'), 500

# 删除循环健身计划
@app.route('/delete_recurring_plan/<int:plan_id>', methods=['POST'])
@login_required
def delete_recurring_plan(plan_id):
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute('DELETE FROM recurring_plans WHERE id = ? AND user_id = ?', (plan_id, current_user.id))
        db.commit()
        if cur.rowcount > 0: return jsonify(status='success', message='循环计划已删除')
        else: return jsonify(status='error', message='未找到循环计划'), 404
    except sqlite3.Error as e: return jsonify(status='error', message=f'数据库错误: {e}'), 500

# -------------------- API: 循环待办 (管理员/用户) --------------------

# 获取所有循环待办
@app.route('/get_recurring_todos', methods=['GET'])
@login_required
def get_recurring_todos():
    db = get_db()
    # 非管理员只看自己的
    user_id = current_user.id if not current_user.is_admin else '%'

    todos = db.execute("""
        SELECT rt.id, rt.content, rt.repeat_interval, rt.repeat_day, rt.start_date, rt.end_date, u.username
        FROM recurring_todos rt
        JOIN users u ON rt.user_id = u.id
        WHERE u.id LIKE ?
        ORDER BY rt.id DESC
    """, (user_id,)).fetchall()
    
    return jsonify(status='success', todos=[dict(t) for t in todos])

# 添加循环待办
@app.route('/add_recurring_todo', methods=['POST'])
@login_required
def add_recurring_todo():
    data = request.get_json()
    content = data.get('content')
    repeat_interval = data.get('repeat_interval')
    repeat_day = data.get('repeat_day')
    start_date = data.get('start_date')
    end_date = data.get('end_date') # 可选

    if not all([content, repeat_interval, start_date]):
        return jsonify(status='error', message='缺少必要的字段'), 400

    try:
        db = get_db()
        db.execute("""
            INSERT INTO recurring_todos (user_id, content, repeat_interval, repeat_day, start_date, end_date)
            VALUES (?, ?, ?, ?, ?, ?)
        """, (current_user.id, content, repeat_interval, repeat_day, start_date, end_date))
        db.commit()
        return jsonify(status='success', message='循环待办已添加')
    except sqlite3.Error as e:
        return jsonify(status='error', message=f'数据库错误: {e}'), 500

# 删除循环待办
@app.route('/delete_recurring_todo/<int:todo_id>', methods=['POST'])
@login_required
def delete_recurring_todo(todo_id):
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute('DELETE FROM recurring_todos WHERE id = ? AND user_id = ?', (todo_id, current_user.id))
        db.commit()
        if cur.rowcount > 0: return jsonify(status='success', message='循环待办已删除')
        else: return jsonify(status='error', message='未找到循环待办'), 404
    except sqlite3.Error as e: return jsonify(status='error', message=f'数据库错误: {e}'), 500


# -------------------- API: 数据统计 --------------------

@app.route('/api/stats/completion', methods=['GET'])
@login_required
def get_completion_stats():
    start_date = request.args.get('start_date')
    end_date = request.args.get('end_date')
    group_by = request.args.get('group_by', 'daily') # 'daily', 'weekly', 'monthly'

    if not start_date or not end_date:
        # 默认查询最近 30 天
        today = datetime.now()
        start_date = (today - timedelta(days=30)).strftime('%Y-%m-%d')
        end_date = today.strftime('%Y-%m-%d')

    db = get_db()
    
    # 辅助函数：根据分组类型确定 SQL 格式
    if group_by == 'monthly':
        date_format = '%Y-%m'
        order_by = 'date_group'
    elif group_by == 'weekly':
        # SQLite 的 strftime 不支持 %W (周数)，这里用日期的 start_date 来简化
        date_format = '%Y-%m-%d' # 实际前端可能需要进一步处理，但后端先按天获取
        order_by = 'date_group'
    else: # daily
        date_format = '%Y-%m-%d'
        order_by = 'date_group'
        
    date_group_sql = f"strftime('{date_format}', date)"

    # 1. 待办统计 (Todos)
    todo_stats = db.execute(f"""
        SELECT 
            {date_group_sql} as date_group,
            COUNT(CASE WHEN is_completed = 1 THEN 1 END) as completed_count,
            COUNT(id) as total_count
        FROM todos
        WHERE user_id = ? AND date BETWEEN ? AND ?
        GROUP BY date_group
        ORDER BY {order_by}
    """, (current_user.id, start_date, end_date)).fetchall()
    
    # 2. 健身计划统计 (Fitness Plans)
    plan_stats = db.execute(f"""
        SELECT 
            {date_group_sql} as date_group,
            COUNT(CASE WHEN is_completed = 1 THEN 1 END) as completed_count,
            COUNT(id) as total_count
        FROM fitness_plan
        WHERE user_id = ? AND date BETWEEN ? AND ?
        GROUP BY date_group
        ORDER BY {order_by}
    """, (current_user.id, start_date, end_date)).fetchall()

    # 格式化输出 (合并并计算完成率)
    stats_map = defaultdict(lambda: {'todos': {'completed': 0, 'total': 0}, 'plans': {'completed': 0, 'total': 0}})

    for row in todo_stats:
        key = row['date_group']
        stats_map[key]['todos']['completed'] = row['completed_count']
        stats_map[key]['todos']['total'] = row['total_count']

    for row in plan_stats:
        key = row['date_group']
        stats_map[key]['plans']['completed'] = row['completed_count']
        stats_map[key]['plans']['total'] = row['total_count']

    # 转换为列表并计算完成率
    results = []
    for date_group in sorted(stats_map.keys()):
        data = stats_map[date_group]
        
        todo_total = data['todos']['total']
        todo_completed = data['todos']['completed']
        todo_rate = (todo_completed / todo_total) * 100 if todo_total > 0 else 0
        
        plan_total = data['plans']['total']
        plan_completed = data['plans']['completed']
        plan_rate = (plan_completed / plan_total) * 100 if plan_total > 0 else 0
        
        results.append({
            'date_group': date_group,
            'todos': {
                'completed': todo_completed,
                'total': todo_total,
                'rate': round(todo_rate, 2)
            },
            'plans': {
                'completed': plan_completed,
                'total': plan_total,
                'rate': round(plan_rate, 2)
            }
        })

    return jsonify(status='success', stats=results, group_by=group_by)


# -------------------- Compatibility headers & health (保持不变) --------------------
@app.after_request
def add_common_headers(response):
    if 'Content-Type' not in response.headers:
        response.headers['Content-Type'] = 'text/html; charset=utf-8'
    else:
        c = response.headers['Content-Type']
        if c.startswith('text/') and 'charset' not in c:
            response.headers['Content-Type'] = c + '; charset=utf-8'
    response.headers.setdefault('X-Content-Type-Options', 'nosniff')
    response.headers.setdefault('X-Frame-Options', 'SAMEORIGIN')
    path = request.path or ''
    if path.startswith('/static/') or path.endswith(('.css', '.js', '.png', '.jpg', '.svg', '.webp')):\
        response.headers['Cache-Control'] = 'public, max-age=3600'
    else:
        response.headers['Cache-Control'] = 'no-cache, no-store, must-revalidate'
    return response

@app.route('/health', methods=['GET'])
def health():
    return jsonify(status='ok', time=datetime.now().isoformat())

# -------------------- Run --------------------
if __name__ == '__main__':
    # 确保数据库和默认用户已初始化
    init_db() 
    
    app.run(host='0.0.0.0', port=8000, debug=True)
