# backend/app.py (v2.0 - 修复 404 并集成 GPIO 提醒功能)
import sqlite3
from datetime import datetime, timedelta
from flask import Flask, render_template, request, jsonify, g, redirect, url_for, make_response, send_from_directory
from flask_login import LoginManager, UserMixin, login_user, logout_user, login_required, current_user
import os, json, uuid
from flask_cors import CORS
from functools import wraps
import sys 
from collections import defaultdict
import atexit

# --- 新增：定时任务和 GPIO 控制 ---
from apscheduler.schedulers.background import BackgroundScheduler
try:
    import RPi.GPIO as GPIO
    GPIO.setmode(GPIO.BCM) # 使用 BCM 引脚编号
    GPIO_AVAILABLE = True
    print("RPi.GPIO imported successfully. GPIO control enabled.")
except ImportError:
    GPIO_AVAILABLE = False
    print("RPi.GPIO not found. GPIO control disabled (running in simulated mode).")
except Exception as e:
    GPIO_AVAILABLE = False
    print(f"Error initializing RPi.GPIO: {e}. GPIO control disabled (running in simulated mode).")

# 全局状态：维护哪些引脚被任务激活 { 'fitness_ID': pin, 'todo_ID': pin }
TASK_TO_GPIO = {} 
# --- 结束新增 ---


# --- 路径逻辑 (已修复) ---
is_frozen = getattr(sys, 'frozen', False)
is_in_onefile_temp = 'onefile_' in os.path.abspath(__file__) 

if is_frozen or is_in_onefile_temp:
    temp_base_dir = os.path.dirname(os.path.abspath(sys.executable))
    template_dir = os.path.join(temp_base_dir, 'templates')
    static_dir = os.path.join(temp_base_dir, 'static')
    exe_path = os.path.realpath(sys.argv[0])
    exe_dir = os.path.dirname(exe_path)
    DATABASE = os.path.join(exe_dir, 'fitness_manager.db')
    print("--- Detected Nuitka Onefile Path ---")
    print(f"--- **DATABASE is PERMANENT at {DATABASE}** ---")
else:
    app_file_path = os.path.abspath(__file__)
    backend_dir = os.path.dirname(app_file_path)
    project_root_dir = os.path.dirname(backend_dir)
    # 关键修复：正确设置模板和静态文件路径
    template_dir = os.path.join(project_root_dir, 'frontend', 'templates')
    static_dir = os.path.join(project_root_dir, 'frontend', 'static')
    DATABASE = os.path.join(backend_dir, 'fitness_manager.db')
    print("--- Detected Standard Development Path ---")
# --- 结束路径逻辑 ---


print("Template folder:", template_dir)
print("Static folder:", static_dir)
print("DB path:", DATABASE)

app = Flask(__name__, template_folder=template_dir, static_folder=static_dir)
CORS(app)
app.secret_key = str(uuid.uuid4()) # 保持不变

# --- Flask-Login 配置 (保持不变) ---
login_manager = LoginManager()
login_manager.init_app(app)
login_manager.login_view = 'login' # 修复：指向 /login 路由
login_manager.login_message = "请先登录以访问此页面。"

class User(UserMixin):
    def __init__(self, id, username, is_admin):
        self.id = id
        self.username = username
        self.is_admin = is_admin

@login_manager.user_loader
def load_user(user_id):
    db = get_db()
    user_row = db.execute("SELECT id, username, is_admin FROM users WHERE id = ?", (user_id,)).fetchone()
    if user_row:
        return User(id=user_row['id'], username=user_row['username'], is_admin=user_row['is_admin'])
    return None

# -------------------- DB --------------------
def get_db():
    db = getattr(g, '_database', None)
    if db is None:
        db = g._database = sqlite3.connect(DATABASE, check_same_thread=False)
        db.row_factory = sqlite3.Row
    return db

@app.teardown_appcontext
def close_connection(exception):
    db = getattr(g, '_database', None)
    if db is not None:
        db.close()
        
def get_setting(key, default=None):
    db = get_db()
    row = db.execute("SELECT value FROM settings WHERE key = ?", (key,)).fetchone()
    if row is None:
        return default
    return row["value"]

def set_setting(key, value):
    db = get_db()
    cur = db.cursor()
    cur.execute(
        "INSERT INTO settings (key, value) VALUES (?, ?) "
        "ON CONFLICT(key) DO UPDATE SET value = excluded.value",
        (key, str(value)),
    )
    db.commit()

# --- 数据库初始化与迁移 (已更新) ---
def init_db():
    db = get_db()
    cur = db.cursor()
    cur.execute("""
        CREATE TABLE IF NOT EXISTS todos (
            id INTEGER PRIMARY KEY AUTOINCREMENT, 
            date TEXT NOT NULL, 
            content TEXT NOT NULL,
            is_completed INTEGER DEFAULT 0, 
            user_id INTEGER, 
            recurring_todo_id INTEGER
        )
    """)
    cur.execute("""
        CREATE TABLE IF NOT EXISTS fitness_plans (
            id INTEGER PRIMARY KEY AUTOINCREMENT, 
            date TEXT NOT NULL, 
            exercise_name TEXT NOT NULL,
            sets_reps_duration TEXT, 
            is_completed INTEGER DEFAULT 0, 
            user_id INTEGER, 
            recurring_plan_id INTEGER
        )
    """)
    cur.execute("""
        CREATE TABLE IF NOT EXISTS recurring_plans (
            id INTEGER PRIMARY KEY AUTOINCREMENT, 
            user_id INTEGER, 
            plan_name TEXT NOT NULL,
            exercises_json TEXT NOT NULL, 
            start_date TEXT NOT NULL, 
            end_date TEXT,
            repeat_interval TEXT NOT NULL, 
            repeat_day INTEGER
        )
    """)
    cur.execute("""
        CREATE TABLE IF NOT EXISTS recurring_todos (
            id INTEGER PRIMARY KEY AUTOINCREMENT, 
            user_id INTEGER, 
            content TEXT NOT NULL,
            start_date TEXT NOT NULL, 
            end_date TEXT, 
            repeat_interval TEXT NOT NULL, 
            repeat_day INTEGER
        )
    """)
    cur.execute("""
        CREATE TABLE IF NOT EXISTS exercise_library (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            user_id INTEGER,
            name TEXT NOT NULL,
            details TEXT
        )
    """)
    cur.execute("""
        CREATE TABLE IF NOT EXISTS settings (
            key   TEXT PRIMARY KEY,
            value TEXT
        )
    """)
    cur.execute("""
        CREATE TABLE IF NOT EXISTS todo_library (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            user_id INTEGER,
            content TEXT NOT NULL
        )
    """)
    cur.execute("""
        CREATE TABLE IF NOT EXISTS users (
            id INTEGER PRIMARY KEY AUTOINCREMENT, username TEXT UNIQUE NOT NULL,
            password TEXT NOT NULL, is_admin INTEGER DEFAULT 0
        )
    """)
    db.commit()

def ensure_db_schema():
    """执行数据库迁移，添加新列"""
    db = get_db()
    cur = db.cursor()
    
    # --- 新增：GPIO 和 Reminder 字段迁移检查 ---
    for table in ['fitness_plans', 'todos']:
        try: cur.execute(f"SELECT reminder_time FROM {table} LIMIT 1") 
        except sqlite3.OperationalError:
            print(f"Migrating {table} table: Adding reminder_time column.")
            cur.execute(f"ALTER TABLE {table} ADD COLUMN reminder_time TEXT") 
        
        try: cur.execute(f"SELECT gpio_pin FROM {table} LIMIT 1") 
        except sqlite3.OperationalError:
            print(f"Migrating {table} table: Adding gpio_pin column.")
            cur.execute(f"ALTER TABLE {table} ADD COLUMN gpio_pin INTEGER") 

        try: cur.execute(f"SELECT sound_alert FROM {table} LIMIT 1") 
        except sqlite3.OperationalError:
            print(f"Migrating {table} table: Adding sound_alert column.")
            cur.execute(f"ALTER TABLE {table} ADD COLUMN sound_alert INTEGER DEFAULT 0") 
            
    # 确保 recurring_plans 也包含提醒设置（用于默认设置）
    for table in ['recurring_plans', 'recurring_todos']:
        try: cur.execute(f"SELECT reminder_time FROM {table} LIMIT 1") 
        except sqlite3.OperationalError:
            print(f"Migrating {table} table: Adding reminder_time column.")
            cur.execute(f"ALTER TABLE {table} ADD COLUMN reminder_time TEXT") 
        
        try: cur.execute(f"SELECT gpio_pin FROM {table} LIMIT 1") 
        except sqlite3.OperationalError:
            print(f"Migrating {table} table: Adding gpio_pin column.")
            cur.execute(f"ALTER TABLE {table} ADD COLUMN gpio_pin INTEGER") 

        try: cur.execute(f"SELECT sound_alert FROM {table} LIMIT 1") 
        except sqlite3.OperationalError:
            print(f"Migrating {table} table: Adding sound_alert column.")
            cur.execute(f"ALTER TABLE {table} ADD COLUMN sound_alert INTEGER DEFAULT 0") 

    # --- (原有用户和默认数据逻辑 - 保持不变) ---
    try: cur.execute("SELECT is_admin FROM users LIMIT 1")
    except sqlite3.OperationalError:
        print("Migrating users table: Adding is_admin column.")
        cur.execute("ALTER TABLE users ADD COLUMN is_admin INTEGER DEFAULT 0")
    
    cur.execute("SELECT COUNT(*) FROM users WHERE username = 'superuser'")
    if cur.fetchone()[0] == 0:
        print("Inserting default superuser: superuser/adminpassword (Admin)")
        cur.execute("INSERT INTO users (username, password, is_admin) VALUES (?, ?, ?)", ('superuser', 'adminpassword', 1))
        
    cur.execute("SELECT COUNT(*) FROM users WHERE username = 'user1'")
    if cur.fetchone()[0] == 0:
        print("Inserting default regular user: user1/password (Regular)")
        cur.execute("INSERT INTO users (username, password, is_admin) VALUES (?, ?, ?)", ('user1', 'password', 0))

    try: cur.execute("SELECT repeat_interval FROM recurring_plans LIMIT 1")
    except sqlite3.OperationalError:
        print("Migrating recurring_plans table...")
        cur.execute("ALTER TABLE recurring_plans ADD COLUMN repeat_interval TEXT DEFAULT 'daily'")
        cur.execute("ALTER TABLE recurring_plans ADD COLUMN repeat_day INTEGER")
        cur.execute("UPDATE recurring_plans SET repeat_interval = 'daily' WHERE repeat_interval IS NULL")

    try: cur.execute("SELECT recurring_plan_id FROM fitness_plans LIMIT 1")
    except sqlite3.OperationalError:
        cur.execute("ALTER TABLE fitness_plans ADD COLUMN recurring_plan_id INTEGER")

    try: cur.execute("SELECT recurring_todo_id FROM todos LIMIT 1")
    except sqlite3.OperationalError:
        cur.execute("ALTER TABLE todos ADD COLUMN recurring_todo_id INTEGER")

    # 确保 user_id 列存在并填充
    default_user_id_row = db.execute("SELECT id FROM users WHERE username = 'user1'").fetchone()
    if default_user_id_row:
        default_user_id = default_user_id_row['id']
        for table in ['fitness_plans', 'todos', 'recurring_plans', 'recurring_todos', 'exercise_library', 'todo_library']:
            try: cur.execute(f"SELECT user_id FROM {table} LIMIT 1") 
            except sqlite3.OperationalError:
                try: 
                    cur.execute(f"ALTER TABLE {table} ADD COLUMN user_id INTEGER")
                    print(f"Migrating {table} table: Adding user_id column.")
                except sqlite3.OperationalError: pass 
            cur.execute(f"UPDATE {table} SET user_id = ? WHERE user_id IS NULL", (default_user_id,))
    db.commit()
    print("Database schema migration check complete.")


# -------------------- GPIO & Scheduler Setup --------------------

def set_gpio_state(pin, state):
    """设置指定引脚的GPIO状态 (HIGH/LOW)"""
    if pin is None: return
    
    pin = int(pin)
    if not GPIO_AVAILABLE:
        print(f"GPIO SIM: Setting pin {pin} to {'HIGH' if state else 'LOW'}")
        return
    
    try:
        # 确保引脚设置为输出，并设置初始值为 LOW
        GPIO.setup(pin, GPIO.OUT, initial=GPIO.LOW)
        
        gpio_state = GPIO.HIGH if state else GPIO.LOW
        GPIO.output(pin, gpio_state)
        print(f"GPIO CTRL: Set pin {pin} to {'HIGH' if state else 'LOW'}")
    except Exception as e:
        print(f"GPIO ERROR: Failed to set pin {pin}: {e}")

def cleanup_gpio():
    """程序退出时清理所有 GPIO 状态"""
    if GPIO_AVAILABLE:
        print("Cleaning up GPIO pins...")
        # 确保所有被占用的引脚先设置为 LOW
        for pin in set(TASK_TO_GPIO.values()):
             try:
                 set_gpio_state(pin, False)
             except Exception:
                 pass
        GPIO.cleanup()
        
atexit.register(cleanup_gpio) # 注册清理函数

def check_reminders():
    """
    定时任务：检查所有未完成且时间已到的任务，并触发 GPIO。
    每分钟的第 0 秒执行。
    """
    with app.app_context(): # 确保在应用上下文中操作数据库
        db = get_db()
        cursor = db.cursor()
        
        # 提醒时间精度为秒，我们检查 HH:MM:00
        current_time_str = datetime.now().strftime("%H:%M:00") 
        today_str = datetime.now().strftime('%Y-%m-%d')
        
        # print(f"Running reminder check at {datetime.now().strftime('%H:%M:%S')}...") # 调试输出过多，暂时注释

        # --- 1. 检查健身计划 ---
        # 仅检查当天的计划
        cursor.execute("""
            SELECT id, gpio_pin FROM fitness_plans 
            WHERE date = ?
              AND is_completed = 0 
              AND reminder_time = ?
              AND gpio_pin IS NOT NULL
        """, (today_str, current_time_str))
        
        for task_id, pin in cursor.fetchall():
            task_key = f'fitness_{task_id}'
            # 只有当该任务未激活且该引脚未被其他任务占用时才触发
            if task_key not in TASK_TO_GPIO: 
                set_gpio_state(pin, True) # 触发高电平（提醒）
                TASK_TO_GPIO[task_key] = pin
                print(f"Reminder ON: Fitness ID {task_id}, Pin {pin}")

        # --- 2. 检查待办事项 ---
        cursor.execute("""
            SELECT id, gpio_pin FROM todos 
            WHERE date = ?
              AND is_completed = 0 
              AND reminder_time = ?
              AND gpio_pin IS NOT NULL
        """, (today_str, current_time_str))
        
        for task_id, pin in cursor.fetchall():
            task_key = f'todo_{task_id}'
            if task_key not in TASK_TO_GPIO:
                set_gpio_state(pin, True)
                TASK_TO_GPIO[task_key] = pin
                print(f"Reminder ON: Todo ID {task_id}, Pin {pin}")

# 启动调度器
scheduler = BackgroundScheduler()
# 每分钟的第 0 秒执行一次检查
scheduler.add_job(check_reminders, 'cron', second='0', id='reminder_checker', max_instances=1)

# -------------------- DB 初始化执行 --------------------
with app.app_context():
    init_db() 
    ensure_db_schema() # 执行迁移
    # 确保在 app context 中启动调度器
    try:
        scheduler.start()
        print("APScheduler started for reminder checks.")
    except Exception as e:
        if 'already started' not in str(e):
            print(f"APScheduler failed to start: {e}")
            
# -------------------- Auth Routes (保持不变) --------------------

@app.route('/login', methods=['GET', 'POST'])
def login():
    if current_user.is_authenticated:
        return redirect(url_for('index'))
        
    if request.method == 'POST':
        username = request.form.get('username') or request.json.get('username')
        password = request.form.get('password') or request.json.get('password')
        db = get_db()
        user_row = db.execute("SELECT id, username, password, is_admin FROM users WHERE username = ?", (username,)).fetchone()
        
        if user_row and user_row['password'] == password: 
            user_obj = User(id=user_row['id'], username=user_row['username'], is_admin=user_row['is_admin'])
            login_user(user_obj, remember=True) # 保持会话
            return jsonify({'status': 'success', 'username': user_obj.username, 'is_admin': user_obj.is_admin})
        
        return jsonify({'status': 'error', 'message': '用户名或密码无效'}), 401
    
    # GET 请求
    return render_template('login.html')

@app.route('/logout', methods=['POST'])
@login_required
def logout():
    logout_user() 
    return jsonify({'status': 'success'})

@app.route('/get_current_user', methods=['GET'])
@login_required
def get_current_user():
    if current_user.is_authenticated:
        return jsonify({
            'status': 'logged_in', 
            'user_id': current_user.id, 
            'username': current_user.username, 
            'is_admin': current_user.is_admin
        })
    return jsonify({'status': 'logged_out'}), 401

# -------------------- UI Routes (保持不变) --------------------
@app.route('/')
def index():
    if not current_user.is_authenticated:
        # 修复：重定向到 /login 路由 (login_view)
        return redirect(url_for('login', next=request.path))
        
    today_str = datetime.now().strftime('%Y-%m-%d')
    _generate_daily_fitness_plan_if_not_exists(current_user.id, today_str)
    _generate_daily_todos_if_not_exists(current_user.id, today_str)
    
    # 修复：显式传递 display.html
    return render_template('display.html', username=current_user.username, is_admin=current_user.is_admin) 

@app.route('/mobile_config')
@login_required
def mobile_config():
    return render_template('mobile_config.html', username=current_user.username)

@app.route('/manage')
@login_required
def manage():
    return render_template('manage.html', username=current_user.username, is_admin=current_user.is_admin)

# -------------------- API: Helpers (Data Generation Logic - 已更新) --------------------
def _get_daily_fitness_plans(user_id, date_str):
    db = get_db()
    plans = db.execute("SELECT * FROM fitness_plans WHERE date = ? AND user_id = ? ORDER BY id", (date_str, user_id)).fetchall()
    return [dict(plan) for plan in plans]

def _generate_daily_fitness_plan_if_not_exists(user_id, date_str):
    db = get_db()
    cur = db.cursor()
    cur.execute("SELECT COUNT(*) FROM fitness_plans WHERE date = ? AND user_id = ?", (date_str, user_id))
    if cur.fetchone()[0] > 0: return 
    today_date = datetime.strptime(date_str, '%Y-%m-%d').date()
    today_weekday = today_date.isoweekday() 
    recurring_plans = db.execute("SELECT * FROM recurring_plans WHERE user_id = ? AND (start_date <= ?) AND (end_date IS NULL OR end_date >= ?)", (user_id, date_str, date_str)).fetchall()
    new_plans_to_insert = []
    for plan in recurring_plans:
        should_add = False
        if plan['repeat_interval'] == 'daily': should_add = True
        elif plan['repeat_interval'] == 'weekly' and plan['repeat_day'] == today_weekday: should_add = True
        if should_add:
            exercises = json.loads(plan['exercises_json'])
            # ★ 新增：从循环计划中继承提醒设置
            reminder_time = plan['reminder_time']
            gpio_pin = plan['gpio_pin']
            sound_alert = plan['sound_alert']
            for ex in exercises: 
                new_plans_to_insert.append({
                    'name': ex['name'], 
                    'details': ex['details'], 
                    'recurring_id': plan['id'],
                    'reminder_time': reminder_time,
                    'gpio_pin': gpio_pin,
                    'sound_alert': sound_alert,
                })
    for plan in new_plans_to_insert:
        cur.execute(
            """INSERT INTO fitness_plans (date, exercise_name, sets_reps_duration, is_completed, user_id, recurring_plan_id, reminder_time, gpio_pin, sound_alert) 
               VALUES (?, ?, ?, 0, ?, ?, ?, ?, ?)""",
            (date_str, plan['name'], plan['details'], user_id, plan['recurring_id'], plan['reminder_time'], plan['gpio_pin'], plan['sound_alert']))
    db.commit()

def _generate_daily_todos_if_not_exists(user_id, date_str):
    db = get_db()
    cur = db.cursor()
    cur.execute("SELECT COUNT(*) FROM todos WHERE date = ? AND user_id = ?", (date_str, user_id))
    if cur.fetchone()[0] > 0: return
    today_date = datetime.strptime(date_str, '%Y-%m-%d').date()
    today_weekday = today_date.isoweekday() 
    today_day_of_month = today_date.day 
    recurring_todos = db.execute("SELECT * FROM recurring_todos WHERE user_id = ? AND (start_date <= ?) AND (end_date IS NULL OR end_date >= ?)", (user_id, date_str, date_str)).fetchall()
    new_todos_to_insert = []
    for todo in recurring_todos:
        should_add = False
        if todo['repeat_interval'] == 'daily': should_add = True
        elif todo['repeat_interval'] == 'weekly' and todo['repeat_day'] == today_weekday: should_add = True
        elif todo['repeat_interval'] == 'monthly' and todo['repeat_day'] == today_day_of_month: should_add = True
        if should_add: 
            # ★ 新增：从循环待办中继承提醒设置
            reminder_time = todo['reminder_time']
            gpio_pin = todo['gpio_pin']
            sound_alert = todo['sound_alert']
            new_todos_to_insert.append({
                'content': todo['content'], 
                'recurring_id': todo['id'],
                'reminder_time': reminder_time,
                'gpio_pin': gpio_pin,
                'sound_alert': sound_alert,
            })
    for todo in new_todos_to_insert:
        cur.execute(
            """INSERT INTO todos (date, content, is_completed, user_id, recurring_todo_id, reminder_time, gpio_pin, sound_alert) 
               VALUES (?, ?, 0, ?, ?, ?, ?, ?)""",
            (date_str, todo['content'], user_id, todo['recurring_id'], todo['reminder_time'], todo['gpio_pin'], todo['sound_alert']))
    db.commit()
    
# 辅助函数：确保返回字典，并处理 None
def row_to_dict(row):
    d = dict(row)
    if 'gpio_pin' in d and d['gpio_pin'] is not None:
        d['gpio_pin'] = int(d['gpio_pin'])
    if 'reminder_time' in d and d['reminder_time'] is not None:
        # 确保返回 HH:MM:SS 格式
        d['reminder_time'] = d['reminder_time'].split('.')[0] 
    if 'sound_alert' in d and d['sound_alert'] is not None:
        d['sound_alert'] = int(d['sound_alert'])
    return d

# -------------------- 修复 404 API: 健身计划 --------------------
@app.route('/fitness_plan/<date_str>', methods=['GET'])
@login_required
def get_fitness_plan(date_str):
    user_id = current_user.id
    _generate_daily_fitness_plan_if_not_exists(user_id, date_str)
    db = get_db()
    # 修复：查询新字段
    plans = db.execute("SELECT *, reminder_time, gpio_pin, sound_alert FROM fitness_plans WHERE date = ? AND user_id = ? ORDER BY id", (date_str, user_id)).fetchall()
    return jsonify([row_to_dict(plan) for plan in plans])

@app.route('/delete_fitness_plan_exercise/<int:plan_id>', methods=['POST'])
@login_required
def delete_fitness_plan_exercise(plan_id):
    user_id = current_user.id
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute("DELETE FROM fitness_plans WHERE id = ? AND user_id = ?", (plan_id, user_id))
        db.commit()
        if cur.rowcount > 0: return jsonify(status='success', message='每日健身已删除')
        else: return jsonify(status='error', message='未找到每日健身计划'), 404
    except sqlite3.Error as e: return jsonify(status='error', message=f'数据库错误: {e}'), 500

# -------------------- 修复 404 API: 待办事项 --------------------
@app.route('/todos', methods=['GET'])
@login_required
def get_todos():
    user_id = current_user.id
    date_str = request.args.get('date') or datetime.now().strftime('%Y-%m-%d')
    _generate_daily_todos_if_not_exists(user_id, date_str) # 确保同步
    db = get_db()
    # 修复：查询新字段
    todos = db.execute(
        "SELECT *, reminder_time, gpio_pin, sound_alert FROM todos WHERE date = ? AND user_id = ? ORDER BY id",
        (date_str, user_id)
    ).fetchall()
    return jsonify([row_to_dict(todo) for todo in todos])

@app.route('/add_todo', methods=['POST'])
@login_required
def add_todo():
    user_id = current_user.id
    data = request.get_json()
    date_str = data.get('date')
    content = data.get('content')
    if not date_str or not content: return jsonify(status='error', message='缺少日期或内容'), 400
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute("INSERT INTO todos (date, content, is_completed, user_id) VALUES (?, ?, 0, ?)", (date_str, content, user_id))
        db.commit()
        return jsonify(status='success', todo_id=cur.lastrowid)
    except sqlite3.Error as e: return jsonify(status='error', message=f'数据库错误: {e}'), 500
    
@app.route('/delete_todo/<int:todo_id>', methods=['POST'])
@login_required
def delete_todo(todo_id):
    user_id = current_user.id
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute("DELETE FROM todos WHERE id = ? AND user_id = ?", (todo_id, user_id))
        db.commit()
        if cur.rowcount > 0: return jsonify(status='success', message='待办事项已删除')
        else: return jsonify(status='error', message='未找到待办事项'), 404
    except sqlite3.Error as e: return jsonify(status='error', message=f'数据库错误: {e}'), 500
    
# -------------------- API: Mark Complete (已更新 GPIO 逻辑) --------------------

@app.route('/mark_exercise_completed_mobile', methods=['POST'])
@login_required
def mark_exercise_completed_mobile():
    user_id = current_user.id
    data = request.get_json()
    plan_id = data.get('plan_id')
    is_completed = data.get('is_completed') 
    
    if plan_id is None or is_completed is None: 
        return jsonify(status='error', message='缺少健身计划ID或完成状态'), 400
    
    task_key = f'fitness_{plan_id}'
    # ★ 关键：如果任务被标记，检查是否需要关闭 GPIO
    if task_key in TASK_TO_GPIO:
        pin = TASK_TO_GPIO.pop(task_key) # 从激活列表中移除
        set_gpio_state(pin, False) # 关闭 GPIO (低电平)
        print(f"Reminder OFF: Fitness ID {plan_id}, Pin {pin}")

    # 更新数据库状态
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute("UPDATE fitness_plans SET is_completed = ? WHERE id = ? AND user_id = ?", (is_completed, plan_id, user_id))
        db.commit()
        if cur.rowcount > 0:
            return jsonify(status='success') 
        else:
            return jsonify(status='error', message='未找到健身计划或您无权修改'), 404
    except sqlite3.Error as e:
        return jsonify(status='error', message=f'数据库错误: {e}'), 500

@app.route('/mark_todo_completed', methods=['POST'])
@login_required
def mark_todo_completed():
    user_id = current_user.id
    data = request.get_json()
    todo_id = data.get('todo_id')
    is_completed = data.get('is_completed') 

    if todo_id is None or is_completed is None: 
        return jsonify(status='error', message='缺少待办ID或完成状态'), 400
    
    task_key = f'todo_{todo_id}'
    # ★ 关键：如果任务被标记，检查是否需要关闭 GPIO
    if task_key in TASK_TO_GPIO:
        pin = TASK_TO_GPIO.pop(task_key) # 从激活列表中移除
        set_gpio_state(pin, False) # 关闭 GPIO (低电平)
        print(f"Reminder OFF: Todo ID {todo_id}, Pin {pin}")

    # 更新数据库状态
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute("UPDATE todos SET is_completed = ? WHERE id = ? AND user_id = ?", (is_completed, todo_id, user_id))
        db.commit()
        if cur.rowcount > 0:
            return jsonify(status='success') 
        else:
            return jsonify(status='error', message='未找到待办事项或您无权修改'), 404
    except sqlite3.Error as e:
        return jsonify(status='error', message=f'数据库错误: {e}'), 500

# -------------------- 修复 404 API: 循环计划 --------------------

@app.route('/get_recurring_plans', methods=['GET'])
@login_required
def get_recurring_plans():
    user_id = current_user.id
    db = get_db()
    plans = db.execute("SELECT * FROM recurring_plans WHERE user_id = ? ORDER BY start_date DESC", (user_id,)).fetchall()
    return jsonify([dict(plan) for plan in plans])

@app.route('/add_recurring_plan', methods=['POST'])
@login_required
def add_recurring_plan():
    user_id = current_user.id
    data = request.get_json()
    if not data.get('plan_name') or not data.get('start_date') or not data.get('repeat_interval') or not data.get('exercises'):
        return jsonify(status='error', message='缺少必填字段'), 400
    if data.get('repeat_interval') == 'weekly' and not data.get('repeat_day'):
        return jsonify(status='error', message='“每周”循环缺少“星期几”设置'), 400
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute(
            """INSERT INTO recurring_plans (user_id, plan_name, exercises_json, start_date, end_date, repeat_interval, repeat_day, reminder_time, gpio_pin, sound_alert) 
               VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)""",
            (user_id, data.get('plan_name'), json.dumps(data.get('exercises')), data.get('start_date'), data.get('end_date'), data.get('repeat_interval'), data.get('repeat_day'),
             data.get('reminder_time'), data.get('gpio_pin'), data.get('sound_alert', 0))
        )
        db.commit()
        return jsonify(status='success', plan_id=cur.lastrowid)
    except sqlite3.Error as e: return jsonify(status='error', message=f'数据库错误: {e}'), 500

@app.route('/delete_recurring_plan/<int:plan_id>', methods=['POST'])
@login_required
def delete_recurring_plan(plan_id):
    user_id = current_user.id
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute("DELETE FROM recurring_plans WHERE id = ? AND user_id = ?", (plan_id, user_id))
        db.commit()
        if cur.rowcount > 0: return jsonify(status='success', message='循环计划已删除')
        else: return jsonify(status='error', message='未找到循环计划'), 404
    except sqlite3.Error as e: return jsonify(status='error', message=f'数据库错误: {e}'), 500

# -------------------- 修复 404 API: 循环待办 --------------------

@app.route('/get_recurring_todos', methods=['GET'])
@login_required
def get_recurring_todos():
    user_id = current_user.id
    db = get_db()
    todos = db.execute("SELECT * FROM recurring_todos WHERE user_id = ? ORDER BY start_date DESC", (user_id,)).fetchall()
    return jsonify([dict(todo) for todo in todos])

@app.route('/add_recurring_todo', methods=['POST'])
@login_required
def add_recurring_todo():
    user_id = current_user.id
    data = request.get_json()
    if not data.get('content') or not data.get('start_date') or not data.get('repeat_interval'):
        return jsonify(status='error', message='缺少必填字段'), 400
    if (data.get('repeat_interval') == 'weekly' or data.get('repeat_interval') == 'monthly') and not data.get('repeat_day'):
        return jsonify(status='error', message='循环缺少“星期几”或“日期”设置'), 400
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute(
            """INSERT INTO recurring_todos (user_id, content, start_date, end_date, repeat_interval, repeat_day, reminder_time, gpio_pin, sound_alert) 
               VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)""",
            (user_id, data.get('content'), data.get('start_date'), data.get('end_date'), data.get('repeat_interval'), data.get('repeat_day'),
             data.get('reminder_time'), data.get('gpio_pin'), data.get('sound_alert', 0))
        )
        db.commit()
        return jsonify(status='success', todo_id=cur.lastrowid)
    except sqlite3.Error as e: return jsonify(status='error', message=f'数据库错误: {e}'), 500

@app.route('/delete_recurring_todo/<int:todo_id>', methods=['POST'])
@login_required
def delete_recurring_todo(todo_id):
    user_id = current_user.id
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute("DELETE FROM recurring_todos WHERE id = ? AND user_id = ?", (todo_id, user_id))
        db.commit()
        if cur.rowcount > 0: return jsonify(status='success', message='循环待办已删除')
        else: return jsonify(status='error', message='未找到循环待办'), 404
    except sqlite3.Error as e: return jsonify(status='error', message=f'数据库错误: {e}'), 500

# -------------------- 修复 404 API: 库 (Library) --------------------

@app.route('/api/exercises', methods=['GET'])
@login_required
def list_exercises():
    user_id = current_user.id
    db = get_db()
    rows = db.execute(
        "SELECT id, name, details FROM exercise_library WHERE user_id = ? ORDER BY id DESC",
        (user_id,)
    ).fetchall()
    return jsonify([dict(r) for r in rows])

@app.route('/api/exercises', methods=['POST'])
@login_required
def create_exercise():
    user_id = current_user.id
    data = request.get_json() or {}
    name = (data.get('name') or '').strip()
    details = (data.get('details') or '').strip()
    if not name: return jsonify(status='error', message='缺少训练项目名称'), 400
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute( "INSERT INTO exercise_library (user_id, name, details) VALUES (?, ?, ?)", (user_id, name, details) )
        db.commit()
        return jsonify(status='success', exercise_id=cur.lastrowid)
    except sqlite3.Error as e: return jsonify(status='error', message=f'数据库错误: {e}'), 500

@app.route('/api/exercises/<int:exercise_id>', methods=['POST']) # 使用 POST 保持一致
@login_required
def delete_exercise(exercise_id):
    user_id = current_user.id
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute( "DELETE FROM exercise_library WHERE id = ? AND user_id = ?", (exercise_id, user_id) )
        db.commit()
        if cur.rowcount > 0: return jsonify(status='success', message='训练项目已删除')
        else: return jsonify(status='error', message='未找到训练项目'), 404
    except sqlite3.Error as e: return jsonify(status='error', message=f'数据库错误: {e}'), 500

@app.route('/api/todo_library', methods=['GET'])
@login_required
def list_todo_library():
    user_id = current_user.id
    db = get_db()
    rows = db.execute( "SELECT id, content FROM todo_library WHERE user_id = ? ORDER BY id DESC", (user_id,) ).fetchall()
    return jsonify([dict(r) for r in rows])

@app.route('/api/todo_library', methods=['POST'])
@login_required
def create_todo_library_item():
    user_id = current_user.id
    data = request.get_json() or {}
    content = (data.get('content') or '').strip()
    if not content: return jsonify(status='error', message='缺少待办内容'), 400
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute( "INSERT INTO todo_library (user_id, content) VALUES (?, ?)", (user_id, content) )
        db.commit()
        return jsonify(status='success', item_id=cur.lastrowid)
    except sqlite3.Error as e: return jsonify(status='error', message=f'数据库错误: {e}'), 500

@app.route('/api/todo_library/<int:item_id>', methods=['POST']) # 使用 POST 保持一致
@login_required
def delete_todo_library_item(item_id):
    user_id = current_user.id
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute( "DELETE FROM todo_library WHERE id = ? AND user_id = ?", (item_id, user_id) )
        db.commit()
        if cur.rowcount > 0: return jsonify(status='success', message='常用待办已删除')
        else: return jsonify(status='error', message='未找到该待办'), 404
    except sqlite3.Error as e: return jsonify(status='error', message=f'数据库错误: {e}'), 500
    
# -------------------- 修复 404 API: 统计 --------------------

@app.route('/api/stats/daily_completion', methods=['GET'])
@login_required
def get_daily_completion_stats():
    user_id = current_user.id
    start_date = request.args.get('start_date')
    end_date = request.args.get('end_date')
    if not start_date or not end_date: return jsonify(status='error', message='缺少开始日期或结束日期'), 400
    db = get_db()
    fitness_rows = db.execute("SELECT date, is_completed FROM fitness_plans WHERE user_id = ? AND date BETWEEN ? AND ?", (user_id, start_date, end_date)).fetchall()
    todo_rows = db.execute("SELECT date, is_completed FROM todos WHERE user_id = ? AND date BETWEEN ? AND ?", (user_id, start_date, end_date)).fetchall()
    
    fitness_data = defaultdict(lambda: {"total": 0, "completed": 0})
    for row in fitness_rows:
        fitness_data[row['date']]["total"] += 1
        if row['is_completed']: fitness_data[row['date']]["completed"] += 1
        
    todo_data = defaultdict(lambda: {"total": 0, "completed": 0})
    for row in todo_rows:
        todo_data[row['date']]["total"] += 1
        if row['is_completed']: todo_data[row['date']]["completed"] += 1
        
    return jsonify({"status": "success", "fitness_data": fitness_data, "todo_data": todo_data})

# -------------------- API: 提醒设置 (新增) --------------------

@app.route('/api/update_task_reminder', methods=['POST'])
@login_required
def update_task_reminder():
    """ 新增：保存每日任务的提醒时间和GPIO引脚 """
    data = request.json
    task_id = data.get('id')
    task_type = data.get('type') # 'fitness' or 'todo'
    
    # 前端传递空字符串时，将其转为 None
    reminder_time = data.get('reminder_time') if data.get('reminder_time') else None 
    gpio_pin = data.get('gpio_pin') if data.get('gpio_pin') else None 
    sound_alert = data.get('sound_alert', 0)
    user_id = current_user.id
    
    if not task_id or not task_type:
        return jsonify(status='error', message='缺少任务ID或类型'), 400

    if task_type == 'fitness':
        table = 'fitness_plans'
    elif task_type == 'todo':
        table = 'todos'
    else:
        return jsonify(status='error', message='无效的任务类型'), 400

    conn = get_db()
    try:
        conn.execute(f"""
            UPDATE {table} SET 
                reminder_time = ?,
                gpio_pin = ?,
                sound_alert = ?
            WHERE id = ? AND user_id = ?
        """, (reminder_time, gpio_pin, sound_alert, task_id, user_id))
        conn.commit()
        return jsonify(status='success', message='提醒设置已更新')
    except sqlite3.Error as e:
        return jsonify(status='error', message=f'数据库错误: {e}'), 500

# -------------------- Compatibility headers & health (保持不变) --------------------
@app.after_request
def add_common_headers(response):
    if 'Content-Type' not in response.headers:
        response.headers['Content-Type'] = 'text/html; charset=utf-8'
    else:
        c = response.headers['Content-Type']
        if c.startswith('text/') and 'charset' not in c:
            response.headers['Content-Type'] = c + '; charset=utf-8'
    response.headers.setdefault('X-Content-Type-Options', 'nosniff')
    response.headers.setdefault('X-Frame-Options', 'SAMEORIGIN')
    path = request.path or ''
    # 新增 .mp3
    if path.startswith('/static/') or path.endswith(('.css', '.js', '.png', '.jpg', '.svg', '.webp', '.mp3')):
        response.headers['Cache-Control'] = 'public, max-age=3600'
    else:
        response.headers['Cache-Control'] = 'no-cache, no-store, must-revalidate'
    return response

# 修复：添加 /static 路由
@app.route('/static/<path:filename>')
def serve_static(filename):
    """确保静态文件路径正确"""
    return send_from_directory(app.static_folder, filename)

@app.route('/health', methods=['GET'])
def health():
    return jsonify(status='ok', time=datetime.now().isoformat())

# -------------------- Run (保持不变) --------------------
if __name__ == '__main__':
    # 修复：使用您日志中的 8000 端口
    app.run(host='0.0.0.0', port=8000, debug=True)
