# backend/app.py (v1.3 - 引入统计 API)
import sqlite3
from datetime import datetime, timedelta
from flask import Flask, render_template, request, jsonify, g, session, redirect, url_for, make_response
import os, json, uuid
from flask_cors import CORS
from functools import wraps
import sys 
from collections import defaultdict # NEW: 引入 defaultdict 用于统计

# --- REVISED PATH LOGIC FOR NUITHA ---
is_frozen = getattr(sys, 'frozen', False)
is_in_onefile_temp = 'onefile_' in os.path.abspath(__file__) 

if is_frozen or is_in_onefile_temp:
    # === 编译 (Nuitka Onefile) 环境 ===
    temp_base_dir = os.path.dirname(os.path.abspath(sys.executable))
    template_dir = os.path.join(temp_base_dir, 'templates')
    static_dir = os.path.join(temp_base_dir, 'static')
    exe_path = os.path.realpath(sys.argv[0])
    exe_dir = os.path.dirname(exe_path)
    DATABASE = os.path.join(exe_dir, 'fitness_manager.db')
    print("--- Detected Nuitka Onefile Path ---")
    print(f"--- **DATABASE is PERMANENT at {DATABASE}** ---")
else:
    # === 开发环境 (Python 脚本运行) ===
    app_file_path = os.path.abspath(__file__)
    backend_dir = os.path.dirname(app_file_path)
    project_root_dir = os.path.dirname(backend_dir)
    template_dir = os.path.join(project_root_dir, 'frontend', 'templates')
    static_dir = os.path.join(project_root_dir, 'frontend', 'static')
    DATABASE = os.path.join(backend_dir, 'fitness_manager.db')
    print("--- Detected Development Path ---")
# --- END REVISED PATH LOGIC ---


print("Template folder:", template_dir)
print("Static folder:", static_dir)
print("DB path:", DATABASE)

app = Flask(__name__, template_folder=template_dir, static_folder=static_dir)
CORS(app)

# --- Session Configuration ---
app.secret_key = str(uuid.uuid4())
# -----------------------------

# -------------------- DB --------------------
def get_db():
    db = getattr(g, '_database', None)
    if db is None:
        db = g._database = sqlite3.connect(DATABASE, check_same_thread=False)
        db.row_factory = sqlite3.Row
    return db

@app.teardown_appcontext
def close_connection(exception):
    db = getattr(g, '_database', None)
    if db is not None:
        db.close()

def init_db():
    db = get_db()
    cur = db.cursor()
    
    cur.execute("""
        CREATE TABLE IF NOT EXISTS todos (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            date TEXT NOT NULL,
            content TEXT NOT NULL,
            is_completed INTEGER DEFAULT 0,
            user_id INTEGER,
            recurring_todo_id INTEGER
        )
    """)
    
    cur.execute("""
        CREATE TABLE IF NOT EXISTS fitness_plans (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            date TEXT NOT NULL,
            exercise_name TEXT NOT NULL,
            sets_reps_duration TEXT,
            is_completed INTEGER DEFAULT 0,
            user_id INTEGER,
            recurring_plan_id INTEGER
        )
    """)
    
    cur.execute("""
        CREATE TABLE IF NOT EXISTS recurring_plans (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            user_id INTEGER,
            plan_name TEXT NOT NULL,
            exercises_json TEXT NOT NULL,
            start_date TEXT NOT NULL,
            end_date TEXT,
            repeat_interval TEXT NOT NULL, -- 'daily', 'weekly'
            repeat_day INTEGER -- 1-7 for weekly (1=Monday), NULL for daily
        )
    """)
    
    cur.execute("""
        CREATE TABLE IF NOT EXISTS recurring_todos (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            user_id INTEGER,
            content TEXT NOT NULL,
            start_date TEXT NOT NULL,
            end_date TEXT,
            repeat_interval TEXT NOT NULL, -- 'daily', 'weekly', 'monthly'
            repeat_day INTEGER -- 1-7 for weekly (1=Monday), 1-31 for monthly
        )
    """)

# --- 用户表和 schema 保证函数 (UPDATED) ---
def ensure_user_schema():
    db = get_db()
    cur = db.cursor()
    
    cur.execute("""
        CREATE TABLE IF NOT EXISTS users (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            username TEXT UNIQUE NOT NULL,
            password TEXT NOT NULL,
            is_admin INTEGER DEFAULT 0
        )
    """)

    # 1. 自动添加 users.is_admin 字段
    try: cur.execute("SELECT is_admin FROM users LIMIT 1")
    except sqlite3.OperationalError:
        print("Migrating users table: Adding is_admin column.")
        cur.execute("ALTER TABLE users ADD COLUMN is_admin INTEGER DEFAULT 0")
    
    # 2. 确保默认用户存在
    cur.execute("SELECT COUNT(*) FROM users WHERE username = 'superuser'")
    if cur.fetchone()[0] == 0:
        print("Inserting default superuser: superuser/adminpassword (Admin)")
        cur.execute("INSERT INTO users (username, password, is_admin) VALUES (?, ?, ?)", ('superuser', 'adminpassword', 1))
        
    cur.execute("SELECT COUNT(*) FROM users WHERE username = 'user1'")
    if cur.fetchone()[0] == 0:
        print("Inserting default regular user: user1/password (Regular)")
        cur.execute("INSERT INTO users (username, password, is_admin) VALUES (?, ?, ?)", ('user1', 'password', 0))

    # 3. 自动更新 (添加) recurring_plans 的新字段
    try: cur.execute("SELECT repeat_interval FROM recurring_plans LIMIT 1")
    except sqlite3.OperationalError:
        print("Migrating recurring_plans table: Adding repeat_interval and repeat_day columns.")
        cur.execute("ALTER TABLE recurring_plans ADD COLUMN repeat_interval TEXT DEFAULT 'daily'")
        cur.execute("ALTER TABLE recurring_plans ADD COLUMN repeat_day INTEGER")
        cur.execute("UPDATE recurring_plans SET repeat_interval = 'daily' WHERE repeat_interval IS NULL")

    # 4. 自动更新 (添加) fitness_plans 的新字段
    try: cur.execute("SELECT recurring_plan_id FROM fitness_plans LIMIT 1")
    except sqlite3.OperationalError:
        print("Migrating fitness_plans table: Adding recurring_plan_id column.")
        cur.execute("ALTER TABLE fitness_plans ADD COLUMN recurring_plan_id INTEGER")

    # 5. 自动更新 (添加) todos 的新字段
    try: cur.execute("SELECT recurring_todo_id FROM todos LIMIT 1")
    except sqlite3.OperationalError:
        print("Migrating todos table: Adding recurring_todo_id column.")
        cur.execute("ALTER TABLE todos ADD COLUMN recurring_todo_id INTEGER")

    # 6. 为所有数据表添加 user_id 字段
    default_user_id_row = db.execute("SELECT id FROM users WHERE username = 'user1'").fetchone()
    if default_user_id_row:
        default_user_id = default_user_id_row['id']
        for table in ['fitness_plans', 'todos', 'recurring_plans', 'recurring_todos']:
            if table == 'users': continue
            try:
                cur.execute(f"SELECT user_id FROM {table} LIMIT 1") 
            except sqlite3.OperationalError:
                try: cur.execute(f"ALTER TABLE {table} ADD COLUMN user_id INTEGER")
                except sqlite3.OperationalError: pass 
            
            cur.execute(f"UPDATE {table} SET user_id = ? WHERE user_id IS NULL", (default_user_id,))
    
    db.commit()
# -----------------------------------------

def ensure_todos_is_completed_column():
    db = get_db()
    cur = db.cursor()
    try: cur.execute("SELECT is_completed FROM todos LIMIT 1")
    except sqlite3.OperationalError:
        cur.execute("ALTER TABLE todos ADD COLUMN is_completed INTEGER DEFAULT 0")
        db.commit()

def ensure_fitness_plans_is_completed_column():
    db = get_db()
    cur = db.cursor()
    try: cur.execute("SELECT is_completed FROM fitness_plans LIMIT 1")
    except sqlite3.OperationalError:
        cur.execute("ALTER TABLE fitness_plans ADD COLUMN is_completed INTEGER DEFAULT 0")
        db.commit()


# 在应用启动时执行所有 Schema 检查
with app.app_context():
    init_db() 
    ensure_todos_is_completed_column()
    ensure_fitness_plans_is_completed_column()
    ensure_user_schema() 


# -------------------- Decorator & Auth --------------------
# (此部分代码无变化)
def login_required(f):
    @wraps(f)
    def decorated_function(*args, **kwargs):
        user_id = session.get('user_id')
        if user_id is None:
            if request.path.startswith('/api/') or request.path.startswith('/fitness_plan/'):
                 return jsonify(status='error', message='Unauthorized'), 401
            return redirect(url_for('index', next=request.url))
        kwargs['user_id'] = user_id
        return f(*args, **kwargs)
    return decorated_function

def admin_required(f):
    @wraps(f)
    def decorated_function(*args, **kwargs):
        user_id = session.get('user_id')
        is_admin = session.get('is_admin', 0)
        if user_id is None:
            return jsonify(status='error', message='Unauthorized'), 401
        if is_admin != 1:
            return jsonify(status='error', message='Admin access required'), 403
        kwargs['user_id'] = user_id
        return f(*args, **kwargs)
    return decorated_function


@app.route('/login', methods=['POST'])
def login():
    username = request.form.get('username') or request.json.get('username')
    password = request.form.get('password') or request.json.get('password')
    db = get_db()
    user = db.execute("SELECT id, username, password, is_admin FROM users WHERE username = ?", (username,)).fetchone()
    if user and user['password'] == password: 
        session.clear()
        session['user_id'] = user['id']
        session['username'] = user['username']
        session['is_admin'] = user['is_admin']
        return jsonify({'status': 'success', 'username': user['username'], 'is_admin': user['is_admin']})
    return jsonify({'status': 'error', 'message': 'Invalid Credentials'}), 401

@app.route('/logout', methods=['POST'])
def logout():
    session.pop('user_id', None)
    session.pop('username', None)
    session.pop('is_admin', None)
    return jsonify({'status': 'success'})

@app.route('/get_current_user', methods=['GET'])
def get_current_user():
    user_id = session.get('user_id')
    username = session.get('username')
    is_admin = session.get('is_admin', 0)
    if user_id:
        return jsonify({'status': 'logged_in', 'user_id': user_id, 'username': username, 'is_admin': is_admin})
    return jsonify({'status': 'logged_out'}), 401

# -------------------- UI Routes --------------------
# (此部分代码无变化)
@app.route('/')
def index():
    if 'user_id' not in session:
        return render_template('login.html') 
    today_str = datetime.now().strftime('%Y-%m-%d')
    _generate_daily_fitness_plan_if_not_exists(session['user_id'], today_str)
    _generate_daily_todos_if_not_exists(session['user_id'], today_str)
    return render_template('display.html', username=session['username'], is_admin=session.get('is_admin', 0)) 

@app.route('/mobile_config')
@login_required
def mobile_config(user_id):
    return render_template('mobile_config.html', username=session['username'])

@app.route('/manage')
@login_required
def manage(user_id):
    return render_template('manage.html', username=session['username'], is_admin=session.get('is_admin', 0))

# -------------------- User Management API (Admin Only) --------------------
# (此部分代码无变化)
@app.route('/api/users', methods=['GET'])
@admin_required
def list_users(user_id):
    db = get_db()
    users = db.execute("SELECT id, username, is_admin FROM users ORDER BY is_admin DESC, username").fetchall()
    return jsonify([dict(u) for u in users])

@app.route('/api/users', methods=['POST'])
@admin_required
def create_user(user_id):
    data = request.get_json()
    username = data.get('username')
    password = data.get('password')
    is_admin = data.get('is_admin', 0)
    if not username or not password: return jsonify(status='error', message='Missing username or password'), 400
    try:
        db = get_db()
        db.execute( "INSERT INTO users (username, password, is_admin) VALUES (?, ?, ?)", (username, password, is_admin))
        db.commit()
        return jsonify(status='success', message='User created')
    except sqlite3.IntegrityError: return jsonify(status='error', message='Username already exists'), 409
    except sqlite3.Error as e: return jsonify(status='error', message=f'Database error: {e}'), 500

@app.route('/api/users/<int:target_user_id>', methods=['DELETE'])
@admin_required
def delete_user(user_id, target_user_id):
    if user_id == target_user_id: return jsonify(status='error', message='Cannot delete self.'), 400
    db = get_db()
    cur = db.cursor()
    target_user = cur.execute("SELECT is_admin FROM users WHERE id = ?", (target_user_id,)).fetchone()
    if target_user and target_user['is_admin'] == 1: return jsonify(status='error', message='Cannot delete another admin.'), 403
    cur.execute("DELETE FROM users WHERE id = ?", (target_user_id,))
    cur.execute("DELETE FROM todos WHERE user_id = ?", (target_user_id,))
    cur.execute("DELETE FROM fitness_plans WHERE user_id = ?", (target_user_id,))
    cur.execute("DELETE FROM recurring_plans WHERE user_id = ?", (target_user_id,))
    cur.execute("DELETE FROM recurring_todos WHERE user_id = ?", (target_user_id,))
    db.commit()
    if cur.rowcount > 0: return jsonify(status='success', message='User and all data deleted.')
    return jsonify(status='error', message='User not found.'), 404

# -------------------- Profile Update API (Any User) --------------------
# (此部分代码无变化)
@app.route('/api/profile', methods=['POST'])
@login_required
def update_profile(user_id):
    data = request.get_json()
    new_username = data.get('username')
    new_password = data.get('password')
    db = get_db()
    cur = db.cursor()
    update_count = 0
    if new_password:
        cur.execute("UPDATE users SET password = ? WHERE id = ?", (new_password, user_id))
        update_count += 1
    if new_username:
        current_user = cur.execute("SELECT username FROM users WHERE id = ?", (user_id,)).fetchone()
        if current_user and current_user['username'] != new_username:
            try:
                cur.execute("UPDATE users SET username = ? WHERE id = ?", (new_username, user_id))
                session['username'] = new_username
                update_count += 1
            except sqlite3.IntegrityError:
                db.rollback()
                return jsonify(status='error', message='Username already exists'), 409
    db.commit()
    if update_count > 0: return jsonify(status='success', message='Profile updated successfully.')
    return jsonify(status='error', message='No fields to update.'), 400

# -------------------- NEW: Stats API --------------------

@app.route('/api/stats/daily_completion', methods=['GET'])
@login_required
def get_daily_completion_stats(user_id):
    start_date = request.args.get('start_date')
    end_date = request.args.get('end_date')

    if not start_date or not end_date:
        return jsonify(status='error', message='Missing start_date or end_date'), 400

    db = get_db()
    
    # 1. 获取 Fitness 数据
    fitness_rows = db.execute(
        "SELECT date, is_completed FROM fitness_plans WHERE user_id = ? AND date BETWEEN ? AND ?",
        (user_id, start_date, end_date)
    ).fetchall()
    
    # 2. 获取 Todos 数据
    todo_rows = db.execute(
        "SELECT date, is_completed FROM todos WHERE user_id = ? AND date BETWEEN ? AND ?",
        (user_id, start_date, end_date)
    ).fetchall()
    
    # 3. 在 Python 中聚合
    fitness_data = defaultdict(lambda: {"total": 0, "completed": 0})
    for row in fitness_rows:
        fitness_data[row['date']]["total"] += 1
        if row['is_completed']:
            fitness_data[row['date']]["completed"] += 1
            
    todo_data = defaultdict(lambda: {"total": 0, "completed": 0})
    for row in todo_rows:
        todo_data[row['date']]["total"] += 1
        if row['is_completed']:
            todo_data[row['date']]["completed"] += 1

    return jsonify({
        "status": "success",
        "fitness_data": fitness_data,
        "todo_data": todo_data
    })

# -------------------- Helpers (Data Generation Logic) --------------------
# (此部分代码无变化)
def _generate_daily_fitness_plan_if_not_exists(user_id, date_str):
    db = get_db()
    cur = db.cursor()
    cur.execute("SELECT COUNT(*) FROM fitness_plans WHERE date = ? AND user_id = ?", (date_str, user_id))
    if cur.fetchone()[0] > 0: return 

    today_date = datetime.strptime(date_str, '%Y-%m-%d').date()
    today_weekday = today_date.isoweekday() 

    recurring_plans = db.execute(
        "SELECT * FROM recurring_plans WHERE user_id = ? AND (start_date <= ?) AND (end_date IS NULL OR end_date >= ?)", 
        (user_id, date_str, date_str)
    ).fetchall()
    
    new_plans_to_insert = []
    for plan in recurring_plans:
        should_add = False
        if plan['repeat_interval'] == 'daily':
            should_add = True
        elif plan['repeat_interval'] == 'weekly' and plan['repeat_day'] == today_weekday:
            should_add = True
        
        if should_add:
            exercises = json.loads(plan['exercises_json'])
            for ex in exercises:
                new_plans_to_insert.append({'name': ex['name'], 'details': ex['details'], 'recurring_id': plan['id']})

    for plan in new_plans_to_insert:
        cur.execute(
            """INSERT INTO fitness_plans (date, exercise_name, sets_reps_duration, is_completed, user_id, recurring_plan_id) VALUES (?, ?, ?, 0, ?, ?)""",
            (date_str, plan['name'], plan['details'], user_id, plan['recurring_id'])
        )
    db.commit()

def _generate_daily_todos_if_not_exists(user_id, date_str):
    db = get_db()
    cur = db.cursor()
    cur.execute("SELECT COUNT(*) FROM todos WHERE date = ? AND user_id = ?", (date_str, user_id))
    if cur.fetchone()[0] > 0: return

    today_date = datetime.strptime(date_str, '%Y-%m-%d').date()
    today_weekday = today_date.isoweekday() 
    today_day_of_month = today_date.day 

    recurring_todos = db.execute(
        "SELECT * FROM recurring_todos WHERE user_id = ? AND (start_date <= ?) AND (end_date IS NULL OR end_date >= ?)", 
        (user_id, date_str, date_str)
    ).fetchall()
    
    new_todos_to_insert = []
    for todo in recurring_todos:
        should_add = False
        if todo['repeat_interval'] == 'daily': should_add = True
        elif todo['repeat_interval'] == 'weekly' and todo['repeat_day'] == today_weekday: should_add = True
        elif todo['repeat_interval'] == 'monthly' and todo['repeat_day'] == today_day_of_month: should_add = True
        if should_add:
            new_todos_to_insert.append({'content': todo['content'], 'recurring_id': todo['id']})

    for todo in new_todos_to_insert:
        cur.execute(
            """INSERT INTO todos (date, content, is_completed, user_id, recurring_todo_id) VALUES (?, ?, 0, ?, ?)""",
            (date_str, todo['content'], user_id, todo['recurring_id'])
        )
    db.commit()


# -------------------- Fitness API --------------------
# (此部分代码无变化)
@app.route('/fitness_plan/<date_str>', methods=['GET'])
@login_required
def get_fitness_plan(date_str, user_id):
    _generate_daily_fitness_plan_if_not_exists(user_id, date_str)
    db = get_db()
    plans = db.execute("SELECT * FROM fitness_plans WHERE date = ? AND user_id = ? ORDER BY id", (date_str, user_id)).fetchall()
    return jsonify([dict(plan) for plan in plans])

@app.route('/mark_exercise_completed_mobile', methods=['POST'])
@login_required
def mark_exercise_completed_mobile(user_id):
    data = request.get_json()
    plan_id = data.get('plan_id')
    is_completed = data.get('is_completed')
    if plan_id is None or is_completed is None: return jsonify(status='error', message='Missing plan_id or is_completed'), 400
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute("UPDATE fitness_plans SET is_completed = ? WHERE id = ? AND user_id = ?", (is_completed, plan_id, user_id))
        db.commit()
        if cur.rowcount > 0: return jsonify(status='success')
        else: return jsonify(status='error', message='Plan not found or not owned by user.'), 404
    except sqlite3.Error as e: return jsonify(status='error', message=f'Database error: {e}'), 500

@app.route('/delete_fitness_plan_exercise/<int:plan_id>', methods=['POST'])
@login_required
def delete_fitness_plan_exercise(plan_id, user_id):
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute("DELETE FROM fitness_plans WHERE id = ? AND user_id = ?", (plan_id, user_id))
        db.commit()
        if cur.rowcount > 0: return jsonify(status='success', message='Daily exercise deleted successfully.')
        else: return jsonify(status='error', message='Daily exercise not found or not owned by user.'), 404
    except sqlite3.Error as e: return jsonify(status='error', message=f'Database error: {e}'), 500

# -------------------- Todo API --------------------
# (此部分代码无变化)
@app.route('/todos', methods=['GET'])
@login_required
def get_todos(user_id):
    date_str = request.args.get('date') or datetime.now().strftime('%Y-%m-%d')
    _generate_daily_todos_if_not_exists(user_id, date_str)
    db = get_db()
    todos = db.execute("SELECT * FROM todos WHERE date = ? AND user_id = ? ORDER BY id", (date_str, user_id)).fetchall()
    return jsonify([dict(todo) for todo in todos])

@app.route('/add_todo', methods=['POST'])
@login_required
def add_todo(user_id):
    data = request.get_json()
    date_str = data.get('date')
    content = data.get('content')
    if not date_str or not content: return jsonify(status='error', message='Missing date or content'), 400
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute("INSERT INTO todos (date, content, is_completed, user_id) VALUES (?, ?, 0, ?)", (date_str, content, user_id))
        db.commit()
        return jsonify(status='success', todo_id=cur.lastrowid)
    except sqlite3.Error as e: return jsonify(status='error', message=f'Database error: {e}'), 500

@app.route('/mark_todo_completed', methods=['POST'])
@login_required
def mark_todo_completed(user_id):
    data = request.get_json()
    todo_id = data.get('todo_id')
    is_completed = data.get('is_completed')
    if todo_id is None or is_completed is None: return jsonify(status='error', message='Missing todo_id or is_completed'), 400
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute("UPDATE todos SET is_completed = ? WHERE id = ? AND user_id = ?", (is_completed, todo_id, user_id))
        db.commit()
        if cur.rowcount > 0: return jsonify(status='success')
        else: return jsonify(status='error', message='Todo not found or not owned by user.'), 404
    except sqlite3.Error as e: return jsonify(status='error', message=f'Database error: {e}'), 500

@app.route('/delete_todo/<int:todo_id>', methods=['POST'])
@login_required
def delete_todo(todo_id, user_id):
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute("DELETE FROM todos WHERE id = ? AND user_id = ?", (todo_id, user_id))
        db.commit()
        if cur.rowcount > 0: return jsonify(status='success', message='Todo deleted successfully.')
        else: return jsonify(status='error', message='Todo not found or not owned by user.'), 404
    except sqlite3.Error as e: return jsonify(status='error', message=f'Database error: {e}'), 500

# -------------------- Recurring Plans API --------------------
# (此部分代码无变化)
@app.route('/get_recurring_plans', methods=['GET'])
@login_required
def get_recurring_plans(user_id):
    db = get_db()
    plans = db.execute("SELECT * FROM recurring_plans WHERE user_id = ? ORDER BY start_date DESC", (user_id,)).fetchall()
    return jsonify([dict(plan) for plan in plans])

@app.route('/add_recurring_plan', methods=['POST'])
@login_required
def add_recurring_plan(user_id):
    data = request.get_json()
    if not data.get('plan_name') or not data.get('start_date') or not data.get('repeat_interval') or not data.get('exercises'):
        return jsonify(status='error', message='Missing required fields'), 400
    if data.get('repeat_interval') == 'weekly' and not data.get('repeat_day'):
        return jsonify(status='error', message='Missing repeat_day for weekly interval'), 400
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute(
            """INSERT INTO recurring_plans (user_id, plan_name, exercises_json, start_date, end_date, repeat_interval, repeat_day) VALUES (?, ?, ?, ?, ?, ?, ?)""",
            (user_id, data.get('plan_name'), json.dumps(data.get('exercises')), data.get('start_date'), data.get('end_date'), data.get('repeat_interval'), data.get('repeat_day'))
        )
        db.commit()
        return jsonify(status='success', plan_id=cur.lastrowid)
    except sqlite3.Error as e: return jsonify(status='error', message=f'Database error: {e}'), 500

@app.route('/delete_recurring_plan/<int:plan_id>', methods=['POST'])
@login_required
def delete_recurring_plan(plan_id, user_id):
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute("DELETE FROM recurring_plans WHERE id = ? AND user_id = ?", (plan_id, user_id))
        db.commit()
        if cur.rowcount > 0: return jsonify(status='success', message='Recurring plan deleted successfully.')
        else: return jsonify(status='error', message='Recurring plan not found or not owned by user.'), 404
    except sqlite3.Error as e: return jsonify(status='error', message=f'Database error: {e}'), 500

# -------------------- Recurring Todos API --------------------
# (此部分代码无变化)
@app.route('/get_recurring_todos', methods=['GET'])
@login_required
def get_recurring_todos(user_id):
    db = get_db()
    todos = db.execute("SELECT * FROM recurring_todos WHERE user_id = ? ORDER BY start_date DESC", (user_id,)).fetchall()
    return jsonify([dict(todo) for todo in todos])

@app.route('/add_recurring_todo', methods=['POST'])
@login_required
def add_recurring_todo(user_id):
    data = request.get_json()
    if not data.get('content') or not data.get('start_date') or not data.get('repeat_interval'):
        return jsonify(status='error', message='Missing required fields'), 400
    if (data.get('repeat_interval') == 'weekly' or data.get('repeat_interval') == 'monthly') and not data.get('repeat_day'):
        return jsonify(status='error', message='Missing repeat_day for interval'), 400
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute(
            """INSERT INTO recurring_todos (user_id, content, start_date, end_date, repeat_interval, repeat_day) VALUES (?, ?, ?, ?, ?, ?)""",
            (user_id, data.get('content'), data.get('start_date'), data.get('end_date'), data.get('repeat_interval'), data.get('repeat_day'))
        )
        db.commit()
        return jsonify(status='success', todo_id=cur.lastrowid)
    except sqlite3.Error as e: return jsonify(status='error', message=f'Database error: {e}'), 500

@app.route('/delete_recurring_todo/<int:todo_id>', methods=['POST'])
@login_required
def delete_recurring_todo(todo_id, user_id):
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute("DELETE FROM recurring_todos WHERE id = ? AND user_id = ?", (todo_id, user_id))
        db.commit()
        if cur.rowcount > 0: return jsonify(status='success', message='Recurring todo deleted successfully.')
        else: return jsonify(status='error', message='Recurring todo not found or not owned by user.'), 404
    except sqlite3.Error as e: return jsonify(status='error', message=f'Database error: {e}'), 500


# -------------------- Compatibility headers & health --------------------
# (此部分代码无变化)
@app.after_request
def add_common_headers(response):
    if 'Content-Type' not in response.headers:
        response.headers['Content-Type'] = 'text/html; charset=utf-8'
    else:
        c = response.headers['Content-Type']
        if c.startswith('text/') and 'charset' not in c:
            response.headers['Content-Type'] = c + '; charset=utf-8'
    response.headers.setdefault('X-Content-Type-Options', 'nosniff')
    response.headers.setdefault('X-Frame-Options', 'SAMEORIGIN')
    path = request.path or ''
    if path.startswith('/static/') or path.endswith(('.css', '.js', '.png', '.jpg', '.svg', '.webp')):
        response.headers['Cache-Control'] = 'public, max-age=3600'
    else:
        response.headers['Cache-Control'] = 'no-cache, no-store, must-revalidate'
    return response

@app.route('/health', methods=['GET'])
def health():
    return jsonify(status='ok', time=datetime.now().isoformat())

# -------------------- Run --------------------
if __name__ == '__main__':
    app.run(host='0.0.0.0', port=8000, debug=True)
