# backend/app.py (v1.7 - Fixed Timezone & Indentation)
import sqlite3
from datetime import datetime, timedelta, timezone
from flask import Flask, render_template, request, jsonify, g, redirect, url_for, make_response
from flask_login import LoginManager, UserMixin, login_user, logout_user, login_required, current_user
import os, json, uuid
from flask_cors import CORS
from functools import wraps
import sys 
from collections import defaultdict

# --- 时区设置 (强制使用 UTC+8 中国标准时间) ---
CN_TZ = timezone(timedelta(hours=8))

# --- Nuitka 路径逻辑 (保持不变) ---
is_frozen = getattr(sys, 'frozen', False)
is_in_onefile_temp = 'onefile_' in os.path.abspath(__file__) 

if is_frozen or is_in_onefile_temp:
    temp_base_dir = os.path.dirname(os.path.abspath(sys.executable))
    template_dir = os.path.join(temp_base_dir, 'templates')
    static_dir = os.path.join(temp_base_dir, 'static')
    exe_path = os.path.realpath(sys.argv[0])
    exe_dir = os.path.dirname(exe_path)
    DATABASE = os.path.join(exe_dir, 'fitness_manager.db')
    print("--- Detected Nuitka Onefile Path ---")
    print(f"--- **DATABASE is PERMANENT at {DATABASE}** ---")
else:
    app_file_path = os.path.abspath(__file__)
    backend_dir = os.path.dirname(app_file_path)
    project_root_dir = os.path.dirname(backend_dir)
    template_dir = os.path.join(project_root_dir, 'frontend', 'templates')
    static_dir = os.path.join(project_root_dir, 'frontend', 'static')
    DATABASE = os.path.join(backend_dir, 'fitness_manager.db')
    print("--- Detected Development Path ---")

print("Template folder:", template_dir)
print("Static folder:", static_dir)
print("DB path:", DATABASE)

app = Flask(__name__, template_folder=template_dir, static_folder=static_dir)
CORS(app)
app.secret_key = str(uuid.uuid4())

# --- Flask-Login 配置 ---
login_manager = LoginManager()
login_manager.init_app(app)
login_manager.login_view = 'index' 
login_manager.login_message = "请先登录以访问此页面。"

# ★★★ User 类
class User(UserMixin):
    def __init__(self, id, username, is_admin, manager_id):
        self.id = id
        self.username = username
        self.is_admin = is_admin
        self.manager_id = manager_id 
    
    @property
    def is_super_admin(self):
        return self.is_admin and (self.manager_id is None or self.manager_id == 0)

@login_manager.user_loader
def load_user(user_id):
    db = get_db()
    user_row = db.execute("SELECT id, username, is_admin, manager_id FROM users WHERE id = ?", (user_id,)).fetchone()
    if user_row:
        return User(
            id=user_row['id'], 
            username=user_row['username'], 
            is_admin=user_row['is_admin'], 
            manager_id=user_row['manager_id']
        )
    return None

# -------------------- DB --------------------
def get_db():
    db = getattr(g, '_database', None)
    if db is None:
        db = g._database = sqlite3.connect(DATABASE, check_same_thread=False)
        db.row_factory = sqlite3.Row
    return db

def get_setting(key, default=None):
    db = get_db()
    row = db.execute("SELECT value FROM settings WHERE key = ?", (key,)).fetchone()
    if row is None:
        return default
    return row["value"]

def set_setting(key, value):
    db = get_db()
    cur = db.cursor()
    cur.execute(
        "INSERT INTO settings (key, value) VALUES (?, ?) "
        "ON CONFLICT(key) DO UPDATE SET value = excluded.value",
        (key, str(value)),
    )
    db.commit()

@app.teardown_appcontext
def close_connection(exception):
    db = getattr(g, '_database', None)
    if db is not None:
        db.close()

def init_db():
    db = get_db()
    cur = db.cursor()
    cur.execute("""
        CREATE TABLE IF NOT EXISTS todos (
            id INTEGER PRIMARY KEY AUTOINCREMENT, 
            date TEXT NOT NULL, 
            content TEXT NOT NULL,
            is_completed INTEGER DEFAULT 0, 
            user_id INTEGER, 
            recurring_todo_id INTEGER
        )
    """)
    cur.execute("""
        CREATE TABLE IF NOT EXISTS fitness_plans (
            id INTEGER PRIMARY KEY AUTOINCREMENT, 
            date TEXT NOT NULL, 
            exercise_name TEXT NOT NULL,
            sets_reps_duration TEXT, 
            is_completed INTEGER DEFAULT 0, 
            user_id INTEGER, 
            recurring_plan_id INTEGER
        )
    """)
    cur.execute("""
        CREATE TABLE IF NOT EXISTS recurring_plans (
            id INTEGER PRIMARY KEY AUTOINCREMENT, 
            user_id INTEGER, 
            plan_name TEXT NOT NULL,
            exercises_json TEXT NOT NULL, 
            start_date TEXT NOT NULL, 
            end_date TEXT,
            repeat_interval TEXT NOT NULL, 
            repeat_day INTEGER
        )
    """)
    cur.execute("""
        CREATE TABLE IF NOT EXISTS recurring_todos (
            id INTEGER PRIMARY KEY AUTOINCREMENT, 
            user_id INTEGER, 
            content TEXT NOT NULL,
            start_date TEXT NOT NULL, 
            end_date TEXT, 
            repeat_interval TEXT NOT NULL, 
            repeat_day INTEGER
        )
    """)
    cur.execute("""
        CREATE TABLE IF NOT EXISTS exercise_library (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            user_id INTEGER,
            name TEXT NOT NULL,
            details TEXT
        )
    """)
    cur.execute("""
        CREATE TABLE IF NOT EXISTS settings (
            key   TEXT PRIMARY KEY,
            value TEXT
        )
    """)
    cur.execute("""
        CREATE TABLE IF NOT EXISTS todo_library (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            user_id INTEGER,
            content TEXT NOT NULL
        )
    """)

def ensure_user_schema():
    db = get_db()
    cur = db.cursor()
    cur.execute("""
        CREATE TABLE IF NOT EXISTS users (
            id INTEGER PRIMARY KEY AUTOINCREMENT, 
            username TEXT UNIQUE NOT NULL,
            password TEXT NOT NULL, 
            is_admin INTEGER DEFAULT 0,
            manager_id INTEGER REFERENCES users(id)
        )
    """)
    try: cur.execute("SELECT is_admin FROM users LIMIT 1")
    except sqlite3.OperationalError:
        cur.execute("ALTER TABLE users ADD COLUMN is_admin INTEGER DEFAULT 0")
    
    try: cur.execute("SELECT manager_id FROM users LIMIT 1")
    except sqlite3.OperationalError:
        cur.execute("ALTER TABLE users ADD COLUMN manager_id INTEGER REFERENCES users(id)")
    
    cur.execute("SELECT COUNT(*) FROM users WHERE username = 'superuser'")
    if cur.fetchone()[0] == 0:
        cur.execute("INSERT INTO users (username, password, is_admin, manager_id) VALUES (?, ?, ?, ?)", 
                    ('superuser', 'adminpassword', 1, None))
        
    su_row = cur.execute("SELECT id FROM users WHERE username = 'superuser'").fetchone()
    su_id = su_row['id'] if su_row else 1
    
    cur.execute("SELECT COUNT(*) FROM users WHERE username = 'user1'")
    if cur.fetchone()[0] == 0:
        cur.execute("INSERT INTO users (username, password, is_admin, manager_id) VALUES (?, ?, ?, ?)", 
                    ('user1', 'password', 0, su_id))

    try: cur.execute("SELECT repeat_interval FROM recurring_plans LIMIT 1")
    except sqlite3.OperationalError:
        cur.execute("ALTER TABLE recurring_plans ADD COLUMN repeat_interval TEXT DEFAULT 'daily'")
        cur.execute("ALTER TABLE recurring_plans ADD COLUMN repeat_day INTEGER")
        cur.execute("UPDATE recurring_plans SET repeat_interval = 'daily' WHERE repeat_interval IS NULL")

    try: cur.execute("SELECT recurring_plan_id FROM fitness_plans LIMIT 1")
    except sqlite3.OperationalError:
        cur.execute("ALTER TABLE fitness_plans ADD COLUMN recurring_plan_id INTEGER")

    try: cur.execute("SELECT recurring_todo_id FROM todos LIMIT 1")
    except sqlite3.OperationalError:
        cur.execute("ALTER TABLE todos ADD COLUMN recurring_todo_id INTEGER")

    default_user_id_row = db.execute("SELECT id FROM users WHERE username = 'user1'").fetchone()
    if default_user_id_row:
        default_user_id = default_user_id_row['id']
        for table in ['fitness_plans', 'todos', 'recurring_plans', 'recurring_todos', 'exercise_library', 'todo_library']:
            if table == 'users': continue
            try: cur.execute(f"SELECT user_id FROM {table} LIMIT 1") 
            except sqlite3.OperationalError:
                try: cur.execute(f"ALTER TABLE {table} ADD COLUMN user_id INTEGER")
                except sqlite3.OperationalError: pass 
            cur.execute(f"UPDATE {table} SET user_id = ? WHERE user_id IS NULL", (default_user_id,))
    db.commit()

def ensure_todos_is_completed_column():
    db = get_db()
    cur = db.cursor()
    try: cur.execute("SELECT is_completed FROM todos LIMIT 1")
    except sqlite3.OperationalError:
        cur.execute("ALTER TABLE todos ADD COLUMN is_completed INTEGER DEFAULT 0")
        db.commit()

def ensure_fitness_plans_is_completed_column():
    db = get_db()
    cur = db.cursor()
    try: cur.execute("SELECT is_completed FROM fitness_plans LIMIT 1")
    except sqlite3.OperationalError:
        cur.execute("ALTER TABLE fitness_plans ADD COLUMN is_completed INTEGER DEFAULT 0")
        db.commit()

with app.app_context():
    init_db() 
    ensure_todos_is_completed_column()
    ensure_fitness_plans_is_completed_column()
    ensure_user_schema() 


# -------------------- Auth Routes --------------------

@app.route('/login', methods=['POST'])
def login():
    username = request.form.get('username') or request.json.get('username')
    password = request.form.get('password') or request.json.get('password')
    db = get_db()
    user_row = db.execute("SELECT id, username, password, is_admin, manager_id FROM users WHERE username = ?", (username,)).fetchone()
    
    if user_row and user_row['password'] == password: 
        user_obj = User(
            id=user_row['id'], 
            username=user_row['username'], 
            is_admin=user_row['is_admin'],
            manager_id=user_row['manager_id']
        )
        login_user(user_obj, remember=True) 
        return jsonify({
            'status': 'success', 
            'username': user_obj.username, 
            'is_admin': user_obj.is_admin,
            'manager_id': user_obj.manager_id
        })
    
    return jsonify({'status': 'error', 'message': '用户名或密码无效'}), 401

@app.route('/logout', methods=['POST'])
@login_required
def logout():
    logout_user() 
    return jsonify({'status': 'success'})

@app.route('/get_current_user', methods=['GET'])
def get_current_user():
    if current_user.is_authenticated:
        return jsonify({
            'status': 'logged_in', 
            'user_id': current_user.id, 
            'username': current_user.username, 
            'is_admin': current_user.is_admin,
            'manager_id': current_user.manager_id
        })
    return jsonify({'status': 'logged_out'}), 401

# -------------------- UI Routes --------------------
@app.route('/')
def index():
    if not current_user.is_authenticated:
        return render_template('login.html') 
    
    view_mode = request.args.get('view')
    if view_mode == 'display':
        pass
    elif 'Mobile' in request.headers.get('User-Agent', '') or 'Android' in request.headers.get('User-Agent', ''):
        return redirect(url_for('mobile_config'))

    # ★ 核心修改：使用中国时区获取“今天”的日期
    today_str = datetime.now(CN_TZ).strftime('%Y-%m-%d')
    
    _generate_daily_fitness_plan_if_not_exists(current_user.id, today_str)
    _generate_daily_todos_if_not_exists(current_user.id, today_str)
    return render_template('display.html', username=current_user.username, is_admin=current_user.is_admin) 

@app.route('/mobile_config')
@login_required
def mobile_config():
    return render_template('mobile_config.html', username=current_user.username)

@app.route('/api/settings/scroll_speed', methods=['GET', 'POST'])
def scroll_speed_setting():
    if request.method == 'GET':
        value = get_setting('scroll_speed', '25')
        try: speed = int(value)
        except (TypeError, ValueError): speed = 25
        return jsonify({"scroll_speed": speed})

    data = request.get_json() or {}
    try: speed = int(data.get('scroll_speed', 25))
    except (TypeError, ValueError):
        return jsonify({"status": "error", "message": "滚动速度必须是整数"}), 400
    if speed < 5: speed = 5
    if speed > 120: speed = 120
    set_setting('scroll_speed', speed)
    return jsonify({"status": "success", "scroll_speed": speed})


@app.route('/manage')
@login_required
def manage():
    return render_template(
        'manage.html', 
        username=current_user.username, 
        is_admin=current_user.is_admin,
        manager_id=current_user.manager_id
    )

# -------------------- User Management API --------------------

def admin_required(f):
    @wraps(f)
    def decorated_function(*args, **kwargs):
        if not current_user.is_authenticated or not current_user.is_admin:
            return jsonify(status='error', message='需要管理员权限'), 403
        return f(*args, **kwargs)
    return decorated_function

@app.route('/api/users', methods=['GET'])
@login_required
@admin_required 
def list_users():
    db = get_db()
    users = db.execute(
        "SELECT id, username, is_admin FROM users WHERE manager_id = ? ORDER BY is_admin DESC, username", 
        (current_user.id,)
    ).fetchall()
    return jsonify([dict(u) for u in users])

@app.route('/api/users', methods=['POST'])
@login_required
@admin_required
def create_user():
    data = request.get_json()
    username = data.get('username')
    password = data.get('password')
    is_admin_from_form = data.get('is_admin', 0)
    
    is_super_admin = current_user.is_super_admin
    
    is_admin_to_create = 0
    if is_super_admin:
        is_admin_to_create = 1 if is_admin_from_form else 0
    
    manager_id = current_user.id 

    if not username or not password: return jsonify(status='error', message='缺少用户名或密码'), 400
    try:
        db = get_db()
        db.execute(
            "INSERT INTO users (username, password, is_admin, manager_id) VALUES (?, ?, ?, ?)", 
            (username, password, is_admin_to_create, manager_id)
        )
        db.commit()
        return jsonify(status='success', message='用户创建成功')
    except sqlite3.IntegrityError: 
        return jsonify(status='error', message='用户名已存在'), 409
    except sqlite3.Error as e: 
        return jsonify(status='error', message=f'数据库错误: {e}'), 500

@app.route('/api/users/<int:target_user_id>', methods=['DELETE'])
@login_required
@admin_required
def delete_user(target_user_id):
    if current_user.id == target_user_id: 
        return jsonify(status='error', message='不能删除自己'), 400
    
    db = get_db()
    cur = db.cursor()
    
    target_user = cur.execute(
        "SELECT id FROM users WHERE id = ? AND manager_id = ?", 
        (target_user_id, current_user.id)
    ).fetchone()

    if not target_user:
        return jsonify(status='error', message='未找到用户或您无权删除此用户'), 403

    cur.execute("DELETE FROM users WHERE id = ?", (target_user_id,))
    cur.execute("DELETE FROM todos WHERE user_id = ?", (target_user_id,))
    cur.execute("DELETE FROM fitness_plans WHERE user_id = ?", (target_user_id,))
    cur.execute("DELETE FROM recurring_plans WHERE user_id = ?", (target_user_id,))
    cur.execute("DELETE FROM recurring_todos WHERE user_id = ?", (target_user_id,))
    cur.execute("DELETE FROM exercise_library WHERE user_id = ?", (target_user_id,))
    cur.execute("DELETE FROM todo_library WHERE user_id = ?", (target_user_id,))
    
    db.commit()
    
    if cur.rowcount > 0: 
        return jsonify(status='success', message='用户及所有数据已删除')
    
    return jsonify(status='error', message='删除失败'), 500

# -------------------- Profile Update API --------------------
@app.route('/api/profile', methods=['POST'])
@login_required
def update_profile():
    user_id = current_user.id
    data = request.get_json()
    new_username = data.get('username')
    new_password = data.get('password')
    db = get_db()
    cur = db.cursor()
    update_count = 0
    if new_password:
        cur.execute("UPDATE users SET password = ? WHERE id = ?", (new_password, user_id))
        update_count += 1
    if new_username:
        current_user_db = cur.execute("SELECT username FROM users WHERE id = ?", (user_id,)).fetchone()
        if current_user_db and current_user_db['username'] != new_username:
            try:
                cur.execute("UPDATE users SET username = ? WHERE id = ?", (new_username, user_id))
                update_count += 1
            except sqlite3.IntegrityError:
                db.rollback()
                return jsonify(status='error', message='用户名已存在'), 409
    db.commit()
    if update_count > 0: 
        return jsonify(status='success', message='个人资料更新成功。用户名的更改将在下次登录时生效。')
    return jsonify(status='error', message='没有要更新的字段'), 400

# -------------------- Stats API --------------------
@app.route('/api/stats/daily_completion', methods=['GET'])
@login_required
def get_daily_completion_stats():
    user_id = current_user.id
    start_date = request.args.get('start_date')
    end_date = request.args.get('end_date')
    if not start_date or not end_date: return jsonify(status='error', message='缺少开始日期或结束日期'), 400
    db = get_db()
    fitness_rows = db.execute("SELECT date, is_completed FROM fitness_plans WHERE user_id = ? AND date BETWEEN ? AND ?", (user_id, start_date, end_date)).fetchall()
    todo_rows = db.execute("SELECT date, is_completed FROM todos WHERE user_id = ? AND date BETWEEN ? AND ?", (user_id, start_date, end_date)).fetchall()
    fitness_data = defaultdict(lambda: {"total": 0, "completed": 0})
    for row in fitness_rows:
        fitness_data[row['date']]["total"] += 1
        if row['is_completed']: fitness_data[row['date']]["completed"] += 1
    todo_data = defaultdict(lambda: {"total": 0, "completed": 0})
    for row in todo_rows:
        todo_data[row['date']]["total"] += 1
        if row['is_completed']: todo_data[row['date']]["completed"] += 1
    return jsonify({"status": "success", "fitness_data": fitness_data, "todo_data": todo_data})

# -------------------- Helpers --------------------
def _get_daily_fitness_plans(user_id, date_str):
    db = get_db()
    plans = db.execute("SELECT * FROM fitness_plans WHERE date = ? AND user_id = ? ORDER BY id", (date_str, user_id)).fetchall()
    return [dict(plan) for plan in plans]
def _generate_daily_fitness_plan_if_not_exists(user_id, date_str):
    db = get_db()
    cur = db.cursor()
    cur.execute("SELECT COUNT(*) FROM fitness_plans WHERE date = ? AND user_id = ?", (date_str, user_id))
    if cur.fetchone()[0] > 0: return 
    today_date = datetime.strptime(date_str, '%Y-%m-%d').date()
    today_weekday = today_date.isoweekday() 
    recurring_plans = db.execute("SELECT * FROM recurring_plans WHERE user_id = ? AND (start_date <= ?) AND (end_date IS NULL OR end_date >= ?)", (user_id, date_str, date_str)).fetchall()
    new_plans_to_insert = []
    for plan in recurring_plans:
        should_add = False
        if plan['repeat_interval'] == 'daily': should_add = True
        elif plan['repeat_interval'] == 'weekly' and plan['repeat_day'] == today_weekday: should_add = True
        if should_add:
            exercises = json.loads(plan['exercises_json'])
            for ex in exercises: new_plans_to_insert.append({'name': ex['name'], 'details': ex['details'], 'recurring_id': plan['id']})
    for plan in new_plans_to_insert:
        cur.execute(
            """INSERT INTO fitness_plans (date, exercise_name, sets_reps_duration, is_completed, user_id, recurring_plan_id) VALUES (?, ?, ?, 0, ?, ?)""",
            (date_str, plan['name'], plan['details'], user_id, plan['recurring_id']))
    db.commit()
def _generate_daily_todos_if_not_exists(user_id, date_str):
    db = get_db()
    cur = db.cursor()
    cur.execute("SELECT COUNT(*) FROM todos WHERE date = ? AND user_id = ?", (date_str, user_id))
    if cur.fetchone()[0] > 0: return
    today_date = datetime.strptime(date_str, '%Y-%m-%d').date()
    today_weekday = today_date.isoweekday() 
    today_day_of_month = today_date.day 
    recurring_todos = db.execute("SELECT * FROM recurring_todos WHERE user_id = ? AND (start_date <= ?) AND (end_date IS NULL OR end_date >= ?)", (user_id, date_str, date_str)).fetchall()
    new_todos_to_insert = []
    for todo in recurring_todos:
        should_add = False
        if todo['repeat_interval'] == 'daily': should_add = True
        elif todo['repeat_interval'] == 'weekly' and todo['repeat_day'] == today_weekday: should_add = True
        elif todo['repeat_interval'] == 'monthly' and todo['repeat_day'] == today_day_of_month: should_add = True
        if should_add: new_todos_to_insert.append({'content': todo['content'], 'recurring_id': todo['id']})
    for todo in new_todos_to_insert:
        cur.execute(
            """INSERT INTO todos (date, content, is_completed, user_id, recurring_todo_id) VALUES (?, ?, 0, ?, ?)""",
            (date_str, todo['content'], user_id, todo['recurring_id']))
    db.commit()

# -------------------- API Routes --------------------

@app.route('/fitness_plan/<date_str>', methods=['GET'])
@login_required
def get_fitness_plan(date_str):
    user_id = current_user.id
    _generate_daily_fitness_plan_if_not_exists(user_id, date_str)
    db = get_db()
    plans = db.execute("SELECT * FROM fitness_plans WHERE date = ? AND user_id = ? ORDER BY id", (date_str, user_id)).fetchall()
    return jsonify([dict(plan) for plan in plans])

@app.route('/mark_exercise_completed_mobile', methods=['POST'])
@login_required
def mark_exercise_completed_mobile():
    user_id = current_user.id
    data = request.get_json()
    plan_id = data.get('plan_id')
    is_completed = data.get('is_completed')
    if plan_id is None or is_completed is None: return jsonify(status='error', message='缺少计划ID或完成状态'), 400
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute("UPDATE fitness_plans SET is_completed = ? WHERE id = ? AND user_id = ?", (is_completed, plan_id, user_id))
        db.commit()
        if cur.rowcount > 0: return jsonify(status='success')
        else: return jsonify(status='error', message='未找到计划或您无权修改'), 404
    except sqlite3.Error as e: return jsonify(status='error', message=f'数据库错误: {e}'), 500

@app.route('/delete_fitness_plan_exercise/<int:plan_id>', methods=['POST'])
@login_required
def delete_fitness_plan_exercise(plan_id):
    user_id = current_user.id
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute("DELETE FROM fitness_plans WHERE id = ? AND user_id = ?", (plan_id, user_id))
        db.commit()
        if cur.rowcount > 0: return jsonify(status='success', message='每日健身已删除')
        else: return jsonify(status='error', message='未找到每日健身计划'), 404
    except sqlite3.Error as e: return jsonify(status='error', message=f'数据库错误: {e}'), 500

@app.route('/todos', methods=['GET'])
@login_required
def get_todos():
    user_id = current_user.id
    # ★ 核心修改：默认使用中国时区日期
    date_str = request.args.get('date') or datetime.now(CN_TZ).strftime('%Y-%m-%d')
    db = get_db()
    todos = db.execute(
        "SELECT * FROM todos WHERE date = ? AND user_id = ? ORDER BY id",
        (date_str, user_id)
    ).fetchall()
    return jsonify([dict(todo) for todo in todos])

@app.route('/add_todo', methods=['POST'])
@login_required
def add_todo():
    user_id = current_user.id
    data = request.get_json()
    date_str = data.get('date')
    content = data.get('content')
    if not date_str or not content: return jsonify(status='error', message='缺少日期或内容'), 400
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute("INSERT INTO todos (date, content, is_completed, user_id) VALUES (?, ?, 0, ?)", (date_str, content, user_id))
        db.commit()
        return jsonify(status='success', todo_id=cur.lastrowid)
    except sqlite3.Error as e: return jsonify(status='error', message=f'数据库错误: {e}'), 500

@app.route('/mark_todo_completed', methods=['POST'])
@login_required
def mark_todo_completed():
    user_id = current_user.id
    data = request.get_json()
    todo_id = data.get('todo_id')
    is_completed = data.get('is_completed')
    if todo_id is None or is_completed is None: return jsonify(status='error', message='缺少待办ID或完成状态'), 400
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute("UPDATE todos SET is_completed = ? WHERE id = ? AND user_id = ?", (is_completed, todo_id, user_id))
        db.commit()
        if cur.rowcount > 0: return jsonify(status='success')
        else: return jsonify(status='error', message='未找到待办事项或您无权修改'), 404
    except sqlite3.Error as e: return jsonify(status='error', message=f'数据库错误: {e}'), 500


@app.route('/delete_todo/<int:todo_id>', methods=['POST'])
@login_required
def delete_todo(todo_id):
    user_id = current_user.id
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute("DELETE FROM todos WHERE id = ? AND user_id = ?", (todo_id, user_id))
        db.commit()
        if cur.rowcount > 0:
            return jsonify(status='success', message='待办事项已删除')
        else:
            return jsonify(status='error', message='未找到待办事项'), 404
    except sqlite3.Error as e:
        return jsonify(status='error', message=f'数据库错误: {e}'), 500


@app.route('/get_recurring_plans', methods=['GET'])
@login_required
def get_recurring_plans():
    user_id = current_user.id
    db = get_db()
    plans = db.execute("SELECT * FROM recurring_plans WHERE user_id = ? ORDER BY start_date DESC", (user_id,)).fetchall()
    return jsonify([dict(plan) for plan in plans])

@app.route('/add_recurring_plan', methods=['POST'])
@login_required
def add_recurring_plan():
    user_id = current_user.id
    data = request.get_json()
    if not data.get('plan_name') or not data.get('start_date') or not data.get('repeat_interval') or not data.get('exercises'):
        return jsonify(status='error', message='缺少必填字段'), 400
    if data.get('repeat_interval') == 'weekly' and not data.get('repeat_day'):
        return jsonify(status='error', message='“每周”循环缺少“星期几”设置'), 400
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute(
            """INSERT INTO recurring_plans (user_id, plan_name, exercises_json, start_date, end_date, repeat_interval, repeat_day) VALUES (?, ?, ?, ?, ?, ?, ?)""",
            (user_id, data.get('plan_name'), json.dumps(data.get('exercises')), data.get('start_date'), data.get('end_date'), data.get('repeat_interval'), data.get('repeat_day'))
        )
        db.commit()
        return jsonify(status='success', plan_id=cur.lastrowid)
    except sqlite3.Error as e: return jsonify(status='error', message=f'数据库错误: {e}'), 500

@app.route('/delete_recurring_plan/<int:plan_id>', methods=['POST'])
@login_required
def delete_recurring_plan(plan_id):
    user_id = current_user.id
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute("DELETE FROM recurring_plans WHERE id = ? AND user_id = ?", (plan_id, user_id))
        db.commit()
        if cur.rowcount > 0: return jsonify(status='success', message='循环计划已删除')
        else: return jsonify(status='error', message='未找到循环计划'), 404
    except sqlite3.Error as e: return jsonify(status='error', message=f'数据库错误: {e}'), 500

@app.route('/get_recurring_todos', methods=['GET'])
@login_required
def get_recurring_todos():
    user_id = current_user.id
    db = get_db()
    todos = db.execute("SELECT * FROM recurring_todos WHERE user_id = ? ORDER BY start_date DESC", (user_id,)).fetchall()
    return jsonify([dict(todo) for todo in todos])

@app.route('/add_recurring_todo', methods=['POST'])
@login_required
def add_recurring_todo():
    user_id = current_user.id
    data = request.get_json()
    if not data.get('content') or not data.get('start_date') or not data.get('repeat_interval'):
        return jsonify(status='error', message='缺少必填字段'), 400
    if (data.get('repeat_interval') == 'weekly' or data.get('repeat_interval') == 'monthly') and not data.get('repeat_day'):
        return jsonify(status='error', message='循环缺少“星期几”或“日期”设置'), 400
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute(
            """INSERT INTO recurring_todos (user_id, content, start_date, end_date, repeat_interval, repeat_day) VALUES (?, ?, ?, ?, ?, ?)""",
            (user_id, data.get('content'), data.get('start_date'), data.get('end_date'), data.get('repeat_interval'), data.get('repeat_day'))
        )
        db.commit()
        return jsonify(status='success', todo_id=cur.lastrowid)
    except sqlite3.Error as e: return jsonify(status='error', message=f'数据库错误: {e}'), 500

@app.route('/delete_recurring_todo/<int:todo_id>', methods=['POST'])
@login_required
def delete_recurring_todo(todo_id):
    user_id = current_user.id
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute("DELETE FROM recurring_todos WHERE id = ? AND user_id = ?", (todo_id, user_id))
        db.commit()
        if cur.rowcount > 0: return jsonify(status='success', message='循环待办已删除')
        else: return jsonify(status='error', message='未找到循环待办'), 404
    except sqlite3.Error as e: return jsonify(status='error', message=f'数据库错误: {e}'), 500

@app.route('/api/exercises', methods=['GET'])
@login_required
def list_exercises():
    user_id = current_user.id
    db = get_db()
    rows = db.execute(
        "SELECT id, name, details FROM exercise_library WHERE user_id = ? ORDER BY id DESC",
        (user_id,)
    ).fetchall()
    return jsonify([dict(r) for r in rows])


@app.route('/api/exercises', methods=['POST'])
@login_required
def create_exercise():
    user_id = current_user.id
    data = request.get_json() or {}
    name = (data.get('name') or '').strip()
    details = (data.get('details') or '').strip()
    if not name:
        return jsonify(status='error', message='缺少训练项目名称'), 400
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute(
            "INSERT INTO exercise_library (user_id, name, details) VALUES (?, ?, ?)",
            (user_id, name, details)
        )
        db.commit()
        return jsonify(status='success', exercise_id=cur.lastrowid)
    except sqlite3.Error as e:
        return jsonify(status='error', message=f'数据库错误: {e}'), 500


@app.route('/api/exercises/<int:exercise_id>', methods=['POST', 'DELETE'])
@login_required
def delete_exercise(exercise_id):
    user_id = current_user.id
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute(
            "DELETE FROM exercise_library WHERE id = ? AND user_id = ?",
            (exercise_id, user_id)
        )
        db.commit()
        if cur.rowcount > 0:
            return jsonify(status='success', message='训练项目已删除')
        else:
            return jsonify(status='error', message='未找到训练项目'), 404
    except sqlite3.Error as e:
        return jsonify(status='error', message=f'数据库错误: {e}'), 500


@app.route('/api/todo_library', methods=['GET'])
@login_required
def list_todo_library():
    user_id = current_user.id
    db = get_db()
    rows = db.execute(
        "SELECT id, content FROM todo_library WHERE user_id = ? ORDER BY id DESC",
        (user_id,)
    ).fetchall()
    return jsonify([dict(r) for r in rows])


@app.route('/api/todo_library', methods=['POST'])
@login_required
def create_todo_library_item():
    user_id = current_user.id
    data = request.get_json() or {}
    content = (data.get('content') or '').strip()
    if not content:
        return jsonify(status='error', message='缺少待办内容'), 400
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute(
            "INSERT INTO todo_library (user_id, content) VALUES (?, ?)",
            (user_id, content)
        )
        db.commit()
        return jsonify(status='success', item_id=cur.lastrowid)
    except sqlite3.Error as e:
        return jsonify(status='error', message=f'数据库错误: {e}'), 500


@app.route('/api/todo_library/<int:item_id>', methods=['POST', 'DELETE'])
@login_required
def delete_todo_library_item(item_id):
    user_id = current_user.id
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute(
            "DELETE FROM todo_library WHERE id = ? AND user_id = ?",
            (item_id, user_id)
        )
        db.commit()
        if cur.rowcount > 0:
            return jsonify(status='success', message='常用待办已删除')
        else:
            return jsonify(status='error', message='未找到该待办'), 404
    except sqlite3.Error as e:
        return jsonify(status='error', message=f'数据库错误: {e}'), 500

@app.after_request
def add_common_headers(response):
    if 'Content-Type' not in response.headers:
        response.headers['Content-Type'] = 'text/html; charset=utf-8'
    else:
        c = response.headers['Content-Type']
        if c.startswith('text/') and 'charset' not in c:
            response.headers['Content-Type'] = c + '; charset=utf-8'
    response.headers.setdefault('X-Content-Type-Options', 'nosniff')
    response.headers.setdefault('X-Frame-Options', 'SAMEORIGIN')
    path = request.path or ''
    if path.startswith('/static/') or path.endswith(('.css', '.js', '.png', '.jpg', '.svg', '.webp')):
        response.headers['Cache-Control'] = 'public, max-age=3600'
    else:
        response.headers['Cache-Control'] = 'no-cache, no-store, must-revalidate'
    return response

@app.route('/health', methods=['GET'])
def health():
    # ★ 核心修改：健康检查也显示中国时区时间
    return jsonify(status='ok', time=datetime.now(CN_TZ).isoformat())

if __name__ == '__main__':
    app.run(host='0.0.0.0', port=8000, debug=True)
