# backend/app.py (v1.9 - Full Features)
import sqlite3
from datetime import datetime, timedelta, timezone
from flask import Flask, render_template, request, jsonify, g, redirect, url_for, make_response
from flask_login import LoginManager, UserMixin, login_user, logout_user, login_required, current_user
import os, json, uuid
from flask_cors import CORS
from functools import wraps
import sys 
from collections import defaultdict

CN_TZ = timezone(timedelta(hours=8))

is_frozen = getattr(sys, 'frozen', False)
is_in_onefile_temp = 'onefile_' in os.path.abspath(__file__) 

if is_frozen or is_in_onefile_temp:
    temp_base_dir = os.path.dirname(os.path.abspath(sys.executable))
    template_dir = os.path.join(temp_base_dir, 'templates')
    static_dir = os.path.join(temp_base_dir, 'static')
    exe_path = os.path.realpath(sys.argv[0])
    exe_dir = os.path.dirname(exe_path)
    DATABASE = os.path.join(exe_dir, 'fitness_manager.db')
else:
    app_file_path = os.path.abspath(__file__)
    backend_dir = os.path.dirname(app_file_path)
    project_root_dir = os.path.dirname(backend_dir)
    template_dir = os.path.join(project_root_dir, 'frontend', 'templates')
    static_dir = os.path.join(project_root_dir, 'frontend', 'static')
    DATABASE = os.path.join(backend_dir, 'fitness_manager.db')

print("DB:", DATABASE)

app = Flask(__name__, template_folder=template_dir, static_folder=static_dir)
CORS(app)
app.secret_key = str(uuid.uuid4())

login_manager = LoginManager()
login_manager.init_app(app)
login_manager.login_view = 'index' 

class User(UserMixin):
    def __init__(self, id, username, is_admin, manager_id):
        self.id = id
        self.username = username
        self.is_admin = is_admin
        self.manager_id = manager_id 
    @property
    def is_super_admin(self):
        return self.is_admin and (self.manager_id is None or self.manager_id == 0)

@login_manager.user_loader
def load_user(user_id):
    db = get_db()
    user_row = db.execute("SELECT id, username, is_admin, manager_id FROM users WHERE id = ?", (user_id,)).fetchone()
    if user_row:
        return User(id=user_row['id'], username=user_row['username'], is_admin=user_row['is_admin'], manager_id=user_row['manager_id'])
    return None

def get_db():
    db = getattr(g, '_database', None)
    if db is None:
        db = g._database = sqlite3.connect(DATABASE, check_same_thread=False)
        db.row_factory = sqlite3.Row
    return db

def get_setting(key, default=None):
    row = get_db().execute("SELECT value FROM settings WHERE key = ?", (key,)).fetchone()
    return default if row is None else row["value"]

def set_setting(key, value):
    db = get_db()
    db.execute("INSERT INTO settings (key, value) VALUES (?, ?) ON CONFLICT(key) DO UPDATE SET value = excluded.value", (key, str(value)))
    db.commit()

@app.teardown_appcontext
def close_connection(exception):
    db = getattr(g, '_database', None)
    if db is not None: db.close()

def init_db():
    db = get_db()
    cur = db.cursor()
    
    # Tables with reminder support
    cur.execute("""
        CREATE TABLE IF NOT EXISTS todos (
            id INTEGER PRIMARY KEY AUTOINCREMENT, 
            date TEXT NOT NULL, 
            content TEXT NOT NULL,
            is_completed INTEGER DEFAULT 0, 
            user_id INTEGER, 
            recurring_todo_id INTEGER,
            reminder_at TEXT
        )
    """)
    cur.execute("""
        CREATE TABLE IF NOT EXISTS fitness_plans (
            id INTEGER PRIMARY KEY AUTOINCREMENT, 
            date TEXT NOT NULL, 
            exercise_name TEXT NOT NULL,
            sets_reps_duration TEXT, 
            is_completed INTEGER DEFAULT 0, 
            user_id INTEGER, 
            recurring_plan_id INTEGER,
            reminder_at TEXT
        )
    """)
    cur.execute("""
        CREATE TABLE IF NOT EXISTS recurring_plans (
            id INTEGER PRIMARY KEY AUTOINCREMENT, 
            user_id INTEGER, 
            plan_name TEXT NOT NULL,
            exercises_json TEXT NOT NULL, 
            start_date TEXT NOT NULL, 
            end_date TEXT,
            repeat_interval TEXT NOT NULL, 
            repeat_day INTEGER,
            reminder_time TEXT
        )
    """)
    cur.execute("""
        CREATE TABLE IF NOT EXISTS recurring_todos (
            id INTEGER PRIMARY KEY AUTOINCREMENT, 
            user_id INTEGER, 
            content TEXT NOT NULL,
            start_date TEXT NOT NULL, 
            end_date TEXT, 
            repeat_interval TEXT NOT NULL, 
            repeat_day INTEGER,
            reminder_time TEXT
        )
    """)
    cur.execute("CREATE TABLE IF NOT EXISTS exercise_library (id INTEGER PRIMARY KEY AUTOINCREMENT, user_id INTEGER, name TEXT NOT NULL, details TEXT)")
    cur.execute("CREATE TABLE IF NOT EXISTS settings (key TEXT PRIMARY KEY, value TEXT)")
    cur.execute("CREATE TABLE IF NOT EXISTS todo_library (id INTEGER PRIMARY KEY AUTOINCREMENT, user_id INTEGER, content TEXT NOT NULL)")
    cur.execute("CREATE TABLE IF NOT EXISTS users (id INTEGER PRIMARY KEY AUTOINCREMENT, username TEXT UNIQUE NOT NULL, password TEXT NOT NULL, is_admin INTEGER DEFAULT 0, manager_id INTEGER REFERENCES users(id))")
    
    # Migrations for new columns
    cols_todos = [row[1] for row in cur.execute("PRAGMA table_info(todos)").fetchall()]
    if 'reminder_at' not in cols_todos: cur.execute("ALTER TABLE todos ADD COLUMN reminder_at TEXT")

    cols_fit = [row[1] for row in cur.execute("PRAGMA table_info(fitness_plans)").fetchall()]
    if 'reminder_at' not in cols_fit: cur.execute("ALTER TABLE fitness_plans ADD COLUMN reminder_at TEXT")
    
    cols_rec_todo = [row[1] for row in cur.execute("PRAGMA table_info(recurring_todos)").fetchall()]
    if 'reminder_time' not in cols_rec_todo: cur.execute("ALTER TABLE recurring_todos ADD COLUMN reminder_time TEXT")

    cols_rec_plan = [row[1] for row in cur.execute("PRAGMA table_info(recurring_plans)").fetchall()]
    if 'reminder_time' not in cols_rec_plan: cur.execute("ALTER TABLE recurring_plans ADD COLUMN reminder_time TEXT")

    # Superuser check
    if cur.execute("SELECT COUNT(*) FROM users WHERE username = 'superuser'").fetchone()[0] == 0:
        cur.execute("INSERT INTO users (username, password, is_admin, manager_id) VALUES (?, ?, ?, ?)", ('superuser', 'adminpassword', 1, None))
        
    # User1 check
    su_id = cur.execute("SELECT id FROM users WHERE username = 'superuser'").fetchone()['id']
    if cur.execute("SELECT COUNT(*) FROM users WHERE username = 'user1'").fetchone()[0] == 0:
        cur.execute("INSERT INTO users (username, password, is_admin, manager_id) VALUES (?, ?, ?, ?)", ('user1', 'password', 0, su_id))

    db.commit()

with app.app_context():
    init_db()

# -------------------- Auth --------------------
@app.route('/login', methods=['POST'])
def login():
    data = request.form if request.form else request.json
    u_row = get_db().execute("SELECT * FROM users WHERE username = ?", (data.get('username'),)).fetchone()
    if u_row and u_row['password'] == data.get('password'):
        login_user(User(u_row['id'], u_row['username'], u_row['is_admin'], u_row['manager_id']), remember=True)
        return jsonify({'status': 'success'})
    return jsonify({'status': 'error', 'message': 'Fail'}), 401

@app.route('/logout', methods=['POST'])
@login_required
def logout():
    logout_user()
    return jsonify({'status': 'success'})

@app.route('/get_current_user', methods=['GET'])
def get_current_user():
    if current_user.is_authenticated:
        return jsonify({'status':'logged_in', 'username':current_user.username, 'is_admin':current_user.is_admin})
    return jsonify({'status':'logged_out'}), 401

# -------------------- Views --------------------
@app.route('/')
def index():
    if not current_user.is_authenticated: return render_template('login.html')
    if 'Mobile' in request.headers.get('User-Agent', '') and request.args.get('view') != 'display':
        return redirect(url_for('mobile_config'))
    today = datetime.now(CN_TZ).strftime('%Y-%m-%d')
    _gen_fitness(current_user.id, today)
    _gen_todos(current_user.id, today)
    return render_template('display.html', username=current_user.username)

@app.route('/mobile_config')
@login_required
def mobile_config(): return render_template('mobile_config.html', username=current_user.username)

@app.route('/manage')
@login_required
def manage():
    return render_template('manage.html', username=current_user.username, is_admin=current_user.is_admin, manager_id=current_user.manager_id)

# -------------------- API: Todos --------------------
@app.route('/todos', methods=['GET'])
@login_required
def get_todos():
    d = request.args.get('date') or datetime.now(CN_TZ).strftime('%Y-%m-%d')
    rows = get_db().execute("SELECT * FROM todos WHERE date=? AND user_id=? ORDER BY is_completed ASC, reminder_at ASC, id DESC", (d, current_user.id)).fetchall()
    return jsonify([dict(r) for r in rows])

@app.route('/add_todo', methods=['POST'])
@login_required
def add_todo():
    d = request.json
    rem = d.get('reminder_at') # "2023-11-22 10:00"
    if rem and len(rem) == 5: rem = f"{d.get('date')} {rem}" # Handle "10:00"
    
    get_db().execute("INSERT INTO todos (date, content, is_completed, user_id, reminder_at) VALUES (?,?,0,?,?)", 
                     (d.get('date'), d.get('content'), current_user.id, rem)).connection.commit()
    return jsonify(status='success')

@app.route('/mark_todo_completed', methods=['POST'])
@login_required
def mark_todo():
    d = request.json
    get_db().execute("UPDATE todos SET is_completed=? WHERE id=? AND user_id=?", (d.get('is_completed'), d.get('todo_id'), current_user.id)).connection.commit()
    return jsonify(status='success')

@app.route('/delete_todo/<int:id>', methods=['POST'])
@login_required
def del_todo(id):
    get_db().execute("DELETE FROM todos WHERE id=? AND user_id=?", (id, current_user.id)).connection.commit()
    return jsonify(status='success')

# -------------------- API: Fitness --------------------
@app.route('/fitness_plan/<date_str>', methods=['GET'])
@login_required
def get_fitness(date_str):
    _gen_fitness(current_user.id, date_str)
    rows = get_db().execute("SELECT * FROM fitness_plans WHERE date=? AND user_id=? ORDER BY is_completed ASC, reminder_at ASC, id DESC", (date_str, current_user.id)).fetchall()
    return jsonify([dict(r) for r in rows])

@app.route('/add_fitness_manual', methods=['POST'])
@login_required
def add_fitness_manual():
    # New endpoint to manually add a daily fitness item with time
    d = request.json
    rem = d.get('reminder_at')
    if rem and len(rem) == 5: rem = f"{d.get('date')} {rem}"
    
    get_db().execute("INSERT INTO fitness_plans (date, exercise_name, sets_reps_duration, is_completed, user_id, reminder_at) VALUES (?,?,?,0,?,?)",
                     (d.get('date'), d.get('exercise_name'), d.get('sets_reps_duration'), current_user.id, rem)).connection.commit()
    return jsonify(status='success')

@app.route('/mark_exercise_completed_mobile', methods=['POST'])
@login_required
def mark_fit():
    d = request.json
    get_db().execute("UPDATE fitness_plans SET is_completed=? WHERE id=? AND user_id=?", (d.get('is_completed'), d.get('plan_id'), current_user.id)).connection.commit()
    return jsonify(status='success')

@app.route('/delete_fitness_plan_exercise/<int:id>', methods=['POST'])
@login_required
def del_fit(id):
    get_db().execute("DELETE FROM fitness_plans WHERE id=? AND user_id=?", (id, current_user.id)).connection.commit()
    return jsonify(status='success')

# -------------------- API: Recurring --------------------
@app.route('/get_recurring_plans', methods=['GET'])
@login_required
def get_rec_plans():
    return jsonify([dict(r) for r in get_db().execute("SELECT * FROM recurring_plans WHERE user_id=?", (current_user.id,)).fetchall()])

@app.route('/add_recurring_plan', methods=['POST'])
@login_required
def add_rec_plan():
    d = request.json
    get_db().execute("INSERT INTO recurring_plans (user_id, plan_name, exercises_json, start_date, end_date, repeat_interval, repeat_day, reminder_time) VALUES (?,?,?,?,?,?,?,?)",
                     (current_user.id, d.get('plan_name'), json.dumps(d.get('exercises')), d.get('start_date'), d.get('end_date'), d.get('repeat_interval'), d.get('repeat_day'), d.get('reminder_time'))).connection.commit()
    return jsonify(status='success')

@app.route('/delete_recurring_plan/<int:id>', methods=['POST'])
@login_required
def del_rec_plan(id):
    get_db().execute("DELETE FROM recurring_plans WHERE id=? AND user_id=?", (id, current_user.id)).connection.commit()
    return jsonify(status='success')

@app.route('/get_recurring_todos', methods=['GET'])
@login_required
def get_rec_todos():
    return jsonify([dict(r) for r in get_db().execute("SELECT * FROM recurring_todos WHERE user_id=?", (current_user.id,)).fetchall()])

@app.route('/add_recurring_todo', methods=['POST'])
@login_required
def add_rec_todo():
    d = request.json
    get_db().execute("INSERT INTO recurring_todos (user_id, content, start_date, end_date, repeat_interval, repeat_day, reminder_time) VALUES (?,?,?,?,?,?,?)",
                     (current_user.id, d.get('content'), d.get('start_date'), d.get('end_date'), d.get('repeat_interval'), d.get('repeat_day'), d.get('reminder_time'))).connection.commit()
    return jsonify(status='success')

@app.route('/delete_recurring_todo/<int:id>', methods=['POST'])
@login_required
def del_rec_todo(id):
    get_db().execute("DELETE FROM recurring_todos WHERE id=? AND user_id=?", (id, current_user.id)).connection.commit()
    return jsonify(status='success')

# -------------------- Libraries & Utils --------------------
@app.route('/api/exercises', methods=['GET','POST'])
@app.route('/api/exercises/<int:id>', methods=['DELETE'])
@login_required
def ex_lib(id=None):
    db = get_db()
    if request.method=='GET': return jsonify([dict(r) for r in db.execute("SELECT * FROM exercise_library WHERE user_id=?",(current_user.id,))])
    if request.method=='POST': 
        db.execute("INSERT INTO exercise_library (user_id, name, details) VALUES (?,?,?)", (current_user.id, request.json.get('name'), request.json.get('details')))
        db.commit()
        return jsonify(status='success')
    if request.method=='DELETE':
        db.execute("DELETE FROM exercise_library WHERE id=? AND user_id=?", (id, current_user.id))
        db.commit()
        return jsonify(status='success')

@app.route('/api/todo_library', methods=['GET','POST'])
@app.route('/api/todo_library/<int:id>', methods=['DELETE'])
@login_required
def todo_lib(id=None):
    db = get_db()
    if request.method=='GET': return jsonify([dict(r) for r in db.execute("SELECT * FROM todo_library WHERE user_id=?",(current_user.id,))])
    if request.method=='POST': 
        db.execute("INSERT INTO todo_library (user_id, content) VALUES (?,?)", (current_user.id, request.json.get('content')))
        db.commit()
        return jsonify(status='success')
    if request.method=='DELETE':
        db.execute("DELETE FROM todo_library WHERE id=? AND user_id=?", (id, current_user.id))
        db.commit()
        return jsonify(status='success')

@app.route('/api/settings/scroll_speed', methods=['GET','POST'])
def settings_scroll():
    if request.method=='POST': set_setting('scroll_speed', request.json.get('scroll_speed')); return jsonify(status='success')
    return jsonify(scroll_speed=int(get_setting('scroll_speed',25) or 25))

@app.route('/api/users', methods=['GET','POST'])
@app.route('/api/users/<int:id>', methods=['DELETE'])
@login_required
def user_mgmt(id=None):
    if not current_user.is_admin: return jsonify(status='error'), 403
    db = get_db()
    if request.method=='GET': return jsonify([dict(u) for u in db.execute("SELECT id,username,is_admin FROM users WHERE manager_id=?",(current_user.id,))])
    if request.method=='POST':
        d=request.json
        is_adm = 1 if current_user.is_super_admin and d.get('is_admin') else 0
        try: db.execute("INSERT INTO users (username,password,is_admin,manager_id) VALUES (?,?,?,?)", (d.get('username'),d.get('password'),is_adm,current_user.id)); db.commit()
        except: return jsonify(status='error',message='Exists')
        return jsonify(status='success')
    if request.method=='DELETE':
        if id==current_user.id: return jsonify(status='error'), 400
        db.execute("DELETE FROM users WHERE id=?",(id,))
        db.commit()
        return jsonify(status='success')

@app.route('/api/profile', methods=['POST'])
@login_required
def profile():
    d=request.json
    if d.get('password'): get_db().execute("UPDATE users SET password=? WHERE id=?", (d.get('password'), current_user.id))
    if d.get('username'): 
        try: get_db().execute("UPDATE users SET username=? WHERE id=?", (d.get('username'), current_user.id))
        except: return jsonify(status='error')
    get_db().commit()
    return jsonify(status='success')

@app.route('/api/stats/daily_completion')
@login_required
def stats():
    s, e = request.args.get('start_date'), request.args.get('end_date')
    db = get_db()
    fr = db.execute("SELECT date, is_completed FROM fitness_plans WHERE user_id=? AND date BETWEEN ? AND ?", (current_user.id, s, e))
    tr = db.execute("SELECT date, is_completed FROM todos WHERE user_id=? AND date BETWEEN ? AND ?", (current_user.id, s, e))
    fd = defaultdict(lambda: {"total":0,"completed":0})
    td = defaultdict(lambda: {"total":0,"completed":0})
    for r in fr: 
        fd[r['date']]['total']+=1
        if r['is_completed']: fd[r['date']]['completed']+=1
    for r in tr:
        td[r['date']]['total']+=1
        if r['is_completed']: td[r['date']]['completed']+=1
    return jsonify(fitness_data=fd, todo_data=td)

# -------------------- Logic --------------------
def _gen_fitness(uid, date_str):
    db = get_db()
    if db.execute("SELECT COUNT(*) FROM fitness_plans WHERE date=? AND user_id=?", (date_str, uid)).fetchone()[0]>0: return
    dt = datetime.strptime(date_str, '%Y-%m-%d').date()
    plans = db.execute("SELECT * FROM recurring_plans WHERE user_id=? AND start_date<=? AND (end_date IS NULL OR end_date>=?)", (uid, date_str, date_str)).fetchall()
    for p in plans:
        if (p['repeat_interval']=='daily') or (p['repeat_interval']=='weekly' and p['repeat_day']==dt.isoweekday()):
            # Calculate specific reminder datetime if time is set
            rem_at = f"{date_str} {p['reminder_time']}" if p['reminder_time'] else None
            exs = json.loads(p['exercises_json'])
            for ex in exs:
                db.execute("INSERT INTO fitness_plans (date,exercise_name,sets_reps_duration,user_id,recurring_plan_id,reminder_at) VALUES (?,?,?,?,?,?)",
                           (date_str, ex['name'], ex['details'], uid, p['id'], rem_at))
    db.commit()

def _gen_todos(uid, date_str):
    db = get_db()
    if db.execute("SELECT COUNT(*) FROM todos WHERE date=? AND user_id=?", (date_str, uid)).fetchone()[0]>0: return
    dt = datetime.strptime(date_str, '%Y-%m-%d').date()
    dom = dt.day
    recs = db.execute("SELECT * FROM recurring_todos WHERE user_id=? AND start_date<=? AND (end_date IS NULL OR end_date>=?)", (uid, date_str, date_str)).fetchall()
    for r in recs:
        if (r['repeat_interval']=='daily') or (r['repeat_interval']=='weekly' and r['repeat_day']==dt.isoweekday()) or (r['repeat_interval']=='monthly' and r['repeat_day']==dom):
            rem_at = f"{date_str} {r['reminder_time']}" if r['reminder_time'] else None
            db.execute("INSERT INTO todos (date,content,is_completed,user_id,recurring_todo_id,reminder_at) VALUES (?,?,0,?,?,?)",
                       (date_str, r['content'], uid, r['id'], rem_at))
    db.commit()

@app.route('/health', methods=['GET'])
def health(): return jsonify(status='ok', time=datetime.now(CN_TZ).isoformat())

if __name__ == '__main__':
    app.run(host='0.0.0.0', port=8000, debug=True)
