import os

# ==========================================
# 1. backend/app.py 的完整内容 (已修复时区)
# ==========================================
app_py_content = r'''# backend/app.py (v1.7 - Fixed Timezone & Indentation)
import sqlite3
from datetime import datetime, timedelta, timezone
from flask import Flask, render_template, request, jsonify, g, redirect, url_for, make_response
from flask_login import LoginManager, UserMixin, login_user, logout_user, login_required, current_user
import os, json, uuid
from flask_cors import CORS
from functools import wraps
import sys 
from collections import defaultdict

# --- 时区设置 (强制使用 UTC+8 中国标准时间) ---
CN_TZ = timezone(timedelta(hours=8))

# --- Nuitka 路径逻辑 (保持不变) ---
is_frozen = getattr(sys, 'frozen', False)
is_in_onefile_temp = 'onefile_' in os.path.abspath(__file__) 

if is_frozen or is_in_onefile_temp:
    temp_base_dir = os.path.dirname(os.path.abspath(sys.executable))
    template_dir = os.path.join(temp_base_dir, 'templates')
    static_dir = os.path.join(temp_base_dir, 'static')
    exe_path = os.path.realpath(sys.argv[0])
    exe_dir = os.path.dirname(exe_path)
    DATABASE = os.path.join(exe_dir, 'fitness_manager.db')
    print("--- Detected Nuitka Onefile Path ---")
    print(f"--- **DATABASE is PERMANENT at {DATABASE}** ---")
else:
    app_file_path = os.path.abspath(__file__)
    backend_dir = os.path.dirname(app_file_path)
    project_root_dir = os.path.dirname(backend_dir)
    template_dir = os.path.join(project_root_dir, 'frontend', 'templates')
    static_dir = os.path.join(project_root_dir, 'frontend', 'static')
    DATABASE = os.path.join(backend_dir, 'fitness_manager.db')
    print("--- Detected Development Path ---")

print("Template folder:", template_dir)
print("Static folder:", static_dir)
print("DB path:", DATABASE)

app = Flask(__name__, template_folder=template_dir, static_folder=static_dir)
CORS(app)
app.secret_key = str(uuid.uuid4())

# --- Flask-Login 配置 ---
login_manager = LoginManager()
login_manager.init_app(app)
login_manager.login_view = 'index' 
login_manager.login_message = "请先登录以访问此页面。"

# ★★★ User 类
class User(UserMixin):
    def __init__(self, id, username, is_admin, manager_id):
        self.id = id
        self.username = username
        self.is_admin = is_admin
        self.manager_id = manager_id 
    
    @property
    def is_super_admin(self):
        return self.is_admin and (self.manager_id is None or self.manager_id == 0)

@login_manager.user_loader
def load_user(user_id):
    db = get_db()
    user_row = db.execute("SELECT id, username, is_admin, manager_id FROM users WHERE id = ?", (user_id,)).fetchone()
    if user_row:
        return User(
            id=user_row['id'], 
            username=user_row['username'], 
            is_admin=user_row['is_admin'], 
            manager_id=user_row['manager_id']
        )
    return None

# -------------------- DB --------------------
def get_db():
    db = getattr(g, '_database', None)
    if db is None:
        db = g._database = sqlite3.connect(DATABASE, check_same_thread=False)
        db.row_factory = sqlite3.Row
    return db

def get_setting(key, default=None):
    db = get_db()
    row = db.execute("SELECT value FROM settings WHERE key = ?", (key,)).fetchone()
    if row is None:
        return default
    return row["value"]

def set_setting(key, value):
    db = get_db()
    cur = db.cursor()
    cur.execute(
        "INSERT INTO settings (key, value) VALUES (?, ?) "
        "ON CONFLICT(key) DO UPDATE SET value = excluded.value",
        (key, str(value)),
    )
    db.commit()

@app.teardown_appcontext
def close_connection(exception):
    db = getattr(g, '_database', None)
    if db is not None:
        db.close()

def init_db():
    db = get_db()
    cur = db.cursor()
    cur.execute("""
        CREATE TABLE IF NOT EXISTS todos (
            id INTEGER PRIMARY KEY AUTOINCREMENT, 
            date TEXT NOT NULL, 
            content TEXT NOT NULL,
            is_completed INTEGER DEFAULT 0, 
            user_id INTEGER, 
            recurring_todo_id INTEGER
        )
    """)
    cur.execute("""
        CREATE TABLE IF NOT EXISTS fitness_plans (
            id INTEGER PRIMARY KEY AUTOINCREMENT, 
            date TEXT NOT NULL, 
            exercise_name TEXT NOT NULL,
            sets_reps_duration TEXT, 
            is_completed INTEGER DEFAULT 0, 
            user_id INTEGER, 
            recurring_plan_id INTEGER
        )
    """)
    cur.execute("""
        CREATE TABLE IF NOT EXISTS recurring_plans (
            id INTEGER PRIMARY KEY AUTOINCREMENT, 
            user_id INTEGER, 
            plan_name TEXT NOT NULL,
            exercises_json TEXT NOT NULL, 
            start_date TEXT NOT NULL, 
            end_date TEXT,
            repeat_interval TEXT NOT NULL, 
            repeat_day INTEGER
        )
    """)
    cur.execute("""
        CREATE TABLE IF NOT EXISTS recurring_todos (
            id INTEGER PRIMARY KEY AUTOINCREMENT, 
            user_id INTEGER, 
            content TEXT NOT NULL,
            start_date TEXT NOT NULL, 
            end_date TEXT, 
            repeat_interval TEXT NOT NULL, 
            repeat_day INTEGER
        )
    """)
    cur.execute("""
        CREATE TABLE IF NOT EXISTS exercise_library (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            user_id INTEGER,
            name TEXT NOT NULL,
            details TEXT
        )
    """)
    cur.execute("""
        CREATE TABLE IF NOT EXISTS settings (
            key   TEXT PRIMARY KEY,
            value TEXT
        )
    """)
    cur.execute("""
        CREATE TABLE IF NOT EXISTS todo_library (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            user_id INTEGER,
            content TEXT NOT NULL
        )
    """)

def ensure_user_schema():
    db = get_db()
    cur = db.cursor()
    cur.execute("""
        CREATE TABLE IF NOT EXISTS users (
            id INTEGER PRIMARY KEY AUTOINCREMENT, 
            username TEXT UNIQUE NOT NULL,
            password TEXT NOT NULL, 
            is_admin INTEGER DEFAULT 0,
            manager_id INTEGER REFERENCES users(id)
        )
    """)
    try: cur.execute("SELECT is_admin FROM users LIMIT 1")
    except sqlite3.OperationalError:
        cur.execute("ALTER TABLE users ADD COLUMN is_admin INTEGER DEFAULT 0")
    
    try: cur.execute("SELECT manager_id FROM users LIMIT 1")
    except sqlite3.OperationalError:
        cur.execute("ALTER TABLE users ADD COLUMN manager_id INTEGER REFERENCES users(id)")
    
    cur.execute("SELECT COUNT(*) FROM users WHERE username = 'superuser'")
    if cur.fetchone()[0] == 0:
        cur.execute("INSERT INTO users (username, password, is_admin, manager_id) VALUES (?, ?, ?, ?)", 
                    ('superuser', 'adminpassword', 1, None))
        
    su_row = cur.execute("SELECT id FROM users WHERE username = 'superuser'").fetchone()
    su_id = su_row['id'] if su_row else 1
    
    cur.execute("SELECT COUNT(*) FROM users WHERE username = 'user1'")
    if cur.fetchone()[0] == 0:
        cur.execute("INSERT INTO users (username, password, is_admin, manager_id) VALUES (?, ?, ?, ?)", 
                    ('user1', 'password', 0, su_id))

    try: cur.execute("SELECT repeat_interval FROM recurring_plans LIMIT 1")
    except sqlite3.OperationalError:
        cur.execute("ALTER TABLE recurring_plans ADD COLUMN repeat_interval TEXT DEFAULT 'daily'")
        cur.execute("ALTER TABLE recurring_plans ADD COLUMN repeat_day INTEGER")
        cur.execute("UPDATE recurring_plans SET repeat_interval = 'daily' WHERE repeat_interval IS NULL")

    try: cur.execute("SELECT recurring_plan_id FROM fitness_plans LIMIT 1")
    except sqlite3.OperationalError:
        cur.execute("ALTER TABLE fitness_plans ADD COLUMN recurring_plan_id INTEGER")

    try: cur.execute("SELECT recurring_todo_id FROM todos LIMIT 1")
    except sqlite3.OperationalError:
        cur.execute("ALTER TABLE todos ADD COLUMN recurring_todo_id INTEGER")

    default_user_id_row = db.execute("SELECT id FROM users WHERE username = 'user1'").fetchone()
    if default_user_id_row:
        default_user_id = default_user_id_row['id']
        for table in ['fitness_plans', 'todos', 'recurring_plans', 'recurring_todos', 'exercise_library', 'todo_library']:
            if table == 'users': continue
            try: cur.execute(f"SELECT user_id FROM {table} LIMIT 1") 
            except sqlite3.OperationalError:
                try: cur.execute(f"ALTER TABLE {table} ADD COLUMN user_id INTEGER")
                except sqlite3.OperationalError: pass 
            cur.execute(f"UPDATE {table} SET user_id = ? WHERE user_id IS NULL", (default_user_id,))
    db.commit()

def ensure_todos_is_completed_column():
    db = get_db()
    cur = db.cursor()
    try: cur.execute("SELECT is_completed FROM todos LIMIT 1")
    except sqlite3.OperationalError:
        cur.execute("ALTER TABLE todos ADD COLUMN is_completed INTEGER DEFAULT 0")
        db.commit()

def ensure_fitness_plans_is_completed_column():
    db = get_db()
    cur = db.cursor()
    try: cur.execute("SELECT is_completed FROM fitness_plans LIMIT 1")
    except sqlite3.OperationalError:
        cur.execute("ALTER TABLE fitness_plans ADD COLUMN is_completed INTEGER DEFAULT 0")
        db.commit()

with app.app_context():
    init_db() 
    ensure_todos_is_completed_column()
    ensure_fitness_plans_is_completed_column()
    ensure_user_schema() 


# -------------------- Auth Routes --------------------

@app.route('/login', methods=['POST'])
def login():
    username = request.form.get('username') or request.json.get('username')
    password = request.form.get('password') or request.json.get('password')
    db = get_db()
    user_row = db.execute("SELECT id, username, password, is_admin, manager_id FROM users WHERE username = ?", (username,)).fetchone()
    
    if user_row and user_row['password'] == password: 
        user_obj = User(
            id=user_row['id'], 
            username=user_row['username'], 
            is_admin=user_row['is_admin'],
            manager_id=user_row['manager_id']
        )
        login_user(user_obj, remember=True) 
        return jsonify({
            'status': 'success', 
            'username': user_obj.username, 
            'is_admin': user_obj.is_admin,
            'manager_id': user_obj.manager_id
        })
    
    return jsonify({'status': 'error', 'message': '用户名或密码无效'}), 401

@app.route('/logout', methods=['POST'])
@login_required
def logout():
    logout_user() 
    return jsonify({'status': 'success'})

@app.route('/get_current_user', methods=['GET'])
def get_current_user():
    if current_user.is_authenticated:
        return jsonify({
            'status': 'logged_in', 
            'user_id': current_user.id, 
            'username': current_user.username, 
            'is_admin': current_user.is_admin,
            'manager_id': current_user.manager_id
        })
    return jsonify({'status': 'logged_out'}), 401

# -------------------- UI Routes --------------------
@app.route('/')
def index():
    if not current_user.is_authenticated:
        return render_template('login.html') 
    
    view_mode = request.args.get('view')
    if view_mode == 'display':
        pass
    elif 'Mobile' in request.headers.get('User-Agent', '') or 'Android' in request.headers.get('User-Agent', ''):
        return redirect(url_for('mobile_config'))

    # ★ 核心修改：使用中国时区获取“今天”的日期
    today_str = datetime.now(CN_TZ).strftime('%Y-%m-%d')
    
    _generate_daily_fitness_plan_if_not_exists(current_user.id, today_str)
    _generate_daily_todos_if_not_exists(current_user.id, today_str)
    return render_template('display.html', username=current_user.username, is_admin=current_user.is_admin) 

@app.route('/mobile_config')
@login_required
def mobile_config():
    return render_template('mobile_config.html', username=current_user.username)

@app.route('/api/settings/scroll_speed', methods=['GET', 'POST'])
def scroll_speed_setting():
    if request.method == 'GET':
        value = get_setting('scroll_speed', '25')
        try: speed = int(value)
        except (TypeError, ValueError): speed = 25
        return jsonify({"scroll_speed": speed})

    data = request.get_json() or {}
    try: speed = int(data.get('scroll_speed', 25))
    except (TypeError, ValueError):
        return jsonify({"status": "error", "message": "滚动速度必须是整数"}), 400
    if speed < 5: speed = 5
    if speed > 120: speed = 120
    set_setting('scroll_speed', speed)
    return jsonify({"status": "success", "scroll_speed": speed})


@app.route('/manage')
@login_required
def manage():
    return render_template(
        'manage.html', 
        username=current_user.username, 
        is_admin=current_user.is_admin,
        manager_id=current_user.manager_id
    )

# -------------------- User Management API --------------------

def admin_required(f):
    @wraps(f)
    def decorated_function(*args, **kwargs):
        if not current_user.is_authenticated or not current_user.is_admin:
            return jsonify(status='error', message='需要管理员权限'), 403
        return f(*args, **kwargs)
    return decorated_function

@app.route('/api/users', methods=['GET'])
@login_required
@admin_required 
def list_users():
    db = get_db()
    users = db.execute(
        "SELECT id, username, is_admin FROM users WHERE manager_id = ? ORDER BY is_admin DESC, username", 
        (current_user.id,)
    ).fetchall()
    return jsonify([dict(u) for u in users])

@app.route('/api/users', methods=['POST'])
@login_required
@admin_required
def create_user():
    data = request.get_json()
    username = data.get('username')
    password = data.get('password')
    is_admin_from_form = data.get('is_admin', 0)
    
    is_super_admin = current_user.is_super_admin
    
    is_admin_to_create = 0
    if is_super_admin:
        is_admin_to_create = 1 if is_admin_from_form else 0
    
    manager_id = current_user.id 

    if not username or not password: return jsonify(status='error', message='缺少用户名或密码'), 400
    try:
        db = get_db()
        db.execute(
            "INSERT INTO users (username, password, is_admin, manager_id) VALUES (?, ?, ?, ?)", 
            (username, password, is_admin_to_create, manager_id)
        )
        db.commit()
        return jsonify(status='success', message='用户创建成功')
    except sqlite3.IntegrityError: 
        return jsonify(status='error', message='用户名已存在'), 409
    except sqlite3.Error as e: 
        return jsonify(status='error', message=f'数据库错误: {e}'), 500

@app.route('/api/users/<int:target_user_id>', methods=['DELETE'])
@login_required
@admin_required
def delete_user(target_user_id):
    if current_user.id == target_user_id: 
        return jsonify(status='error', message='不能删除自己'), 400
    
    db = get_db()
    cur = db.cursor()
    
    target_user = cur.execute(
        "SELECT id FROM users WHERE id = ? AND manager_id = ?", 
        (target_user_id, current_user.id)
    ).fetchone()

    if not target_user:
        return jsonify(status='error', message='未找到用户或您无权删除此用户'), 403

    cur.execute("DELETE FROM users WHERE id = ?", (target_user_id,))
    cur.execute("DELETE FROM todos WHERE user_id = ?", (target_user_id,))
    cur.execute("DELETE FROM fitness_plans WHERE user_id = ?", (target_user_id,))
    cur.execute("DELETE FROM recurring_plans WHERE user_id = ?", (target_user_id,))
    cur.execute("DELETE FROM recurring_todos WHERE user_id = ?", (target_user_id,))
    cur.execute("DELETE FROM exercise_library WHERE user_id = ?", (target_user_id,))
    cur.execute("DELETE FROM todo_library WHERE user_id = ?", (target_user_id,))
    
    db.commit()
    
    if cur.rowcount > 0: 
        return jsonify(status='success', message='用户及所有数据已删除')
    
    return jsonify(status='error', message='删除失败'), 500

# -------------------- Profile Update API --------------------
@app.route('/api/profile', methods=['POST'])
@login_required
def update_profile():
    user_id = current_user.id
    data = request.get_json()
    new_username = data.get('username')
    new_password = data.get('password')
    db = get_db()
    cur = db.cursor()
    update_count = 0
    if new_password:
        cur.execute("UPDATE users SET password = ? WHERE id = ?", (new_password, user_id))
        update_count += 1
    if new_username:
        current_user_db = cur.execute("SELECT username FROM users WHERE id = ?", (user_id,)).fetchone()
        if current_user_db and current_user_db['username'] != new_username:
            try:
                cur.execute("UPDATE users SET username = ? WHERE id = ?", (new_username, user_id))
                update_count += 1
            except sqlite3.IntegrityError:
                db.rollback()
                return jsonify(status='error', message='用户名已存在'), 409
    db.commit()
    if update_count > 0: 
        return jsonify(status='success', message='个人资料更新成功。用户名的更改将在下次登录时生效。')
    return jsonify(status='error', message='没有要更新的字段'), 400

# -------------------- Stats API --------------------
@app.route('/api/stats/daily_completion', methods=['GET'])
@login_required
def get_daily_completion_stats():
    user_id = current_user.id
    start_date = request.args.get('start_date')
    end_date = request.args.get('end_date')
    if not start_date or not end_date: return jsonify(status='error', message='缺少开始日期或结束日期'), 400
    db = get_db()
    fitness_rows = db.execute("SELECT date, is_completed FROM fitness_plans WHERE user_id = ? AND date BETWEEN ? AND ?", (user_id, start_date, end_date)).fetchall()
    todo_rows = db.execute("SELECT date, is_completed FROM todos WHERE user_id = ? AND date BETWEEN ? AND ?", (user_id, start_date, end_date)).fetchall()
    fitness_data = defaultdict(lambda: {"total": 0, "completed": 0})
    for row in fitness_rows:
        fitness_data[row['date']]["total"] += 1
        if row['is_completed']: fitness_data[row['date']]["completed"] += 1
    todo_data = defaultdict(lambda: {"total": 0, "completed": 0})
    for row in todo_rows:
        todo_data[row['date']]["total"] += 1
        if row['is_completed']: todo_data[row['date']]["completed"] += 1
    return jsonify({"status": "success", "fitness_data": fitness_data, "todo_data": todo_data})

# -------------------- Helpers --------------------
def _get_daily_fitness_plans(user_id, date_str):
    db = get_db()
    plans = db.execute("SELECT * FROM fitness_plans WHERE date = ? AND user_id = ? ORDER BY id", (date_str, user_id)).fetchall()
    return [dict(plan) for plan in plans]
def _generate_daily_fitness_plan_if_not_exists(user_id, date_str):
    db = get_db()
    cur = db.cursor()
    cur.execute("SELECT COUNT(*) FROM fitness_plans WHERE date = ? AND user_id = ?", (date_str, user_id))
    if cur.fetchone()[0] > 0: return 
    today_date = datetime.strptime(date_str, '%Y-%m-%d').date()
    today_weekday = today_date.isoweekday() 
    recurring_plans = db.execute("SELECT * FROM recurring_plans WHERE user_id = ? AND (start_date <= ?) AND (end_date IS NULL OR end_date >= ?)", (user_id, date_str, date_str)).fetchall()
    new_plans_to_insert = []
    for plan in recurring_plans:
        should_add = False
        if plan['repeat_interval'] == 'daily': should_add = True
        elif plan['repeat_interval'] == 'weekly' and plan['repeat_day'] == today_weekday: should_add = True
        if should_add:
            exercises = json.loads(plan['exercises_json'])
            for ex in exercises: new_plans_to_insert.append({'name': ex['name'], 'details': ex['details'], 'recurring_id': plan['id']})
    for plan in new_plans_to_insert:
        cur.execute(
            """INSERT INTO fitness_plans (date, exercise_name, sets_reps_duration, is_completed, user_id, recurring_plan_id) VALUES (?, ?, ?, 0, ?, ?)""",
            (date_str, plan['name'], plan['details'], user_id, plan['recurring_id']))
    db.commit()
def _generate_daily_todos_if_not_exists(user_id, date_str):
    db = get_db()
    cur = db.cursor()
    cur.execute("SELECT COUNT(*) FROM todos WHERE date = ? AND user_id = ?", (date_str, user_id))
    if cur.fetchone()[0] > 0: return
    today_date = datetime.strptime(date_str, '%Y-%m-%d').date()
    today_weekday = today_date.isoweekday() 
    today_day_of_month = today_date.day 
    recurring_todos = db.execute("SELECT * FROM recurring_todos WHERE user_id = ? AND (start_date <= ?) AND (end_date IS NULL OR end_date >= ?)", (user_id, date_str, date_str)).fetchall()
    new_todos_to_insert = []
    for todo in recurring_todos:
        should_add = False
        if todo['repeat_interval'] == 'daily': should_add = True
        elif todo['repeat_interval'] == 'weekly' and todo['repeat_day'] == today_weekday: should_add = True
        elif todo['repeat_interval'] == 'monthly' and todo['repeat_day'] == today_day_of_month: should_add = True
        if should_add: new_todos_to_insert.append({'content': todo['content'], 'recurring_id': todo['id']})
    for todo in new_todos_to_insert:
        cur.execute(
            """INSERT INTO todos (date, content, is_completed, user_id, recurring_todo_id) VALUES (?, ?, 0, ?, ?)""",
            (date_str, todo['content'], user_id, todo['recurring_id']))
    db.commit()

# -------------------- API Routes --------------------

@app.route('/fitness_plan/<date_str>', methods=['GET'])
@login_required
def get_fitness_plan(date_str):
    user_id = current_user.id
    _generate_daily_fitness_plan_if_not_exists(user_id, date_str)
    db = get_db()
    plans = db.execute("SELECT * FROM fitness_plans WHERE date = ? AND user_id = ? ORDER BY id", (date_str, user_id)).fetchall()
    return jsonify([dict(plan) for plan in plans])

@app.route('/mark_exercise_completed_mobile', methods=['POST'])
@login_required
def mark_exercise_completed_mobile():
    user_id = current_user.id
    data = request.get_json()
    plan_id = data.get('plan_id')
    is_completed = data.get('is_completed')
    if plan_id is None or is_completed is None: return jsonify(status='error', message='缺少计划ID或完成状态'), 400
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute("UPDATE fitness_plans SET is_completed = ? WHERE id = ? AND user_id = ?", (is_completed, plan_id, user_id))
        db.commit()
        if cur.rowcount > 0: return jsonify(status='success')
        else: return jsonify(status='error', message='未找到计划或您无权修改'), 404
    except sqlite3.Error as e: return jsonify(status='error', message=f'数据库错误: {e}'), 500

@app.route('/delete_fitness_plan_exercise/<int:plan_id>', methods=['POST'])
@login_required
def delete_fitness_plan_exercise(plan_id):
    user_id = current_user.id
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute("DELETE FROM fitness_plans WHERE id = ? AND user_id = ?", (plan_id, user_id))
        db.commit()
        if cur.rowcount > 0: return jsonify(status='success', message='每日健身已删除')
        else: return jsonify(status='error', message='未找到每日健身计划'), 404
    except sqlite3.Error as e: return jsonify(status='error', message=f'数据库错误: {e}'), 500

@app.route('/todos', methods=['GET'])
@login_required
def get_todos():
    user_id = current_user.id
    # ★ 核心修改：默认使用中国时区日期
    date_str = request.args.get('date') or datetime.now(CN_TZ).strftime('%Y-%m-%d')
    db = get_db()
    todos = db.execute(
        "SELECT * FROM todos WHERE date = ? AND user_id = ? ORDER BY id",
        (date_str, user_id)
    ).fetchall()
    return jsonify([dict(todo) for todo in todos])

@app.route('/add_todo', methods=['POST'])
@login_required
def add_todo():
    user_id = current_user.id
    data = request.get_json()
    date_str = data.get('date')
    content = data.get('content')
    if not date_str or not content: return jsonify(status='error', message='缺少日期或内容'), 400
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute("INSERT INTO todos (date, content, is_completed, user_id) VALUES (?, ?, 0, ?)", (date_str, content, user_id))
        db.commit()
        return jsonify(status='success', todo_id=cur.lastrowid)
    except sqlite3.Error as e: return jsonify(status='error', message=f'数据库错误: {e}'), 500

@app.route('/mark_todo_completed', methods=['POST'])
@login_required
def mark_todo_completed():
    user_id = current_user.id
    data = request.get_json()
    todo_id = data.get('todo_id')
    is_completed = data.get('is_completed')
    if todo_id is None or is_completed is None: return jsonify(status='error', message='缺少待办ID或完成状态'), 400
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute("UPDATE todos SET is_completed = ? WHERE id = ? AND user_id = ?", (is_completed, todo_id, user_id))
        db.commit()
        if cur.rowcount > 0: return jsonify(status='success')
        else: return jsonify(status='error', message='未找到待办事项或您无权修改'), 404
    except sqlite3.Error as e: return jsonify(status='error', message=f'数据库错误: {e}'), 500


@app.route('/delete_todo/<int:todo_id>', methods=['POST'])
@login_required
def delete_todo(todo_id):
    user_id = current_user.id
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute("DELETE FROM todos WHERE id = ? AND user_id = ?", (todo_id, user_id))
        db.commit()
        if cur.rowcount > 0:
            return jsonify(status='success', message='待办事项已删除')
        else:
            return jsonify(status='error', message='未找到待办事项'), 404
    except sqlite3.Error as e:
        return jsonify(status='error', message=f'数据库错误: {e}'), 500


@app.route('/get_recurring_plans', methods=['GET'])
@login_required
def get_recurring_plans():
    user_id = current_user.id
    db = get_db()
    plans = db.execute("SELECT * FROM recurring_plans WHERE user_id = ? ORDER BY start_date DESC", (user_id,)).fetchall()
    return jsonify([dict(plan) for plan in plans])

@app.route('/add_recurring_plan', methods=['POST'])
@login_required
def add_recurring_plan():
    user_id = current_user.id
    data = request.get_json()
    if not data.get('plan_name') or not data.get('start_date') or not data.get('repeat_interval') or not data.get('exercises'):
        return jsonify(status='error', message='缺少必填字段'), 400
    if data.get('repeat_interval') == 'weekly' and not data.get('repeat_day'):
        return jsonify(status='error', message='“每周”循环缺少“星期几”设置'), 400
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute(
            """INSERT INTO recurring_plans (user_id, plan_name, exercises_json, start_date, end_date, repeat_interval, repeat_day) VALUES (?, ?, ?, ?, ?, ?, ?)""",
            (user_id, data.get('plan_name'), json.dumps(data.get('exercises')), data.get('start_date'), data.get('end_date'), data.get('repeat_interval'), data.get('repeat_day'))
        )
        db.commit()
        return jsonify(status='success', plan_id=cur.lastrowid)
    except sqlite3.Error as e: return jsonify(status='error', message=f'数据库错误: {e}'), 500

@app.route('/delete_recurring_plan/<int:plan_id>', methods=['POST'])
@login_required
def delete_recurring_plan(plan_id):
    user_id = current_user.id
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute("DELETE FROM recurring_plans WHERE id = ? AND user_id = ?", (plan_id, user_id))
        db.commit()
        if cur.rowcount > 0: return jsonify(status='success', message='循环计划已删除')
        else: return jsonify(status='error', message='未找到循环计划'), 404
    except sqlite3.Error as e: return jsonify(status='error', message=f'数据库错误: {e}'), 500

@app.route('/get_recurring_todos', methods=['GET'])
@login_required
def get_recurring_todos():
    user_id = current_user.id
    db = get_db()
    todos = db.execute("SELECT * FROM recurring_todos WHERE user_id = ? ORDER BY start_date DESC", (user_id,)).fetchall()
    return jsonify([dict(todo) for todo in todos])

@app.route('/add_recurring_todo', methods=['POST'])
@login_required
def add_recurring_todo():
    user_id = current_user.id
    data = request.get_json()
    if not data.get('content') or not data.get('start_date') or not data.get('repeat_interval'):
        return jsonify(status='error', message='缺少必填字段'), 400
    if (data.get('repeat_interval') == 'weekly' or data.get('repeat_interval') == 'monthly') and not data.get('repeat_day'):
        return jsonify(status='error', message='循环缺少“星期几”或“日期”设置'), 400
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute(
            """INSERT INTO recurring_todos (user_id, content, start_date, end_date, repeat_interval, repeat_day) VALUES (?, ?, ?, ?, ?, ?)""",
            (user_id, data.get('content'), data.get('start_date'), data.get('end_date'), data.get('repeat_interval'), data.get('repeat_day'))
        )
        db.commit()
        return jsonify(status='success', todo_id=cur.lastrowid)
    except sqlite3.Error as e: return jsonify(status='error', message=f'数据库错误: {e}'), 500

@app.route('/delete_recurring_todo/<int:todo_id>', methods=['POST'])
@login_required
def delete_recurring_todo(todo_id):
    user_id = current_user.id
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute("DELETE FROM recurring_todos WHERE id = ? AND user_id = ?", (todo_id, user_id))
        db.commit()
        if cur.rowcount > 0: return jsonify(status='success', message='循环待办已删除')
        else: return jsonify(status='error', message='未找到循环待办'), 404
    except sqlite3.Error as e: return jsonify(status='error', message=f'数据库错误: {e}'), 500

@app.route('/api/exercises', methods=['GET'])
@login_required
def list_exercises():
    user_id = current_user.id
    db = get_db()
    rows = db.execute(
        "SELECT id, name, details FROM exercise_library WHERE user_id = ? ORDER BY id DESC",
        (user_id,)
    ).fetchall()
    return jsonify([dict(r) for r in rows])


@app.route('/api/exercises', methods=['POST'])
@login_required
def create_exercise():
    user_id = current_user.id
    data = request.get_json() or {}
    name = (data.get('name') or '').strip()
    details = (data.get('details') or '').strip()
    if not name:
        return jsonify(status='error', message='缺少训练项目名称'), 400
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute(
            "INSERT INTO exercise_library (user_id, name, details) VALUES (?, ?, ?)",
            (user_id, name, details)
        )
        db.commit()
        return jsonify(status='success', exercise_id=cur.lastrowid)
    except sqlite3.Error as e:
        return jsonify(status='error', message=f'数据库错误: {e}'), 500


@app.route('/api/exercises/<int:exercise_id>', methods=['POST', 'DELETE'])
@login_required
def delete_exercise(exercise_id):
    user_id = current_user.id
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute(
            "DELETE FROM exercise_library WHERE id = ? AND user_id = ?",
            (exercise_id, user_id)
        )
        db.commit()
        if cur.rowcount > 0:
            return jsonify(status='success', message='训练项目已删除')
        else:
            return jsonify(status='error', message='未找到训练项目'), 404
    except sqlite3.Error as e:
        return jsonify(status='error', message=f'数据库错误: {e}'), 500


@app.route('/api/todo_library', methods=['GET'])
@login_required
def list_todo_library():
    user_id = current_user.id
    db = get_db()
    rows = db.execute(
        "SELECT id, content FROM todo_library WHERE user_id = ? ORDER BY id DESC",
        (user_id,)
    ).fetchall()
    return jsonify([dict(r) for r in rows])


@app.route('/api/todo_library', methods=['POST'])
@login_required
def create_todo_library_item():
    user_id = current_user.id
    data = request.get_json() or {}
    content = (data.get('content') or '').strip()
    if not content:
        return jsonify(status='error', message='缺少待办内容'), 400
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute(
            "INSERT INTO todo_library (user_id, content) VALUES (?, ?)",
            (user_id, content)
        )
        db.commit()
        return jsonify(status='success', item_id=cur.lastrowid)
    except sqlite3.Error as e:
        return jsonify(status='error', message=f'数据库错误: {e}'), 500


@app.route('/api/todo_library/<int:item_id>', methods=['POST', 'DELETE'])
@login_required
def delete_todo_library_item(item_id):
    user_id = current_user.id
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute(
            "DELETE FROM todo_library WHERE id = ? AND user_id = ?",
            (item_id, user_id)
        )
        db.commit()
        if cur.rowcount > 0:
            return jsonify(status='success', message='常用待办已删除')
        else:
            return jsonify(status='error', message='未找到该待办'), 404
    except sqlite3.Error as e:
        return jsonify(status='error', message=f'数据库错误: {e}'), 500

@app.after_request
def add_common_headers(response):
    if 'Content-Type' not in response.headers:
        response.headers['Content-Type'] = 'text/html; charset=utf-8'
    else:
        c = response.headers['Content-Type']
        if c.startswith('text/') and 'charset' not in c:
            response.headers['Content-Type'] = c + '; charset=utf-8'
    response.headers.setdefault('X-Content-Type-Options', 'nosniff')
    response.headers.setdefault('X-Frame-Options', 'SAMEORIGIN')
    path = request.path or ''
    if path.startswith('/static/') or path.endswith(('.css', '.js', '.png', '.jpg', '.svg', '.webp')):
        response.headers['Cache-Control'] = 'public, max-age=3600'
    else:
        response.headers['Cache-Control'] = 'no-cache, no-store, must-revalidate'
    return response

@app.route('/health', methods=['GET'])
def health():
    # ★ 核心修改：健康检查也显示中国时区时间
    return jsonify(status='ok', time=datetime.now(CN_TZ).isoformat())

if __name__ == '__main__':
    app.run(host='0.0.0.0', port=8000, debug=True)
'''

# ==========================================
# 2. frontend/templates/manage.html 的完整内容 (已修复UI逻辑)
# ==========================================
manage_html_content = r'''<!DOCTYPE html>
<html lang="zh-CN">
<head>
  <meta charset="utf-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1.0" />
  <title>健身看板 - 后台管理</title>
  <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
  <link rel="preconnect" href="https://fonts.googleapis.com">
  <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
  <link href="https://fonts.googleapis.com/css2?family=Noto+Sans+SC:wght@300;400;500;600;700&family=Roboto:wght@400;500;700&display=swap" rel="stylesheet">
  <style>
    :root {
        --bg-color: #f1f5f9;
        --panel-color: #ffffff;
        --accent-color: #0f766e;
        --accent-color-light: #ccfbf1;
        --accent-color-soft: #ecfdf5;
        --accent-danger: #dc2626;
        --accent-warning: #f97316;
        --text-primary: #0f172a;
        --text-secondary: #64748b;
        --border-soft: #e2e8f0;
        --shadow-soft: 0 18px 40px rgba(15, 23, 42, 0.12);
        --radius-lg: 18px;
        --radius-md: 12px;
        --radius-sm: 8px;
        --spacing-base: 20px;
    }
.hidden {
  display: none !important;
}

    * { box-sizing: border-box; }
    body {
      margin: 0;
      font-family: 'Noto Sans SC', 'Roboto', system-ui, -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif;
      background-color: var(--bg-color);
      color: var(--text-primary);
    }
    .app-container { display: flex; min-height: 100vh; }
    .sidebar {
      width: 260px;
      background: linear-gradient(180deg, #0f172a, #020617);
      color: #e5e7eb;
      display: flex;
      flex-direction: column;
      padding: 24px 18px;
      box-shadow: 6px 0 25px rgba(15, 23, 42, 0.7);
      position: sticky;
      top: 0;
      height: 100vh;
    }
    .sidebar-header {
      display: flex;
      align-items: center;
      gap: 12px;
      margin-bottom: 32px;
    }
    .logo-circle {
      width: 40px;
      height: 40px;
      border-radius: 50%;
      background: radial-gradient(circle at 30% 20%, #a7f3d0, #16a34a);
      display: flex;
      align-items: center;
      justify-content: center;
      color: #022c22;
      font-weight: 800;
      font-size: 20px;
      box-shadow: 0 0 0 3px rgba(34, 197, 94, 0.2);
    }
    .sidebar-title { font-size: 18px; font-weight: 700; }
    .sidebar-subtitle {
      font-size: 12px;
      color: #9ca3af;
      margin-top: 2px;
    }
    .nav-section-title {
      font-size: 12px;
      text-transform: uppercase;
      letter-spacing: 0.08em;
      color: #6b7280;
      margin: 18px 10px 8px;
    }
    .nav-list {
      list-style: none;
      margin: 0;
      padding: 0;
      flex: 1;
    }
    .nav-item { margin-bottom: 6px; }
    .nav-button {
      width: 100%;
      padding: 10px 12px;
      border-radius: 12px;
      border: none;
      background: transparent;
      color: #d1d5db;
      font-size: 14px;
      text-align: left;
      display: flex;
      align-items: center;
      cursor: pointer;
      transition: all 0.18s ease;
      position: relative;
      gap: 8px;
    }
    .nav-button::before {
      content: '';
      position: absolute;
      left: 8px;
      top: 50%;
      width: 4px;
      height: 4px;
      border-radius: 999px;
      transform: translateY(-50%);
      background: transparent;
    }
    .nav-button:hover {
      background: rgba(15, 23, 42, 0.8);
      transform: translateX(2px);
    }
    .nav-button.active {
      background: linear-gradient(90deg, rgba(34, 197, 94, 0.22), rgba(34, 197, 94, 0.04));
      color: #ecfdf3;
      box-shadow: 0 10px 25px rgba(22, 163, 74, 0.5);
    }
    .nav-button.active::before {
      background: #22c55e;
      height: 20px;
    }
    .nav-icon { font-size: 16px; width: 20px; text-align: center; }
    .nav-footer {
      margin-top: 12px;
      padding: 10px 12px;
      border-radius: 12px;
      background: rgba(15, 23, 42, 0.75);
      font-size: 12px;
      color: #9ca3af;
    }
    .nav-footer strong { color: #e5e7eb; }
    .main {
      flex: 1;
      padding: 24px 30px;
      max-width: calc(100vw - 260px);
    }
    .top-bar {
      display: flex;
      justify-content: space-between;
      align-items: center;
      margin-bottom: 18px;
    }
    .top-title { display: flex; flex-direction: column; gap: 4px; }
    .top-title h1 {
      margin: 0;
      font-size: 26px;
      letter-spacing: 0.02em;
    }
    .top-title span {
      font-size: 13px;
      color: var(--text-secondary);
    }
    .top-actions {
      display: flex;
      align-items: center;
      gap: 8px;
      font-size: 13px;
      color: var(--text-secondary);
    }
    .pill-tag {
      padding: 6px 10px;
      border-radius: 999px;
      background: rgba(45, 212, 191, 0.1);
      border: 1px solid rgba(34, 197, 94, 0.35);
      color: #047857;
      font-size: 12px;
      display: inline-flex;
      align-items: center;
      gap: 6px;
    }
    .pill-tag span.dot {
      width: 7px;
      height: 7px;
      border-radius: 50%;
      background: #22c55e;
      box-shadow: 0 0 12px rgba(34, 197, 94, 0.8);
    }
    .main-grid {
      display: grid;
      grid-template-columns: minmax(0, 3fr) minmax(0, 2fr);
      gap: 18px;
      align-items: flex-start;
    }
    .panel {
      background-color: var(--panel-color);
      border-radius: var(--radius-lg);
      box-shadow: var(--shadow-soft);
      padding: 20px 22px;
      margin-bottom: 18px;
      border: 1px solid rgba(148, 163, 184, 0.18);
    }
    .panel-header {
      display: flex;
      justify-content: space-between;
      align-items: baseline;
      margin-bottom: 14px;
    }
    .panel-header h2 {
      margin: 0;
      font-size: 18px;
      color: var(--text-primary);
      display: flex;
      align-items: center;
      gap: 6px;
    }
    .panel-header h2 span.badge {
      font-size: 10px;
      text-transform: uppercase;
      letter-spacing: 0.08em;
      padding: 2px 8px;
      border-radius: 999px;
      background: var(--accent-color-soft);
      color: #047857;
      border: 1px solid rgba(16, 185, 129, 0.25);
    }
    .panel-header p {
      margin: 4px 0 0;
      font-size: 13px;
      color: var(--text-secondary);
    }
    .panel-body { margin-top: 6px; }
    .content-section { display: none; }
    .content-section.active { display: block; }
    .form-group { margin-bottom: 12px; }
    .form-group label {
      display: block;
      margin-bottom: 4px;
      font-size: 13px;
      color: var(--text-secondary);
    }
    .form-group input, .form-group select, .form-group textarea {
      width: 100%;
      padding: 12px;
      border: 1px solid #cbd5e1;
      border-radius: var(--radius-md);
      font-size: 1rem;
      color: var(--text-primary);
      background: #f8fafc;
    }
    .form-group input:focus, .form-group select:focus {
        outline: none;
        border-color: var(--accent-color);
        background: var(--panel-color);
        box-shadow: 0 0 0 3px var(--accent-color-light);
    }
    .btn-submit, .btn-primary {
      background-color: var(--accent-color);
      color: #ecfdf5;
      border: none;
      padding: 10px 18px;
      border-radius: 999px;
      cursor: pointer;
      font-size: 14px;
      font-weight: 600;
      display: inline-flex;
      align-items: center;
      gap: 6px;
      box-shadow: 0 12px 25px rgba(16, 185, 129, 0.35);
      transition: all 0.15s ease;
    }
    .btn-submit:hover, .btn-primary:hover {
      transform: translateY(-1px);
      box-shadow: 0 16px 30px rgba(16, 185, 129, 0.55);
      background-color: #0d9488;
    }
    .btn-secondary {
      background-color: #e5e7eb;
      color: #111827;
      border: none;
      padding: 6px 12px;
      border-radius: 999px;
      cursor: pointer;
      font-size: 12px;
      font-weight: 500;
    }
    .btn-secondary:hover { background-color: #d1d5db; }
    .btn-danger {
      background-color: #ef4444;
      color: #fff;
      border: none;
      padding: 6px 12px;
      border-radius: 999px;
      cursor: pointer;
      font-size: 12px;
    }
    .btn-danger:hover { background-color: #b91c1c; }
    .btn-add-ex {
      background-color: #e0f2fe;
      color: #075985;
      border: none;
      padding: 8px 12px;
      border-radius: 999px;
      cursor: pointer;
      font-size: 12px;
      font-weight: 500;
    }
    .btn-add-ex:hover { background-color: #bfdbfe; }
    .list-item {
      padding: 10px 12px;
      border-radius: 10px;
      border: 1px solid var(--border-soft);
      margin-bottom: 8px;
      background: #f9fafb;
      display: flex;
      justify-content: space-between;
      align-items: center;
      gap: 10px;
    }
    .list-item span { font-size: 13px; }
    .list-item .item-actions { display: flex; gap: 6px; }
    .alert-success {
      padding: 10px 12px;
      border-radius: var(--radius-md);
      background-color: #ecfdf5;
      border: 1px solid #bbf7d0;
      color: #166534;
      font-size: 13px;
      margin-bottom: 10px;
    }
    .alert-error {
      padding: 10px 12px;
      border-radius: var(--radius-md);
      background-color: #fef2f2;
      border: 1px solid #fecaca;
      color: #b91c1c;
      font-size: 13px;
      margin-bottom: 10px;
    }
    .exercise-group {
      display: grid;
      grid-template-columns: 1.1fr 1fr;
      gap: 8px;
      margin-bottom: 8px;
    }
    .section-grid {
      display: grid;
      grid-template-columns: 1.2fr 1fr;
      gap: 16px;
    }
    .small-hint { font-size: 12px; color: var(--text-secondary); }
    .readonly-box {
      padding: 10px 12px;
      border-radius: 10px;
      background: #f1f5f9;
      border: 1px dashed #cbd5e1;
      font-size: 13px;
      color: var(--text-secondary);
      margin: 6px 0 10px;
    }
    .divider {
      height: 1px;
      background: linear-gradient(90deg, transparent, rgba(148, 163, 184, 0.6), transparent);
      margin: 14px 0;
    }
    .stats-buttons {
      display: flex;
      gap: 10px;
      margin-bottom: 12px;
    }
    .stats-buttons button { flex: 1; }
    .stats-layout {
      display: grid;
      grid-template-columns: 1.6fr 1.4fr;
      gap: 16px;
      margin-top: 12px;
    }
    .stats-card {
      padding: 12px 14px;
      border-radius: 14px;
      background: #f8fafc;
      border: 1px solid #e2e8f0;
      font-size: 13px;
      color: var(--text-secondary);
    }
    .stats-card h4 {
      margin: 0 0 8px;
      font-size: 14px;
      color: var(--text-primary);
    }
    .stats-summary { margin-top: 10px; font-size: 14px; color: var(--text-secondary); }
    .stats-highlight { font-weight: 600; color: #0f766e; }
    .stats-chip-row {
      display: flex;
      flex-wrap: wrap;
      gap: 6px;
      margin-top: 6px;
    }
    .stats-chip {
      padding: 4px 8px;
      border-radius: 999px;
      font-size: 11px;
      background: #e0f2fe;
      color: #075985;
    }
    .profile-grid {
      display: grid;
      grid-template-columns: minmax(0, 1.2fr) minmax(0, 1.2fr);
      gap: 16px;
    }
    .profile-box {
      padding: 14px 16px;
      border-radius: 14px;
      border: 1px solid #e2e8f0;
      background: #f9fafb;
      font-size: 13px;
      color: var(--text-secondary);
    }
    .profile-box h3 { margin: 0 0 6px; font-size: 15px; color: var(--text-primary); }
    .footer-note { font-size: 12px; text-align: right; color: #9ca3af; margin-top: 6px; }

    @media (max-width: 1024px) {
      .app-container { flex-direction: column; }
      .sidebar {
        width: 100%;
        height: auto;
        position: static;
        flex-direction: row;
        overflow-x: auto;
      }
      .sidebar-header { margin-bottom: 0; }
      .nav-section-title, .nav-footer { display: none; }
      .nav-list { display: flex; flex-wrap: nowrap; gap: 4px; }
      .nav-item { flex: 1; }
      .main {
        max-width: 100%;
        padding: 16px;
      }
      .main-grid { grid-template-columns: minmax(0, 1fr); }
      .section-grid { grid-template-columns: minmax(0, 1fr); }
      .stats-layout { grid-template-columns: minmax(0, 1fr); }
      .profile-grid { grid-template-columns: minmax(0, 1fr); }
      .panel { padding: 16px; }
    }
  </style>
</head>
<body>
  <div class="app-container">
    <aside class="sidebar">
      <div class="sidebar-header">
        <div class="logo-circle">F</div>
        <div>
          <div class="sidebar-title">智能健身 & 待办 管理</div>
          <div class="sidebar-subtitle">循环计划 · 大屏展示 · 多设备同步</div>
        </div>
      </div>
      <div class="nav-section-title">导航</div>
      <ul class="nav-list" id="navList">
        <li class="nav-item">
          <button class="nav-button active" data-tab="daily-todos">
            <span class="nav-icon">✅</span> 每日待办 (T)
          </button>
        </li>
        <li class="nav-item">
          <button class="nav-button" data-tab="daily-fitness">
            <span class="nav-icon">💪</span> 每日健身计划 (T)
          </button>
        </li>
        <li class="nav-item">
          <button class="nav-button" data-tab="recurring-todos">
            <span class="nav-icon">🔁</span> 循环待办 (R)
          </button>
        </li>
        <li class="nav-item">
          <button class="nav-button" data-tab="recurring-fitness">
            <span class="nav-icon">📅</span> 循环健身计划 (R)
          </button>
        </li>
        <li class="nav-item">
          <button class="nav-button" data-tab="stats">
            <span class="nav-icon">📊</span> 统计与分析
          </button>
        </li>
        <li class="nav-item">
          <button class="nav-button" data-tab="mobile-config">
            <span class="nav-icon">📱</span> 手机 & 平板配置
          </button>
        </li>
        <li class="nav-item" id="userManagementNavItem" {% if not is_admin %}style="display:none;"{% endif %}>
          <button class="nav-button" data-tab="user-management">
            <span class="nav-icon">👥</span> 用户管理
          </button>
        </li>
        <li class="nav-item">
          <button class="nav-button" data-tab="profile">
            <span class="nav-icon">⚙️</span> 个人资料
          </button>
        </li>
      </ul>
      <div class="nav-footer">
        <div><strong>提示：</strong>可在「循环计划」中设置长期模式，系统会每天自动生成每日任务。</div>
      </div>
    </aside>

    <main class="main">
      <div class="top-bar">
        <div class="top-title">
          <h1>后台管理中心</h1>
          <span>配置每日/循环健身 & 待办 · 查看统计 · 管理显示设备</span>
        </div>
        <div class="top-actions">
          <span class="pill-tag"><span class="dot"></span> 大屏看板已联机</span>
          <span id="current-user-label">当前用户：{{ username }}</span>
        </div>
      </div>

      <div class="panel">
        <div class="panel-header">
          <div>
            <h2>今日总览 <span class="badge">OVERVIEW</span></h2>
            <p>快速查看当天的待办和健身计划完成情况。</p>
          </div>
        </div>
        <div class="panel-body">
          <div class="main-grid">
            <div>
              <div id="todaySummary" class="stats-card">
                <h4>今日任务完成情况</h4>
                <div id="todaySummaryContent">加载中...</div>
              </div>
            </div>
            <div>
              <div class="stats-card">
                <h4>使用建议</h4>
                <div class="stats-chip-row">
                  <span class="stats-chip">✅ 每日待办 (T) 管具体某天</span>
                  <span class="stats-chip">📅 循环待办/健身 (R) 管长期模式</span>
                  <span class="stats-chip">🖥 display 负责展示 + 勾选</span>
                </div>
                <div class="divider"></div>
                <div class="small-hint">
                  每天用「每日」两个页面检视当日任务；<br>
                  定期用「循环」两个页面维护长期习惯；<br>
                  手机/平板则在「手机 & 平板配置」中查看说明。
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>

      <div class="panel">
        <div class="panel-body">
          <div id="daily-todos" class="content-section active">
            <div class="panel-header">
              <div>
                <h2>每日待办事项 (T) <span class="badge">TODAY</span></h2>
                <p>查看和管理具体某一天的待办事项。</p>
              </div>
            </div>
            <div class="panel-body">
              <div class="section-grid">
                <div>
                  <h3 style="margin-top:0;">选择日期</h3>
                  <div class="form-group">
                    <label for="date-todo">选择日期:</label>
                    <input type="date" id="date-todo">
                  </div>
                  <h3>当日待办列表</h3>
                  <div id="todoList"></div>
                </div>
                <div>
                  <h3 style="margin-top:0;">添加新的待办事项</h3>
                  <form id="addTodoForm">
                    <div class="form-group">
                      <label for="todo-content">待办内容:</label>
                      <textarea id="todo-content" rows="3" required></textarea>
                    </div>
                    <button type="submit" class="btn-submit">添加当日待办</button>
                  </form>
                  <div class="readonly-box">
                    提示：每天首次打开首页或大屏时，会根据「循环待办 (R)」自动生成当天的基础待办列表。
                    这里的增删改只影响当天，不会改变循环设置。
                  </div>
                </div>
              </div>
            </div>
          </div>

          <div id="daily-fitness" class="content-section">
            <div class="panel-header">
              <div>
                <h2>每日健身计划 (T) <span class="badge">TODAY</span></h2>
                <p>查看和管理特定某一天的健身计划 (通常由循环计划生成)。</p>
              </div>
            </div>
            <div class="panel-body">
              <div class="section-grid">
                <div>
                  <h3 style="margin-top:0;">选择日期</h3>
                  <div class="form-group">
                    <label for="date-daily">选择日期:</label>
                    <input type="date" id="date-daily">
                  </div>
                  <h3>计划列表:</h3>
                  <div id="dailyFitnessList"></div>
                </div>
                <div>
                  <h3 style="margin-top:0;">说明</h3>
                  <div class="readonly-box">
                    每日健身计划通常由「循环健身计划 (R)」自动生成。<br>
                    如需修改长期模式（例如每周一练胸、每周三跑步），请前往下方「循环健身计划 (R)」设置。
                  </div>
                </div>
              </div>
            </div>
          </div>

          <div id="recurring-fitness" class="content-section">
            <div class="panel-header">
              <div>
                <h2>循环健身计划 (R)</h2>
                <p>设置自动生成的循环健身计划。 (例如：每周一练胸)</p>
              </div>
            </div>
            <div class="panel-body">
              <form id="addRecurringPlanForm">
                <div class="form-group">
                  <label for="rec-plan-name">计划名称 (例如: 周一练胸日):</label>
                  <input type="text" id="rec-plan-name" required>
                </div>
                <div class="form-group">
                  <label for="rec-plan-start">开始日期:</label>
                  <input type="date" id="rec-plan-start" required>
                </div>
                <div class="form-group">
                  <label for="rec-plan-end">结束日期 (可选):</label>
                  <input type="date" id="rec-plan-end">
                </div>
                <div class="form-group">
                  <label for="rec-plan-interval">循环方式:</label>
                  <select id="rec-plan-interval">
                    <option value="daily">每日</option>
                    <option value="weekly">每周</option>
                  </select>
                </div>
                <div class="form-group hidden" id="rec-plan-day-weekly">
                  <label for="rec-plan-day-w">星期几:</label>
                  <select id="rec-plan-day-w">
                    <option value="1">星期一</option>
                    <option value="2">星期二</option>
                    <option value="3">星期三</option>
                    <option value="4">星期四</option>
                    <option value="5">星期五</option>
                    <option value="6">星期六</option>
                    <option value="7">星期日</option>
                  </select>
                </div>

                <label>训练项目列表:</label>

                <div class="form-group">
                  <div style="display:flex; gap:8px; flex-wrap:wrap; margin-bottom:6px;">
                    <select id="exercise-library-select">
                      <option value="">从训练项目库中选择...</option>
                    </select>
                    <button type="button" id="addFromLibraryBtn" class="btn-add-ex">添加到上方列表</button>
                  </div>
                  <p class="small-hint">
                    建议先在下方“训练项目库”维护常用动作，然后在这里直接选择，无需每次手动输入。
                  </p>
                </div>

                <div id="exercise-inputs">
                  <div class="exercise-group">
                    <input type="text" class="rec-ex-name" placeholder="训练 1: 名称">
                    <input type="text" class="rec-ex-details" placeholder="训练 1: 组数/时长">
                  </div>
                </div>
                <button type="button" id="addExerciseBtn" class="btn-add-ex">添加训练项目</button>

                <div style="margin-top: 15px;">
                  <button type="submit" class="btn-submit">保存循环计划</button>
                </div>
              </form>

              <div class="divider"></div>

              <h3>已保存的循环健身计划</h3>
              <div id="recurringPlanList"></div>

              <div class="divider"></div>

              <h3>训练项目库</h3>
              <p class="small-hint">
                在这里维护常用的训练动作，例如“俯卧撑 3×15”、“平板支撑 60 秒”，然后在上方循环计划中选择使用。
              </p>
              <form id="exerciseLibraryForm">
                <div class="form-group">
                  <label for="lib-ex-name">项目名称:</label>
                  <input type="text" id="lib-ex-name" required placeholder="例如：俯卧撑">
                </div>
                <div class="form-group">
                  <label for="lib-ex-details">默认组数/时长 (可选):</label>
                  <input type="text" id="lib-ex-details" placeholder="例如：3×15 或 60 秒">
                </div>
                <button type="submit" class="btn-submit">添加到项目库</button>
              </form>
              <div id="exerciseLibraryList" style="margin-top: 12px;"></div>
            </div>
          </div>

          <div id="recurring-todos" class="content-section">
            <div class="panel-header">
              <div>
                <h2>循环待办事项 (R)</h2>
                <p>设置自动生成的循环待办事项，例如“每周一开会”。</p>
              </div>
            </div>
            <div class="panel-body">
              <form id="addRecurringTodoForm">
                <div class="form-group">
                  <label for="rec-todo-content">待办内容:</label>
                  <div style="display:flex; gap:8px; flex-wrap:wrap; margin-bottom:6px;">
                    <select id="todo-library-select">
                      <option value="">从常用待办库中选择...</option>
                    </select>
                    <button type="button" id="fillTodoFromLibraryBtn" class="btn-add-ex">填入上方输入框</button>
                  </div>
                  <input type="text" id="rec-todo-content" required>
                </div>
                <div class="form-group">
                  <label for="rec-todo-start">开始日期:</label>
                  <input type="date" id="rec-todo-start" required>
                </div>
                <div class="form-group">
                  <label for="rec-todo-end">结束日期 (可选):</label>
                  <input type="date" id="rec-todo-end">
                </div>
                <div class="form-group">
                  <label for="rec-todo-interval">循环方式:</label>
                  <select id="rec-todo-interval">
                    <option value="daily">每日</option>
                    <option value="weekly">每周</option>
                    <option value="monthly">每月</option>
                  </select>
                </div>
                <div class="form-group hidden" id="rec-todo-day-weekly">
                  <label for="rec-todo-day-w">星期几:</label>
                  <select id="rec-todo-day-w">
                    <option value="1">星期一</option>
                    <option value="2">星期二</option>
                    <option value="3">星期三</option>
                    <option value="4">星期四</option>
                    <option value="5">星期五</option>
                    <option value="6">星期六</option>
                    <option value="7">星期日</option>
                  </select>
                </div>
                <div class="form-group hidden" id="rec-todo-day-monthly">
                  <label for="rec-todo-day-m">几号 (1-31):</label>
                  <input type="number" id="rec-todo-day-m" min="1" max="31" value="1">
                </div>
                <button type="submit" class="btn-submit">保存循环待办</button>
              </form>

              <div class="divider"></div>

              <h3>已保存的循环待办</h3>
              <div id="recurringTodoList"></div>

              <div class="divider"></div>

              <h3>常用待办库</h3>
              <p class="small-hint">
                在这里维护常用的待办项，例如“喝水 500ml”、“拉伸 5 分钟”等，上方循环待办可直接选择使用。
              </p>
              <form id="todoLibraryForm">
                <div class="form-group">
                  <label for="lib-todo-content">待办内容:</label>
                  <input type="text" id="lib-todo-content" required placeholder="例如：喝水 500ml">
                </div>
                <button type="submit" class="btn-submit">添加到常用待办库</button>
              </form>
              <div id="todoLibraryList" style="margin-top: 12px;"></div>
            </div>
          </div>

          <div id="stats" class="content-section">
            <div class="panel-header">
              <div>
                <h2>统计与分析</h2>
                <p>按时间范围统计健身与待办的完成情况。</p>
              </div>
            </div>
            <div class="panel-body">
              <div class="form-group">
                <label>统计区间 (起止日期):</label>
                <div style="display:flex; gap:8px; flex-wrap:wrap; align-items:center;">
                  <input type="date" id="stats-start">
                  <span style="font-size:13px;color:var(--text-secondary);">至</span>
                  <input type="date" id="stats-end">
                  <button id="stats-refresh" class="btn-secondary" type="button">刷新区间</button>
                </div>
              </div>
              <div class="stats-buttons">
                <button id="stats-btn-week" class="btn-secondary" type="button">最近 7 天</button>
                <button id="stats-btn-month" class="btn-secondary" type="button">最近 30 天</button>
                <button id="stats-btn-range" class="btn-secondary" type="button">使用上方日期</button>
              </div>
              <div class="stats-layout">
                <div>
                  <canvas id="statsChart"></canvas>
                </div>
                <div>
                  <div class="stats-card">
                    <h4>完成率概览</h4>
                    <div id="statsSummary" class="stats-summary">请选择时间范围后查看。</div>
                  </div>
                  <div class="stats-card" style="margin-top:10px;">
                    <h4>小贴士</h4>
                    <p class="small-hint">
                      建议：<br>
                      ① 每天至少完成 1 条健身任务或 2 条日常待办；<br>
                      ② 将长期事项（如“每周三跑步”）设为循环计划，方便系统自动生成。
                    </p>
                  </div>
                </div>
              </div>
              <div class="footer-note">统计数据来源于每日健身 & 待办完成记录，仅用于个人自我监督。</div>
            </div>
          </div>

          <div id="mobile-config" class="content-section">
            <div class="panel-header">
              <div>
                <h2>大屏 & 移动端配置</h2>
                <p>在这里可以调整大屏字幕滚动速度，并查看手机/平板的使用说明。</p>
              </div>
            </div>
            <div class="panel-body">
              <div class="profile-box" style="margin-bottom:18px;">
                <h3>大屏字幕滚动速度</h3>
                <p class="small-hint">
                  数值表示 <strong>滚完整屏列表所需的秒数</strong>。数值越小滚动越快，建议在
                  <strong>10 ~ 60 秒</strong> 之间根据观看距离微调。
                </p>
                <form id="scrollSpeedForm">
                  <div class="form-group" style="max-width:260px;">
                    <label for="scroll-speed-input">滚动速度（秒/一轮）:</label>
                    <input type="number" id="scroll-speed-input" min="5" max="120" step="1" value="25">
                  </div>
                  <button type="submit" class="btn-submit">保存滚动速度</button>
                  <span id="scroll-speed-status" style="margin-left:10px;font-size:13px;color:var(--text-secondary);"></span>
                </form>
              </div>
              
              <iframe src="/mobile_config"
                      style="width:100%;height:650px;border:none;border-radius:14px;box-shadow:0 12px 30px rgba(15,23,42,.15);"></iframe>
            </div>
          </div>
 
          <div id="user-management" class="content-section">
            <div class="panel-header">
              <div>
                <h2>用户管理</h2>
                <p>
                  {% if manager_id is none or manager_id == 0 %}
                  (超级管理员) 您可以创建其他管理员或普通用户。
                  {% else %}
                  (管理员) 您可以创建归属于您的普通用户。
                  {% endif %}
                </p>
              </div>
            </div>
            <div class="panel-body">
              
              {% if manager_id is none or manager_id == 0 %}
              <div class="alert-success">您是超级管理员 (Super Admin)。</div>
              {% endif %}

              <h3>
                {% if manager_id is none or manager_id == 0 %}
                创建新管理员 / 新用户
                {% else %}
                创建新用户
                {% endif %}
              </h3>
              <form id="createUserForm">
                <div class="form-group">
                  <label for="new-username">用户名:</label>
                  <input type="text" id="new-username" required>
                </div>
                <div class="form-group">
                  <label for="new-password">密码:</label>
                  <input type="password" id="new-password" required>
                </div>
                
                {% if manager_id is none or manager_id == 0 %}
                <div class="form-group">
                  <label style="font-size:13px;color:var(--text-secondary);">
                    <input type="checkbox" id="is-admin" value="1"> 设为管理员 (Manager)
                  </label>
                </div>
                {% endif %}
                
                <button type="submit" class="btn-submit">创建用户</button>
              </form>
              <h3 style="margin-top: 24px;">
                {% if manager_id is none or manager_id == 0 %}
                我创建的管理员/用户列表
                {% else %}
                我创建的用户列表
                {% endif %}
              </h3>
              <div id="userList"></div>
            </div>
          </div>

          <div id="profile" class="content-section">
            <div class="panel-header">
              <div>
                <h2>个人资料</h2>
                <p>查看和修改当前登录用户的基本信息。</p>
              </div>
            </div>
            <div class="panel-body">
              <div class="profile-grid">
                <div>
                  <div class="profile-box">
                    <h3>修改我的资料</h3>
                    <p class="small-hint">
                      当前登录账号: <strong>{{ username }}</strong>
                      <br>
                      角色：<strong>
                        {% if is_admin %}
                          {% if manager_id is none or manager_id == 0 %}
                            超级管理员 (Super Admin)
                          {% else %}
                            管理员 (Manager)
                          {% endif %}
                        {% else %}
                          普通用户
                        {% endif %}
                      </strong>
                    </p>
                    <div class="divider"></div>
                    <form id="profileUpdateForm">
                      <div class="form-group">
                        <label for="profile-username">新用户名:</label>
                        <input type="text" id="profile-username" name="username" placeholder="留空则不修改">
                      </div>
                      <div class="form-group">
                        <label for="profile-password">新密码:</label>
                        <input type="password" id="profile-password" name="password" placeholder="留空则不修改">
                      </div>
                      <button type="submit" class="btn-submit">保存更改</button>
                      <span id="profileUpdateStatus" class="small-hint" style="margin-left: 10px;"></span>
                    </form>
                  </div>
                </div>
                <div>
                  <div class="profile-box">
                    <h3>数据存储位置</h3>
                    <p>所有数据保存在 <code>fitness_manager.db</code> 中，位于后台目录下。</p>
                    <p>备份方法：在停止程序后，复制该文件保存到安全位置即可。</p>
                  </div>
                </div>
              </div>
            </div>
          </div></div>
      </div></main>
  </div>

  <script>
    // ★★★ 核心修复：使用本地时间获取“今天” (而不是 UTC) ★★★
    const today = new Date();
    const year = today.getFullYear();
    const month = String(today.getMonth() + 1).padStart(2, '0');
    const day = String(today.getDate()).padStart(2, '0');
    const todayStr = `${year}-${month}-${day}`;

    async function apiFetch(url, options = {}) {
      const finalUrl = url.startsWith('http') ? url : (location.origin + url);
      const resp = await fetch(finalUrl, {
        credentials: 'include',
        ...options
      });
      return resp;
    }

    const navButtons = document.querySelectorAll('.nav-button');
    const sections = document.querySelectorAll('.content-section');
    navButtons.forEach(btn => {
      btn.addEventListener('click', () => {
        navButtons.forEach(b => b.classList.remove('active'));
        sections.forEach(s => s.classList.remove('active'));
        btn.classList.add('active');
        const targetId = btn.getAttribute('data-tab');
        document.getElementById(targetId).classList.add('active');
        loadScrollSpeedSetting();
        if (targetId === 'daily-todos') loadTodos(datePickerTodo.value);
        if (targetId === 'daily-fitness') loadDailyFitness(datePickerDaily.value);
        if (targetId === 'recurring-todos') {
          loadRecurringTodos();
          loadTodoLibrary();
        }
        if (targetId === 'recurring-fitness') {
          loadRecurringPlans();
          loadExerciseLibrary();
        }
        if (targetId === 'stats') document.getElementById('stats-btn-week').click();
        if (targetId === 'mobile-config') {
          const iframe = document.querySelector('#mobile-config iframe');
          iframe.contentWindow.location.reload();
        }
        if (targetId === 'user-management') loadUsers();
      });
    });

    const datePickerTodo = document.getElementById('date-todo');
    const datePickerDaily = document.getElementById('date-daily');
    datePickerTodo.addEventListener('change', () => loadTodos(datePickerTodo.value));
    datePickerDaily.addEventListener('change', () => loadDailyFitness(datePickerDaily.value));

    function showAlert(message, isError = false) {
      let el = document.getElementById('globalAlert');
      if (!el) {
        el = document.createElement('div');
        el.id = 'globalAlert';
        el.style.position = 'fixed';
        el.style.top = '16px';
        el.style.right = '16px';
        el.style.zIndex = '9999';
        document.body.appendChild(el);
      }
      el.className = isError ? 'alert-error' : 'alert-success';
      el.textContent = message;
      el.style.display = 'block';
      setTimeout(() => el.style.display = 'none', 4000);
    }

    async function loadTodos(date_str) {
      try {
        const response = await apiFetch(`/todos?date=${date_str}`);
        const todos = await response.json();
        const listHtml = todos.map(todo => `
          <div class="list-item">
            <span>[${todo.is_completed ? '完成' : '未完成'}] ${todo.content} ${todo.recurring_todo_id ? '(R)' : ''}</span>
            <div class="item-actions">
              <button class="btn-danger btn-delete" data-type="todo" data-id="${todo.id}">删除</button>
            </div>
          </div>`).join('');
        document.getElementById('todoList').innerHTML = listHtml || '<div class="small-hint">该日暂无待办事项。</div>';
      } catch (e) {
        console.error('loadTodos error', e);
        document.getElementById('todoList').innerHTML = '<div class="alert-error">加载失败</div>';
      }
    }

    async function loadDailyFitness(date_str) {
      try {
        const response = await apiFetch(`/fitness_plan/${date_str}`);
        const plans = await response.json();
        const listHtml = plans.map(plan => `
          <div class="list-item">
            <span>[${plan.is_completed ? '完成' : '未完成'}] ${plan.exercise_name} - ${plan.sets_reps_duration}</span>
            <div class="item-actions">
              <button class="btn-danger btn-delete" data-type="daily-fitness" data-id="${plan.id}">删除</button>
            </div>
          </div>`).join('');
        document.getElementById('dailyFitnessList').innerHTML = listHtml || '<div class="small-hint">该日暂无健身计划。</div>';
      } catch (e) {
        console.error('loadDailyFitness error', e);
        document.getElementById('dailyFitnessList').innerHTML = '<div class="alert-error">加载失败</div>';
      }
    }

    async function loadRecurringPlans() {
      try {
        const response = await apiFetch('/get_recurring_plans');
        const plans = await response.json();
        const listHtml = plans.map(plan => `
          <div class="list-item">
            <span>${plan.plan_name} [${plan.repeat_interval === 'daily' ? '每日' : '每周' + '周' + '一二三四五六日'[plan.repeat_day - 1]}]
              (${plan.start_date} ~ ${plan.end_date || '无限期'})</span>
            <div class="item-actions">
              <button class="btn-danger btn-delete" data-type="recurring-fitness" data-id="${plan.id}">删除</button>
            </div>
          </div>`).join('');
        document.getElementById('recurringPlanList').innerHTML = listHtml || '<div class="small-hint">尚未设置循环健身计划。</div>';
      } catch (e) {
        console.error('loadRecurringPlans error', e);
        document.getElementById('recurringPlanList').innerHTML = '<div class="alert-error">加载失败</div>';
      }
    }

    async function loadRecurringTodos() {
      try {
        const response = await apiFetch('/get_recurring_todos');
        const todos = await response.json();
        const listHtml = todos.map(todo => `
          <div class="list-item">
            <span>${todo.content} [${
              todo.repeat_interval === 'daily'
                ? '每日'
                : (todo.repeat_interval === 'weekly'
                    ? '每周' + '周' + '一二三四五六日'[todo.repeat_day - 1]
                    : '每月' + todo.repeat_day + '号')}]
              (${todo.start_date} ~ ${todo.end_date || '无限期'})</span>
            <div class="item-actions">
              <button class="btn-danger btn-delete" data-type="recurring-todo" data-id="${todo.id}">删除</button>
            </div>
          </div>`).join('');
        document.getElementById('recurringTodoList').innerHTML = listHtml || '<div class="small-hint">尚未设置循环待办。</div>';
      } catch (e) {
        console.error('loadRecurringTodos error', e);
        document.getElementById('recurringTodoList').innerHTML = '<div class="alert-error">加载失败</div>';
      }
    }

    document.getElementById('addTodoForm').addEventListener('submit', async (e) => {
      e.preventDefault();
      const content = document.getElementById('todo-content').value.trim();
      if (!content) { alert('请输入待办内容'); return; }
      const date = datePickerTodo.value;
      try {
        await apiFetch('/add_todo', {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({ date, content })
        });
        document.getElementById('todo-content').value = '';
        loadTodos(date);
        showAlert('已添加每日待办');
      } catch (e) {
        console.error('add_todo error', e);
        showAlert('添加失败', true);
      }
    });

    const recPlanInterval = document.getElementById('rec-plan-interval');
    const recPlanDayWeekly = document.getElementById('rec-plan-day-weekly');
    recPlanInterval.addEventListener('change', (e) => {
      recPlanDayWeekly.classList.toggle('hidden', e.target.value !== 'weekly');
    });

    const recTodoInterval = document.getElementById('rec-todo-interval');
    const recTodoWeekly = document.getElementById('rec-todo-day-weekly');
    const recTodoMonthly = document.getElementById('rec-todo-day-monthly');
    recTodoInterval.addEventListener('change', (e) => {
      const val = e.target.value;
      recTodoWeekly.classList.toggle('hidden', val !== 'weekly');
      recTodoMonthly.classList.toggle('hidden', val !== 'monthly');
    });

    // ===== 循环健身：训练项目输入 + 提交逻辑 =====
    let exerciseCount = 1;

    document.getElementById('addExerciseBtn').addEventListener('click', () => {
      exerciseCount++;
      const container = document.getElementById('exercise-inputs');
      const newGroup = document.createElement('div');
      newGroup.classList.add('exercise-group');
      newGroup.innerHTML = `
        <input type="text" class="rec-ex-name" placeholder="训练 ${exerciseCount}: 名称">
        <input type="text" class="rec-ex-details" placeholder="训练 ${exerciseCount}: 组数/时长">
      `;
      container.appendChild(newGroup);
    });

    document.getElementById('addRecurringPlanForm').addEventListener('submit', async (e) => {
      e.preventDefault();
      const interval = recPlanInterval.value;
      let repeat_day = null;
      if (interval === 'weekly') {
        repeat_day = document.getElementById('rec-plan-day-w').value;
      }

      const names = document.querySelectorAll('.rec-ex-name');
      const details = document.querySelectorAll('.rec-ex-details');
      const exercises = [];
      for (let i = 0; i < names.length; i++) {
        const nameVal = names[i].value.trim();
        const detailVal = details[i].value.trim();
        if (nameVal || detailVal) {
          exercises.push({ name: nameVal, details: detailVal });
        }
      }
      if (exercises.length === 0) {
        alert('请至少添加一个训练项目（可以通过下方“训练项目库”添加）。');
        return;
      }

      const data = {
        plan_name: document.getElementById('rec-plan-name').value,
        start_date: document.getElementById('rec-plan-start').value,
        end_date: document.getElementById('rec-plan-end').value || null,
        repeat_interval: interval,
        repeat_day: repeat_day,
        exercises: exercises
      };

      await apiFetch('/add_recurring_plan', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(data)
      });

      e.target.reset();
      exerciseCount = 1;
      document.getElementById('exercise-inputs').innerHTML = `
        <div class="exercise-group">
          <input type="text" class="rec-ex-name" placeholder="训练 1: 名称">
          <input type="text" class="rec-ex-details" placeholder="训练 1: 组数/时长">
        </div>
      `;
      document.getElementById('rec-plan-start').value = todayStr;
      loadRecurringPlans();
    });

    // ===== 训练项目库 =====
    const exerciseLibrarySelect = document.getElementById('exercise-library-select');

    async function loadExerciseLibrary() {
      if (!exerciseLibrarySelect) return;
      try {
        const response = await apiFetch('/api/exercises');
        const exercises = await response.json();

        exerciseLibrarySelect.innerHTML = '<option value="">从训练项目库中选择...</option>';
        exercises.forEach(ex => {
          const opt = document.createElement('option');
          opt.value = ex.id;
          opt.textContent = ex.details ? `${ex.name} (${ex.details})` : ex.name;
          opt.dataset.name = ex.name;
          opt.dataset.details = ex.details || '';
          exerciseLibrarySelect.appendChild(opt);
        });

        const listDiv = document.getElementById('exerciseLibraryList');
        if (!exercises.length) {
          listDiv.innerHTML = '<div class="alert-error" style="margin-top:8px;">暂未添加任何训练项目。</div>';
        } else {
          listDiv.innerHTML = exercises.map(ex => `
            <div class="list-item">
              <span>${ex.name}${ex.details ? '（' + ex.details + '）' : ''}</span>
              <div class="item-actions">
                <button class="btn-danger btn-delete" data-type="exercise-lib" data-id="${ex.id}">删除</button>
              </div>
            </div>
          `).join('');
        }
      } catch (e) {
        console.error('loadExerciseLibrary error', e);
      }
    }

    const exerciseLibraryForm = document.getElementById('exerciseLibraryForm');
    if (exerciseLibraryForm) {
      exerciseLibraryForm.addEventListener('submit', async (e) => {
        e.preventDefault();
        const name = document.getElementById('lib-ex-name').value.trim();
        const details = document.getElementById('lib-ex-details').value.trim();
        if (!name) { alert('请输入项目名称'); return; }
        try {
          await apiFetch('/api/exercises', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ name, details })
          });
          exerciseLibraryForm.reset();
          loadExerciseLibrary();
        } catch (err) {
          console.error('add exercise error', err);
          alert('添加训练项目失败');
        }
      });
    }

    const addFromLibraryBtn = document.getElementById('addFromLibraryBtn');
    if (addFromLibraryBtn) {
      addFromLibraryBtn.addEventListener('click', () => {
        if (!exerciseLibrarySelect.value) {
          alert('请先在下拉框中选择一个训练项目');
          return;
        }
        const opt = exerciseLibrarySelect.options[exerciseLibrarySelect.selectedIndex];
        const name = opt.dataset.name || opt.textContent;
        const details = opt.dataset.details || '';

        exerciseCount++;
        const container = document.getElementById('exercise-inputs');
        const newGroup = document.createElement('div');
        newGroup.classList.add('exercise-group');
        newGroup.innerHTML = `
          <input type="text" class="rec-ex-name" placeholder="训练 ${exerciseCount}: 名称" value="${name}">
          <input type="text" class="rec-ex-details" placeholder="训练 ${exerciseCount}: 组数/时长" value="${details}">
        `;
        container.appendChild(newGroup);
      });
    }

    // ===== 常用待办库 =====
    const todoLibrarySelect = document.getElementById('todo-library-select');

    async function loadTodoLibrary() {
      if (!todoLibrarySelect) return;
      try {
        const response = await apiFetch('/api/todo_library');
        const items = await response.json();

        todoLibrarySelect.innerHTML = '<option value="">从常用待办库中选择...</option>';
        items.forEach(it => {
          const opt = document.createElement('option');
          opt.value = it.id;
          opt.textContent = it.content;
          opt.dataset.content = it.content;
          todoLibrarySelect.appendChild(opt);
        });

        const listDiv = document.getElementById('todoLibraryList');
        if (!items.length) {
          listDiv.innerHTML = '<div class="alert-error" style="margin-top:8px;">暂未添加任何常用待办。</div>';
        } else {
          listDiv.innerHTML = items.map(it => `
            <div class="list-item">
              <span>${it.content}</span>
              <div class="item-actions">
                <button class="btn-danger btn-delete" data-type="todo-lib" data-id="${it.id}">删除</button>
              </div>
            </div>
          `).join('');
        }
      } catch (e) {
        console.error('loadTodoLibrary error', e);
      }
    }

    const todoLibraryForm = document.getElementById('todoLibraryForm');
    if (todoLibraryForm) {
      todoLibraryForm.addEventListener('submit', async (e) => {
        e.preventDefault();
        const content = document.getElementById('lib-todo-content').value.trim();
        if (!content) { alert('请输入待办内容'); return; }
        try {
          await apiFetch('/api/todo_library', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ content })
          });
          todoLibraryForm.reset();
          loadTodoLibrary();
        } catch (err) {
          console.error('add todo lib error', err);
          alert('添加常用待办失败');
        }
      });
    }

    const fillTodoFromLibraryBtn = document.getElementById('fillTodoFromLibraryBtn');
    if (fillTodoFromLibraryBtn) {
      fillTodoFromLibraryBtn.addEventListener('click', () => {
        if (!todoLibrarySelect.value) {
          alert('请先在下拉框中选择一个常用待办');
          return;
        }
        const opt = todoLibrarySelect.options[todoLibrarySelect.selectedIndex];
        const content = opt.dataset.content || opt.textContent;
        document.getElementById('rec-todo-content').value = content;
      });
    }

    document.getElementById('addRecurringTodoForm').addEventListener('submit', async (e) => {
      e.preventDefault();
      const content = document.getElementById('rec-todo-content').value.trim();
      const start_date = document.getElementById('rec-todo-start').value;
      const end_date = document.getElementById('rec-todo-end').value || null;
      const interval = recTodoInterval.value;
      let repeat_day = null;
      if (interval === 'weekly') repeat_day = document.getElementById('rec-todo-day-w').value;
      else if (interval === 'monthly') repeat_day = document.getElementById('rec-todo-day-m').value;
      if (!content) { alert('请填写待办内容'); return; }
      const data = { content, start_date, end_date, repeat_interval: interval, repeat_day };
      try {
        await apiFetch('/add_recurring_todo', {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify(data)
        });
        e.target.reset();
        document.getElementById('rec-todo-start').value = todayStr;
        loadRecurringTodos();
      } catch (err) {
        console.error('add_recurring_todo error', err);
        alert('保存循环待办失败');
      }
    });

    // 用户管理 (JS)
    async function loadUsers() {
      try {
        const response = await apiFetch('/api/users');
        if (!response.ok) return;
        const users = await response.json();
        const userList = document.getElementById('userList');
        userList.innerHTML = users.map(u => `
          <div class="list-item">
            <span>${u.username} (${u.is_admin ? '管理员' : '普通用户'})</span>
            <div class="item-actions">
              <button class="btn-danger btn-delete" data-type="user" data-id="${u.id}">删除</button>
            </div>
          </div>
        `).join('');
      } catch (err) {
        console.error('loadUsers error', err);
      }
    }

    const createUserForm = document.getElementById('createUserForm');
    if (createUserForm) {
      createUserForm.addEventListener('submit', async (e) => {
        e.preventDefault();
        const username = document.getElementById('new-username').value.trim();
        const password = document.getElementById('new-password').value;
        
        const isAdminCheckbox = document.getElementById('is-admin');
        const is_admin = (isAdminCheckbox && isAdminCheckbox.checked) ? 1 : 0;
        
        if (!username || !password) { alert('请输入用户名和密码'); return; }
        try {
          const resp = await apiFetch('/api/users', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ username, password, is_admin })
          });
          const data = await resp.json().catch(()=>({}));
          if (resp.ok && data.status === 'success') {
            createUserForm.reset();
            loadUsers();
            showAlert('用户创建成功');
          } else {
            alert(data.message || '创建失败');
          }
        } catch (err) {
          console.error('create user error', err);
          alert('创建用户失败');
        }
      });
    }

    // ===== 统计与分析 (保持不变) =====
    let statsChartInstance = null;
    async function fetchStats(startDate, endDate) {
      try {
        const response = await apiFetch(`/api/stats/daily_completion?start_date=${startDate}&end_date=${endDate}`);
        const stats = await response.json();
        if (!response.ok) {
          document.getElementById('statsSummary').textContent = stats.message || '统计数据加载失败。';
          return null;
        }
        if (stats.status && stats.status !== 'success') {
          document.getElementById('statsSummary').textContent = stats.message || '统计数据加载失败。';
          return null;
        }
        return stats;
      } catch (e) {
        console.error('fetchStats error', e);
        document.getElementById('statsSummary').textContent = '统计数据加载失败。';
        return null;
      }
    }

    async function loadStats(startDate, endDate) {
      document.getElementById('statsSummary').textContent = '统计数据加载中...';
      const stats = await fetchStats(startDate, endDate);
      if (!stats) return;

      let labels = [];
      let todoCompleted = [];
      let todoTotal = [];
      let fitnessCompleted = [];
      let fitnessTotal = [];

      if (stats.fitness_data || stats.todo_data) {
        const fitnessData = stats.fitness_data || {};
        const todoData = stats.todo_data || {};
        const allDates = Array.from(new Set([
          ...Object.keys(fitnessData),
          ...Object.keys(todoData),
        ])).sort();

        labels = allDates;
        todoCompleted = labels.map(d => (todoData[d] && todoData[d].completed) || 0);
        todoTotal     = labels.map(d => (todoData[d] && todoData[d].total)     || 0);
        fitnessCompleted = labels.map(d => (fitnessData[d] && fitnessData[d].completed) || 0);
        fitnessTotal     = labels.map(d => (fitnessData[d] && fitnessData[d].total)     || 0);
      } else {
        document.getElementById('statsSummary').textContent = '统计数据格式不支持。';
        return;
      }

      renderStatsChart(labels, todoCompleted, todoTotal, fitnessCompleted, fitnessTotal);

      const totalTodoCompleted = todoCompleted.reduce((a,b)=>a+b,0);
      const totalTodo = todoTotal.reduce((a,b)=>a+b,0);
      const totalFitCompleted = fitnessCompleted.reduce((a,b)=>a+b,0);
      const totalFit = fitnessTotal.reduce((a,b)=>a+b,0);
      const todoRate = totalTodo ? Math.round(totalTodoCompleted/totalTodo*100) : 0;
      const fitRate = totalFit ? Math.round(totalFitCompleted/totalFit*100) : 0;

      document.getElementById('statsSummary').innerHTML = `
        时间范围内，总待办完成率约为 <span class="stats-highlight">${todoRate}%</span>，
        健身计划完成率约为 <span class="stats-highlight">${fitRate}%</span>。<br>
        已完成待办 <strong>${totalTodoCompleted}</strong> 条 / 共 ${totalTodo} 条；
        已完成健身计划 <strong>${totalFitCompleted}</strong> 条 / 共 ${totalFit} 条。
      `;
    }

    function renderStatsChart(labels, todoCompleted, todoTotal, fitnessCompleted, fitnessTotal) {
      const ctx = document.getElementById('statsChart').getContext('2d');
      if (statsChartInstance) statsChartInstance.destroy();
      statsChartInstance = new Chart(ctx, {
        type: 'line',
        data: {
          labels,
          datasets: [
            {
              label: '待办完成数',
              data: todoCompleted,
              borderColor: '#22c55e',
              backgroundColor: 'rgba(34, 197, 94, 0.15)',
              tension: 0.3,
              fill: true,
            },
            {
              label: '健身完成数',
              data: fitnessCompleted,
              borderColor: '#3b82f6',
              backgroundColor: 'rgba(59, 130, 246, 0.15)',
              tension: 0.3,
              fill: true,
            },
          ]
        },
        options: {
          responsive: true,
          plugins: { legend: { position: 'top' } },
          scales: {
            y: { beginAtZero: true, ticks: { stepSize: 1 } }
          }
        }
      });
    }
    function getDateNDaysAgo(n) {
      const d = new Date();
      d.setDate(d.getDate() - n);
      const y = d.getFullYear();
      const m = String(d.getMonth() + 1).padStart(2, '0');
      const da = String(d.getDate()).padStart(2, '0');
      return `${y}-${m}-${da}`;
    }
    document.getElementById('stats-btn-week').addEventListener('click', () => {
      const end = todayStr;
      const start = getDateNDaysAgo(6);
      document.getElementById('stats-start').value = start;
      document.getElementById('stats-end').value = end;
      loadStats(start, end);
    });
    document.getElementById('stats-btn-month').addEventListener('click', () => {
      const end = todayStr;
      const start = getDateNDaysAgo(29);
      document.getElementById('stats-start').value = start;
      document.getElementById('stats-end').value = end;
      loadStats(start, end);
    });
    document.getElementById('stats-btn-range').addEventListener('click', () => {
      const start = document.getElementById('stats-start').value;
      const end = document.getElementById('stats-end').value;
      if (!start || !end) { alert('请先选择起止日期'); return; }
      loadStats(start, end);
    });
    document.getElementById('stats-refresh').addEventListener('click', () => {
      const start = document.getElementById('stats-start').value;
      const end = document.getElementById('stats-end').value;
      if (!start || !end) { alert('请先选择起止日期'); return; }
      loadStats(start, end);
    });

    // ===== 每日概览 (保持不变) =====
    async function loadTodaySummary() {
      try {
        const [todoResp, fitResp] = await Promise.all([
          apiFetch(`/todos?date=${todayStr}`),
          apiFetch(`/fitness_plan/${todayStr}`)
        ]);
        const todos = await todoResp.json();
        const plans = await fitResp.json();
        const todoTotal = todos.length;
        const todoCompleted = todos.filter(t => t.is_completed).length;
        const fitTotal = plans.length;
        const fitCompleted = plans.filter(p => p.is_completed).length;
        document.getElementById('todaySummaryContent').innerHTML = `
          今日待办：完成 <strong>${todoCompleted}</strong> / ${todoTotal} 条<br>
          今日健身：完成 <strong>${fitCompleted}</strong> / ${fitTotal} 项<br>
          <span class="small-hint">你可以在“每日待办 (T)”和“每日健身计划 (T)”中手动调整今日任务。</span>
        `;
      } catch (e) {
        console.error('today summary error', e);
        document.getElementById('todaySummaryContent').textContent = '加载失败';
      }
    }

    // ===== 大屏滚动速度设置 =====
    const scrollSpeedInput = document.getElementById('scroll-speed-input');
    const scrollSpeedStatus = document.getElementById('scroll-speed-status');
    const scrollSpeedForm = document.getElementById('scrollSpeedForm');

    async function loadScrollSpeedSetting() {
      if (!scrollSpeedInput) return;
      try {
        const resp = await apiFetch('/api/settings/scroll_speed');
        if (!resp.ok) throw new Error('load scroll speed failed');
        const data = await resp.json();
        if (typeof data.scroll_speed === 'number') {
          scrollSpeedInput.value = data.scroll_speed;
        }
        if (scrollSpeedStatus) {
          scrollSpeedStatus.textContent = `当前速度：${scrollSpeedInput.value} 秒/一轮`;
        }
      } catch (e) {
        console.error('loadScrollSpeedSetting error', e);
        if (scrollSpeedStatus) {
          scrollSpeedStatus.textContent = '加载失败，使用默认 25 秒';
        }
      }
    }

    if (scrollSpeedForm) {
      scrollSpeedForm.addEventListener('submit', async (e) => {
        e.preventDefault();
        let value = parseInt(scrollSpeedInput.value, 10);
        if (isNaN(value)) {
          alert('请填写一个有效的整数秒数');
          return;
        }
        if (value < 5) value = 5;
        if (value > 120) value = 120;
        scrollSpeedInput.value = value;
        try {
          const resp = await apiFetch('/api/settings/scroll_speed', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ scroll_speed: value })
          });
          const data = await resp.json();
          if (!resp.ok || data.status !== 'success') {
            throw new Error(data.message || '保存失败');
          }
          if (scrollSpeedStatus) {
            scrollSpeedStatus.textContent = `已保存：${value} 秒/一轮。请刷新大屏页面生效。`;
          }
        } catch (err) {
          console.error('save scroll speed error', err);
          alert('保存滚动速度失败，请稍后重试。');
        }
      });
    }

    // ===== 删除按钮 =====
    document.body.addEventListener('click', async (e) => {
      if (!e.target.classList.contains('btn-delete')) return;
      if (!confirm('确定要删除吗？')) return;
      const type = e.target.dataset.type;
      const id = e.target.dataset.id;
      try {
        if (type === 'todo') {
          await apiFetch(`/delete_todo/${id}`, { method: 'POST' });
          loadTodos(datePickerTodo.value);
        } else if (type === 'daily-fitness') {
          await apiFetch(`/delete_fitness_plan_exercise/${id}`, { method: 'POST' });
          loadDailyFitness(datePickerDaily.value);
        } else if (type === 'recurring-todo') {
          await apiFetch(`/delete_recurring_todo/${id}`, { method: 'POST' });
          loadRecurringTodos();
        } else if (type === 'recurring-fitness') {
          await apiFetch(`/delete_recurring_plan/${id}`, { method: 'POST' });
          loadRecurringPlans();
        } else if (type === 'exercise-lib') {
          await apiFetch(`/api/exercises/${id}`, { method: 'DELETE' });
          loadExerciseLibrary();
        } else if (type === 'todo-lib') {
          await apiFetch(`/api/todo_library/${id}`, { method: 'DELETE' });
          loadTodoLibrary();
        } else if (type === 'user') {
          await apiFetch(`/api/users/${id}`, { method: 'DELETE' });
          showAlert('用户及其数据已删除');
          loadUsers();
        }
      } catch (error) {
        console.error('Delete error:', error);
      }
    });

    const profileUpdateForm = document.getElementById('profileUpdateForm');
    if (profileUpdateForm) {
        profileUpdateForm.addEventListener('submit', async (e) => {
            e.preventDefault();
            const statusEl = document.getElementById('profileUpdateStatus');
            statusEl.textContent = '保存中...';
            
            const newUsername = document.getElementById('profile-username').value.trim();
            const newPassword = document.getElementById('profile-password').value.trim();
            
            const payload = {};
            if (newUsername) {
                payload.username = newUsername;
            }
            if (newPassword) {
                payload.password = newPassword;
            }

            if (Object.keys(payload).length === 0) {
                statusEl.textContent = '未输入任何更改。';
                return;
            }

            try {
                const resp = await apiFetch('/api/profile', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify(payload)
                });
                const data = await resp.json().catch(() => ({}));

                if (resp.ok && data.status === 'success') {
                    showAlert('个人资料更新成功！用户名更改将在下次登录后完全生效。');
                    statusEl.textContent = '保存成功！';
                    
                    if (newUsername) {
                        setTimeout(() => {
                            showAlert('用户名已更改，将刷新页面...', false);
                            window.location.reload();
                        }, 2000);
                    } else {
                        profileUpdateForm.reset();
                    }
                } else {
                    showAlert(data.message || '更新失败', true);
                    statusEl.textContent = `失败: ${data.message}`;
                }
            } catch (err) {
                console.error('Update profile error:', err);
                showAlert('更新失败，网络或服务器错误。', true);
                statusEl.textContent = '网络错误。';
            }
        });
    }

    // ★★★ 核心修复：页面加载完成后执行初始化 ★★★
    document.addEventListener('DOMContentLoaded', () => {
        // 1. 填充日期选择器
        if(document.getElementById('date-todo')) document.getElementById('date-todo').value = todayStr;
        if(document.getElementById('date-daily')) document.getElementById('date-daily').value = todayStr;
        if(document.getElementById('rec-plan-start')) document.getElementById('rec-plan-start').value = todayStr;
        if(document.getElementById('rec-todo-start')) document.getElementById('rec-todo-start').value = todayStr;
        if(document.getElementById('stats-start')) document.getElementById('stats-start').value = todayStr;
        if(document.getElementById('stats-end')) document.getElementById('stats-end').value = todayStr;

        // 2. 强制触发一次 "change" 事件，修复循环方式下拉框 UI 状态 (解决“选了每日却显示星期几”的Bug)
        const event = new Event('change');
        const recPlanInterval = document.getElementById('rec-plan-interval');
        if (recPlanInterval) {
            recPlanInterval.dispatchEvent(event);
        }
        const recTodoInterval = document.getElementById('rec-todo-interval');
        if (recTodoInterval) {
            recTodoInterval.dispatchEvent(event);
        }

        // 3. 初始加载数据
        loadTodos(todayStr);
        loadDailyFitness(todayStr);
        loadRecurringPlans();
        loadRecurringTodos();
        loadExerciseLibrary();
        loadTodoLibrary();
        loadTodaySummary();
    });
  </script>
</body>
</html>
'''

# ==========================================
# 3. 主程序逻辑：写入文件
# ==========================================
def write_file(filename, content, folder="."):
    filepath = os.path.join(folder, filename)
    try:
        with open(filepath, 'w', encoding='utf-8') as f:
            f.write(content)
        print(f"[成功] 已写入: {filepath}")
    except Exception as e:
        print(f"[错误] 写入 {filepath} 失败: {e}")

def main():
    print("=== 正在更新项目文件 (v1.7 修复版) ===")
    
    # 1. 写入 app.py (覆盖当前目录的 app.py)
    write_file('app.py', app_py_content)
    
    # 2. 写入 manage.html (需要先确认路径结构)
    # 尝试查找 templates 目录，如果是标准结构
    target_template_dir = "."
    
    # 简单逻辑：如果当前目录下有 templates 文件夹，就写进去；否则写在当前目录
    if os.path.exists("templates"):
        target_template_dir = "templates"
    elif os.path.exists("frontend/templates"):
         target_template_dir = "frontend/templates"
    
    write_file('manage.html', manage_html_content, target_template_dir)
    
    print("\n=== 更新完成 ===")
    print("请重新启动 Flask 应用以使更改生效。")
    print("python app.py")

if __name__ == '__main__':
    main()
