import os

# ==========================================
# 1. backend/app.py (v1.8 - Reminders & Migration)
# ==========================================
app_py_content = r'''# backend/app.py (v1.8 - Reminders)
import sqlite3
from datetime import datetime, timedelta, timezone
from flask import Flask, render_template, request, jsonify, g, redirect, url_for, make_response
from flask_login import LoginManager, UserMixin, login_user, logout_user, login_required, current_user
import os, json, uuid
from flask_cors import CORS
from functools import wraps
import sys 
from collections import defaultdict

# --- 时区设置 (强制使用 UTC+8 中国标准时间) ---
CN_TZ = timezone(timedelta(hours=8))

# --- Nuitka 路径逻辑 ---
is_frozen = getattr(sys, 'frozen', False)
is_in_onefile_temp = 'onefile_' in os.path.abspath(__file__) 

if is_frozen or is_in_onefile_temp:
    temp_base_dir = os.path.dirname(os.path.abspath(sys.executable))
    template_dir = os.path.join(temp_base_dir, 'templates')
    static_dir = os.path.join(temp_base_dir, 'static')
    exe_path = os.path.realpath(sys.argv[0])
    exe_dir = os.path.dirname(exe_path)
    DATABASE = os.path.join(exe_dir, 'fitness_manager.db')
else:
    app_file_path = os.path.abspath(__file__)
    backend_dir = os.path.dirname(app_file_path)
    project_root_dir = os.path.dirname(backend_dir)
    template_dir = os.path.join(project_root_dir, 'frontend', 'templates')
    static_dir = os.path.join(project_root_dir, 'frontend', 'static')
    DATABASE = os.path.join(backend_dir, 'fitness_manager.db')

print("DB path:", DATABASE)

app = Flask(__name__, template_folder=template_dir, static_folder=static_dir)
CORS(app)
app.secret_key = str(uuid.uuid4())

# --- Flask-Login ---
login_manager = LoginManager()
login_manager.init_app(app)
login_manager.login_view = 'index' 

class User(UserMixin):
    def __init__(self, id, username, is_admin, manager_id):
        self.id = id
        self.username = username
        self.is_admin = is_admin
        self.manager_id = manager_id 
    
    @property
    def is_super_admin(self):
        return self.is_admin and (self.manager_id is None or self.manager_id == 0)

@login_manager.user_loader
def load_user(user_id):
    db = get_db()
    user_row = db.execute("SELECT id, username, is_admin, manager_id FROM users WHERE id = ?", (user_id,)).fetchone()
    if user_row:
        return User(id=user_row['id'], username=user_row['username'], is_admin=user_row['is_admin'], manager_id=user_row['manager_id'])
    return None

# -------------------- DB --------------------
def get_db():
    db = getattr(g, '_database', None)
    if db is None:
        db = g._database = sqlite3.connect(DATABASE, check_same_thread=False)
        db.row_factory = sqlite3.Row
    return db

def get_setting(key, default=None):
    db = get_db()
    row = db.execute("SELECT value FROM settings WHERE key = ?", (key,)).fetchone()
    return default if row is None else row["value"]

def set_setting(key, value):
    db = get_db()
    db.execute("INSERT INTO settings (key, value) VALUES (?, ?) ON CONFLICT(key) DO UPDATE SET value = excluded.value", (key, str(value)))
    db.commit()

@app.teardown_appcontext
def close_connection(exception):
    db = getattr(g, '_database', None)
    if db is not None: db.close()

def init_db():
    db = get_db()
    cur = db.cursor()
    # todos 表增加 reminder_at (YYYY-MM-DD HH:MM)
    cur.execute("""
        CREATE TABLE IF NOT EXISTS todos (
            id INTEGER PRIMARY KEY AUTOINCREMENT, 
            date TEXT NOT NULL, 
            content TEXT NOT NULL,
            is_completed INTEGER DEFAULT 0, 
            user_id INTEGER, 
            recurring_todo_id INTEGER,
            reminder_at TEXT
        )
    """)
    cur.execute("""
        CREATE TABLE IF NOT EXISTS fitness_plans (
            id INTEGER PRIMARY KEY AUTOINCREMENT, 
            date TEXT NOT NULL, 
            exercise_name TEXT NOT NULL,
            sets_reps_duration TEXT, 
            is_completed INTEGER DEFAULT 0, 
            user_id INTEGER, 
            recurring_plan_id INTEGER
        )
    """)
    # recurring_todos 表增加 reminder_time (HH:MM)
    cur.execute("""
        CREATE TABLE IF NOT EXISTS recurring_plans (
            id INTEGER PRIMARY KEY AUTOINCREMENT, 
            user_id INTEGER, 
            plan_name TEXT NOT NULL,
            exercises_json TEXT NOT NULL, 
            start_date TEXT NOT NULL, 
            end_date TEXT,
            repeat_interval TEXT NOT NULL, 
            repeat_day INTEGER
        )
    """)
    cur.execute("""
        CREATE TABLE IF NOT EXISTS recurring_todos (
            id INTEGER PRIMARY KEY AUTOINCREMENT, 
            user_id INTEGER, 
            content TEXT NOT NULL,
            start_date TEXT NOT NULL, 
            end_date TEXT, 
            repeat_interval TEXT NOT NULL, 
            repeat_day INTEGER,
            reminder_time TEXT
        )
    """)
    cur.execute("CREATE TABLE IF NOT EXISTS exercise_library (id INTEGER PRIMARY KEY AUTOINCREMENT, user_id INTEGER, name TEXT NOT NULL, details TEXT)")
    cur.execute("CREATE TABLE IF NOT EXISTS settings (key TEXT PRIMARY KEY, value TEXT)")
    cur.execute("CREATE TABLE IF NOT EXISTS todo_library (id INTEGER PRIMARY KEY AUTOINCREMENT, user_id INTEGER, content TEXT NOT NULL)")
    cur.execute("CREATE TABLE IF NOT EXISTS users (id INTEGER PRIMARY KEY AUTOINCREMENT, username TEXT UNIQUE NOT NULL, password TEXT NOT NULL, is_admin INTEGER DEFAULT 0, manager_id INTEGER REFERENCES users(id))")

    # --- Migrations ---
    try: cur.execute("SELECT reminder_at FROM todos LIMIT 1")
    except sqlite3.OperationalError:
        print("Migrating todos: adding reminder_at")
        cur.execute("ALTER TABLE todos ADD COLUMN reminder_at TEXT")
    
    try: cur.execute("SELECT reminder_time FROM recurring_todos LIMIT 1")
    except sqlite3.OperationalError:
        print("Migrating recurring_todos: adding reminder_time")
        cur.execute("ALTER TABLE recurring_todos ADD COLUMN reminder_time TEXT")

    # Check superuser
    cur.execute("SELECT COUNT(*) FROM users WHERE username = 'superuser'")
    if cur.fetchone()[0] == 0:
        cur.execute("INSERT INTO users (username, password, is_admin, manager_id) VALUES (?, ?, ?, ?)", ('superuser', 'adminpassword', 1, None))
    
    db.commit()

with app.app_context():
    init_db()

# -------------------- Auth Routes --------------------
@app.route('/login', methods=['POST'])
def login():
    username = request.form.get('username') or request.json.get('username')
    password = request.form.get('password') or request.json.get('password')
    db = get_db()
    user_row = db.execute("SELECT id, username, password, is_admin, manager_id FROM users WHERE username = ?", (username,)).fetchone()
    if user_row and user_row['password'] == password: 
        user_obj = User(id=user_row['id'], username=user_row['username'], is_admin=user_row['is_admin'], manager_id=user_row['manager_id'])
        login_user(user_obj, remember=True) 
        return jsonify({'status': 'success', 'username': user_obj.username})
    return jsonify({'status': 'error', 'message': '用户名或密码无效'}), 401

@app.route('/logout', methods=['POST'])
@login_required
def logout():
    logout_user() 
    return jsonify({'status': 'success'})

# -------------------- UI Routes --------------------
@app.route('/')
def index():
    if not current_user.is_authenticated: return render_template('login.html') 
    if 'Mobile' in request.headers.get('User-Agent', '') and request.args.get('view') != 'display':
        return redirect(url_for('mobile_config'))
    
    today_str = datetime.now(CN_TZ).strftime('%Y-%m-%d')
    _generate_daily_fitness_plan_if_not_exists(current_user.id, today_str)
    _generate_daily_todos_if_not_exists(current_user.id, today_str)
    return render_template('display.html', username=current_user.username) 

@app.route('/mobile_config')
@login_required
def mobile_config(): return render_template('mobile_config.html', username=current_user.username)

@app.route('/manage')
@login_required
def manage(): return render_template('manage.html', username=current_user.username, is_admin=current_user.is_admin, manager_id=current_user.manager_id)

# -------------------- API Routes --------------------

# 1. Todos
@app.route('/todos', methods=['GET'])
@login_required
def get_todos():
    date_str = request.args.get('date') or datetime.now(CN_TZ).strftime('%Y-%m-%d')
    db = get_db()
    # 按 reminder_at 排序，有时间的排前面 (NULL 放最后或最前看需求，这里默认混排)
    todos = db.execute("SELECT * FROM todos WHERE date = ? AND user_id = ? ORDER BY is_completed ASC, reminder_at ASC, id DESC", (date_str, current_user.id)).fetchall()
    return jsonify([dict(t) for t in todos])

@app.route('/add_todo', methods=['POST'])
@login_required
def add_todo():
    data = request.get_json()
    date_str = data.get('date')
    content = data.get('content')
    reminder_at = data.get('reminder_at') # Format: "2023-11-22 10:00" or just date
    if not date_str or not content: return jsonify(status='error', message='缺少日期或内容'), 400
    
    # 如果 reminder_at 只是时间 "10:00"，需要拼上日期
    if reminder_at and len(reminder_at) == 5: 
        reminder_at = f"{date_str} {reminder_at}"

    db = get_db()
    db.execute("INSERT INTO todos (date, content, is_completed, user_id, reminder_at) VALUES (?, ?, 0, ?, ?)", (date_str, content, current_user.id, reminder_at))
    db.commit()
    return jsonify(status='success')

@app.route('/mark_todo_completed', methods=['POST'])
@login_required
def mark_todo_completed():
    data = request.get_json()
    db = get_db()
    db.execute("UPDATE todos SET is_completed = ? WHERE id = ? AND user_id = ?", (data.get('is_completed'), data.get('todo_id'), current_user.id))
    db.commit()
    return jsonify(status='success')

@app.route('/delete_todo/<int:todo_id>', methods=['POST'])
@login_required
def delete_todo(todo_id):
    get_db().execute("DELETE FROM todos WHERE id = ? AND user_id = ?", (todo_id, current_user.id)).connection.commit()
    return jsonify(status='success')

# 2. Recurring Todos
@app.route('/get_recurring_todos', methods=['GET'])
@login_required
def get_recurring_todos():
    todos = get_db().execute("SELECT * FROM recurring_todos WHERE user_id = ? ORDER BY start_date DESC", (current_user.id,)).fetchall()
    return jsonify([dict(t) for t in todos])

@app.route('/add_recurring_todo', methods=['POST'])
@login_required
def add_recurring_todo():
    data = request.get_json()
    reminder_time = data.get('reminder_time') # Format "HH:MM" or empty
    if not data.get('content'): return jsonify(status='error'), 400
    
    db = get_db()
    db.execute("""INSERT INTO recurring_todos (user_id, content, start_date, end_date, repeat_interval, repeat_day, reminder_time) 
                  VALUES (?, ?, ?, ?, ?, ?, ?)""",
               (current_user.id, data.get('content'), data.get('start_date'), data.get('end_date'), 
                data.get('repeat_interval'), data.get('repeat_day'), reminder_time))
    db.commit()
    return jsonify(status='success')

@app.route('/delete_recurring_todo/<int:todo_id>', methods=['POST'])
@login_required
def delete_recurring_todo(todo_id):
    get_db().execute("DELETE FROM recurring_todos WHERE id = ? AND user_id = ?", (todo_id, current_user.id)).connection.commit()
    return jsonify(status='success')

# 3. Fitness (Standard)
@app.route('/fitness_plan/<date_str>', methods=['GET'])
@login_required
def get_fitness_plan(date_str):
    _generate_daily_fitness_plan_if_not_exists(current_user.id, date_str)
    plans = get_db().execute("SELECT * FROM fitness_plans WHERE date = ? AND user_id = ? ORDER BY id", (date_str, current_user.id)).fetchall()
    return jsonify([dict(p) for p in plans])

@app.route('/mark_exercise_completed_mobile', methods=['POST'])
@login_required
def mark_exercise_completed_mobile():
    data = request.get_json()
    get_db().execute("UPDATE fitness_plans SET is_completed = ? WHERE id = ? AND user_id = ?", (data.get('is_completed'), data.get('plan_id'), current_user.id)).connection.commit()
    return jsonify(status='success')

@app.route('/delete_fitness_plan_exercise/<int:plan_id>', methods=['POST'])
@login_required
def delete_fitness_plan_exercise(plan_id):
    get_db().execute("DELETE FROM fitness_plans WHERE id = ? AND user_id = ?", (plan_id, current_user.id)).connection.commit()
    return jsonify(status='success')

# 4. Recurring Fitness
@app.route('/get_recurring_plans', methods=['GET'])
@login_required
def get_recurring_plans():
    plans = get_db().execute("SELECT * FROM recurring_plans WHERE user_id = ? ORDER BY start_date DESC", (current_user.id,)).fetchall()
    return jsonify([dict(p) for p in plans])

@app.route('/add_recurring_plan', methods=['POST'])
@login_required
def add_recurring_plan():
    data = request.get_json()
    get_db().execute("""INSERT INTO recurring_plans (user_id, plan_name, exercises_json, start_date, end_date, repeat_interval, repeat_day) 
                        VALUES (?, ?, ?, ?, ?, ?, ?)""",
                     (current_user.id, data.get('plan_name'), json.dumps(data.get('exercises')), data.get('start_date'), data.get('end_date'), data.get('repeat_interval'), data.get('repeat_day'))).connection.commit()
    return jsonify(status='success')

@app.route('/delete_recurring_plan/<int:plan_id>', methods=['POST'])
@login_required
def delete_recurring_plan(plan_id):
    get_db().execute("DELETE FROM recurring_plans WHERE id = ? AND user_id = ?", (plan_id, current_user.id)).connection.commit()
    return jsonify(status='success')

# 5. Libraries & Settings & Stats (Simplified for brevity as they didn't change logic)
@app.route('/api/todo_library', methods=['GET', 'POST', 'DELETE'])
@app.route('/api/todo_library/<int:item_id>', methods=['POST', 'DELETE'])
@login_required
def todo_library_api(item_id=None):
    db = get_db()
    if request.method == 'GET':
        return jsonify([dict(r) for r in db.execute("SELECT * FROM todo_library WHERE user_id=?",(current_user.id,))])
    if request.method == 'POST' and not item_id: # Add
        db.execute("INSERT INTO todo_library (user_id, content) VALUES (?, ?)", (current_user.id, request.get_json().get('content')))
        db.commit()
        return jsonify(status='success')
    if (request.method == 'DELETE' or request.method == 'POST') and item_id: # Delete
        db.execute("DELETE FROM todo_library WHERE id=? AND user_id=?", (item_id, current_user.id))
        db.commit()
        return jsonify(status='success')
    return jsonify(status='error')

@app.route('/api/exercises', methods=['GET', 'POST'])
@app.route('/api/exercises/<int:ex_id>', methods=['POST', 'DELETE'])
@login_required
def exercise_lib_api(ex_id=None):
    db = get_db()
    if request.method == 'GET': return jsonify([dict(r) for r in db.execute("SELECT * FROM exercise_library WHERE user_id=?",(current_user.id,))])
    if request.method == 'POST' and not ex_id:
        d=request.json
        db.execute("INSERT INTO exercise_library (user_id, name, details) VALUES (?,?,?)", (current_user.id, d.get('name'), d.get('details')))
        db.commit()
        return jsonify(status='success')
    if ex_id:
        db.execute("DELETE FROM exercise_library WHERE id=? AND user_id=?", (ex_id, current_user.id))
        db.commit()
        return jsonify(status='success')
    return jsonify(status='error')

@app.route('/api/settings/scroll_speed', methods=['GET', 'POST'])
def scroll_speed():
    if request.method == 'POST':
        set_setting('scroll_speed', request.json.get('scroll_speed', 25))
        return jsonify(status='success')
    return jsonify(scroll_speed=int(get_setting('scroll_speed', 25) or 25))

@app.route('/api/users', methods=['GET', 'POST'])
@app.route('/api/users/<int:uid>', methods=['DELETE'])
@login_required
def users_api(uid=None):
    if not current_user.is_admin: return jsonify(status='error'), 403
    db = get_db()
    if request.method == 'GET':
        return jsonify([dict(u) for u in db.execute("SELECT id,username,is_admin FROM users WHERE manager_id=?", (current_user.id,))])
    if request.method == 'POST':
        d=request.json
        is_adm = 1 if (current_user.is_super_admin and d.get('is_admin')) else 0
        try:
            db.execute("INSERT INTO users (username,password,is_admin,manager_id) VALUES (?,?,?,?)", (d.get('username'), d.get('password'), is_adm, current_user.id))
            db.commit()
            return jsonify(status='success')
        except: return jsonify(status='error', message='exists')
    if uid:
        if uid == current_user.id: return jsonify(status='error'), 400
        db.execute("DELETE FROM users WHERE id=?", (uid,)) # Cascade handled by manual delete in full code, simplified here for brevity but ensuring deletes works
        for t in ['todos','fitness_plans','recurring_plans','recurring_todos','exercise_library','todo_library']:
            db.execute(f"DELETE FROM {t} WHERE user_id=?", (uid,))
        db.commit()
        return jsonify(status='success')

@app.route('/api/stats/daily_completion')
@login_required
def stats():
    sd, ed = request.args.get('start_date'), request.args.get('end_date')
    db = get_db()
    # Logic same as before
    f_rows = db.execute("SELECT date, is_completed FROM fitness_plans WHERE user_id=? AND date BETWEEN ? AND ?", (current_user.id, sd, ed))
    t_rows = db.execute("SELECT date, is_completed FROM todos WHERE user_id=? AND date BETWEEN ? AND ?", (current_user.id, sd, ed))
    fd = defaultdict(lambda: {"total":0,"completed":0})
    td = defaultdict(lambda: {"total":0,"completed":0})
    for r in f_rows: 
        fd[r['date']]['total']+=1
        if r['is_completed']: fd[r['date']]['completed']+=1
    for r in t_rows:
        td[r['date']]['total']+=1
        if r['is_completed']: td[r['date']]['completed']+=1
    return jsonify(fitness_data=fd, todo_data=td)

@app.route('/api/profile', methods=['POST'])
@login_required
def profile():
    d = request.json
    db = get_db()
    if d.get('password'): db.execute("UPDATE users SET password=? WHERE id=?", (d.get('password'), current_user.id))
    if d.get('username'): 
        try: db.execute("UPDATE users SET username=? WHERE id=?", (d.get('username'), current_user.id))
        except: return jsonify(status='error', message='exists')
    db.commit()
    return jsonify(status='success')


# -------------------- Generation Logic (Updated) --------------------
def _generate_daily_fitness_plan_if_not_exists(user_id, date_str):
    # Fitness plans generation logic (No reminder logic requested for fitness, keeping standard)
    db = get_db()
    if db.execute("SELECT COUNT(*) FROM fitness_plans WHERE date=? AND user_id=?", (date_str, user_id)).fetchone()[0] > 0: return
    today_date = datetime.strptime(date_str, '%Y-%m-%d').date()
    wd = today_date.isoweekday()
    plans = db.execute("SELECT * FROM recurring_plans WHERE user_id=? AND start_date<=? AND (end_date IS NULL OR end_date>=?)", (user_id, date_str, date_str)).fetchall()
    for p in plans:
        add = False
        if p['repeat_interval'] == 'daily': add = True
        elif p['repeat_interval'] == 'weekly' and p['repeat_day'] == wd: add = True
        if add:
            exs = json.loads(p['exercises_json'])
            for ex in exs:
                db.execute("INSERT INTO fitness_plans (date, exercise_name, sets_reps_duration, user_id, recurring_plan_id) VALUES (?,?,?,?,?)",
                           (date_str, ex['name'], ex['details'], user_id, p['id']))
    db.commit()

def _generate_daily_todos_if_not_exists(user_id, date_str):
    db = get_db()
    if db.execute("SELECT COUNT(*) FROM todos WHERE date=? AND user_id=?", (date_str, user_id)).fetchone()[0] > 0: return
    
    today_date = datetime.strptime(date_str, '%Y-%m-%d').date()
    wd = today_date.isoweekday()
    dom = today_date.day
    recs = db.execute("SELECT * FROM recurring_todos WHERE user_id=? AND start_date<=? AND (end_date IS NULL OR end_date>=?)", (user_id, date_str, date_str)).fetchall()
    
    for r in recs:
        add = False
        if r['repeat_interval'] == 'daily': add = True
        elif r['repeat_interval'] == 'weekly' and r['repeat_day'] == wd: add = True
        elif r['repeat_interval'] == 'monthly' and r['repeat_day'] == dom: add = True
        
        if add:
            # ★ 生成提醒时间：如果循环待办有 reminder_time (如 07:00)，则生成具体的 reminder_at (2025-11-22 07:00)
            reminder_at = None
            if r['reminder_time']:
                reminder_at = f"{date_str} {r['reminder_time']}"
                
            db.execute("INSERT INTO todos (date, content, is_completed, user_id, recurring_todo_id, reminder_at) VALUES (?,?,?,?,?,?)",
                       (date_str, r['content'], 0, user_id, r['id'], reminder_at))
    db.commit()

@app.route('/health', methods=['GET'])
def health(): return jsonify(status='ok', time=datetime.now(CN_TZ).isoformat())

if __name__ == '__main__':
    app.run(host='0.0.0.0', port=8000, debug=True)
'''

# ==========================================
# 2. frontend/templates/manage.html (v1.8 - UI for Reminders)
# ==========================================
manage_html_content = r'''<!DOCTYPE html>
<html lang="zh-CN">
<head>
  <meta charset="utf-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1.0" />
  <title>健身看板 - 后台管理</title>
  <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
  <link href="https://fonts.googleapis.com/css2?family=Noto+Sans+SC:wght@300;400;500;700&display=swap" rel="stylesheet">
  <style>
    :root {
        --bg-color: #f1f5f9; --panel-color: #ffffff; --accent-color: #0f766e;
        --text-primary: #0f172a; --text-secondary: #64748b; --border-soft: #e2e8f0;
    }
    * { box-sizing: border-box; }
    body { margin: 0; font-family: 'Noto Sans SC', sans-serif; background: var(--bg-color); color: var(--text-primary); }
    .app-container { display: flex; min-height: 100vh; }
    .sidebar { width: 260px; background: #0f172a; color: #e2e8f0; padding: 20px; flex-shrink: 0; }
    .main { flex: 1; padding: 24px; overflow-y: auto; }
    .panel { background: var(--panel-color); border-radius: 12px; padding: 20px; margin-bottom: 20px; box-shadow: 0 4px 6px rgba(0,0,0,0.05); }
    h2 { margin-top: 0; color: var(--accent-color); border-bottom: 1px solid var(--border-soft); padding-bottom: 10px; }
    .form-group { margin-bottom: 12px; }
    .form-group label { display: block; font-size: 13px; color: var(--text-secondary); margin-bottom: 4px; }
    input, select, textarea { width: 100%; padding: 10px; border: 1px solid #cbd5e1; border-radius: 8px; background: #f8fafc; }
    .btn-submit { background: var(--accent-color); color: white; border: none; padding: 10px 20px; border-radius: 8px; cursor: pointer; }
    .btn-danger { background: #ef4444; color: white; border: none; padding: 5px 10px; border-radius: 4px; cursor: pointer; font-size: 12px; }
    .list-item { display: flex; justify-content: space-between; align-items: center; padding: 10px; background: #f9fafb; margin-bottom: 8px; border-radius: 8px; border: 1px solid var(--border-soft); }
    .hidden { display: none; }
    .nav-button { display: block; width: 100%; text-align: left; padding: 12px; background: none; border: none; color: #94a3b8; cursor: pointer; }
    .nav-button.active { color: #ffffff; background: rgba(255,255,255,0.1); border-radius: 8px; }
    .content-section { display: none; }
    .content-section.active { display: block; }
    .tag-time { background: #e0f2fe; color: #0369a1; padding: 2px 6px; border-radius: 4px; font-size: 12px; margin-left: 6px; }
  </style>
</head>
<body>
<div class="app-container">
  <aside class="sidebar">
    <h3>管理后台</h3>
    <nav>
      <button class="nav-button active" data-tab="daily-todos">每日待办 (Today)</button>
      <button class="nav-button" data-tab="daily-fitness">每日健身 (Today)</button>
      <button class="nav-button" data-tab="recurring-todos">循环待办 (Recurring)</button>
      <button class="nav-button" data-tab="recurring-fitness">循环健身 (Recurring)</button>
      <button class="nav-button" data-tab="mobile-config">手机/大屏配置</button>
    </nav>
  </aside>
  <main class="main">
    
    <div id="daily-todos" class="content-section active">
      <div class="panel">
        <h2>每日待办 (单次)</h2>
        <div class="form-group"><input type="date" id="date-todo"></div>
        <div id="todoList"></div>
        <hr>
        <h3>添加待办</h3>
        <form id="addTodoForm">
          <div class="form-group"><input type="text" id="todo-content" placeholder="内容" required></div>
          <div class="form-group">
            <label>提醒时间 (可选，留空则不提醒):</label>
            <input type="datetime-local" id="todo-reminder">
          </div>
          <button type="submit" class="btn-submit">添加</button>
        </form>
      </div>
    </div>

    <div id="daily-fitness" class="content-section">
      <div class="panel">
        <h2>每日健身</h2>
        <div class="form-group"><input type="date" id="date-daily"></div>
        <div id="dailyFitnessList"></div>
      </div>
    </div>

    <div id="recurring-todos" class="content-section">
      <div class="panel">
        <h2>循环待办 (自动生成)</h2>
        <div id="recurringTodoList"></div>
        <hr>
        <h3>添加循环待办</h3>
        <form id="addRecurringTodoForm">
          <div class="form-group"><input type="text" id="rec-todo-content" placeholder="内容" required></div>
          <div class="form-group"><input type="date" id="rec-todo-start" required></div>
          <div class="form-group">
            <label>循环方式:</label>
            <select id="rec-todo-interval">
              <option value="daily">每日</option>
              <option value="weekly">每周</option>
              <option value="monthly">每月</option>
            </select>
          </div>
          <div class="form-group hidden" id="rec-todo-day-weekly-box">
            <select id="rec-todo-day-w"><option value="1">周一</option><option value="2">周二</option><option value="3">周三</option><option value="4">周四</option><option value="5">周五</option><option value="6">周六</option><option value="7">周日</option></select>
          </div>
          <div class="form-group hidden" id="rec-todo-day-monthly-box">
            <input type="number" id="rec-todo-day-m" placeholder="几号 (1-31)" min="1" max="31">
          </div>
          <div class="form-group">
            <label>提醒时间点 (可选, 例 07:00):</label>
            <input type="time" id="rec-todo-time">
          </div>
          <button type="submit" class="btn-submit">保存循环设置</button>
        </form>
      </div>
    </div>

    <div id="recurring-fitness" class="content-section">
      <div class="panel">
        <h2>循环健身计划</h2>
        <div id="recurringPlanList"></div>
        <hr>
        <h3>添加循环计划</h3>
        <form id="addRecurringPlanForm">
            <input type="text" id="rec-plan-name" placeholder="计划名 (如: 练胸)" required style="margin-bottom:10px;">
            <input type="date" id="rec-plan-start" required style="margin-bottom:10px;">
            <select id="rec-plan-interval" style="margin-bottom:10px;">
                <option value="daily">每日</option>
                <option value="weekly">每周</option>
            </select>
            <select id="rec-plan-day-w" class="hidden" style="margin-bottom:10px;">
                <option value="1">周一</option><option value="2">周二</option><option value="3">周三</option><option value="4">周四</option><option value="5">周五</option><option value="6">周六</option><option value="7">周日</option>
            </select>
            <div id="ex-inputs"><input type="text" class="ex-name" placeholder="动作名"><input type="text" class="ex-dets" placeholder="组数"></div>
            <button type="button" id="addExBtn" style="margin:10px 0;">+ 动作</button>
            <button type="submit" class="btn-submit">保存</button>
        </form>
      </div>
    </div>

    <div id="mobile-config" class="content-section">
        <iframe src="/mobile_config" style="width:100%; height:600px; border:none;"></iframe>
    </div>

  </main>
</div>

<script>
const todayStr = new Date().toLocaleDateString('zh-CN', {year:'numeric',month:'2-digit',day:'2-digit'}).replace(/\//g, '-');
const api = async (u, o) => (await fetch(u, {credentials:'include', ...o})).json();

// UI Logic
document.querySelectorAll('.nav-button').forEach(b => b.addEventListener('click', () => {
    document.querySelectorAll('.nav-button').forEach(x=>x.classList.remove('active'));
    document.querySelectorAll('.content-section').forEach(x=>x.classList.remove('active'));
    b.classList.add('active');
    document.getElementById(b.dataset.tab).classList.add('active');
    refreshData();
}));

// Init values
document.getElementById('date-todo').value = todayStr;
document.getElementById('date-daily').value = todayStr;
document.getElementById('rec-todo-start').value = todayStr;
document.getElementById('rec-plan-start').value = todayStr;

document.getElementById('date-todo').addEventListener('change', ()=>loadTodos(document.getElementById('date-todo').value));
document.getElementById('date-daily').addEventListener('change', ()=>loadFitness(document.getElementById('date-daily').value));

// Loaders
async function loadTodos(d) {
    const list = await api(`/todos?date=${d}`);
    document.getElementById('todoList').innerHTML = list.map(t => `
        <div class="list-item" style="${t.is_completed?'opacity:0.5':''}">
            <div>
                ${t.is_completed?'✅':'⬜'} <strong>${t.content}</strong> 
                ${t.reminder_at ? `<span class="tag-time">🔔 ${t.reminder_at.slice(5)}</span>` : ''}
                ${t.recurring_todo_id ? '(循环)' : ''}
            </div>
            <button class="btn-danger" onclick="delItem('delete_todo', ${t.id})">删</button>
        </div>
    `).join('') || '<div style="padding:10px;color:#999">无待办</div>';
}

async function loadFitness(d) {
    const list = await api(`/fitness_plan/${d}`);
    document.getElementById('dailyFitnessList').innerHTML = list.map(p => `
        <div class="list-item">
            <span>${p.is_completed?'✅':'⬜'} ${p.exercise_name} - ${p.sets_reps_duration}</span>
            <button class="btn-danger" onclick="delItem('delete_fitness_plan_exercise', ${p.id})">删</button>
        </div>
    `).join('');
}

async function loadRecTodos() {
    const list = await api('/get_recurring_todos');
    document.getElementById('recurringTodoList').innerHTML = list.map(t => `
        <div class="list-item">
            <div>
                <strong>${t.content}</strong> 
                [${t.repeat_interval}${t.repeat_interval!=='daily'?'...':''}]
                ${t.reminder_time ? `<span class="tag-time">⏰ ${t.reminder_time}</span>` : ''}
            </div>
            <button class="btn-danger" onclick="delItem('delete_recurring_todo', ${t.id})">删</button>
        </div>
    `).join('');
}

async function loadRecPlans() {
    const list = await api('/get_recurring_plans');
    document.getElementById('recurringPlanList').innerHTML = list.map(p => `
        <div class="list-item">
            <span>${p.plan_name} [${p.repeat_interval}]</span>
            <button class="btn-danger" onclick="delItem('delete_recurring_plan', ${p.id})">删</button>
        </div>
    `).join('');
}

// Actions
document.getElementById('addTodoForm').onsubmit = async (e) => {
    e.preventDefault();
    let reminder = document.getElementById('todo-reminder').value; // "2023-11-22T10:00"
    if(reminder) reminder = reminder.replace('T', ' ');
    await api('/add_todo', {method:'POST', headers:{'Content-Type':'application/json'}, body:JSON.stringify({
        date: document.getElementById('date-todo').value,
        content: document.getElementById('todo-content').value,
        reminder_at: reminder
    })});
    e.target.reset();
    loadTodos(document.getElementById('date-todo').value);
};

document.getElementById('addRecurringTodoForm').onsubmit = async (e) => {
    e.preventDefault();
    await api('/add_recurring_todo', {method:'POST', headers:{'Content-Type':'application/json'}, body:JSON.stringify({
        content: document.getElementById('rec-todo-content').value,
        start_date: document.getElementById('rec-todo-start').value,
        repeat_interval: document.getElementById('rec-todo-interval').value,
        repeat_day: document.getElementById('rec-todo-interval').value === 'weekly' ? document.getElementById('rec-todo-day-w').value : document.getElementById('rec-todo-day-m').value,
        reminder_time: document.getElementById('rec-todo-time').value
    })});
    e.target.reset();
    loadRecTodos();
};

document.getElementById('rec-todo-interval').onchange = (e) => {
    document.getElementById('rec-todo-day-weekly-box').classList.toggle('hidden', e.target.value !== 'weekly');
    document.getElementById('rec-todo-day-monthly-box').classList.toggle('hidden', e.target.value !== 'monthly');
};

document.getElementById('rec-plan-interval').onchange = (e) => {
    document.getElementById('rec-plan-day-w').classList.toggle('hidden', e.target.value !== 'weekly');
};

document.getElementById('addExBtn').onclick = () => {
    document.getElementById('ex-inputs').insertAdjacentHTML('beforeend', '<div style="margin-top:5px"><input type="text" class="ex-name" placeholder="动作"><input type="text" class="ex-dets" placeholder="组数"></div>');
};

document.getElementById('addRecurringPlanForm').onsubmit = async (e) => {
    e.preventDefault();
    const names = [...document.querySelectorAll('.ex-name')].map(i=>i.value);
    const dets = [...document.querySelectorAll('.ex-dets')].map(i=>i.value);
    const exercises = names.filter(n=>n).map((n,i)=>({name:n, details:dets[i]||''}));
    
    await api('/add_recurring_plan', {method:'POST', headers:{'Content-Type':'application/json'}, body:JSON.stringify({
        plan_name: document.getElementById('rec-plan-name').value,
        start_date: document.getElementById('rec-plan-start').value,
        repeat_interval: document.getElementById('rec-plan-interval').value,
        repeat_day: document.getElementById('rec-plan-day-w').value,
        exercises: exercises
    })});
    e.target.reset();
    loadRecPlans();
};

async function delItem(endpoint, id) {
    if(!confirm('删除?')) return;
    await api(`/${endpoint}/${id}`, {method:'POST'});
    refreshData();
}

function refreshData() {
    loadTodos(document.getElementById('date-todo').value);
    loadFitness(document.getElementById('date-daily').value);
    loadRecTodos();
    loadRecPlans();
}

// Init
refreshData();
</script>
</body>
</html>
'''

# ==========================================
# 3. frontend/templates/display.html (v1.8 - Display & Triple Click & Sound)
# ==========================================
display_html_content = r'''<!DOCTYPE html>
<html lang="zh-CN">
<head>
<meta charset="utf-8" />
<meta name="viewport" content="width=device-width, initial-scale=1.0" />
<title>智能看板</title>
<style>
:root { --bg:#0f1724; --panel:#1e293b; --accent:#4CAF50; --font:56px; --alert:#ff4444; }
body { background:var(--bg); color:#fff; font-family:sans-serif; margin:0; overflow:hidden; height:100vh; display:flex; flex-direction:column; }
.header { padding:20px 40px; background:linear-gradient(90deg,#071029,#0b2540); display:flex; justify-content:space-between; align-items:center; box-shadow:0 5px 15px rgba(0,0,0,0.5); }
.brand { font-size:40px; color:var(--accent); font-weight:bold; }
.datetime { text-align:right; cursor:pointer; user-select:none; }
.time { font-size:50px; font-weight:bold; }
.main { flex:1; display:flex; gap:30px; padding:30px; overflow:hidden; }
.panel { flex:1; background:var(--panel); border-radius:20px; padding:30px; display:flex; flex-direction:column; box-shadow:0 10px 30px rgba(0,0,0,0.3); overflow:hidden; }
h2 { margin:0 0 20px 0; color:var(--accent); border-bottom:2px solid #333; padding-bottom:10px; }
.list-box { flex:1; overflow:hidden; position:relative; }
.scroll-content { position:absolute; width:100%; }
.item { padding:20px; border-bottom:1px solid #333; display:flex; justify-content:space-between; align-items:center; font-size:32px; }
.item.completed { text-decoration:line-through; color:#666; }
.item.alert { background: rgba(255, 68, 68, 0.2); border-left: 8px solid var(--alert); animation: flash 2s infinite; }
.btn { padding:10px 20px; font-size:24px; background:#6a1b9a; color:fff; border:none; border-radius:10px; cursor:pointer; }
@keyframes flash { 0%{opacity:1} 50%{opacity:0.7} 100%{opacity:1} }
</style>
</head>
<body>
<div class="header">
    <div class="brand">智能看板</div>
    <div class="datetime" id="clockArea">
        <div id="dateStr">...</div>
        <div class="time" id="timeStr">...</div>
    </div>
</div>
<div class="main">
    <div class="panel"><h2>今日健身</h2><div class="list-box"><div id="fitList" class="scroll-content"></div></div></div>
    <div class="panel"><h2>今日待办</h2><div class="list-box"><div id="todoList" class="scroll-content"></div></div></div>
</div>

<!-- Simple Beep Sound (Base64) -->
<audio id="alarmSound" src="data:audio/wav;base64,UklGRl9vT19XQVZFZm10IBAAAAABAAEAQB8AAEAfAAABAAgAZGF0YU..." preload="auto"></audio>

<script>
const API = location.origin.includes('localhost') ? 'http://localhost:8000' : location.origin;
const TODAY = () => new Date().toLocaleDateString('zh-CN', {year:'numeric',month:'2-digit',day:'2-digit'}).replace(/\//g, '-');

// 1. Clock & Triple Click Logic
let clicks = 0;
document.getElementById('clockArea').addEventListener('click', () => {
    clicks++;
    if(clicks === 1) setTimeout(() => clicks = 0, 800); // 0.8s window
    if(clicks >= 3) window.location.href = '/manage';
});

setInterval(() => {
    const now = new Date();
    document.getElementById('dateStr').innerText = now.toLocaleDateString();
    document.getElementById('timeStr').innerText = now.toLocaleTimeString();
    checkReminders(); // Check alerts every second
}, 1000);

// 2. Data & Reminder Logic
let todos = [];
let hasAlert = false;

// Simple beep generation if base64 fails, or just rely on visual for now if audio blocked
const audioCtx = new (window.AudioContext || window.webkitAudioContext)();
function beep() {
    if(audioCtx.state === 'suspended') audioCtx.resume();
    const osc = audioCtx.createOscillator();
    const gain = audioCtx.createGain();
    osc.connect(gain);
    gain.connect(audioCtx.destination);
    osc.frequency.value = 880; // A5
    gain.gain.value = 0.1;
    osc.start();
    setTimeout(() => osc.stop(), 200);
}

async function loadData() {
    try {
        const [r1, r2] = await Promise.all([
            fetch(`${API}/fitness_plan/${TODAY()}`, {credentials:'include'}),
            fetch(`${API}/todos?date=${TODAY()}`, {credentials:'include'})
        ]);
        if(r1.status===401 || r2.status===401) return window.location.href='/';
        
        const fits = await r1.json();
        todos = await r2.json(); // Global for checker
        
        render('fitList', fits, false);
        render('todoList', todos, true);
    } catch(e) { console.error(e); }
}

function render(id, list, isTodo) {
    const el = document.getElementById(id);
    if(!list.length) { el.innerHTML='<div style="padding:20px;color:#666">无项目</div>'; return; }
    
    el.innerHTML = list.map(i => {
        // Reminder Check for rendering class
        let alertClass = '';
        if(isTodo && !i.is_completed && i.reminder_at) {
            const remindTime = new Date(i.reminder_at).getTime();
            const now = new Date().getTime();
            if(now >= remindTime) alertClass = 'alert';
        }

        return `
        <div class="item ${i.is_completed?'completed':''} ${alertClass}">
            <div>
                ${isTodo ? i.content : i.exercise_name}
                ${!isTodo ? `<div style="font-size:20px;color:#999">${i.sets_reps_duration}</div>` : ''}
                ${isTodo && i.reminder_at ? `<div style="font-size:18px;color:#aaa">⏰ ${i.reminder_at.slice(5)}</div>` : ''}
            </div>
            <button class="btn" onclick="toggle(${i.id}, ${isTodo?1:0}, ${i.is_completed?1:0})">
                ${i.is_completed ? '已完成' : '标记'}
            </button>
        </div>`;
    }).join('');
}

async function toggle(id, isTodo, state) {
    const ep = isTodo ? 'mark_todo_completed' : 'mark_exercise_completed_mobile';
    const body = isTodo ? {todo_id:id, is_completed:!state} : {plan_id:id, is_completed:!state};
    await fetch(`${API}/${ep}`, {method:'POST', headers:{'Content-Type':'application/json'}, body:JSON.stringify(body), credentials:'include'});
    loadData();
}

// Check alerts loop
let lastBeep = 0;
function checkReminders() {
    const now = new Date().getTime();
    const overdue = todos.some(t => !t.is_completed && t.reminder_at && now >= new Date(t.reminder_at).getTime());
    
    if(overdue) {
        // Flash visual handled by CSS .alert in render()
        // Sound every 10 seconds
        if(now - lastBeep > 10000) {
            beep();
            lastBeep = now;
        }
        // Re-render to update alert classes immediately if time crossed just now
        if(!document.querySelector('.alert')) render('todoList', todos, true);
    }
}

// Auto Scroll Logic
function scroll(id) {
    const el = document.getElementById(id);
    let pos = 0;
    setInterval(() => {
        const max = el.scrollHeight - el.parentElement.clientHeight;
        if(max <= 0) return;
        pos += 1;
        if(pos > max + 50) pos = -50; // Wrap around delay
        el.style.transform = `translateY(-${pos<0?0:pos}px)`;
    }, 50); // Speed
}

loadData();
setInterval(loadData, 10000); // Refresh data
setTimeout(() => { scroll('fitList'); scroll('todoList'); }, 2000);
</script>
</body>
</html>
'''

# ==========================================
# 4. Write Files Utility
# ==========================================
def write_file(filename, content, folder="."):
    filepath = os.path.join(folder, filename)
    try:
        with open(filepath, 'w', encoding='utf-8') as f:
            f.write(content)
        print(f"[成功] 已更新: {filepath}")
    except Exception as e:
        print(f"[错误] 写入 {filepath} 失败: {e}")

def main():
    print("=== 正在升级系统 (v1.8 提醒功能版) ===")
    
    # Write Backend
    write_file('app.py', app_py_content)
    
    # Determine template folder
    target_dir = "."
    if os.path.exists("templates"): target_dir = "templates"
    elif os.path.exists("frontend/templates"): target_dir = "frontend/templates"
    
    # Write Frontend
    write_file('manage.html', manage_html_content, target_dir)
    write_file('display.html', display_html_content, target_dir)
    
    print("\n=== 升级完成 ===")
    print("1. 请重启 Flask 程序 (python app.py) 以应用数据库变更。")
    print("2. 打开主界面，连续点击右上角时间3次即可进入后台。")

if __name__ == '__main__':
    main()
