import os

# ==============================================================================
# 1. backend/app.py (v1.9 - Full Restore + Fitness/Todo Reminders)
# ==============================================================================
app_py_content = r'''# backend/app.py (v1.9 - Full Features)
import sqlite3
from datetime import datetime, timedelta, timezone
from flask import Flask, render_template, request, jsonify, g, redirect, url_for, make_response
from flask_login import LoginManager, UserMixin, login_user, logout_user, login_required, current_user
import os, json, uuid
from flask_cors import CORS
from functools import wraps
import sys 
from collections import defaultdict

CN_TZ = timezone(timedelta(hours=8))

is_frozen = getattr(sys, 'frozen', False)
is_in_onefile_temp = 'onefile_' in os.path.abspath(__file__) 

if is_frozen or is_in_onefile_temp:
    temp_base_dir = os.path.dirname(os.path.abspath(sys.executable))
    template_dir = os.path.join(temp_base_dir, 'templates')
    static_dir = os.path.join(temp_base_dir, 'static')
    exe_path = os.path.realpath(sys.argv[0])
    exe_dir = os.path.dirname(exe_path)
    DATABASE = os.path.join(exe_dir, 'fitness_manager.db')
else:
    app_file_path = os.path.abspath(__file__)
    backend_dir = os.path.dirname(app_file_path)
    project_root_dir = os.path.dirname(backend_dir)
    template_dir = os.path.join(project_root_dir, 'frontend', 'templates')
    static_dir = os.path.join(project_root_dir, 'frontend', 'static')
    DATABASE = os.path.join(backend_dir, 'fitness_manager.db')

print("DB:", DATABASE)

app = Flask(__name__, template_folder=template_dir, static_folder=static_dir)
CORS(app)
app.secret_key = str(uuid.uuid4())

login_manager = LoginManager()
login_manager.init_app(app)
login_manager.login_view = 'index' 

class User(UserMixin):
    def __init__(self, id, username, is_admin, manager_id):
        self.id = id
        self.username = username
        self.is_admin = is_admin
        self.manager_id = manager_id 
    @property
    def is_super_admin(self):
        return self.is_admin and (self.manager_id is None or self.manager_id == 0)

@login_manager.user_loader
def load_user(user_id):
    db = get_db()
    user_row = db.execute("SELECT id, username, is_admin, manager_id FROM users WHERE id = ?", (user_id,)).fetchone()
    if user_row:
        return User(id=user_row['id'], username=user_row['username'], is_admin=user_row['is_admin'], manager_id=user_row['manager_id'])
    return None

def get_db():
    db = getattr(g, '_database', None)
    if db is None:
        db = g._database = sqlite3.connect(DATABASE, check_same_thread=False)
        db.row_factory = sqlite3.Row
    return db

def get_setting(key, default=None):
    row = get_db().execute("SELECT value FROM settings WHERE key = ?", (key,)).fetchone()
    return default if row is None else row["value"]

def set_setting(key, value):
    db = get_db()
    db.execute("INSERT INTO settings (key, value) VALUES (?, ?) ON CONFLICT(key) DO UPDATE SET value = excluded.value", (key, str(value)))
    db.commit()

@app.teardown_appcontext
def close_connection(exception):
    db = getattr(g, '_database', None)
    if db is not None: db.close()

def init_db():
    db = get_db()
    cur = db.cursor()
    
    # Tables with reminder support
    cur.execute("""
        CREATE TABLE IF NOT EXISTS todos (
            id INTEGER PRIMARY KEY AUTOINCREMENT, 
            date TEXT NOT NULL, 
            content TEXT NOT NULL,
            is_completed INTEGER DEFAULT 0, 
            user_id INTEGER, 
            recurring_todo_id INTEGER,
            reminder_at TEXT
        )
    """)
    cur.execute("""
        CREATE TABLE IF NOT EXISTS fitness_plans (
            id INTEGER PRIMARY KEY AUTOINCREMENT, 
            date TEXT NOT NULL, 
            exercise_name TEXT NOT NULL,
            sets_reps_duration TEXT, 
            is_completed INTEGER DEFAULT 0, 
            user_id INTEGER, 
            recurring_plan_id INTEGER,
            reminder_at TEXT
        )
    """)
    cur.execute("""
        CREATE TABLE IF NOT EXISTS recurring_plans (
            id INTEGER PRIMARY KEY AUTOINCREMENT, 
            user_id INTEGER, 
            plan_name TEXT NOT NULL,
            exercises_json TEXT NOT NULL, 
            start_date TEXT NOT NULL, 
            end_date TEXT,
            repeat_interval TEXT NOT NULL, 
            repeat_day INTEGER,
            reminder_time TEXT
        )
    """)
    cur.execute("""
        CREATE TABLE IF NOT EXISTS recurring_todos (
            id INTEGER PRIMARY KEY AUTOINCREMENT, 
            user_id INTEGER, 
            content TEXT NOT NULL,
            start_date TEXT NOT NULL, 
            end_date TEXT, 
            repeat_interval TEXT NOT NULL, 
            repeat_day INTEGER,
            reminder_time TEXT
        )
    """)
    cur.execute("CREATE TABLE IF NOT EXISTS exercise_library (id INTEGER PRIMARY KEY AUTOINCREMENT, user_id INTEGER, name TEXT NOT NULL, details TEXT)")
    cur.execute("CREATE TABLE IF NOT EXISTS settings (key TEXT PRIMARY KEY, value TEXT)")
    cur.execute("CREATE TABLE IF NOT EXISTS todo_library (id INTEGER PRIMARY KEY AUTOINCREMENT, user_id INTEGER, content TEXT NOT NULL)")
    cur.execute("CREATE TABLE IF NOT EXISTS users (id INTEGER PRIMARY KEY AUTOINCREMENT, username TEXT UNIQUE NOT NULL, password TEXT NOT NULL, is_admin INTEGER DEFAULT 0, manager_id INTEGER REFERENCES users(id))")
    
    # Migrations for new columns
    cols_todos = [row[1] for row in cur.execute("PRAGMA table_info(todos)").fetchall()]
    if 'reminder_at' not in cols_todos: cur.execute("ALTER TABLE todos ADD COLUMN reminder_at TEXT")

    cols_fit = [row[1] for row in cur.execute("PRAGMA table_info(fitness_plans)").fetchall()]
    if 'reminder_at' not in cols_fit: cur.execute("ALTER TABLE fitness_plans ADD COLUMN reminder_at TEXT")
    
    cols_rec_todo = [row[1] for row in cur.execute("PRAGMA table_info(recurring_todos)").fetchall()]
    if 'reminder_time' not in cols_rec_todo: cur.execute("ALTER TABLE recurring_todos ADD COLUMN reminder_time TEXT")

    cols_rec_plan = [row[1] for row in cur.execute("PRAGMA table_info(recurring_plans)").fetchall()]
    if 'reminder_time' not in cols_rec_plan: cur.execute("ALTER TABLE recurring_plans ADD COLUMN reminder_time TEXT")

    # Superuser check
    if cur.execute("SELECT COUNT(*) FROM users WHERE username = 'superuser'").fetchone()[0] == 0:
        cur.execute("INSERT INTO users (username, password, is_admin, manager_id) VALUES (?, ?, ?, ?)", ('superuser', 'adminpassword', 1, None))
        
    # User1 check
    su_id = cur.execute("SELECT id FROM users WHERE username = 'superuser'").fetchone()['id']
    if cur.execute("SELECT COUNT(*) FROM users WHERE username = 'user1'").fetchone()[0] == 0:
        cur.execute("INSERT INTO users (username, password, is_admin, manager_id) VALUES (?, ?, ?, ?)", ('user1', 'password', 0, su_id))

    db.commit()

with app.app_context():
    init_db()

# -------------------- Auth --------------------
@app.route('/login', methods=['POST'])
def login():
    data = request.form if request.form else request.json
    u_row = get_db().execute("SELECT * FROM users WHERE username = ?", (data.get('username'),)).fetchone()
    if u_row and u_row['password'] == data.get('password'):
        login_user(User(u_row['id'], u_row['username'], u_row['is_admin'], u_row['manager_id']), remember=True)
        return jsonify({'status': 'success'})
    return jsonify({'status': 'error', 'message': 'Fail'}), 401

@app.route('/logout', methods=['POST'])
@login_required
def logout():
    logout_user()
    return jsonify({'status': 'success'})

@app.route('/get_current_user', methods=['GET'])
def get_current_user():
    if current_user.is_authenticated:
        return jsonify({'status':'logged_in', 'username':current_user.username, 'is_admin':current_user.is_admin})
    return jsonify({'status':'logged_out'}), 401

# -------------------- Views --------------------
@app.route('/')
def index():
    if not current_user.is_authenticated: return render_template('login.html')
    if 'Mobile' in request.headers.get('User-Agent', '') and request.args.get('view') != 'display':
        return redirect(url_for('mobile_config'))
    today = datetime.now(CN_TZ).strftime('%Y-%m-%d')
    _gen_fitness(current_user.id, today)
    _gen_todos(current_user.id, today)
    return render_template('display.html', username=current_user.username)

@app.route('/mobile_config')
@login_required
def mobile_config(): return render_template('mobile_config.html', username=current_user.username)

@app.route('/manage')
@login_required
def manage():
    return render_template('manage.html', username=current_user.username, is_admin=current_user.is_admin, manager_id=current_user.manager_id)

# -------------------- API: Todos --------------------
@app.route('/todos', methods=['GET'])
@login_required
def get_todos():
    d = request.args.get('date') or datetime.now(CN_TZ).strftime('%Y-%m-%d')
    rows = get_db().execute("SELECT * FROM todos WHERE date=? AND user_id=? ORDER BY is_completed ASC, reminder_at ASC, id DESC", (d, current_user.id)).fetchall()
    return jsonify([dict(r) for r in rows])

@app.route('/add_todo', methods=['POST'])
@login_required
def add_todo():
    d = request.json
    rem = d.get('reminder_at') # "2023-11-22 10:00"
    if rem and len(rem) == 5: rem = f"{d.get('date')} {rem}" # Handle "10:00"
    
    get_db().execute("INSERT INTO todos (date, content, is_completed, user_id, reminder_at) VALUES (?,?,0,?,?)", 
                     (d.get('date'), d.get('content'), current_user.id, rem)).connection.commit()
    return jsonify(status='success')

@app.route('/mark_todo_completed', methods=['POST'])
@login_required
def mark_todo():
    d = request.json
    get_db().execute("UPDATE todos SET is_completed=? WHERE id=? AND user_id=?", (d.get('is_completed'), d.get('todo_id'), current_user.id)).connection.commit()
    return jsonify(status='success')

@app.route('/delete_todo/<int:id>', methods=['POST'])
@login_required
def del_todo(id):
    get_db().execute("DELETE FROM todos WHERE id=? AND user_id=?", (id, current_user.id)).connection.commit()
    return jsonify(status='success')

# -------------------- API: Fitness --------------------
@app.route('/fitness_plan/<date_str>', methods=['GET'])
@login_required
def get_fitness(date_str):
    _gen_fitness(current_user.id, date_str)
    rows = get_db().execute("SELECT * FROM fitness_plans WHERE date=? AND user_id=? ORDER BY is_completed ASC, reminder_at ASC, id DESC", (date_str, current_user.id)).fetchall()
    return jsonify([dict(r) for r in rows])

@app.route('/add_fitness_manual', methods=['POST'])
@login_required
def add_fitness_manual():
    # New endpoint to manually add a daily fitness item with time
    d = request.json
    rem = d.get('reminder_at')
    if rem and len(rem) == 5: rem = f"{d.get('date')} {rem}"
    
    get_db().execute("INSERT INTO fitness_plans (date, exercise_name, sets_reps_duration, is_completed, user_id, reminder_at) VALUES (?,?,?,0,?,?)",
                     (d.get('date'), d.get('exercise_name'), d.get('sets_reps_duration'), current_user.id, rem)).connection.commit()
    return jsonify(status='success')

@app.route('/mark_exercise_completed_mobile', methods=['POST'])
@login_required
def mark_fit():
    d = request.json
    get_db().execute("UPDATE fitness_plans SET is_completed=? WHERE id=? AND user_id=?", (d.get('is_completed'), d.get('plan_id'), current_user.id)).connection.commit()
    return jsonify(status='success')

@app.route('/delete_fitness_plan_exercise/<int:id>', methods=['POST'])
@login_required
def del_fit(id):
    get_db().execute("DELETE FROM fitness_plans WHERE id=? AND user_id=?", (id, current_user.id)).connection.commit()
    return jsonify(status='success')

# -------------------- API: Recurring --------------------
@app.route('/get_recurring_plans', methods=['GET'])
@login_required
def get_rec_plans():
    return jsonify([dict(r) for r in get_db().execute("SELECT * FROM recurring_plans WHERE user_id=?", (current_user.id,)).fetchall()])

@app.route('/add_recurring_plan', methods=['POST'])
@login_required
def add_rec_plan():
    d = request.json
    get_db().execute("INSERT INTO recurring_plans (user_id, plan_name, exercises_json, start_date, end_date, repeat_interval, repeat_day, reminder_time) VALUES (?,?,?,?,?,?,?,?)",
                     (current_user.id, d.get('plan_name'), json.dumps(d.get('exercises')), d.get('start_date'), d.get('end_date'), d.get('repeat_interval'), d.get('repeat_day'), d.get('reminder_time'))).connection.commit()
    return jsonify(status='success')

@app.route('/delete_recurring_plan/<int:id>', methods=['POST'])
@login_required
def del_rec_plan(id):
    get_db().execute("DELETE FROM recurring_plans WHERE id=? AND user_id=?", (id, current_user.id)).connection.commit()
    return jsonify(status='success')

@app.route('/get_recurring_todos', methods=['GET'])
@login_required
def get_rec_todos():
    return jsonify([dict(r) for r in get_db().execute("SELECT * FROM recurring_todos WHERE user_id=?", (current_user.id,)).fetchall()])

@app.route('/add_recurring_todo', methods=['POST'])
@login_required
def add_rec_todo():
    d = request.json
    get_db().execute("INSERT INTO recurring_todos (user_id, content, start_date, end_date, repeat_interval, repeat_day, reminder_time) VALUES (?,?,?,?,?,?,?)",
                     (current_user.id, d.get('content'), d.get('start_date'), d.get('end_date'), d.get('repeat_interval'), d.get('repeat_day'), d.get('reminder_time'))).connection.commit()
    return jsonify(status='success')

@app.route('/delete_recurring_todo/<int:id>', methods=['POST'])
@login_required
def del_rec_todo(id):
    get_db().execute("DELETE FROM recurring_todos WHERE id=? AND user_id=?", (id, current_user.id)).connection.commit()
    return jsonify(status='success')

# -------------------- Libraries & Utils --------------------
@app.route('/api/exercises', methods=['GET','POST'])
@app.route('/api/exercises/<int:id>', methods=['DELETE'])
@login_required
def ex_lib(id=None):
    db = get_db()
    if request.method=='GET': return jsonify([dict(r) for r in db.execute("SELECT * FROM exercise_library WHERE user_id=?",(current_user.id,))])
    if request.method=='POST': 
        db.execute("INSERT INTO exercise_library (user_id, name, details) VALUES (?,?,?)", (current_user.id, request.json.get('name'), request.json.get('details')))
        db.commit()
        return jsonify(status='success')
    if request.method=='DELETE':
        db.execute("DELETE FROM exercise_library WHERE id=? AND user_id=?", (id, current_user.id))
        db.commit()
        return jsonify(status='success')

@app.route('/api/todo_library', methods=['GET','POST'])
@app.route('/api/todo_library/<int:id>', methods=['DELETE'])
@login_required
def todo_lib(id=None):
    db = get_db()
    if request.method=='GET': return jsonify([dict(r) for r in db.execute("SELECT * FROM todo_library WHERE user_id=?",(current_user.id,))])
    if request.method=='POST': 
        db.execute("INSERT INTO todo_library (user_id, content) VALUES (?,?)", (current_user.id, request.json.get('content')))
        db.commit()
        return jsonify(status='success')
    if request.method=='DELETE':
        db.execute("DELETE FROM todo_library WHERE id=? AND user_id=?", (id, current_user.id))
        db.commit()
        return jsonify(status='success')

@app.route('/api/settings/scroll_speed', methods=['GET','POST'])
def settings_scroll():
    if request.method=='POST': set_setting('scroll_speed', request.json.get('scroll_speed')); return jsonify(status='success')
    return jsonify(scroll_speed=int(get_setting('scroll_speed',25) or 25))

@app.route('/api/users', methods=['GET','POST'])
@app.route('/api/users/<int:id>', methods=['DELETE'])
@login_required
def user_mgmt(id=None):
    if not current_user.is_admin: return jsonify(status='error'), 403
    db = get_db()
    if request.method=='GET': return jsonify([dict(u) for u in db.execute("SELECT id,username,is_admin FROM users WHERE manager_id=?",(current_user.id,))])
    if request.method=='POST':
        d=request.json
        is_adm = 1 if current_user.is_super_admin and d.get('is_admin') else 0
        try: db.execute("INSERT INTO users (username,password,is_admin,manager_id) VALUES (?,?,?,?)", (d.get('username'),d.get('password'),is_adm,current_user.id)); db.commit()
        except: return jsonify(status='error',message='Exists')
        return jsonify(status='success')
    if request.method=='DELETE':
        if id==current_user.id: return jsonify(status='error'), 400
        db.execute("DELETE FROM users WHERE id=?",(id,))
        db.commit()
        return jsonify(status='success')

@app.route('/api/profile', methods=['POST'])
@login_required
def profile():
    d=request.json
    if d.get('password'): get_db().execute("UPDATE users SET password=? WHERE id=?", (d.get('password'), current_user.id))
    if d.get('username'): 
        try: get_db().execute("UPDATE users SET username=? WHERE id=?", (d.get('username'), current_user.id))
        except: return jsonify(status='error')
    get_db().commit()
    return jsonify(status='success')

@app.route('/api/stats/daily_completion')
@login_required
def stats():
    s, e = request.args.get('start_date'), request.args.get('end_date')
    db = get_db()
    fr = db.execute("SELECT date, is_completed FROM fitness_plans WHERE user_id=? AND date BETWEEN ? AND ?", (current_user.id, s, e))
    tr = db.execute("SELECT date, is_completed FROM todos WHERE user_id=? AND date BETWEEN ? AND ?", (current_user.id, s, e))
    fd = defaultdict(lambda: {"total":0,"completed":0})
    td = defaultdict(lambda: {"total":0,"completed":0})
    for r in fr: 
        fd[r['date']]['total']+=1
        if r['is_completed']: fd[r['date']]['completed']+=1
    for r in tr:
        td[r['date']]['total']+=1
        if r['is_completed']: td[r['date']]['completed']+=1
    return jsonify(fitness_data=fd, todo_data=td)

# -------------------- Logic --------------------
def _gen_fitness(uid, date_str):
    db = get_db()
    if db.execute("SELECT COUNT(*) FROM fitness_plans WHERE date=? AND user_id=?", (date_str, uid)).fetchone()[0]>0: return
    dt = datetime.strptime(date_str, '%Y-%m-%d').date()
    plans = db.execute("SELECT * FROM recurring_plans WHERE user_id=? AND start_date<=? AND (end_date IS NULL OR end_date>=?)", (uid, date_str, date_str)).fetchall()
    for p in plans:
        if (p['repeat_interval']=='daily') or (p['repeat_interval']=='weekly' and p['repeat_day']==dt.isoweekday()):
            # Calculate specific reminder datetime if time is set
            rem_at = f"{date_str} {p['reminder_time']}" if p['reminder_time'] else None
            exs = json.loads(p['exercises_json'])
            for ex in exs:
                db.execute("INSERT INTO fitness_plans (date,exercise_name,sets_reps_duration,user_id,recurring_plan_id,reminder_at) VALUES (?,?,?,?,?,?)",
                           (date_str, ex['name'], ex['details'], uid, p['id'], rem_at))
    db.commit()

def _gen_todos(uid, date_str):
    db = get_db()
    if db.execute("SELECT COUNT(*) FROM todos WHERE date=? AND user_id=?", (date_str, uid)).fetchone()[0]>0: return
    dt = datetime.strptime(date_str, '%Y-%m-%d').date()
    dom = dt.day
    recs = db.execute("SELECT * FROM recurring_todos WHERE user_id=? AND start_date<=? AND (end_date IS NULL OR end_date>=?)", (uid, date_str, date_str)).fetchall()
    for r in recs:
        if (r['repeat_interval']=='daily') or (r['repeat_interval']=='weekly' and r['repeat_day']==dt.isoweekday()) or (r['repeat_interval']=='monthly' and r['repeat_day']==dom):
            rem_at = f"{date_str} {r['reminder_time']}" if r['reminder_time'] else None
            db.execute("INSERT INTO todos (date,content,is_completed,user_id,recurring_todo_id,reminder_at) VALUES (?,?,0,?,?,?)",
                       (date_str, r['content'], uid, r['id'], rem_at))
    db.commit()

@app.route('/health', methods=['GET'])
def health(): return jsonify(status='ok', time=datetime.now(CN_TZ).isoformat())

if __name__ == '__main__':
    app.run(host='0.0.0.0', port=8000, debug=True)
'''

# ==============================================================================
# 2. frontend/templates/manage.html (v1.9 - Full Restore + New Time Inputs)
# ==============================================================================
manage_html_content = r'''<!DOCTYPE html>
<html lang="zh-CN">
<head>
  <meta charset="utf-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1.0" />
  <title>健身看板 - 后台管理</title>
  <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
  <link href="https://fonts.googleapis.com/css2?family=Noto+Sans+SC:wght@300;400;500;600;700&display=swap" rel="stylesheet">
  <style>
    :root {
        --bg-color: #f1f5f9; --panel-color: #ffffff; --accent-color: #0f766e;
        --accent-color-light: #ccfbf1; --accent-color-soft: #ecfdf5; --text-primary: #0f172a;
        --text-secondary: #64748b; --border-soft: #e2e8f0; --shadow-soft: 0 18px 40px rgba(15, 23, 42, 0.12);
        --radius-lg: 18px; --radius-md: 12px;
    }
    .hidden { display: none !important; }
    * { box-sizing: border-box; }
    body { margin: 0; font-family: 'Noto Sans SC', sans-serif; background-color: var(--bg-color); color: var(--text-primary); }
    .app-container { display: flex; min-height: 100vh; }
    .sidebar { width: 260px; background: linear-gradient(180deg, #0f172a, #020617); color: #e5e7eb; display: flex; flex-direction: column; padding: 24px 18px; box-shadow: 6px 0 25px rgba(15, 23, 42, 0.7); position: sticky; top: 0; height: 100vh; }
    .sidebar-header { display: flex; align-items: center; gap: 12px; margin-bottom: 32px; }
    .logo-circle { width: 40px; height: 40px; border-radius: 50%; background: radial-gradient(circle at 30% 20%, #a7f3d0, #16a34a); display: flex; align-items: center; justify-content: center; color: #022c22; font-weight: 800; font-size: 20px; box-shadow: 0 0 0 3px rgba(34, 197, 94, 0.2); }
    .nav-list { list-style: none; margin: 0; padding: 0; flex: 1; }
    .nav-button { width: 100%; padding: 10px 12px; border-radius: 12px; border: none; background: transparent; color: #d1d5db; font-size: 14px; text-align: left; display: flex; align-items: center; cursor: pointer; transition: all 0.18s ease; margin-bottom: 6px; }
    .nav-button:hover { background: rgba(15, 23, 42, 0.8); }
    .nav-button.active { background: linear-gradient(90deg, rgba(34, 197, 94, 0.22), rgba(34, 197, 94, 0.04)); color: #ecfdf3; box-shadow: 0 10px 25px rgba(22, 163, 74, 0.5); }
    .main { flex: 1; padding: 24px 30px; max-width: calc(100vw - 260px); }
    .panel { background-color: var(--panel-color); border-radius: var(--radius-lg); box-shadow: var(--shadow-soft); padding: 20px 22px; margin-bottom: 18px; border: 1px solid rgba(148, 163, 184, 0.18); }
    .panel-header h2 { margin: 0; font-size: 18px; color: var(--text-primary); }
    .form-group { margin-bottom: 12px; }
    .form-group label { display: block; margin-bottom: 4px; font-size: 13px; color: var(--text-secondary); }
    input, select, textarea { width: 100%; padding: 12px; border: 1px solid #cbd5e1; border-radius: var(--radius-md); background: #f8fafc; font-family: inherit; }
    .btn-submit { background-color: var(--accent-color); color: #ecfdf5; border: none; padding: 10px 18px; border-radius: 999px; cursor: pointer; font-size: 14px; font-weight: 600; }
    .btn-danger { background-color: #ef4444; color: #fff; border: none; padding: 6px 12px; border-radius: 999px; cursor: pointer; font-size: 12px; }
    .list-item { padding: 10px 12px; border-radius: 10px; border: 1px solid var(--border-soft); margin-bottom: 8px; background: #f9fafb; display: flex; justify-content: space-between; align-items: center; }
    .content-section { display: none; }
    .content-section.active { display: block; }
    .tag-time { background: #e0f2fe; color: #0369a1; padding: 2px 6px; border-radius: 4px; font-size: 12px; margin-left: 6px; font-weight: normal; }
    .exercise-group { display: grid; grid-template-columns: 1.1fr 1fr; gap: 8px; margin-bottom: 8px; }
    .alert-success { padding: 10px; background: #ecfdf5; color: #166534; border-radius: 8px; margin-bottom: 10px; display:none; }
  </style>
</head>
<body>
  <div class="app-container">
    <aside class="sidebar">
      <div class="sidebar-header"><div class="logo-circle">F</div><div><div style="font-weight:700">健身看板</div><div style="font-size:12px;color:#9ca3af">后台管理系统</div></div></div>
      <ul class="nav-list">
        <li><button class="nav-button active" data-tab="daily-todos">✅ 每日待办 (T)</button></li>
        <li><button class="nav-button" data-tab="daily-fitness">💪 每日健身 (T)</button></li>
        <li><button class="nav-button" data-tab="recurring-todos">🔁 循环待办 (R)</button></li>
        <li><button class="nav-button" data-tab="recurring-fitness">📅 循环健身 (R)</button></li>
        <li><button class="nav-button" data-tab="stats">📊 统计分析</button></li>
        <li><button class="nav-button" data-tab="mobile-config">📱 手机/大屏配置</button></li>
        <li id="userMgmtBtn"><button class="nav-button" data-tab="user-management">👥 用户管理</button></li>
        <li><button class="nav-button" data-tab="profile">⚙️ 个人资料</button></li>
      </ul>
    </aside>

    <main class="main">
      <div id="globalAlert" class="alert-success"></div>

      <!-- 1. Daily Todos -->
      <div id="daily-todos" class="content-section active">
        <div class="panel">
            <div class="panel-header"><h2>每日待办 (Today)</h2></div>
            <p style="font-size:13px;color:#666">管理具体某一日的待办。支持设置提醒时间。</p>
            <div class="form-group"><label>日期:</label><input type="date" id="date-todo"></div>
            <div id="todoList"></div>
            <hr style="border:0;border-top:1px solid #eee;margin:20px 0">
            <h3>+ 新增待办</h3>
            <form id="addTodoForm">
                <div class="form-group"><input type="text" id="todo-content" placeholder="内容" required></div>
                <div class="form-group"><label>提醒时间 (可选):</label><input type="datetime-local" id="todo-reminder"></div>
                <button type="submit" class="btn-submit">添加</button>
            </form>
        </div>
      </div>

      <!-- 2. Daily Fitness -->
      <div id="daily-fitness" class="content-section">
        <div class="panel">
            <div class="panel-header"><h2>每日健身 (Today)</h2></div>
            <p style="font-size:13px;color:#666">通常由循环计划自动生成，也可以在此手动补充单次训练。</p>
            <div class="form-group"><label>日期:</label><input type="date" id="date-daily"></div>
            <div id="dailyFitnessList"></div>
            <hr style="border:0;border-top:1px solid #eee;margin:20px 0">
            <h3>+ 手动添加单次训练</h3>
            <form id="addManualFitForm">
                <div class="form-group"><input type="text" id="man-fit-name" placeholder="动作名称 (如: 跑步)" required></div>
                <div class="form-group"><input type="text" id="man-fit-det" placeholder="详情 (如: 5公里)"></div>
                <div class="form-group"><label>提醒时间 (可选):</label><input type="datetime-local" id="man-fit-rem"></div>
                <button type="submit" class="btn-submit">添加单次训练</button>
            </form>
        </div>
      </div>

      <!-- 3. Recurring Todos -->
      <div id="recurring-todos" class="content-section">
        <div class="panel">
            <div class="panel-header"><h2>循环待办 (Recurring)</h2></div>
            <div id="recurringTodoList"></div>
            <hr style="border:0;border-top:1px solid #eee;margin:20px 0">
            <h3>+ 新增循环待办</h3>
            <form id="addRecurringTodoForm">
                <div class="form-group"><input type="text" id="rec-todo-content" placeholder="内容" required></div>
                <div class="form-group"><label>开始日期:</label><input type="date" id="rec-todo-start" required></div>
                <div class="form-group">
                    <label>循环频率:</label>
                    <select id="rec-todo-interval">
                        <option value="daily">每日</option>
                        <option value="weekly">每周</option>
                        <option value="monthly">每月</option>
                    </select>
                </div>
                <div class="form-group hidden" id="rec-todo-weekly-box">
                    <select id="rec-todo-day-w"><option value="1">周一</option><option value="2">周二</option><option value="3">周三</option><option value="4">周四</option><option value="5">周五</option><option value="6">周六</option><option value="7">周日</option></select>
                </div>
                <div class="form-group hidden" id="rec-todo-monthly-box">
                    <input type="number" id="rec-todo-day-m" placeholder="几号 (1-31)" min="1" max="31" value="1">
                </div>
                <div class="form-group"><label>提醒时间点 (可选, 例 08:00):</label><input type="time" id="rec-todo-time"></div>
                <button type="submit" class="btn-submit">保存</button>
            </form>
        </div>
      </div>

      <!-- 4. Recurring Fitness -->
      <div id="recurring-fitness" class="content-section">
        <div class="panel">
            <div class="panel-header"><h2>循环健身计划 (Recurring)</h2></div>
            <p style="font-size:13px;color:#666">设置长期计划 (如: 每周一练胸)。可设置固定提醒时间。</p>
            <div id="recurringPlanList"></div>
            <hr style="border:0;border-top:1px solid #eee;margin:20px 0">
            <h3>+ 新增循环计划</h3>
            <form id="addRecurringPlanForm">
                <div class="form-group"><input type="text" id="rec-plan-name" placeholder="计划名 (如: 练腿日)" required></div>
                <div class="form-group"><label>开始日期:</label><input type="date" id="rec-plan-start" required></div>
                <div class="form-group">
                    <label>循环频率:</label>
                    <select id="rec-plan-interval">
                        <option value="daily">每日</option>
                        <option value="weekly">每周</option>
                    </select>
                </div>
                <div class="form-group hidden" id="rec-plan-weekly-box">
                    <select id="rec-plan-day-w"><option value="1">周一</option><option value="2">周二</option><option value="3">周三</option><option value="4">周四</option><option value="5">周五</option><option value="6">周六</option><option value="7">周日</option></select>
                </div>
                <div class="form-group"><label>提醒时间点 (可选, 例 19:00):</label><input type="time" id="rec-plan-time"></div>
                
                <label>动作列表:</label>
                <div id="ex-inputs">
                    <div class="exercise-group"><input type="text" class="ex-name" placeholder="动作名"><input type="text" class="ex-det" placeholder="组数/详情"></div>
                </div>
                <button type="button" id="addExBtn" style="background:#e0f2fe;color:#0369a1;border:none;padding:5px 10px;border-radius:99px;cursor:pointer;margin-bottom:10px">+ 增加动作</button>
                <br>
                <button type="submit" class="btn-submit">保存计划</button>
            </form>
        </div>
      </div>

      <!-- 5. Mobile Config -->
      <div id="mobile-config" class="content-section">
        <div class="panel">
            <h2>大屏/手机配置</h2>
            <iframe src="/mobile_config" style="width:100%;height:600px;border:none;border-radius:12px;"></iframe>
        </div>
      </div>
      
      <!-- 6. User Mgmt -->
      <div id="user-management" class="content-section">
          <div class="panel">
              <h2>用户管理</h2>
              <div id="userList"></div>
              <h3>+ 新增用户</h3>
              <form id="addUserForm">
                  <div class="form-group"><input type="text" id="new-u-name" placeholder="用户名" required></div>
                  <div class="form-group"><input type="password" id="new-u-pass" placeholder="密码" required></div>
                  <label><input type="checkbox" id="new-u-admin"> 设为管理员 (仅超管可用)</label>
                  <br><br>
                  <button type="submit" class="btn-submit">创建</button>
              </form>
          </div>
      </div>

      <!-- 7. Stats -->
      <div id="stats" class="content-section">
          <div class="panel">
              <h2>统计分析</h2>
              <div class="form-group"><label>起止日期:</label><div style="display:flex;gap:10px"><input type="date" id="stats-s"><input type="date" id="stats-e"></div></div>
              <button class="btn-submit" onclick="loadStats()">刷新统计</button>
              <br><br>
              <canvas id="statsChart"></canvas>
              <div id="statsTxt" style="margin-top:10px;color:#666"></div>
          </div>
      </div>

      <!-- 8. Profile -->
      <div id="profile" class="content-section">
          <div class="panel">
              <h2>个人资料</h2>
              <form id="profileForm">
                  <div class="form-group"><input type="text" id="p-name" placeholder="新用户名 (留空不改)"></div>
                  <div class="form-group"><input type="password" id="p-pass" placeholder="新密码 (留空不改)"></div>
                  <button type="submit" class="btn-submit">更新资料</button>
              </form>
          </div>
      </div>

    </main>
  </div>

<script>
// === Common ===
const todayStr = new Date().toLocaleDateString('zh-CN', {year:'numeric',month:'2-digit',day:'2-digit'}).replace(/\//g, '-');
const api = async (u,o) => { const r=await fetch(u,{credentials:'include',...o}); return r.json(); }
function msg(t) { const e=document.getElementById('globalAlert'); e.textContent=t; e.style.display='block'; setTimeout(()=>e.style.display='none',3000); }

// === Init UI ===
document.querySelectorAll('.nav-button').forEach(b => b.addEventListener('click', () => {
    document.querySelectorAll('.nav-button').forEach(x=>x.classList.remove('active'));
    document.querySelectorAll('.content-section').forEach(x=>x.classList.remove('active'));
    b.classList.add('active');
    document.getElementById(b.dataset.tab).classList.add('active');
    refreshAll();
}));

document.addEventListener('DOMContentLoaded', ()=>{
    ['date-todo','date-daily','rec-todo-start','rec-plan-start','stats-s','stats-e'].forEach(i=>{
        const el=document.getElementById(i); if(el) el.value=todayStr;
    });
    refreshAll();
});

function refreshAll() {
    const d = document.getElementById('date-todo').value;
    const dd = document.getElementById('date-daily').value;
    loadTodos(d);
    loadFitness(dd);
    loadRecTodos();
    loadRecPlans();
    loadUsers();
}

// === Logic ===
// 1. Todos
async function loadTodos(d) {
    const l = await api(`/todos?date=${d}`);
    document.getElementById('todoList').innerHTML = l.map(t => `
        <div class="list-item">
            <div>
                [${t.is_completed?'OK':'..'}] ${t.content} 
                ${t.reminder_at ? `<span class="tag-time">🔔 ${t.reminder_at.slice(5)}</span>` : ''}
                ${t.recurring_todo_id ? '<small style="color:#999">(R)</small>' : ''}
            </div>
            <button class="btn-danger" onclick="del('delete_todo',${t.id})">删</button>
        </div>
    `).join('') || '<div style="color:#ccc;padding:10px">无待办</div>';
}
document.getElementById('date-todo').onchange = (e)=>loadTodos(e.target.value);
document.getElementById('addTodoForm').onsubmit = async (e) => {
    e.preventDefault();
    let rem = document.getElementById('todo-reminder').value; if(rem) rem=rem.replace('T',' ');
    await api('/add_todo', {method:'POST', headers:{'Content-Type':'application/json'}, body:JSON.stringify({
        date: document.getElementById('date-todo').value,
        content: document.getElementById('todo-content').value,
        reminder_at: rem
    })});
    e.target.reset(); loadTodos(document.getElementById('date-todo').value); msg('添加成功');
};

// 2. Daily Fitness
async function loadFitness(d) {
    const l = await api(`/fitness_plan/${d}`);
    document.getElementById('dailyFitnessList').innerHTML = l.map(p => `
        <div class="list-item">
            <div>
                [${p.is_completed?'OK':'..'}] ${p.exercise_name} - ${p.sets_reps_duration}
                ${p.reminder_at ? `<span class="tag-time">⏰ ${p.reminder_at.slice(5)}</span>` : ''}
            </div>
            <button class="btn-danger" onclick="del('delete_fitness_plan_exercise',${p.id})">删</button>
        </div>
    `).join('') || '<div style="color:#ccc;padding:10px">无计划</div>';
}
document.getElementById('date-daily').onchange = (e)=>loadFitness(e.target.value);
document.getElementById('addManualFitForm').onsubmit = async (e) => {
    e.preventDefault();
    let rem = document.getElementById('man-fit-rem').value; if(rem) rem=rem.replace('T',' ');
    await api('/add_fitness_manual', {method:'POST', headers:{'Content-Type':'application/json'}, body:JSON.stringify({
        date: document.getElementById('date-daily').value,
        exercise_name: document.getElementById('man-fit-name').value,
        sets_reps_duration: document.getElementById('man-fit-det').value,
        reminder_at: rem
    })});
    e.target.reset(); loadFitness(document.getElementById('date-daily').value); msg('添加成功');
};

// 3. Rec Todos
async function loadRecTodos() {
    const l = await api('/get_recurring_todos');
    document.getElementById('recurringTodoList').innerHTML = l.map(t => `
        <div class="list-item">
            <div>
                ${t.content} [${t.repeat_interval}]
                ${t.reminder_time ? `<span class="tag-time">🔔 ${t.reminder_time}</span>` : ''}
            </div>
            <button class="btn-danger" onclick="del('delete_recurring_todo',${t.id})">删</button>
        </div>
    `).join('');
}
document.getElementById('rec-todo-interval').onchange = (e) => {
    document.getElementById('rec-todo-weekly-box').classList.toggle('hidden', e.target.value!=='weekly');
    document.getElementById('rec-todo-monthly-box').classList.toggle('hidden', e.target.value!=='monthly');
};
document.getElementById('addRecurringTodoForm').onsubmit = async (e) => {
    e.preventDefault();
    await api('/add_recurring_todo', {method:'POST', headers:{'Content-Type':'application/json'}, body:JSON.stringify({
        content: document.getElementById('rec-todo-content').value,
        start_date: document.getElementById('rec-todo-start').value,
        repeat_interval: document.getElementById('rec-todo-interval').value,
        repeat_day: document.getElementById('rec-todo-interval').value==='weekly' ? document.getElementById('rec-todo-day-w').value : document.getElementById('rec-todo-day-m').value,
        reminder_time: document.getElementById('rec-todo-time').value
    })});
    e.target.reset(); loadRecTodos(); msg('保存成功');
};

// 4. Rec Fitness
async function loadRecPlans() {
    const l = await api('/get_recurring_plans');
    document.getElementById('recurringPlanList').innerHTML = l.map(p => `
        <div class="list-item">
            <div>
                ${p.plan_name} [${p.repeat_interval}]
                ${p.reminder_time ? `<span class="tag-time">⏰ ${p.reminder_time}</span>` : ''}
            </div>
            <button class="btn-danger" onclick="del('delete_recurring_plan',${p.id})">删</button>
        </div>
    `).join('');
}
document.getElementById('rec-plan-interval').onchange = (e) => {
    document.getElementById('rec-plan-weekly-box').classList.toggle('hidden', e.target.value!=='weekly');
};
document.getElementById('addExBtn').onclick = () => {
    document.getElementById('ex-inputs').insertAdjacentHTML('beforeend', `<div class="exercise-group"><input type="text" class="ex-name" placeholder="动作名"><input type="text" class="ex-det" placeholder="组数"></div>`);
};
document.getElementById('addRecurringPlanForm').onsubmit = async (e) => {
    e.preventDefault();
    const exs = [...document.querySelectorAll('.ex-name')].map((el,i)=>({name:el.value, details:document.querySelectorAll('.ex-det')[i].value})).filter(x=>x.name);
    await api('/add_recurring_plan', {method:'POST', headers:{'Content-Type':'application/json'}, body:JSON.stringify({
        plan_name: document.getElementById('rec-plan-name').value,
        start_date: document.getElementById('rec-plan-start').value,
        repeat_interval: document.getElementById('rec-plan-interval').value,
        repeat_day: document.getElementById('rec-plan-day-w').value,
        reminder_time: document.getElementById('rec-plan-time').value,
        exercises: exs
    })});
    e.target.reset(); document.getElementById('ex-inputs').innerHTML='<div class="exercise-group"><input type="text" class="ex-name" placeholder="动作名"><input type="text" class="ex-det" placeholder="组数"></div>';
    loadRecPlans(); msg('保存成功');
};

// 5. Users & Stats & Profile
async function loadUsers(){ try{const u=await api('/api/users'); document.getElementById('userList').innerHTML=u.map(x=>`<div class="list-item"><span>${x.username} ${x.is_admin?'(管)':''}</span><button class="btn-danger" onclick="delUser(${x.id})">删</button></div>`).join('');}catch(e){document.getElementById('userMgmtBtn').style.display='none';} }
document.getElementById('addUserForm').onsubmit=async(e)=>{e.preventDefault(); await api('/api/users',{method:'POST',headers:{'Content-Type':'application/json'},body:JSON.stringify({username:document.getElementById('new-u-name').value,password:document.getElementById('new-u-pass').value,is_admin:document.getElementById('new-u-admin').checked?1:0})}); e.target.reset(); loadUsers();}
document.getElementById('profileForm').onsubmit=async(e)=>{e.preventDefault(); await api('/api/profile',{method:'POST',headers:{'Content-Type':'application/json'},body:JSON.stringify({username:document.getElementById('p-name').value,password:document.getElementById('p-pass').value})}); msg('资料更新');}
let chart=null;
async function loadStats(){
    const d=await api(`/api/stats/daily_completion?start_date=${document.getElementById('stats-s').value}&end_date=${document.getElementById('stats-e').value}`);
    const lbls=Object.keys(d.todo_data).sort();
    if(chart) chart.destroy();
    chart=new Chart(document.getElementById('statsChart'),{type:'line',data:{labels:lbls,datasets:[{label:'待办',data:lbls.map(k=>d.todo_data[k].completed),borderColor:'#22c55e'},{label:'健身',data:lbls.map(k=>d.fitness_data[k].completed),borderColor:'#0ea5e9'}]}});
    document.getElementById('statsTxt').innerText = `区间内完成情况已更新`;
}

async function del(ep, id) { if(confirm('删除?')) { await api(`/${ep}/${id}`, {method:'POST'}); refreshAll(); } }
async function delUser(id) { if(confirm('删用户?')) { await api(`/api/users/${id}`, {method:'DELETE'}); loadUsers(); } }
</script>
</body>
</html>
'''

# ==============================================================================
# 3. frontend/templates/display.html (v1.9 - Restore Look & Feel + Alerts)
# ==============================================================================
display_html_content = r'''<!DOCTYPE html>
<html lang="zh-CN">
<head>
  <meta charset="utf-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1.0" />
  <title>智能看板</title>
  <style>
    :root{ --bg:#0f1724; --panel:#1e293b; --accent:#4CAF50; --muted:#cbd5e1; --card-radius:18px; --font-base:56px; --alert-color: #ff5252; }
    html,body{ height:100%;margin:0;padding:0;background:var(--bg);color:#fff; font-family:sans-serif; overflow:hidden; }
    .app{height:100vh;display:flex;flex-direction:column;}
    .header{ display:flex;justify-content:space-between;align-items:center; padding:28px 48px; background:linear-gradient(90deg,#071029,#0b2540); box-shadow:0 8px 20px rgba(0,0,0,0.6);flex-shrink:0; }
    .brand{font-size:calc(var(--font-base)*0.6);color:var(--accent);font-weight:700;}
    .datetime{text-align:right;font-size:calc(var(--font-base)*0.38);color:#dbeafe; cursor: pointer; user-select: none; }
    .datetime .time{font-size:calc(var(--font-base)*0.9);font-weight:700;}
    .main{ padding:28px;display:flex;gap:28px;flex:1;box-sizing:border-box; }
    .panel{ background:var(--panel);border-radius:var(--card-radius);padding:28px; box-shadow:0 12px 30px rgba(0,0,0,0.55); display:flex;flex-direction:column;min-height:0; flex:1; }
    h2{ margin:0 0 18px 0;color:var(--accent); font-size:calc(var(--font-base)*0.48); border-bottom:3px solid rgba(255,255,255,0.03); padding-bottom:12px; }
    .scroll-container{ position:relative;overflow:hidden;flex:1;min-height:120px; }
    .scroll-content{ position:absolute;top:0;left:0;width:100%;will-change:transform; }
    .item{ display:flex;justify-content:space-between;align-items:center; padding:28px 24px; border-bottom:1px solid rgba(255,255,255,0.03); gap:20px; }
    .item .title{ font-size:calc(var(--font-base)*0.65);font-weight:800; }
    .item .meta{ font-size:calc(var(--font-base)*0.4);color:var(--muted); }
    .item.completed{background:rgba(76,175,80,0.08);}
    .item.completed .title{text-decoration:line-through;color:#a7f3d0;}
    
    /* Alert Style */
    .item.alert { background: rgba(255, 82, 82, 0.15); border-left: 10px solid var(--alert-color); animation: pulse 2s infinite; }
    .item.alert .title { color: #ff8a80; }
    @keyframes pulse { 0% { opacity: 1; } 50% { opacity: 0.7; } 100% { opacity: 1; } }

    .btn{ background:#6a1b9a;color:white;border:none; padding:14px 20px;border-radius:12px; cursor:pointer;font-weight:700; font-size:calc(var(--font-base)*0.4); }
    .placeholder{ color:#94a3b8;font-style:italic; padding:28px;font-size:calc(var(--font-base)*0.33); }
  </style>
</head>
<body>
  <div class="app">
    <div class="header">
      <div class="brand">智能健身看板</div>
      <div class="datetime" id="clockArea">
        <div id="currentDate">...</div>
        <div class="time" id="currentTime">...</div>
      </div>
    </div>
    <div class="main">
      <div class="panel">
        <h2>今日健身</h2>
        <div id="fitContainer" class="scroll-container"><div id="fitContent" class="scroll-content"></div></div>
      </div>
      <div class="panel">
        <h2>今日待办</h2>
        <div id="todoContainer" class="scroll-container"><div id="todoContent" class="scroll-content"></div></div>
      </div>
    </div>
  </div>

<script>
const API = location.origin.includes('localhost') ? 'http://localhost:8000' : location.origin;
const TODAY = () => new Date().toLocaleDateString('zh-CN', {year:'numeric',month:'2-digit',day:'2-digit'}).replace(/\//g, '-');

// 1. Clock & Triple Click Logic
let clicks = 0;
document.getElementById('clockArea').addEventListener('click', () => {
    clicks++;
    if(clicks === 1) setTimeout(() => clicks = 0, 800); 
    if(clicks >= 3) window.location.href = '/manage';
});
setInterval(() => {
    const now = new Date();
    document.getElementById('currentDate').innerText = now.toLocaleDateString('zh-CN',{weekday:'long', month:'long', day:'numeric'});
    document.getElementById('currentTime').innerText = now.toLocaleTimeString('zh-CN',{hour12:false});
    checkAlerts(); 
}, 1000);

// 2. Audio Logic
const audioCtx = new (window.AudioContext || window.webkitAudioContext)();
function beep() {
    if(audioCtx.state === 'suspended') audioCtx.resume();
    const osc = audioCtx.createOscillator();
    const gain = audioCtx.createGain();
    osc.connect(gain);
    gain.connect(audioCtx.destination);
    osc.frequency.value = 880; 
    gain.gain.value = 0.1;
    osc.start();
    setTimeout(() => osc.stop(), 200);
}

// 3. Data Logic
let todos = [], fits = [];
let SCROLL_SPEED = 25; 

async function loadData() {
    try {
        const [r1, r2, r3] = await Promise.all([
            fetch(`${API}/fitness_plan/${TODAY()}`, {credentials:'include'}),
            fetch(`${API}/todos?date=${TODAY()}`, {credentials:'include'}),
            fetch(`${API}/api/settings/scroll_speed`)
        ]);
        if(r1.status===401) return window.location.href='/';
        
        fits = await r1.json();
        todos = await r2.json();
        const s = await r3.json(); if(s.scroll_speed) SCROLL_SPEED = s.scroll_speed;

        render('fitContent', fits, false);
        render('todoContent', todos, true);
        
        restartScroll('fitContainer', 'fitContent');
        restartScroll('todoContainer', 'todoContent');
    } catch(e) { console.error(e); }
}

function render(id, list, isTodo) {
    const el = document.getElementById(id);
    if(!list.length) { el.innerHTML='<div class="placeholder">暂无内容</div>'; return; }
    
    const now = new Date().getTime();
    
    el.innerHTML = list.map(i => {
        let alertClass = '';
        // Alert logic: Not completed AND has time AND time passed
        if(!i.is_completed && i.reminder_at && now >= new Date(i.reminder_at).getTime()) {
            alertClass = 'alert';
        }
        
        return `
      <div class="item ${i.is_completed ? 'completed' : ''} ${alertClass}">
        <div>
          <div class="title">
            ${isTodo ? i.content : i.exercise_name}
            ${i.reminder_at ? `<span style="font-size:0.6em;opacity:0.8">⏰ ${i.reminder_at.slice(11,16)}</span>` : ''}
          </div>
          <div class="meta">${isTodo ? '' : i.sets_reps_duration}</div>
        </div>
        <button class="btn" onclick="toggle(${i.id}, ${isTodo}, ${i.is_completed})">
            ${i.is_completed ? '已完成' : '标记'}
        </button>
      </div>`;
    }).join('');
}

async function toggle(id, isTodo, state) {
    const ep = isTodo ? 'mark_todo_completed' : 'mark_exercise_completed_mobile';
    const body = isTodo ? {todo_id:id, is_completed:!state} : {plan_id:id, is_completed:!state};
    await fetch(`${API}/${ep}`, {method:'POST', headers:{'Content-Type':'application/json'}, body:JSON.stringify(body), credentials:'include'});
    loadData();
}

// 4. Alert Loop
let lastBeep = 0;
function checkAlerts() {
    const now = new Date().getTime();
    // Check both todos and fitness plans for alerts
    const hasTodoAlert = todos.some(t => !t.is_completed && t.reminder_at && now >= new Date(t.reminder_at).getTime());
    const hasFitAlert = fits.some(f => !f.is_completed && f.reminder_at && now >= new Date(f.reminder_at).getTime());

    if(hasTodoAlert || hasFitAlert) {
        // Beep every 15 seconds
        if(now - lastBeep > 15000) {
            beep();
            lastBeep = now;
        }
        // Force re-render to show visual alert immediately if just crossed time
        if(!document.querySelector('.alert')) {
             render('fitContent', fits, false);
             render('todoContent', todos, true);
        }
    }
}

// 5. Smooth Scroll
const anims = {};
function restartScroll(conId, contentId) {
    if(anims[contentId]) cancelAnimationFrame(anims[contentId]);
    const con = document.getElementById(conId);
    const content = document.getElementById(contentId);
    
    // Reset
    content.style.transform = 'translateY(0px)';
    if(content.scrollHeight <= con.clientHeight) return;
    
    let pos = 0;
    let dir = 1;
    const speed = (content.scrollHeight - con.clientHeight) / (SCROLL_SPEED * 60); // pixels per frame
    
    function step() {
        pos += dir * speed;
        const max = content.scrollHeight - con.clientHeight;
        if(pos > max) { pos = max; dir = -1; setTimeout(() => { dir=1; pos=0; }, 2000); } // Simple reset loop
        content.style.transform = `translateY(-${pos}px)`;
        anims[contentId] = requestAnimationFrame(step);
    }
    anims[contentId] = requestAnimationFrame(step);
}

loadData();
setInterval(loadData, 10000);
</script>
</body>
</html>
'''

def write_file(filename, content, folder="."):
    filepath = os.path.join(folder, filename)
    try:
        with open(filepath, 'w', encoding='utf-8') as f: f.write(content)
        print(f"[OK] {filepath}")
    except Exception as e: print(f"[ERR] {filepath}: {e}")

def main():
    print("=== V1.9 Restoring & Upgrading... ===")
    write_file('app.py', app_py_content)
    td = "templates" if os.path.exists("templates") else ("frontend/templates" if os.path.exists("frontend/templates") else ".")
    write_file('manage.html', manage_html_content, td)
    write_file('display.html', display_html_content, td)
    print("\nDone. Restart python app.py.")

if __name__ == '__main__': main()
