import os

# ==============================================================================
# 1. backend/app.py (v4.0 - MQTT + Health Analysis)
# ==============================================================================
app_py_content = r'''# backend/app.py (v4.0 - IoT Health Center)
import sqlite3
from datetime import datetime, timedelta, timezone
from flask import Flask, render_template, request, jsonify, g
from flask_login import LoginManager, UserMixin, login_user, logout_user, login_required, current_user
import os, json, uuid
from flask_cors import CORS
import sys 
from collections import defaultdict
import threading
import paho.mqtt.client as mqtt

CN_TZ = timezone(timedelta(hours=8))

# --- Path Setup ---
is_frozen = getattr(sys, 'frozen', False)
is_in_onefile_temp = 'onefile_' in os.path.abspath(__file__) 
if is_frozen or is_in_onefile_temp:
    base_dir = os.path.dirname(os.path.abspath(sys.executable))
    template_dir = os.path.join(base_dir, 'templates')
    static_dir = os.path.join(base_dir, 'static')
    DATABASE = os.path.join(os.path.dirname(os.path.realpath(sys.argv[0])), 'fitness_manager.db')
else:
    base_dir = os.path.dirname(os.path.abspath(__file__))
    project_root = os.path.dirname(base_dir)
    if os.path.exists(os.path.join(project_root, 'frontend', 'templates')):
        template_dir = os.path.join(project_root, 'frontend', 'templates')
        static_dir = os.path.join(project_root, 'frontend', 'static')
    else:
        template_dir = os.path.join(base_dir, 'templates')
        static_dir = os.path.join(base_dir, 'static')
    DATABASE = os.path.join(base_dir, 'fitness_manager.db')

if not os.path.exists(template_dir): os.makedirs(template_dir, exist_ok=True)

app = Flask(__name__, template_folder=template_dir, static_folder=static_dir)
CORS(app)
app.secret_key = str(uuid.uuid4())

login_manager = LoginManager()
login_manager.init_app(app)
login_manager.login_view = 'index' 

# --- User Model Enhanced ---
class User(UserMixin):
    def __init__(self, id, username, is_admin, manager_id, age=None, gender=None, weight=None, height=None):
        self.id = id; self.username = username; self.is_admin = is_admin; self.manager_id = manager_id
        self.age = age; self.gender = gender; self.weight = weight; self.height = height
    @property
    def is_super_admin(self): return self.is_admin and (self.manager_id is None or self.manager_id == 0)

@login_manager.user_loader
def load_user(uid):
    row = get_db().execute("SELECT * FROM users WHERE id=?", (uid,)).fetchone()
    if row:
        return User(row['id'], row['username'], row['is_admin'], row['manager_id'], 
                    row.get('age'), row.get('gender'), row.get('weight'), row.get('height'))
    return None

def get_db():
    db = getattr(g, '_database', None)
    if db is None: db = g._database = sqlite3.connect(DATABASE, check_same_thread=False); db.row_factory = sqlite3.Row
    return db

def set_setting(key, value):
    db = get_db()
    db.execute("INSERT INTO settings (key, value) VALUES (?, ?) ON CONFLICT(key) DO UPDATE SET value = excluded.value", (key, str(value)))
    db.commit()

def get_setting(key, default=None):
    row = get_db().execute("SELECT value FROM settings WHERE key = ?", (key,)).fetchone()
    return default if row is None else row["value"]

@app.teardown_appcontext
def close_conn(e):
    db = getattr(g, '_database', None)
    if db is not None: db.close()

# --- Database Init & Migration ---
def init_db():
    db = get_db()
    cur = db.cursor()
    # Base Tables
    cur.execute("CREATE TABLE IF NOT EXISTS todos (id INTEGER PRIMARY KEY AUTOINCREMENT, date TEXT NOT NULL, content TEXT NOT NULL, is_completed INTEGER DEFAULT 0, user_id INTEGER, recurring_todo_id INTEGER, reminder_at TEXT)")
    cur.execute("CREATE TABLE IF NOT EXISTS fitness_plans (id INTEGER PRIMARY KEY AUTOINCREMENT, date TEXT NOT NULL, exercise_name TEXT NOT NULL, sets_reps_duration TEXT, is_completed INTEGER DEFAULT 0, user_id INTEGER, recurring_plan_id INTEGER, reminder_at TEXT)")
    cur.execute("CREATE TABLE IF NOT EXISTS recurring_plans (id INTEGER PRIMARY KEY AUTOINCREMENT, user_id INTEGER, plan_name TEXT NOT NULL, exercises_json TEXT NOT NULL, start_date TEXT NOT NULL, end_date TEXT, repeat_interval TEXT NOT NULL, repeat_day INTEGER, reminder_time TEXT)")
    cur.execute("CREATE TABLE IF NOT EXISTS recurring_todos (id INTEGER PRIMARY KEY AUTOINCREMENT, user_id INTEGER, content TEXT NOT NULL, start_date TEXT NOT NULL, end_date TEXT, repeat_interval TEXT NOT NULL, repeat_day INTEGER, reminder_time TEXT)")
    cur.execute("CREATE TABLE IF NOT EXISTS exercise_library (id INTEGER PRIMARY KEY AUTOINCREMENT, user_id INTEGER, name TEXT, details TEXT)")
    cur.execute("CREATE TABLE IF NOT EXISTS todo_library (id INTEGER PRIMARY KEY AUTOINCREMENT, user_id INTEGER, content TEXT)")
    cur.execute("CREATE TABLE IF NOT EXISTS settings (key TEXT PRIMARY KEY, value TEXT)")
    
    # Users Table with Bio Data
    cur.execute("CREATE TABLE IF NOT EXISTS users (id INTEGER PRIMARY KEY AUTOINCREMENT, username TEXT UNIQUE, password TEXT, is_admin INTEGER DEFAULT 0, manager_id INTEGER, age INTEGER, gender TEXT, weight REAL, height REAL)")
    
    # Health Logs Table (New)
    cur.execute("""
        CREATE TABLE IF NOT EXISTS health_logs (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            user_id INTEGER,
            metric_type TEXT,  -- e.g., 'bp_sys', 'bp_dia', 'spo2', 'hr', 'glucose', 'water'
            value REAL,
            unit TEXT,
            analysis TEXT,     -- e.g., 'Normal', 'High', 'Low'
            recorded_at TEXT
        )
    """)

    # Migrations for missing columns
    try: cur.execute("SELECT reminder_at FROM todos LIMIT 1"); 
    except: cur.execute("ALTER TABLE todos ADD COLUMN reminder_at TEXT")
    try: cur.execute("SELECT reminder_at FROM fitness_plans LIMIT 1"); 
    except: cur.execute("ALTER TABLE fitness_plans ADD COLUMN reminder_at TEXT")
    try: cur.execute("SELECT reminder_time FROM recurring_plans LIMIT 1"); 
    except: cur.execute("ALTER TABLE recurring_plans ADD COLUMN reminder_time TEXT")
    try: cur.execute("SELECT reminder_time FROM recurring_todos LIMIT 1"); 
    except: cur.execute("ALTER TABLE recurring_todos ADD COLUMN reminder_time TEXT")
    
    # Migration for User Bio Data
    try: cur.execute("SELECT age FROM users LIMIT 1"); 
    except: 
        print("Migrating Users: Adding bio fields...")
        cur.execute("ALTER TABLE users ADD COLUMN age INTEGER")
        cur.execute("ALTER TABLE users ADD COLUMN gender TEXT")
        cur.execute("ALTER TABLE users ADD COLUMN weight REAL")
        cur.execute("ALTER TABLE users ADD COLUMN height REAL")

    # Seeds
    cur.execute("INSERT OR IGNORE INTO settings (key, value) VALUES ('scroll_speed', '25')")
    if cur.execute("SELECT COUNT(*) FROM users WHERE username='superuser'").fetchone()[0]==0:
        cur.execute("INSERT INTO users (username,password,is_admin,manager_id) VALUES (?,?,?,?)",('superuser','adminpassword',1,None))
    su_id = cur.execute("SELECT id FROM users WHERE username='superuser'").fetchone()['id']
    if cur.execute("SELECT COUNT(*) FROM users WHERE username='user1'").fetchone()[0]==0:
        cur.execute("INSERT INTO users (username,password,is_admin,manager_id) VALUES (?,?,?,?)",('user1','password',0,su_id))
    
    db.commit()

with app.app_context(): init_db()

# ==============================================================================
# HEALTH ANALYSIS ENGINE
# ==============================================================================
def analyze_health_data(metric, value, user_id):
    """
    Simple rule-based analysis based on standard medical data.
    Future: Fetch user age/gender to refine logic.
    """
    status = "Normal"
    
    # Fetch User Info for context (using a fresh connection since this runs in MQTT thread)
    try:
        with sqlite3.connect(DATABASE) as conn:
            conn.row_factory = sqlite3.Row
            user = conn.execute("SELECT age, gender, weight FROM users WHERE id=?", (user_id,)).fetchone()
            age = user['age'] if user and user['age'] else 30
            # gender = user['gender'] 
    except:
        age = 30 # Default

    if metric == 'bp_sys': # Systolic BP
        if value < 90: status = "Low (Hypotension)"
        elif value > 140: status = "High (Hypertension)" if age < 60 else "Elevated"
        elif value > 120: status = "Elevated"
    
    elif metric == 'bp_dia': # Diastolic BP
        if value < 60: status = "Low"
        elif value > 90: status = "High"
        elif value > 80: status = "Elevated"
        
    elif metric == 'spo2': # Blood Oxygen
        if value < 90: status = "Critical Low"
        elif value < 95: status = "Low"
        
    elif metric == 'glucose': # Blood Sugar (Assuming mg/dL)
        if value < 70: status = "Hypoglycemia"
        elif value > 140: status = "High (Post-Meal?)"
        
    elif metric == 'hr': # Heart Rate
        if value < 50: status = "Bradycardia (Low)"
        elif value > 100: status = "Tachycardia (High)"
        
    elif metric == 'temp': # Body Temp
        if value > 37.5: status = "Fever"
        elif value < 35.0: status = "Hypothermia"

    return status

# ==============================================================================
# MQTT CLIENT (Background Thread)
# ==============================================================================
def on_connect(client, userdata, flags, rc):
    print(f"Connected to MQTT Broker with result code {rc}")
    # Subscribe to all health topics: health/{user_id}/{metric}
    # Example: health/1/bp_sys
    client.subscribe("health/#")

def on_message(client, userdata, msg):
    try:
        topic_parts = msg.topic.split('/')
        # Expected format: health/<user_id>/<metric>
        if len(topic_parts) < 3: return
        
        try:
            user_id = int(topic_parts[1])
        except:
            user_id = 1 # Default to superuser if ID parsing fails
            
        metric = topic_parts[2]
        payload = msg.payload.decode()
        
        # Try parse JSON payload, otherwise treat as raw value
        try:
            data = json.loads(payload)
            val = float(data.get('value', 0))
            unit = data.get('unit', '')
        except:
            val = float(payload)
            unit = ""

        # Run Analysis
        analysis = analyze_health_data(metric, val, user_id)
        
        # Save to DB (New connection needed for thread)
        with sqlite3.connect(DATABASE) as conn:
            now_str = datetime.now(CN_TZ).strftime('%Y-%m-%d %H:%M:%S')
            conn.execute("INSERT INTO health_logs (user_id, metric_type, value, unit, analysis, recorded_at) VALUES (?,?,?,?,?,?)",
                         (user_id, metric, val, unit, analysis, now_str))
            # Optional: Create a Todo alert if critical?
            if "High" in analysis or "Critical" in analysis:
                content = f"⚠️ 健康警报: {metric} 数值 {val} ({analysis})"
                today = datetime.now(CN_TZ).strftime('%Y-%m-%d')
                conn.execute("INSERT INTO todos (date, content, is_completed, user_id, reminder_at) VALUES (?,?,0,?,?)",
                             (today, content, user_id, now_str))
                
        print(f"[MQTT] Saved {metric}: {val} ({analysis}) for User {user_id}")
        
    except Exception as e:
        print(f"[MQTT Error] {e}")

def start_mqtt():
    client = mqtt.Client()
    client.on_connect = on_connect
    client.on_message = on_message
    try:
        # Assuming Mosquitto is running locally on default port
        client.connect("localhost", 1883, 60)
        client.loop_start() # Runs in background thread
        print("MQTT Client Started")
    except Exception as e:
        print(f"WARNING: MQTT Connection Failed. is Mosquitto running? ({e})")

# Start MQTT on App Launch
start_mqtt()

# ==============================================================================
# EXISTING ROUTES (Kept Intact)
# ==============================================================================
# ... (Standard Auth, Index, Manage Routes - Same as before) ...

@app.route('/login', methods=['POST'])
def login_route():
    d = request.get_json(silent=True) or request.form
    u = get_db().execute("SELECT * FROM users WHERE username=?",(d.get('username'),)).fetchone()
    if u and u['password']==d.get('password'):
        login_user(User(u['id'],u['username'],u['is_admin'],u['manager_id'],u['age'],u['gender'],u['weight'],u['height']), remember=True)
        return jsonify({'status':'success'})
    return jsonify({'status':'error'}), 401

@app.route('/logout', methods=['POST'])
@login_required
def logout_route(): logout_user(); return jsonify({'status':'success'})

@app.route('/')
def index_route():
    if not current_user.is_authenticated: return render_template('login.html')
    if 'Mobile' in request.headers.get('User-Agent','') and request.args.get('view')!='display': return redirect(url_for('mobile_config'))
    today = datetime.now(CN_TZ).strftime('%Y-%m-%d')
    _gen_fit(current_user.id, today); _gen_todo(current_user.id, today)
    return render_template('display.html', username=current_user.username)

@app.route('/manage')
@login_required
def manage_route(): 
    return render_template('manage.html', username=current_user.username, is_admin=current_user.is_admin, 
                           user_info={'age':current_user.age, 'gender':current_user.gender, 'weight':current_user.weight, 'height':current_user.height})

@app.route('/mobile_config')
@login_required
def mobile_config_route(): return render_template('mobile_config.html', username=current_user.username)

# ... (Standard CRUD APIs - Todos, Fitness, Recurring - Same as v3.1) ...
@app.route('/todos', methods=['GET'])
@login_required
def get_todos():
    d = request.args.get('date') or datetime.now(CN_TZ).strftime('%Y-%m-%d')
    return jsonify([dict(r) for r in get_db().execute("SELECT * FROM todos WHERE date=? AND user_id=? ORDER BY reminder_at ASC, id ASC", (d, current_user.id)).fetchall()])

@app.route('/add_todo', methods=['POST'])
@login_required
def add_todo():
    d = request.get_json(silent=True) or request.form
    rem = d.get('reminder_at')
    if rem and len(rem)==5: rem = f"{d.get('date')} {rem}"
    get_db().execute("INSERT INTO todos (date,content,is_completed,user_id,reminder_at) VALUES (?,?,0,?,?)", (d.get('date'),d.get('content'),current_user.id,rem)).connection.commit()
    return jsonify(status='success')

@app.route('/add_recurring_todo', methods=['POST'])
@login_required
def add_rec_todo():
    d = request.get_json(silent=True) or request.form
    end_d = d.get('end_date'); 
    if not end_d: end_d=None
    cur = get_db().cursor()
    cur.execute("INSERT INTO recurring_todos (user_id,content,start_date,end_date,repeat_interval,repeat_day,reminder_time) VALUES (?,?,?,?,?,?,?)",
                (current_user.id, d.get('content'), d.get('start_date'), end_d, d.get('repeat_interval'), d.get('repeat_day'), d.get('reminder_time')))
    get_db().commit()
    _gen_single_rec_todo(current_user.id, datetime.now(CN_TZ).strftime('%Y-%m-%d'), cur.lastrowid)
    return jsonify(status='success')

@app.route('/fitness_plan/<date>', methods=['GET'])
@login_required
def get_fit(date):
    _gen_fit(current_user.id, date)
    return jsonify([dict(r) for r in get_db().execute("SELECT * FROM fitness_plans WHERE date=? AND user_id=? ORDER BY reminder_at ASC, id ASC", (date, current_user.id)).fetchall()])

@app.route('/add_recurring_plan', methods=['POST'])
@login_required
def add_rec_plan():
    d = request.get_json(silent=True) or request.form
    end_d = d.get('end_date'); 
    if not end_d: end_d=None
    cur = get_db().cursor()
    cur.execute("INSERT INTO recurring_plans (user_id,plan_name,exercises_json,start_date,end_date,repeat_interval,repeat_day,reminder_time) VALUES (?,?,?,?,?,?,?,?)",
                (current_user.id, d.get('plan_name'), json.dumps(d.get('exercises')), d.get('start_date'), end_d, d.get('repeat_interval'), d.get('repeat_day'), d.get('reminder_time')))
    get_db().commit()
    _gen_single_rec_plan(current_user.id, datetime.now(CN_TZ).strftime('%Y-%m-%d'), cur.lastrowid)
    return jsonify(status='success')

@app.route('/delete_todo/<int:id>', methods=['POST'])
@login_required
def del_todo(id): get_db().execute("DELETE FROM todos WHERE id=? AND user_id=?",(id,current_user.id)).connection.commit(); return jsonify(status='success')
@app.route('/mark_todo_completed', methods=['POST'])
@login_required
def mark_todo(): 
    d=request.get_json(silent=True) or request.form
    val = 1 if (d.get('is_completed') == True or d.get('is_completed') == 'true' or d.get('is_completed') == 1) else 0
    get_db().execute("UPDATE todos SET is_completed=? WHERE id=? AND user_id=?",(val,d.get('todo_id'),current_user.id)).connection.commit(); return jsonify(status='success')
@app.route('/delete_fitness_plan_exercise/<int:id>', methods=['POST'])
@login_required
def del_fit(id): get_db().execute("DELETE FROM fitness_plans WHERE id=? AND user_id=?",(id,current_user.id)).connection.commit(); return jsonify(status='success')
@app.route('/mark_exercise_completed_mobile', methods=['POST'])
@login_required
def mark_fit(): 
    d=request.get_json(silent=True) or request.form
    val = 1 if (d.get('is_completed') == True or d.get('is_completed') == 'true' or d.get('is_completed') == 1) else 0
    get_db().execute("UPDATE fitness_plans SET is_completed=? WHERE id=? AND user_id=?",(val,d.get('plan_id'),current_user.id)).connection.commit(); return jsonify(status='success')
@app.route('/get_recurring_todos', methods=['GET'])
@login_required
def get_rec_todos_list(): return jsonify([dict(r) for r in get_db().execute("SELECT * FROM recurring_todos WHERE user_id=?",(current_user.id,)).fetchall()])
@app.route('/delete_recurring_todo/<int:id>', methods=['POST'])
@login_required
def del_rec_todo(id): get_db().execute("DELETE FROM recurring_todos WHERE id=? AND user_id=?",(id,current_user.id)).connection.commit(); return jsonify(status='success')
@app.route('/get_recurring_plans', methods=['GET'])
@login_required
def get_rec_plans_list(): return jsonify([dict(r) for r in get_db().execute("SELECT * FROM recurring_plans WHERE user_id=?",(current_user.id,)).fetchall()])
@app.route('/delete_recurring_plan/<int:id>', methods=['POST'])
@login_required
def del_rec_plan(id): get_db().execute("DELETE FROM recurring_plans WHERE id=? AND user_id=?",(id,current_user.id)).connection.commit(); return jsonify(status='success')
@app.route('/add_fitness_manual', methods=['POST'])
@login_required
def add_fit_man():
    d=request.get_json(silent=True) or request.form
    rem = d.get('reminder_at')
    if rem and len(rem)==5: rem = f"{d.get('date')} {rem}"
    get_db().execute("INSERT INTO fitness_plans (date,exercise_name,sets_reps_duration,user_id,reminder_at) VALUES (?,?,?,?,?)",(d.get('date'),d.get('exercise_name'),d.get('sets_reps_duration'),current_user.id,rem)).connection.commit()
    return jsonify(status='success')
@app.route('/api/settings/scroll_speed', methods=['GET','POST'])
def scroll_speed():
    if request.method=='POST': 
        d=request.get_json(silent=True) or request.form
        try: v=int(d.get('scroll_speed'))
        except: v=25
        if v<1: v=1
        if v>120: v=120
        set_setting('scroll_speed', v); return jsonify(status='success', scroll_speed=v)
    try: val = int(get_setting('scroll_speed', 25))
    except: val = 25
    return jsonify(scroll_speed=val)
@app.route('/api/exercises', methods=['GET','POST'])
@app.route('/api/exercises/<int:id>', methods=['DELETE'])
@login_required
def ex_api(id=None):
    if request.method=='GET': return jsonify([dict(r) for r in get_db().execute("SELECT * FROM exercise_library WHERE user_id=?",(current_user.id,)).fetchall()])
    if request.method=='POST': d=request.get_json(silent=True) or request.form; get_db().execute("INSERT INTO exercise_library (user_id,name,details) VALUES (?,?,?)",(current_user.id,d.get('name'),d.get('details'))).connection.commit(); return jsonify(status='success')
    if request.method=='DELETE': get_db().execute("DELETE FROM exercise_library WHERE id=?",(id,)).connection.commit(); return jsonify(status='success')
@app.route('/api/todo_library', methods=['GET','POST'])
@app.route('/api/todo_library/<int:id>', methods=['DELETE'])
@login_required
def td_api(id=None):
    if request.method=='GET': return jsonify([dict(r) for r in get_db().execute("SELECT * FROM todo_library WHERE user_id=?",(current_user.id,)).fetchall()])
    if request.method=='POST': d=request.get_json(silent=True) or request.form; get_db().execute("INSERT INTO todo_library (user_id,content) VALUES (?,?)",(current_user.id,d.get('content'))).connection.commit(); return jsonify(status='success')
    if request.method=='DELETE': get_db().execute("DELETE FROM todo_library WHERE id=?",(id,)).connection.commit(); return jsonify(status='success')
def admin_required(f):
    @wraps(f)
    def dec(*args, **kwargs):
        if not current_user.is_authenticated or not current_user.is_admin: return jsonify(status='error'), 403
        return f(*args, **kwargs)
    return dec
@app.route('/api/users', methods=['GET','POST'])
@app.route('/api/users/<int:id>', methods=['DELETE'])
@login_required
@admin_required
def users_api(id=None):
    if request.method=='GET': return jsonify([dict(u) for u in get_db().execute("SELECT id,username,is_admin FROM users WHERE manager_id=?",(current_user.id,)).fetchall()])
    if request.method=='POST':
        d=request.get_json(silent=True) or request.form
        try: get_db().execute("INSERT INTO users (username,password,is_admin,manager_id) VALUES (?,?,?,?)",(d.get('username'),d.get('password'),1 if d.get('is_admin') else 0, current_user.id)).connection.commit(); return jsonify(status='success')
        except: return jsonify(status='error')
    if request.method=='DELETE':
        if id==current_user.id: return jsonify(status='error')
        get_db().execute("DELETE FROM users WHERE id=?",(id,)).connection.commit(); return jsonify(status='success')

# --- Health API (NEW) ---
@app.route('/api/health_logs', methods=['GET'])
@login_required
def get_health_logs():
    # Get recent logs for charts
    limit = request.args.get('limit', 50)
    rows = get_db().execute("SELECT * FROM health_logs WHERE user_id=? ORDER BY recorded_at DESC LIMIT ?", (current_user.id, limit)).fetchall()
    return jsonify([dict(r) for r in rows])

@app.route('/api/profile/bio', methods=['POST'])
@login_required
def update_bio():
    d = request.get_json(silent=True) or request.form
    get_db().execute("UPDATE users SET age=?, gender=?, weight=?, height=? WHERE id=?", 
                     (d.get('age'), d.get('gender'), d.get('weight'), d.get('height'), current_user.id))
    get_db().commit()
    return jsonify(status='success')

@app.route('/api/profile', methods=['POST'])
@login_required
def profile():
    d=request.get_json(silent=True) or request.form
    if d.get('password'): get_db().execute("UPDATE users SET password=? WHERE id=?",(d.get('password'),current_user.id))
    if d.get('username'): 
        try: get_db().execute("UPDATE users SET username=? WHERE id=?",(d.get('username'),current_user.id))
        except: pass
    get_db().commit(); return jsonify(status='success')

@app.route('/api/stats/daily_completion')
@login_required
def stats():
    s, e = request.args.get('start_date'), request.args.get('end_date')
    db = get_db()
    f = db.execute("SELECT date, is_completed FROM fitness_plans WHERE user_id=? AND date BETWEEN ? AND ?", (current_user.id, s, e)).fetchall()
    t = db.execute("SELECT date, is_completed FROM todos WHERE user_id=? AND date BETWEEN ? AND ?", (current_user.id, s, e)).fetchall()
    fd = defaultdict(lambda: {"total":0,"completed":0}); td = defaultdict(lambda: {"total":0,"completed":0})
    for r in f: fd[r['date']]['total']+=1; 
    for r in f: 
        if r['is_completed']: fd[r['date']]['completed']+=1
    for r in t: td[r['date']]['total']+=1; 
    for r in t:
        if r['is_completed']: td[r['date']]['completed']+=1
    return jsonify(fitness_data=fd, todo_data=td)

# ... (Generators kept same as v3.1) ...
def _gen_fit(uid, date_str):
    db = get_db()
    if db.execute("SELECT COUNT(*) FROM fitness_plans WHERE date=? AND user_id=?",(date_str,uid)).fetchone()[0] > 0: return
    plans = db.execute("SELECT * FROM recurring_plans WHERE user_id=? AND start_date<=? AND (end_date IS NULL OR end_date>=?)", (uid,date_str,date_str)).fetchall()
    dt = datetime.strptime(date_str, '%Y-%m-%d').date()
    for p in plans:
        if _is_due(p, dt): _insert_fit(db, uid, date_str, p)
    db.commit()
def _gen_todo(uid, date_str):
    db = get_db()
    if db.execute("SELECT COUNT(*) FROM todos WHERE date=? AND user_id=?",(date_str,uid)).fetchone()[0] > 0: return
    recs = db.execute("SELECT * FROM recurring_todos WHERE user_id=? AND start_date<=? AND (end_date IS NULL OR end_date>=?)", (uid,date_str,date_str)).fetchall()
    dt = datetime.strptime(date_str, '%Y-%m-%d').date()
    for r in recs:
        if _is_due(r, dt): _insert_todo(db, uid, date_str, r)
    db.commit()
def _gen_single_rec_todo(uid, date_str, rec_id):
    db = get_db()
    r = db.execute("SELECT * FROM recurring_todos WHERE id=?",(rec_id,)).fetchone()
    if not r: return
    dt = datetime.strptime(date_str, '%Y-%m-%d').date()
    if r['start_date'] <= date_str and (not r['end_date'] or r['end_date'] >= date_str):
        if _is_due(r, dt): _insert_todo(db, uid, date_str, r)
    db.commit()
def _gen_single_rec_plan(uid, date_str, rec_id):
    db = get_db()
    p = db.execute("SELECT * FROM recurring_plans WHERE id=?",(rec_id,)).fetchone()
    if not p: return
    dt = datetime.strptime(date_str, '%Y-%m-%d').date()
    if p['start_date'] <= date_str and (not p['end_date'] or p['end_date'] >= date_str):
        if _is_due(p, dt): _insert_fit(db, uid, date_str, p)
    db.commit()
def _is_due(item, dt):
    if item['repeat_interval'] == 'daily': return True
    if item['repeat_interval'] == 'weekly' and item['repeat_day'] == dt.isoweekday(): return True
    if item['repeat_interval'] == 'monthly' and item['repeat_day'] == dt.day: return True
    return False
def _insert_todo(db, uid, date_str, r):
    if db.execute("SELECT COUNT(*) FROM todos WHERE date=? AND recurring_todo_id=?",(date_str, r['id'])).fetchone()[0] > 0: return
    rem = f"{date_str} {r['reminder_time']}" if r['reminder_time'] else None
    db.execute("INSERT INTO todos (date, content, is_completed, user_id, recurring_todo_id, reminder_at) VALUES (?,?,?,?,?,?)",
               (date_str, r['content'], 0, uid, r['id'], rem))
def _insert_fit(db, uid, date_str, p):
    if db.execute("SELECT COUNT(*) FROM fitness_plans WHERE date=? AND recurring_plan_id=?",(date_str, p['id'])).fetchone()[0] > 0: return
    rem = f"{date_str} {p['reminder_time']}" if p['reminder_time'] else None
    for ex in json.loads(p['exercises_json']):
        db.execute("INSERT INTO fitness_plans (date, exercise_name, sets_reps_duration, is_completed, user_id, recurring_plan_id, reminder_at) VALUES (?,?,?,0,?,?,?)",
                   (date_str, ex['name'], ex['details'], uid, p['id'], rem))

@app.after_request
def add_headers(r):
    r.headers['Cache-Control'] = 'no-cache, no-store, must-revalidate'
    return r

@app.route('/health', methods=['GET'])
def health(): return jsonify(status='ok', time=datetime.now(CN_TZ).isoformat())

if __name__ == '__main__':
    app.run(host='0.0.0.0', port=8080, debug=True)
'''

# ==============================================================================
# 2. frontend/templates/manage.html (v4.0 - Added Health Tab)
# ==============================================================================
manage_html_content = r'''<!DOCTYPE html>
<html lang="zh-CN">
<head>
  <meta charset="utf-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1.0" />
  <title>健身看板 - 后台管理</title>
  <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
  <link href="https://fonts.googleapis.com/css2?family=Noto+Sans+SC:wght@300;400;500;600;700&display=swap" rel="stylesheet">
  <style>
    :root { --bg-color: #f1f5f9; --panel-color: #ffffff; --accent-color: #0f766e; --accent-color-light: #ccfbf1; --accent-color-soft: #ecfdf5; --accent-danger: #dc2626; --text-primary: #0f172a; --text-secondary: #64748b; --border-soft: #e2e8f0; --shadow-soft: 0 18px 40px rgba(15, 23, 42, 0.12); --radius-lg: 18px; --radius-md: 12px; }
    .hidden { display: none !important; }
    * { box-sizing: border-box; }
    body { margin: 0; font-family: 'Noto Sans SC', sans-serif; background-color: var(--bg-color); color: var(--text-primary); }
    .app-container { display: flex; min-height: 100vh; }
    .sidebar { width: 260px; background: linear-gradient(180deg, #0f172a, #020617); color: #e5e7eb; display: flex; flex-direction: column; padding: 24px 18px; box-shadow: 6px 0 25px rgba(15, 23, 42, 0.7); position: sticky; top: 0; height: 100vh; }
    .sidebar-header { display: flex; align-items: center; gap: 12px; margin-bottom: 32px; }
    .logo-circle { width: 40px; height: 40px; border-radius: 50%; background: radial-gradient(circle at 30% 20%, #a7f3d0, #16a34a); display: flex; align-items: center; justify-content: center; color: #022c22; font-weight: 800; font-size: 20px; }
    .nav-section-title { font-size: 12px; text-transform: uppercase; letter-spacing: 0.08em; color: #6b7280; margin: 18px 10px 8px; }
    .nav-list { list-style: none; margin: 0; padding: 0; flex: 1; }
    .nav-button { width: 100%; padding: 10px 12px; border-radius: 12px; border: none; background: transparent; color: #d1d5db; font-size: 14px; text-align: left; display: flex; align-items: center; cursor: pointer; transition: all 0.18s ease; margin-bottom: 4px; }
    .nav-button:hover { background: rgba(15, 23, 42, 0.8); transform: translateX(2px); }
    .nav-button.active { background: linear-gradient(90deg, rgba(34, 197, 94, 0.22), rgba(34, 197, 94, 0.04)); color: #ecfdf3; box-shadow: 0 10px 25px rgba(22, 163, 74, 0.5); }
    .nav-button.active::before { content: ''; position: absolute; left: 8px; top: 50%; width: 4px; height: 20px; border-radius: 999px; transform: translateY(-50%); background: #22c55e; }
    .nav-footer { margin-top: 12px; }
    .btn-display { text-decoration:none; font-weight:bold; color:#fff; background:#334155; display:block; text-align:center; padding:12px; border-radius:12px; margin-top:15px; }
    .main { flex: 1; padding: 24px 30px; max-width: calc(100vw - 260px); }
    .panel { background-color: var(--panel-color); border-radius: var(--radius-lg); box-shadow: var(--shadow-soft); padding: 20px 22px; margin-bottom: 18px; border: 1px solid rgba(148, 163, 184, 0.18); }
    .panel-header h2 { margin: 0; font-size: 18px; color: var(--text-primary); }
    .form-group { margin-bottom: 12px; }
    input, select, textarea { width: 100%; padding: 12px; border: 1px solid #cbd5e1; border-radius: var(--radius-md); font-size: 1rem; background: #f8fafc; }
    .btn-submit { background-color: var(--accent-color); color: #ecfdf5; border: none; padding: 10px 18px; border-radius: 999px; cursor: pointer; font-size: 14px; font-weight: 600; }
    .btn-danger { background-color: #ef4444; color: #fff; border: none; padding: 6px 12px; border-radius: 999px; cursor: pointer; font-size: 12px; }
    .list-item { padding: 10px 12px; border-radius: 10px; border: 1px solid var(--border-soft); margin-bottom: 8px; background: #f9fafb; display: flex; justify-content: space-between; align-items: center; gap: 10px; }
    .content-section { display: none; }
    .content-section.active { display: block; }
    .exercise-group { display: grid; grid-template-columns: 1fr 1fr; gap: 8px; margin-bottom: 8px; }
    .alert-success { padding: 10px; background: #ecfdf5; color: #166534; margin-bottom: 10px; display:none; border-radius: 8px; }
    .tag-time { background: #e0f2fe; color: #0369a1; padding: 2px 6px; border-radius: 4px; font-size: 11px; margin-left: 6px; font-weight: 600; }
    
    /* IoT Table */
    .iot-table { width: 100%; border-collapse: collapse; font-size: 13px; }
    .iot-table th, .iot-table td { padding: 10px; border-bottom: 1px solid #eee; text-align: left; }
    .iot-table th { background: #f8fafc; color: #64748b; font-weight: 600; }
    .status-Normal { color: #16a34a; font-weight: bold; }
    .status-High, .status-Fever { color: #dc2626; font-weight: bold; }
    .status-Low { color: #ea580c; font-weight: bold; }
  </style>
</head>
<body>
  <div class="app-container">
    <aside class="sidebar">
      <div class="sidebar-header"><div class="logo-circle">F</div><div><div style="font-weight:700">健身看板</div><div style="font-size:12px">后台管理</div></div></div>
      <div class="nav-section-title">任务</div>
      <ul class="nav-list">
        <li><button class="nav-button active" data-tab="daily-todos">✅ 每日待办</button></li>
        <li><button class="nav-button" data-tab="daily-fitness">💪 每日健身</button></li>
        <li><button class="nav-button" data-tab="recurring-todos">🔁 循环待办</button></li>
        <li><button class="nav-button" data-tab="recurring-fitness">📅 循环健身</button></li>
      </ul>
      <div class="nav-section-title">健康 & 系统</div>
      <ul class="nav-list">
        <li><button class="nav-button" data-tab="health">❤️ 健康档案 (IoT)</button></li>
        <li><button class="nav-button" data-tab="stats">📊 统计分析</button></li>
        <li><button class="nav-button" data-tab="mobile-config">📱 移动配置</button></li>
        <li id="userMgmtBtn"><button class="nav-button" data-tab="user-management">👥 用户管理</button></li>
        <li><button class="nav-button" data-tab="profile">⚙️ 个人资料</button></li>
      </ul>
      <div class="nav-footer"><a href="/?view=display" class="btn-display">🖥️ 切换大屏</a></div>
    </aside>

    <main class="main">
      <div class="top-bar"><h1>后台管理中心</h1><span>用户: {{ username }}</span></div>
      <div id="globalAlert" class="alert-success"></div>

      <!-- 1-4 Tasks (Collapsed for brevity, logic identical to v3.2) -->
      <div id="daily-todos" class="content-section active">
        <div class="panel">
            <div class="panel-header"><h2>每日待办 (Today)</h2></div>
            <div style="display:grid;grid-template-columns:1fr 1fr;gap:20px;">
                <div><div class="form-group"><label>日期:</label><input type="date" id="date-todo"></div><div id="todoList"></div></div>
                <div>
                    <h3>+ 待办</h3><form id="addTodoForm"><div class="form-group"><input id="todo-content" placeholder="内容" required></div><div class="form-group"><label>到期:</label><input type="datetime-local" id="todo-reminder"></div><button class="btn-submit">添加</button></form>
                </div>
            </div>
        </div>
      </div>
      <div id="daily-fitness" class="content-section"><div class="panel"><div class="panel-header"><h2>每日健身 (Today)</h2></div><div style="display:grid;grid-template-columns:1fr 1fr;gap:20px;"><div><div class="form-group"><input type="date" id="date-daily"></div><div id="dailyFitnessList"></div></div><div><h3>+ 训练</h3><form id="addManFitForm"><div class="form-group"><input id="mf-name" placeholder="动作" required></div><div class="form-group"><input id="mf-det" placeholder="详情"></div><div class="form-group"><label>到期:</label><input type="datetime-local" id="mf-rem"></div><button class="btn-submit">添加</button></form></div></div></div></div>
      <div id="recurring-todos" class="content-section"><div class="panel"><div class="panel-header"><h2>循环待办</h2></div><div id="recTodoList"></div><hr style="margin:15px 0"><form id="addRecTodoForm"><div class="form-group"><input id="rec-todo-content" placeholder="内容" required></div><div class="exercise-group"><input type="date" id="rec-todo-start"><input type="date" id="rec-todo-end"></div><div class="form-group"><label>提醒点:</label><input type="time" id="rec-todo-time"></div><div class="form-group"><select id="rec-todo-interval"><option value="daily">每日</option><option value="weekly">每周</option><option value="monthly">每月</option></select></div><div class="hidden" id="rec-todo-week-box"><select id="rec-todo-day-w"><option value="1">一</option><option value="2">二</option><option value="3">三</option><option value="4">四</option><option value="5">五</option><option value="6">六</option><option value="7">日</option></select></div><div class="hidden" id="rec-todo-month-box"><input type="number" id="rec-todo-day-m" min="1" max="31" value="1"></div><button class="btn-submit">保存</button></form></div></div>
      <div id="recurring-fitness" class="content-section"><div class="panel"><div class="panel-header"><h2>循环健身</h2></div><div id="recPlanList"></div><hr style="margin:15px 0"><form id="addRecPlanForm"><div class="form-group"><input id="rec-plan-name" placeholder="计划名" required></div><div class="exercise-group"><input type="date" id="rec-plan-start"><input type="date" id="rec-plan-end"></div><div class="form-group"><label>提醒点:</label><input type="time" id="rec-plan-time"></div><div class="form-group"><select id="rec-plan-interval"><option value="daily">每日</option><option value="weekly">每周</option></select></div><div class="hidden" id="rec-plan-week-box"><select id="rec-plan-day-w"><option value="1">一</option><option value="2">二</option><option value="3">三</option><option value="4">四</option><option value="5">五</option><option value="6">六</option><option value="7">日</option></select></div><div id="ex-container"><div class="exercise-group"><input class="ex-n" placeholder="动作"><input class="ex-d" placeholder="组数"></div></div><button type="button" id="addExBtn" class="btn-add-ex">+ 动作</button><br><br><button class="btn-submit">保存</button></form></div></div>

      <!-- ★★★ HEALTH TAB (NEW) ★★★ -->
      <div id="health" class="content-section">
        <div class="panel">
            <div class="panel-header"><h2>健康档案 (IoT Center)</h2></div>
            <div class="section-grid" style="display:grid;grid-template-columns:1fr 2fr;gap:20px">
                <!-- Bio Config -->
                <div class="profile-box">
                    <h3>身体参数设置</h3>
                    <p class="small-hint">系统将根据这些参数自动分析传感器数据。</p>
                    <form id="bioForm">
                        <div class="form-group"><label>年龄:</label><input type="number" id="bio-age" value="{{ user_info.age }}"></div>
                        <div class="form-group"><label>性别:</label><select id="bio-gender"><option value="M">男</option><option value="F">女</option></select></div>
                        <div class="exercise-group">
                            <div class="form-group"><label>身高(cm):</label><input type="number" id="bio-height" value="{{ user_info.height }}"></div>
                            <div class="form-group"><label>体重(kg):</label><input type="number" id="bio-weight" value="{{ user_info.weight }}"></div>
                        </div>
                        <button type="submit" class="btn-submit">保存参数</button>
                    </form>
                    <hr style="margin:15px 0;border:0;border-top:1px solid #eee">
                    <h3>MQTT 模拟测试</h3>
                    <p class="small-hint">如果没有传感器，可在此模拟发送数据。</p>
                    <button onclick="simMqtt('bp_sys', 135)" class="btn-add-ex">模拟 高血压(135)</button>
                    <button onclick="simMqtt('hr', 110)" class="btn-add-ex" style="margin-top:5px">模拟 心率过速(110)</button>
                </div>
                <!-- Logs -->
                <div>
                    <h3>最近接收的数据</h3>
                    <button class="btn-secondary" onclick="loadHealthLogs()" style="float:right;margin-top:-30px">刷新</button>
                    <div style="overflow-y:auto;max-height:500px;border:1px solid #eee;border-radius:8px;">
                        <table class="iot-table">
                            <thead><tr><th>时间</th><th>指标</th><th>数值</th><th>分析结果</th></tr></thead>
                            <tbody id="healthLogBody"><tr><td colspan="4">加载中...</td></tr></tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
      </div>

      <!-- 5-8 Configs -->
      <div id="stats" class="content-section"><div class="panel"><h2>统计</h2><canvas id="statsChart"></canvas></div></div>
      <div id="mobile-config" class="content-section"><div class="panel"><h2>配置</h2><iframe src="/mobile_config" style="width:100%;height:500px;border:0"></iframe></div></div>
      <div id="user-management" class="content-section"><div class="panel"><h2>用户管理</h2><div id="userList"></div><h3>新增用户</h3><form id="addUserForm"><div class="form-group"><input id="nu-name" placeholder="用户" required></div><div class="form-group"><input id="nu-pass" placeholder="密码" required></div><label><input type="checkbox" id="nu-admin"> 管理员</label><br><br><button class="btn-submit">创建</button></form></div></div>
      <div id="profile" class="content-section"><div class="panel"><h2>资料</h2><form id="profForm"><input id="pu-name" placeholder="新名"><br><input id="pu-pass" placeholder="新密"><br><button class="btn-submit">更新</button></form></div></div>

    </main>
  </div>

<script>
const todayStr = new Date().toISOString().slice(0,10);
const api = async (u, o) => (await fetch(u, {credentials:'include', ...o})).json();
function msg(t) { const e=document.getElementById('globalAlert'); e.textContent=t; e.style.display='block'; setTimeout(()=>e.style.display='none',3000); }

document.querySelectorAll('.nav-button').forEach(b => b.addEventListener('click', () => {
    if(b.classList.contains('btn-display-link')) return;
    document.querySelectorAll('.nav-button').forEach(x=>x.classList.remove('active'));
    document.querySelectorAll('.content-section').forEach(x=>x.classList.remove('active'));
    b.classList.add('active');
    document.getElementById(b.dataset.tab).classList.add('active');
    if(b.dataset.tab === 'health') loadHealthLogs(); // Load health data on tab switch
    else refreshAll();
}));

document.addEventListener('DOMContentLoaded', ()=>{
    ['date-todo','date-daily','rec-todo-start','rec-plan-start'].forEach(i=>{ const el=document.getElementById(i); if(el) el.value=todayStr; });
    document.getElementById('rec-todo-interval').dispatchEvent(new Event('change'));
    document.getElementById('rec-plan-interval').dispatchEvent(new Event('change'));
    // Pre-select gender if available
    const g = "{{ user_info.gender }}";
    if(g && g!=='None') document.getElementById('bio-gender').value = g;
    refreshAll();
});

function refreshAll() {
    loadTodos(document.getElementById('date-todo').value);
    loadFitness(document.getElementById('date-daily').value);
    loadRecTodos(); loadRecPlans(); loadUsers();
}

// --- Health Logic ---
async function loadHealthLogs() {
    const logs = await api('/api/health_logs');
    const tbody = document.getElementById('healthLogBody');
    if(logs.length === 0) { tbody.innerHTML = '<tr><td colspan="4">暂无数据，请等待设备上传。</td></tr>'; return; }
    tbody.innerHTML = logs.map(l => `
        <tr>
            <td>${l.recorded_at}</td>
            <td>${mapMetricName(l.metric_type)}</td>
            <td>${l.value} ${l.unit}</td>
            <td class="status-${l.analysis.split(' ')[0]}">${l.analysis}</td>
        </tr>
    `).join('');
}

function mapMetricName(t) {
    const map = {'bp_sys':'收缩压', 'bp_dia':'舒张压', 'hr':'心率', 'spo2':'血氧', 'glucose':'血糖', 'weight':'体重'};
    return map[t] || t;
}

document.getElementById('bioForm').onsubmit = async (e) => {
    e.preventDefault();
    await api('/api/profile/bio', {method:'POST', headers:{'Content-Type':'application/json'}, body:JSON.stringify({
        age: document.getElementById('bio-age').value,
        gender: document.getElementById('bio-gender').value,
        weight: document.getElementById('bio-weight').value,
        height: document.getElementById('bio-height').value
    })});
    msg('身体参数已保存');
}

// Simulator (For testing without hardware)
// Note: In real life, this would be sent via MQTT from a device, not via HTTP. 
// But for testing, we can trick it or just rely on manual DB entry if we wanted to write a simulation endpoint.
// Since we don't have a simulation endpoint, this button is just a placeholder for the user to know *how* to test.
// Actually, let's make a tiny helper endpoint in app.py to "inject" data for testing? 
// No, let's just instruct the user to use mosquitto_pub command line.
function simMqtt(topic, val) {
    alert(`请在终端运行: mosquitto_pub -t "health/{{ current_user.id }}/${topic}" -m "${val}"`);
}

// --- Original Logic (Collapsed) ---
async function loadTodos(d){ const l=await api(`/todos?date=${d}`); document.getElementById('todoList').innerHTML=l.map(t=>`<div class="list-item"><div>${t.is_completed?'✅':'⬜'} ${t.content}</div><button class="btn-danger" onclick="del('delete_todo',${t.id})">删</button></div>`).join(''); }
document.getElementById('date-todo').onchange=(e)=>loadTodos(e.target.value);
document.getElementById('addTodoForm').onsubmit=async(e)=>{e.preventDefault(); let rem=document.getElementById('todo-reminder').value.replace('T',' '); await api('/add_todo',{method:'POST',headers:{'Content-Type':'application/json'},body:JSON.stringify({date:document.getElementById('date-todo').value,content:document.getElementById('todo-content').value,reminder_at:rem})}); e.target.reset(); loadTodos(document.getElementById('date-todo').value);}
// ... (Rest of the logic for fitness, recs, stats, etc. same as v3.2) ...
async function loadFitness(d){ const l=await api(`/fitness_plan/${d}`); document.getElementById('dailyFitnessList').innerHTML=l.map(p=>`<div class="list-item"><div>${p.exercise_name}</div><button class="btn-danger" onclick="del('delete_fitness_plan_exercise',${p.id})">删</button></div>`).join(''); }
document.getElementById('date-daily').onchange=(e)=>loadFitness(e.target.value);
document.getElementById('addManFitForm').onsubmit=async(e)=>{e.preventDefault(); let rem=document.getElementById('mf-rem').value.replace('T',' '); await api('/add_fitness_manual',{method:'POST',headers:{'Content-Type':'application/json'},body:JSON.stringify({date:document.getElementById('date-daily').value,exercise_name:document.getElementById('mf-name').value,sets_reps_duration:document.getElementById('mf-det').value,reminder_at:rem})}); e.target.reset(); loadFitness(document.getElementById('date-daily').value);}
async function loadRecTodos(){ const l=await api('/get_recurring_todos'); document.getElementById('recTodoList').innerHTML=l.map(t=>`<div class="list-item"><div>${t.content}</div><button class="btn-danger" onclick="del('delete_recurring_todo',${t.id})">删</button></div>`).join(''); }
document.getElementById('addRecTodoForm').onsubmit=async(e)=>{e.preventDefault(); await api('/add_recurring_todo',{method:'POST',headers:{'Content-Type':'application/json'},body:JSON.stringify({content:document.getElementById('rec-todo-content').value,start_date:document.getElementById('rec-todo-start').value,end_date:document.getElementById('rec-todo-end').value||null,repeat_interval:document.getElementById('rec-todo-interval').value,repeat_day:document.getElementById('rec-todo-interval').value==='weekly'?document.getElementById('rec-todo-day-w').value:document.getElementById('rec-todo-day-m').value,reminder_time:document.getElementById('rec-todo-time').value})}); e.target.reset(); loadRecTodos();}
document.getElementById('rec-todo-interval').onchange=(e)=>{document.getElementById('rec-todo-week-box').classList.toggle('hidden',e.target.value!=='weekly');document.getElementById('rec-todo-month-box').classList.toggle('hidden',e.target.value!=='monthly');}
async function loadRecPlans(){ const l=await api('/get_recurring_plans'); document.getElementById('recPlanList').innerHTML=l.map(p=>`<div class="list-item"><div>${p.plan_name}</div><button class="btn-danger" onclick="del('delete_recurring_plan',${p.id})">删</button></div>`).join(''); }
document.getElementById('addExBtn').onclick=()=>{document.getElementById('ex-container').insertAdjacentHTML('beforeend','<div class="exercise-group"><input class="ex-n" placeholder="动作"><input class="ex-d" placeholder="组数"></div>')}
document.getElementById('addRecPlanForm').onsubmit=async(e)=>{e.preventDefault(); const exs=[...document.querySelectorAll('.ex-n')].map((x,i)=>({name:x.value,details:document.querySelectorAll('.ex-d')[i].value})).filter(x=>x.name); await api('/add_recurring_plan',{method:'POST',headers:{'Content-Type':'application/json'},body:JSON.stringify({plan_name:document.getElementById('rec-plan-name').value,start_date:document.getElementById('rec-plan-start').value,end_date:document.getElementById('rec-plan-end').value||null,repeat_interval:document.getElementById('rec-plan-interval').value,repeat_day:document.getElementById('rec-plan-day-w').value,reminder_time:document.getElementById('rec-plan-time').value,exercises:exs})}); e.target.reset(); loadRecPlans();}
document.getElementById('rec-plan-interval').onchange=(e)=>{document.getElementById('rec-plan-week-box').classList.toggle('hidden',e.target.value!=='weekly');}
async function loadUsers(){ try{const u=await api('/api/users'); document.getElementById('userList').innerHTML=u.map(x=>`<div class="list-item"><span>${x.username}</span><button class="btn-danger" onclick="delUser(${x.id})">删</button></div>`).join('');}catch(e){document.getElementById('userMgmtBtn').style.display='none';} }
document.getElementById('addUserForm').onsubmit=async(e)=>{e.preventDefault();await api('/api/users',{method:'POST',headers:{'Content-Type':'application/json'},body:JSON.stringify({username:document.getElementById('nu-name').value,password:document.getElementById('nu-pass').value,is_admin:document.getElementById('nu-admin').checked?1:0})});e.target.reset();loadUsers();}
async function delUser(id){if(confirm('Del?'))await api(`/api/users/${id}`,{method:'DELETE'});loadUsers();}
async function del(ep,id){if(confirm('Del?')){await api(`/${ep}/${id}`,{method:'POST'});refreshAll();}}
let chart; async function loadStats(){const d=await api(`/api/stats/daily_completion?start_date=${document.getElementById('stats-s').value}&end_date=${document.getElementById('stats-e').value}`);const l=Object.keys(d.todo_data).sort();if(chart)chart.destroy();chart=new Chart(document.getElementById('statsChart'),{type:'line',data:{labels:l,datasets:[{label:'Todo',data:l.map(k=>d.todo_data[k].completed),borderColor:'green'},{label:'Fit',data:l.map(k=>d.fitness_data[k].completed),borderColor:'blue'}]}});}
</script>
</body>
</html>
'''

def write_file(filename, content, folder="."):
    filepath = os.path.join(folder, filename)
    try:
        with open(filepath, 'w', encoding='utf-8') as f: f.write(content)
        print(f"[OK] Updated: {filepath}")
    except Exception as e: print(f"[ERR] Failed {filepath}: {e}")

def main():
    print("=== V4.0 IoT Health Upgrade ===")
    # 1. Backend
    backend_dir = "backend" if os.path.exists("backend") else "."
    write_file('app.py', app_py_content, backend_dir)
    
    # 2. Frontend
    td = "templates" if os.path.exists("templates") else ("frontend/templates" if os.path.exists("frontend/templates") else ".")
    write_file('manage.html', manage_html_content, td)
    
    print("\nUpdate Complete.")
    print("1. Ensure Mosquitto is installed: sudo apt install mosquitto")
    print("2. Ensure Python lib is installed: pip install paho-mqtt")
    print("3. Restart app: python backend/app.py")
    print("4. Test by sending MQTT message: mosquitto_pub -t 'health/1/hr' -m '110'")

if __name__ == '__main__': main()
