import paho.mqtt.client as mqtt
import time
import json
import random

# ================= 配置区域 =================
# 请修改为你的树莓派IP地址
BROKER_ADDRESS = "192.168.10.59" 
BROKER_PORT = 1883
USER_ID = 1  # 模拟的用户ID
# ===========================================

def on_connect(client, userdata, flags, rc, properties=None):
    if rc == 0:
        print(f"成功连接到 MQTT 服务器: {BROKER_ADDRESS}")
    else:
        print(f"连接失败，错误码: {rc}")

# 兼容不同版本的 paho-mqtt
try:
    client = mqtt.Client(mqtt.CallbackAPIVersion.VERSION1)
except AttributeError:
    client = mqtt.Client()

client.on_connect = on_connect

print("正在尝试连接 MQTT 服务器...")
try:
    client.connect(BROKER_ADDRESS, BROKER_PORT, 60)
except Exception as e:
    print(f"连接失败: {e}")
    print("请检查: 1. IP地址是否正确 2. Mosquitto服务是否已启动")
    exit()

client.loop_start()

def publish_data(metric, value, unit):
    """封装发送函数，自动打包JSON"""
    topic = f"health/{USER_ID}/{metric}"
    payload = json.dumps({
        "value": value,
        "unit": unit
    })
    client.publish(topic, payload)
    print(f"  [发送] {metric:<10} : {value} {unit}")

# 计时器
tick = 0

print(f"\n=== 全能传感器模拟器已启动 (用户ID: {USER_ID}) ===")
print("功能: 模拟正常数据，偶尔发送异常数据以触发报警")
print("------------------------------------------------")

try:
    while True:
        # 10% 的概率生成异常数据 (用于测试报警)
        is_abnormal = random.random() < 0.1
        
        # --- 1. 高频数据 (每 5 秒): 心率, 血氧 ---
        if tick % 5 == 0:
            # 心率 (正常 60-90, 异常 >110)
            hr = random.randint(110, 130) if is_abnormal else random.randint(60, 90)
            publish_data("hr", hr, "bpm")

            # 血氧 (正常 96-99, 异常 <92)
            spo2 = random.randint(88, 92) if is_abnormal else random.randint(96, 99)
            publish_data("spo2", spo2, "%")

        # --- 2. 中频数据 (每 10 秒): 血压, 体温 ---
        if tick % 10 == 0:
            # 血压 (正常 110/70, 异常 150/95)
            if is_abnormal:
                sys = random.randint(145, 160)
                dia = random.randint(95, 100)
            else:
                sys = random.randint(110, 125)
                dia = random.randint(70, 80)
            
            publish_data("bp_sys", sys, "mmHg")
            publish_data("bp_dia", dia, "mmHg")

            # 体温 (正常 36.5, 异常 38.0)
            temp = round(random.uniform(37.8, 38.5), 1) if is_abnormal else round(random.uniform(36.3, 37.0), 1)
            publish_data("temp", temp, "°C")

        # --- 3. 低频数据 (每 30 秒): 血糖, 体重, 饮水 ---
        if tick % 30 == 0:
            # 血糖 (正常 4.5-6.0, 异常 >8.0)
            glucose = round(random.uniform(7.5, 9.0), 1) if is_abnormal else round(random.uniform(4.5, 6.0), 1)
            publish_data("glucose", glucose, "mmol/L")

            # 体重 (波动)
            weight = round(random.uniform(70.0, 70.5), 1)
            publish_data("weight", weight, "kg")

            # 饮水 (模拟喝了一口水)
            water = 200
            publish_data("water", water, "ml")

        tick += 5
        time.sleep(5)

except KeyboardInterrupt:
    print("\n模拟器已停止")
    client.loop_stop()
    client.disconnect()
