import os

# ==============================================================================
# 1. backend/app.py (v3.0 - Unlock Speed Limit)
# ==============================================================================
app_py_content = r'''# backend/app.py (v3.0 - Speed Limit Unlocked)
import sqlite3
from datetime import datetime, timedelta, timezone
from flask import Flask, render_template, request, jsonify, g, redirect, url_for
from flask_login import LoginManager, UserMixin, login_user, logout_user, login_required, current_user
import os, json, uuid
from flask_cors import CORS
from functools import wraps
import sys 
from collections import defaultdict

CN_TZ = timezone(timedelta(hours=8))

# Path Setup
is_frozen = getattr(sys, 'frozen', False)
is_in_onefile_temp = 'onefile_' in os.path.abspath(__file__) 
if is_frozen or is_in_onefile_temp:
    base_dir = os.path.dirname(os.path.abspath(sys.executable))
    template_dir = os.path.join(base_dir, 'templates')
    static_dir = os.path.join(base_dir, 'static')
    DATABASE = os.path.join(os.path.dirname(os.path.realpath(sys.argv[0])), 'fitness_manager.db')
else:
    base_dir = os.path.dirname(os.path.abspath(__file__))
    project_root = os.path.dirname(base_dir)
    if os.path.exists(os.path.join(project_root, 'frontend', 'templates')):
        template_dir = os.path.join(project_root, 'frontend', 'templates')
        static_dir = os.path.join(project_root, 'frontend', 'static')
    else:
        template_dir = os.path.join(base_dir, 'templates')
        static_dir = os.path.join(base_dir, 'static')
    DATABASE = os.path.join(base_dir, 'fitness_manager.db')

if not os.path.exists(template_dir): os.makedirs(template_dir, exist_ok=True)

app = Flask(__name__, template_folder=template_dir, static_folder=static_dir)
CORS(app)
app.secret_key = str(uuid.uuid4())

login_manager = LoginManager()
login_manager.init_app(app)
login_manager.login_view = 'index' 

class User(UserMixin):
    def __init__(self, id, username, is_admin, manager_id):
        self.id = id; self.username = username; self.is_admin = is_admin; self.manager_id = manager_id 
    @property
    def is_super_admin(self): return self.is_admin and (self.manager_id is None or self.manager_id == 0)

@login_manager.user_loader
def load_user(uid):
    row = get_db().execute("SELECT * FROM users WHERE id=?", (uid,)).fetchone()
    return User(row['id'], row['username'], row['is_admin'], row['manager_id']) if row else None

def get_db():
    db = getattr(g, '_database', None)
    if db is None: db = g._database = sqlite3.connect(DATABASE, check_same_thread=False); db.row_factory = sqlite3.Row
    return db

def get_setting(key, default=None):
    row = get_db().execute("SELECT value FROM settings WHERE key = ?", (key,)).fetchone()
    return default if row is None else row["value"]

def set_setting(key, value):
    db = get_db()
    db.execute("INSERT INTO settings (key, value) VALUES (?, ?) ON CONFLICT(key) DO UPDATE SET value = excluded.value", (key, str(value)))
    db.commit()

@app.teardown_appcontext
def close_conn(e):
    db = getattr(g, '_database', None)
    if db is not None: db.close()

def init_db():
    db = get_db()
    cur = db.cursor()
    cur.execute("CREATE TABLE IF NOT EXISTS todos (id INTEGER PRIMARY KEY AUTOINCREMENT, date TEXT NOT NULL, content TEXT NOT NULL, is_completed INTEGER DEFAULT 0, user_id INTEGER, recurring_todo_id INTEGER, reminder_at TEXT)")
    cur.execute("CREATE TABLE IF NOT EXISTS fitness_plans (id INTEGER PRIMARY KEY AUTOINCREMENT, date TEXT NOT NULL, exercise_name TEXT NOT NULL, sets_reps_duration TEXT, is_completed INTEGER DEFAULT 0, user_id INTEGER, recurring_plan_id INTEGER, reminder_at TEXT)")
    cur.execute("CREATE TABLE IF NOT EXISTS recurring_plans (id INTEGER PRIMARY KEY AUTOINCREMENT, user_id INTEGER, plan_name TEXT NOT NULL, exercises_json TEXT NOT NULL, start_date TEXT NOT NULL, end_date TEXT, repeat_interval TEXT NOT NULL, repeat_day INTEGER, reminder_time TEXT)")
    cur.execute("CREATE TABLE IF NOT EXISTS recurring_todos (id INTEGER PRIMARY KEY AUTOINCREMENT, user_id INTEGER, content TEXT NOT NULL, start_date TEXT NOT NULL, end_date TEXT, repeat_interval TEXT NOT NULL, repeat_day INTEGER, reminder_time TEXT)")
    cur.execute("CREATE TABLE IF NOT EXISTS exercise_library (id INTEGER PRIMARY KEY AUTOINCREMENT, user_id INTEGER, name TEXT, details TEXT)")
    cur.execute("CREATE TABLE IF NOT EXISTS todo_library (id INTEGER PRIMARY KEY AUTOINCREMENT, user_id INTEGER, content TEXT)")
    cur.execute("CREATE TABLE IF NOT EXISTS settings (key TEXT PRIMARY KEY, value TEXT)")
    cur.execute("CREATE TABLE IF NOT EXISTS users (id INTEGER PRIMARY KEY AUTOINCREMENT, username TEXT UNIQUE, password TEXT, is_admin INTEGER DEFAULT 0, manager_id INTEGER)")
    
    try: cur.execute("SELECT reminder_at FROM todos LIMIT 1"); 
    except: cur.execute("ALTER TABLE todos ADD COLUMN reminder_at TEXT")
    try: cur.execute("SELECT reminder_at FROM fitness_plans LIMIT 1"); 
    except: cur.execute("ALTER TABLE fitness_plans ADD COLUMN reminder_at TEXT")
    try: cur.execute("SELECT reminder_time FROM recurring_plans LIMIT 1"); 
    except: cur.execute("ALTER TABLE recurring_plans ADD COLUMN reminder_time TEXT")
    try: cur.execute("SELECT reminder_time FROM recurring_todos LIMIT 1"); 
    except: cur.execute("ALTER TABLE recurring_todos ADD COLUMN reminder_time TEXT")
    
    cur.execute("INSERT OR IGNORE INTO settings (key, value) VALUES ('scroll_speed', '25')")
    if cur.execute("SELECT COUNT(*) FROM users WHERE username='superuser'").fetchone()[0]==0:
        cur.execute("INSERT INTO users (username,password,is_admin,manager_id) VALUES (?,?,?,?)",('superuser','adminpassword',1,None))
    su_id = cur.execute("SELECT id FROM users WHERE username='superuser'").fetchone()['id']
    if cur.execute("SELECT COUNT(*) FROM users WHERE username='user1'").fetchone()[0]==0:
        cur.execute("INSERT INTO users (username,password,is_admin,manager_id) VALUES (?,?,?,?)",('user1','password',0,su_id))
    db.commit()

with app.app_context(): init_db()

@app.route('/login', methods=['POST'])
def login():
    d = request.get_json(silent=True) or request.form
    u = get_db().execute("SELECT * FROM users WHERE username=?",(d.get('username'),)).fetchone()
    if u and u['password']==d.get('password'):
        login_user(User(u['id'],u['username'],u['is_admin'],u['manager_id']), remember=True)
        return jsonify({'status':'success'})
    return jsonify({'status':'error'}), 401

@app.route('/logout', methods=['POST'])
@login_required
def logout(): logout_user(); return jsonify({'status':'success'})

@app.route('/')
def index():
    if not current_user.is_authenticated: return render_template('login.html')
    if 'Mobile' in request.headers.get('User-Agent','') and request.args.get('view')!='display': return redirect(url_for('mobile_config'))
    today = datetime.now(CN_TZ).strftime('%Y-%m-%d')
    _gen_fit(current_user.id, today)
    _gen_todo(current_user.id, today)
    return render_template('display.html', username=current_user.username)

@app.route('/manage')
@login_required
def manage(): return render_template('manage.html', username=current_user.username, is_admin=current_user.is_admin)

@app.route('/mobile_config')
@login_required
def mobile_config(): return render_template('mobile_config.html', username=current_user.username)

@app.route('/todos', methods=['GET'])
@login_required
def get_todos():
    d = request.args.get('date') or datetime.now(CN_TZ).strftime('%Y-%m-%d')
    return jsonify([dict(r) for r in get_db().execute("SELECT * FROM todos WHERE date=? AND user_id=? ORDER BY is_completed ASC, reminder_at ASC, id DESC", (d, current_user.id)).fetchall()])

@app.route('/add_todo', methods=['POST'])
@login_required
def add_todo():
    d = request.get_json(silent=True) or request.form
    rem = d.get('reminder_at')
    if rem and len(rem)==5: rem = f"{d.get('date')} {rem}"
    get_db().execute("INSERT INTO todos (date,content,is_completed,user_id,reminder_at) VALUES (?,?,0,?,?)", (d.get('date'),d.get('content'),current_user.id,rem)).connection.commit()
    return jsonify(status='success')

@app.route('/add_recurring_todo', methods=['POST'])
@login_required
def add_rec_todo():
    d = request.get_json(silent=True) or request.form
    end_d = d.get('end_date')
    if not end_d: end_d = None
    db = get_db()
    cur = db.cursor()
    cur.execute("INSERT INTO recurring_todos (user_id,content,start_date,end_date,repeat_interval,repeat_day,reminder_time) VALUES (?,?,?,?,?,?,?)",
                (current_user.id, d.get('content'), d.get('start_date'), end_d, d.get('repeat_interval'), d.get('repeat_day'), d.get('reminder_time')))
    new_id = cur.lastrowid
    db.commit()
    _gen_single_rec_todo(current_user.id, datetime.now(CN_TZ).strftime('%Y-%m-%d'), new_id)
    return jsonify(status='success')

@app.route('/fitness_plan/<date>', methods=['GET'])
@login_required
def get_fit(date):
    _gen_fit(current_user.id, date)
    return jsonify([dict(r) for r in get_db().execute("SELECT * FROM fitness_plans WHERE date=? AND user_id=? ORDER BY is_completed ASC, reminder_at ASC, id DESC", (date, current_user.id)).fetchall()])

@app.route('/add_recurring_plan', methods=['POST'])
@login_required
def add_rec_plan():
    d = request.get_json(silent=True) or request.form
    end_d = d.get('end_date')
    if not end_d: end_d = None
    db = get_db()
    cur = db.cursor()
    cur.execute("INSERT INTO recurring_plans (user_id,plan_name,exercises_json,start_date,end_date,repeat_interval,repeat_day,reminder_time) VALUES (?,?,?,?,?,?,?,?)",
                (current_user.id, d.get('plan_name'), json.dumps(d.get('exercises')), d.get('start_date'), end_d, d.get('repeat_interval'), d.get('repeat_day'), d.get('reminder_time')))
    new_id = cur.lastrowid
    db.commit()
    _gen_single_rec_plan(current_user.id, datetime.now(CN_TZ).strftime('%Y-%m-%d'), new_id)
    return jsonify(status='success')

@app.route('/delete_todo/<int:id>', methods=['POST'])
@login_required
def del_todo(id): get_db().execute("DELETE FROM todos WHERE id=? AND user_id=?",(id,current_user.id)).connection.commit(); return jsonify(status='success')
@app.route('/mark_todo_completed', methods=['POST'])
@login_required
def mark_todo(): d=request.get_json(silent=True) or request.form; get_db().execute("UPDATE todos SET is_completed=? WHERE id=? AND user_id=?",(d.get('is_completed'),d.get('todo_id'),current_user.id)).connection.commit(); return jsonify(status='success')
@app.route('/delete_fitness_plan_exercise/<int:id>', methods=['POST'])
@login_required
def del_fit(id): get_db().execute("DELETE FROM fitness_plans WHERE id=? AND user_id=?",(id,current_user.id)).connection.commit(); return jsonify(status='success')
@app.route('/mark_exercise_completed_mobile', methods=['POST'])
@login_required
def mark_fit(): d=request.get_json(silent=True) or request.form; get_db().execute("UPDATE fitness_plans SET is_completed=? WHERE id=? AND user_id=?",(d.get('is_completed'),d.get('plan_id'),current_user.id)).connection.commit(); return jsonify(status='success')
@app.route('/get_recurring_todos', methods=['GET'])
@login_required
def get_rec_todos_list(): return jsonify([dict(r) for r in get_db().execute("SELECT * FROM recurring_todos WHERE user_id=?",(current_user.id,)).fetchall()])
@app.route('/delete_recurring_todo/<int:id>', methods=['POST'])
@login_required
def del_rec_todo(id): get_db().execute("DELETE FROM recurring_todos WHERE id=? AND user_id=?",(id,current_user.id)).connection.commit(); return jsonify(status='success')
@app.route('/get_recurring_plans', methods=['GET'])
@login_required
def get_rec_plans_list(): return jsonify([dict(r) for r in get_db().execute("SELECT * FROM recurring_plans WHERE user_id=?",(current_user.id,)).fetchall()])
@app.route('/delete_recurring_plan/<int:id>', methods=['POST'])
@login_required
def del_rec_plan(id): get_db().execute("DELETE FROM recurring_plans WHERE id=? AND user_id=?",(id,current_user.id)).connection.commit(); return jsonify(status='success')
@app.route('/add_fitness_manual', methods=['POST'])
@login_required
def add_fit_man():
    d=request.get_json(silent=True) or request.form
    rem = d.get('reminder_at')
    if rem and len(rem)==5: rem = f"{d.get('date')} {rem}"
    get_db().execute("INSERT INTO fitness_plans (date,exercise_name,sets_reps_duration,user_id,reminder_at) VALUES (?,?,?,?,?)",(d.get('date'),d.get('exercise_name'),d.get('sets_reps_duration'),current_user.id,rem)).connection.commit()
    return jsonify(status='success')

@app.route('/api/settings/scroll_speed', methods=['GET','POST'])
def scroll_speed():
    if request.method=='POST': 
        d=request.get_json(silent=True) or request.form
        try: v=int(d.get('scroll_speed'))
        except: v=25
        # ★★★ MODIFIED: Allow speed down to 1 second ★★★
        if v < 1: v = 1 
        if v > 120: v = 120
        set_setting('scroll_speed', v); return jsonify(status='success', scroll_speed=v)
    try: val = int(get_setting('scroll_speed', 25))
    except: val = 25
    return jsonify(scroll_speed=val)

@app.route('/api/exercises', methods=['GET','POST'])
@app.route('/api/exercises/<int:id>', methods=['DELETE'])
@login_required
def ex_api(id=None):
    if request.method=='GET': return jsonify([dict(r) for r in get_db().execute("SELECT * FROM exercise_library WHERE user_id=?",(current_user.id,)).fetchall()])
    if request.method=='POST': d=request.get_json(silent=True) or request.form; get_db().execute("INSERT INTO exercise_library (user_id,name,details) VALUES (?,?,?)",(current_user.id,d.get('name'),d.get('details'))).connection.commit(); return jsonify(status='success')
    if request.method=='DELETE': get_db().execute("DELETE FROM exercise_library WHERE id=?",(id,)).connection.commit(); return jsonify(status='success')

@app.route('/api/todo_library', methods=['GET','POST'])
@app.route('/api/todo_library/<int:id>', methods=['DELETE'])
@login_required
def td_api(id=None):
    if request.method=='GET': return jsonify([dict(r) for r in get_db().execute("SELECT * FROM todo_library WHERE user_id=?",(current_user.id,)).fetchall()])
    if request.method=='POST': d=request.get_json(silent=True) or request.form; get_db().execute("INSERT INTO todo_library (user_id,content) VALUES (?,?)",(current_user.id,d.get('content'))).connection.commit(); return jsonify(status='success')
    if request.method=='DELETE': get_db().execute("DELETE FROM todo_library WHERE id=?",(id,)).connection.commit(); return jsonify(status='success')

def admin_required(f):
    @wraps(f)
    def dec(*args, **kwargs):
        if not current_user.is_authenticated or not current_user.is_admin: return jsonify(status='error'), 403
        return f(*args, **kwargs)
    return dec

@app.route('/api/users', methods=['GET','POST'])
@app.route('/api/users/<int:id>', methods=['DELETE'])
@login_required
@admin_required
def users_api(id=None):
    if request.method=='GET': return jsonify([dict(u) for u in get_db().execute("SELECT id,username,is_admin FROM users WHERE manager_id=?",(current_user.id,)).fetchall()])
    if request.method=='POST':
        d=request.get_json(silent=True) or request.form
        try: get_db().execute("INSERT INTO users (username,password,is_admin,manager_id) VALUES (?,?,?,?)",(d.get('username'),d.get('password'),1 if d.get('is_admin') else 0, current_user.id)).connection.commit(); return jsonify(status='success')
        except: return jsonify(status='error')
    if request.method=='DELETE':
        if id==current_user.id: return jsonify(status='error')
        get_db().execute("DELETE FROM users WHERE id=?",(id,)).connection.commit(); return jsonify(status='success')

@app.route('/api/profile', methods=['POST'])
@login_required
def profile():
    d=request.get_json(silent=True) or request.form
    if d.get('password'): get_db().execute("UPDATE users SET password=? WHERE id=?",(d.get('password'),current_user.id))
    if d.get('username'): 
        try: get_db().execute("UPDATE users SET username=? WHERE id=?",(d.get('username'),current_user.id))
        except: pass
    get_db().commit(); return jsonify(status='success')

@app.route('/api/stats/daily_completion')
@login_required
def stats():
    s, e = request.args.get('start_date'), request.args.get('end_date')
    db = get_db()
    f = db.execute("SELECT date, is_completed FROM fitness_plans WHERE user_id=? AND date BETWEEN ? AND ?", (current_user.id, s, e)).fetchall()
    t = db.execute("SELECT date, is_completed FROM todos WHERE user_id=? AND date BETWEEN ? AND ?", (current_user.id, s, e)).fetchall()
    fd = defaultdict(lambda: {"total":0,"completed":0}); td = defaultdict(lambda: {"total":0,"completed":0})
    for r in f: fd[r['date']]['total']+=1; 
    for r in f: 
        if r['is_completed']: fd[r['date']]['completed']+=1
    for r in t: td[r['date']]['total']+=1; 
    for r in t:
        if r['is_completed']: td[r['date']]['completed']+=1
    return jsonify(fitness_data=fd, todo_data=td)

def _gen_fit(uid, date_str):
    db = get_db()
    if db.execute("SELECT COUNT(*) FROM fitness_plans WHERE date=? AND user_id=?",(date_str,uid)).fetchone()[0] > 0: return
    plans = db.execute("SELECT * FROM recurring_plans WHERE user_id=? AND start_date<=? AND (end_date IS NULL OR end_date>=?)", (uid,date_str,date_str)).fetchall()
    dt = datetime.strptime(date_str, '%Y-%m-%d').date()
    for p in plans:
        if _is_due(p, dt): _insert_fit(db, uid, date_str, p)
    db.commit()

def _gen_todo(uid, date_str):
    db = get_db()
    if db.execute("SELECT COUNT(*) FROM todos WHERE date=? AND user_id=?",(date_str,uid)).fetchone()[0] > 0: return
    recs = db.execute("SELECT * FROM recurring_todos WHERE user_id=? AND start_date<=? AND (end_date IS NULL OR end_date>=?)", (uid,date_str,date_str)).fetchall()
    dt = datetime.strptime(date_str, '%Y-%m-%d').date()
    for r in recs:
        if _is_due(r, dt): _insert_todo(db, uid, date_str, r)
    db.commit()

def _gen_single_rec_todo(uid, date_str, rec_id):
    db = get_db()
    r = db.execute("SELECT * FROM recurring_todos WHERE id=?",(rec_id,)).fetchone()
    if not r: return
    dt = datetime.strptime(date_str, '%Y-%m-%d').date()
    if r['start_date'] <= date_str and (not r['end_date'] or r['end_date'] >= date_str):
        if _is_due(r, dt): _insert_todo(db, uid, date_str, r)
    db.commit()

def _gen_single_rec_plan(uid, date_str, rec_id):
    db = get_db()
    p = db.execute("SELECT * FROM recurring_plans WHERE id=?",(rec_id,)).fetchone()
    if not p: return
    dt = datetime.strptime(date_str, '%Y-%m-%d').date()
    if p['start_date'] <= date_str and (not p['end_date'] or p['end_date'] >= date_str):
        if _is_due(p, dt): _insert_fit(db, uid, date_str, p)
    db.commit()

def _is_due(item, dt):
    if item['repeat_interval'] == 'daily': return True
    if item['repeat_interval'] == 'weekly' and item['repeat_day'] == dt.isoweekday(): return True
    if item['repeat_interval'] == 'monthly' and item['repeat_day'] == dt.day: return True
    return False

def _insert_todo(db, uid, date_str, r):
    if db.execute("SELECT COUNT(*) FROM todos WHERE date=? AND recurring_todo_id=?",(date_str, r['id'])).fetchone()[0] > 0: return
    rem = f"{date_str} {r['reminder_time']}" if r['reminder_time'] else None
    db.execute("INSERT INTO todos (date, content, is_completed, user_id, recurring_todo_id, reminder_at) VALUES (?,?,?,?,?,?)",
               (date_str, r['content'], 0, uid, r['id'], rem))

def _insert_fit(db, uid, date_str, p):
    if db.execute("SELECT COUNT(*) FROM fitness_plans WHERE date=? AND recurring_plan_id=?",(date_str, p['id'])).fetchone()[0] > 0: return
    rem = f"{date_str} {p['reminder_time']}" if p['reminder_time'] else None
    for ex in json.loads(p['exercises_json']):
        db.execute("INSERT INTO fitness_plans (date, exercise_name, sets_reps_duration, is_completed, user_id, recurring_plan_id, reminder_at) VALUES (?,?,?,0,?,?,?)",
                   (date_str, ex['name'], ex['details'], uid, p['id'], rem))

@app.after_request
def add_headers(r):
    r.headers['Cache-Control'] = 'no-cache, no-store, must-revalidate'
    return r

@app.route('/health', methods=['GET'])
def health(): return jsonify(status='ok', time=datetime.now(CN_TZ).isoformat())

if __name__ == '__main__':
    app.run(host='0.0.0.0', port=8080, debug=True)
'''

# ==============================================================================
# 2. frontend/templates/manage.html (v2.5 - Min Speed 1s)
# ==============================================================================
manage_html_content = r'''<!DOCTYPE html>
<html lang="zh-CN">
<head>
  <meta charset="utf-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1.0" />
  <title>健身看板 - 后台管理</title>
  <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
  <link href="https://fonts.googleapis.com/css2?family=Noto+Sans+SC:wght@300;400;500;600;700&family=Roboto:wght@400;500;700&display=swap" rel="stylesheet">
  <style>
    :root { --bg-color: #f1f5f9; --panel-color: #ffffff; --accent-color: #0f766e; --accent-color-light: #ccfbf1; --accent-color-soft: #ecfdf5; --accent-danger: #dc2626; --text-primary: #0f172a; --text-secondary: #64748b; --border-soft: #e2e8f0; --shadow-soft: 0 18px 40px rgba(15, 23, 42, 0.12); --radius-lg: 18px; --radius-md: 12px; }
    .hidden { display: none !important; }
    * { box-sizing: border-box; }
    body { margin: 0; font-family: 'Noto Sans SC', 'Roboto', sans-serif; background-color: var(--bg-color); color: var(--text-primary); }
    .app-container { display: flex; min-height: 100vh; }
    .sidebar { width: 260px; background: linear-gradient(180deg, #0f172a, #020617); color: #e5e7eb; display: flex; flex-direction: column; padding: 24px 18px; box-shadow: 6px 0 25px rgba(15, 23, 42, 0.7); position: sticky; top: 0; height: 100vh; }
    .sidebar-header { display: flex; align-items: center; gap: 12px; margin-bottom: 32px; }
    .logo-circle { width: 40px; height: 40px; border-radius: 50%; background: radial-gradient(circle at 30% 20%, #a7f3d0, #16a34a); display: flex; align-items: center; justify-content: center; color: #022c22; font-weight: 800; font-size: 20px; box-shadow: 0 0 0 3px rgba(34, 197, 94, 0.2); }
    .sidebar-title { font-size: 18px; font-weight: 700; }
    .sidebar-subtitle { font-size: 12px; color: #9ca3af; margin-top: 2px; }
    .nav-section-title { font-size: 12px; text-transform: uppercase; letter-spacing: 0.08em; color: #6b7280; margin: 18px 10px 8px; }
    .nav-list { list-style: none; margin: 0; padding: 0; flex: 1; }
    .nav-item { margin-bottom: 6px; }
    .nav-button { width: 100%; padding: 10px 12px; border-radius: 12px; border: none; background: transparent; color: #d1d5db; font-size: 14px; text-align: left; display: flex; align-items: center; cursor: pointer; transition: all 0.18s ease; position: relative; gap: 8px; }
    .nav-button:hover { background: rgba(15, 23, 42, 0.8); transform: translateX(2px); }
    .nav-button.active { background: linear-gradient(90deg, rgba(34, 197, 94, 0.22), rgba(34, 197, 94, 0.04)); color: #ecfdf3; box-shadow: 0 10px 25px rgba(22, 163, 74, 0.5); }
    .nav-button.active::before { content: ''; position: absolute; left: 8px; top: 50%; width: 4px; height: 20px; border-radius: 999px; transform: translateY(-50%); background: #22c55e; }
    .nav-footer { margin-top: 12px; padding: 10px 12px; border-radius: 12px; background: rgba(15, 23, 42, 0.75); font-size: 12px; color: #9ca3af; }
    .main { flex: 1; padding: 24px 30px; max-width: calc(100vw - 260px); }
    .top-bar { display: flex; justify-content: space-between; align-items: center; margin-bottom: 18px; }
    .top-title h1 { margin: 0; font-size: 26px; letter-spacing: 0.02em; }
    .top-title span { font-size: 13px; color: var(--text-secondary); }
    .top-actions { display: flex; align-items: center; gap: 8px; font-size: 13px; color: var(--text-secondary); }
    .pill-tag { padding: 6px 10px; border-radius: 999px; background: rgba(45, 212, 191, 0.1); border: 1px solid rgba(34, 197, 94, 0.35); color: #047857; font-size: 12px; display: inline-flex; align-items: center; gap: 6px; }
    .pill-tag span.dot { width: 7px; height: 7px; border-radius: 50%; background: #22c55e; box-shadow: 0 0 12px rgba(34, 197, 94, 0.8); }
    .main-grid { display: grid; grid-template-columns: minmax(0, 3fr) minmax(0, 2fr); gap: 18px; align-items: flex-start; }
    .panel { background-color: var(--panel-color); border-radius: var(--radius-lg); box-shadow: var(--shadow-soft); padding: 20px 22px; margin-bottom: 18px; border: 1px solid rgba(148, 163, 184, 0.18); }
    .panel-header { display: flex; justify-content: space-between; align-items: baseline; margin-bottom: 14px; }
    .panel-header h2 { margin: 0; font-size: 18px; color: var(--text-primary); display: flex; align-items: center; gap: 6px; }
    .panel-header h2 span.badge { font-size: 10px; text-transform: uppercase; letter-spacing: 0.08em; padding: 2px 8px; border-radius: 999px; background: var(--accent-color-soft); color: #047857; border: 1px solid rgba(16, 185, 129, 0.25); }
    .panel-header p { margin: 4px 0 0; font-size: 13px; color: var(--text-secondary); }
    .panel-body { margin-top: 6px; }
    .content-section { display: none; }
    .content-section.active { display: block; }
    .form-group { margin-bottom: 12px; }
    .form-group label { display: block; margin-bottom: 4px; font-size: 13px; color: var(--text-secondary); }
    input, select, textarea { width: 100%; padding: 12px; border: 1px solid #cbd5e1; border-radius: var(--radius-md); font-size: 1rem; color: var(--text-primary); background: #f8fafc; }
    .form-group input:focus, .form-group select:focus { outline: none; border-color: var(--accent-color); background: var(--panel-color); box-shadow: 0 0 0 3px var(--accent-color-light); }
    .btn-submit, .btn-primary { background-color: var(--accent-color); color: #ecfdf5; border: none; padding: 10px 18px; border-radius: 999px; cursor: pointer; font-size: 14px; font-weight: 600; display: inline-flex; align-items: center; gap: 6px; box-shadow: 0 12px 25px rgba(16, 185, 129, 0.35); transition: all 0.15s ease; }
    .btn-submit:hover, .btn-primary:hover { transform: translateY(-1px); box-shadow: 0 16px 30px rgba(16, 185, 129, 0.55); background-color: #0d9488; }
    .btn-secondary { background-color: #e5e7eb; color: #111827; border: none; padding: 6px 12px; border-radius: 999px; cursor: pointer; font-size: 12px; font-weight: 500; }
    .btn-secondary:hover { background-color: #d1d5db; }
    .btn-danger { background-color: #ef4444; color: #fff; border: none; padding: 6px 12px; border-radius: 999px; cursor: pointer; font-size: 12px; }
    .btn-danger:hover { background-color: #b91c1c; }
    .btn-add-ex { background-color: #e0f2fe; color: #075985; border: none; padding: 8px 12px; border-radius: 999px; cursor: pointer; font-size: 12px; font-weight: 500; }
    .btn-add-ex:hover { background-color: #bfdbfe; }
    .list-item { padding: 10px 12px; border-radius: 10px; border: 1px solid var(--border-soft); margin-bottom: 8px; background: #f9fafb; display: flex; justify-content: space-between; align-items: center; gap: 10px; }
    .list-item span { font-size: 13px; }
    .list-item .item-actions { display: flex; gap: 6px; }
    .alert-success { padding: 10px 12px; border-radius: var(--radius-md); background-color: #ecfdf5; border: 1px solid #bbf7d0; color: #166534; font-size: 13px; margin-bottom: 10px; display:none; }
    .exercise-group { display: grid; grid-template-columns: 1fr 1fr; gap: 8px; margin-bottom: 8px; }
    .section-grid { display: grid; grid-template-columns: 1.2fr 1fr; gap: 16px; }
    .small-hint { font-size: 12px; color: var(--text-secondary); }
    .readonly-box { padding: 10px 12px; border-radius: 10px; background: #f1f5f9; border: 1px dashed #cbd5e1; font-size: 13px; color: var(--text-secondary); margin: 6px 0 10px; }
    .divider { height: 1px; background: linear-gradient(90deg, transparent, rgba(148, 163, 184, 0.6), transparent); margin: 14px 0; }
    .stats-buttons { display: flex; gap: 10px; margin-bottom: 12px; } .stats-buttons button { flex: 1; }
    .stats-layout { display: grid; grid-template-columns: 1.6fr 1.4fr; gap: 16px; margin-top: 12px; }
    .stats-card { padding: 12px 14px; border-radius: 14px; background: #f8fafc; border: 1px solid #e2e8f0; font-size: 13px; color: var(--text-secondary); }
    .stats-card h4 { margin: 0 0 8px; font-size: 14px; color: var(--text-primary); }
    .stats-summary { margin-top: 10px; font-size: 14px; color: var(--text-secondary); }
    .stats-highlight { font-weight: 600; color: #0f766e; }
    .stats-chip-row { display: flex; flex-wrap: wrap; gap: 6px; margin-top: 6px; }
    .stats-chip { padding: 4px 8px; border-radius: 999px; font-size: 11px; background: #e0f2fe; color: #075985; }
    .profile-grid { display: grid; grid-template-columns: minmax(0, 1.2fr) minmax(0, 1.2fr); gap: 16px; }
    .profile-box { padding: 14px 16px; border-radius: 14px; border: 1px solid #e2e8f0; background: #f9fafb; font-size: 13px; color: var(--text-secondary); }
    .profile-box h3 { margin: 0 0 6px; font-size: 15px; color: var(--text-primary); }
    .footer-note { font-size: 12px; text-align: right; color: #9ca3af; margin-top: 6px; }
    .btn-display { text-decoration:none; font-weight:bold; color:#fff; background:#334155; display:block; text-align:center; padding:12px; border-radius:12px; margin-top:15px; }
    .tag-time { background: #e0f2fe; color: #0369a1; padding: 2px 6px; border-radius: 4px; font-size: 11px; margin-left: 6px; font-weight: 600; }
    @media (max-width: 1024px) { .app-container { flex-direction: column; } .sidebar { width: 100%; height: auto; position: static; flex-direction: row; overflow-x: auto; } .sidebar-header { margin-bottom: 0; } .nav-section-title, .nav-footer { display: none; } .nav-list { display: flex; flex-wrap: nowrap; gap: 4px; } .nav-item { flex: 1; } .main { max-width: 100%; padding: 16px; } .main-grid { grid-template-columns: minmax(0, 1fr); } .section-grid { grid-template-columns: minmax(0, 1fr); } .stats-layout { grid-template-columns: minmax(0, 1fr); } .profile-grid { grid-template-columns: minmax(0, 1fr); } .panel { padding: 16px; } }
  </style>
</head>
<body>
  <div class="app-container">
    <aside class="sidebar">
      <div class="sidebar-header">
        <div class="logo-circle">F</div>
        <div><div class="sidebar-title">健身看板</div><div class="sidebar-subtitle">后台管理</div></div>
      </div>
      <div class="nav-section-title">导航</div>
      <ul class="nav-list">
        <li class="nav-item"><button class="nav-button active" data-tab="daily-todos"><span class="nav-icon">✅</span> 每日待办 (T)</button></li>
        <li class="nav-item"><button class="nav-button" data-tab="daily-fitness"><span class="nav-icon">💪</span> 每日健身 (T)</button></li>
        <li class="nav-item"><button class="nav-button" data-tab="recurring-todos"><span class="nav-icon">🔁</span> 循环待办 (R)</button></li>
        <li class="nav-item"><button class="nav-button" data-tab="recurring-fitness"><span class="nav-icon">📅</span> 循环健身 (R)</button></li>
        <li class="nav-item"><button class="nav-button" data-tab="stats"><span class="nav-icon">📊</span> 统计分析</button></li>
        <li class="nav-item"><button class="nav-button" data-tab="mobile-config"><span class="nav-icon">📱</span> 移动配置</button></li>
        <li class="nav-item" id="userMgmtBtn"><button class="nav-button" data-tab="user-management"><span class="nav-icon">👥</span> 用户管理</button></li>
        <li class="nav-item"><button class="nav-button" data-tab="profile"><span class="nav-icon">⚙️</span> 个人资料</button></li>
      </ul>
      <div class="nav-footer">
        <a href="/?view=display" class="btn-display">🖥️ 切换到大屏模式</a>
      </div>
    </aside>

    <main class="main">
      <div class="top-bar">
        <div class="top-title"><h1>后台管理中心</h1><span>当前用户: {{ username }}</span></div>
        <div id="globalAlert" class="alert-success"></div>
      </div>

      <!-- 1. Daily Todos -->
      <div id="daily-todos" class="content-section active">
        <div class="panel">
            <div class="panel-header"><h2>每日待办 (Today)</h2></div>
            <div class="section-grid">
                <div>
                    <div class="form-group"><label>日期:</label><input type="date" id="date-todo"></div>
                    <div id="todoList"></div>
                </div>
                <div>
                    <h3>+ 添加待办</h3>
                    <form id="addTodoForm">
                        <div class="form-group"><input type="text" id="todo-content" placeholder="内容" required></div>
                        <div class="form-group"><label>到期 (可选):</label><input type="datetime-local" id="todo-reminder"></div>
                        <button type="submit" class="btn-submit">添加待办</button>
                    </form>
                </div>
            </div>
        </div>
      </div>

      <!-- 2. Daily Fitness -->
      <div id="daily-fitness" class="content-section">
        <div class="panel">
            <div class="panel-header"><h2>每日健身 (Today)</h2></div>
            <div class="section-grid">
                <div>
                    <div class="form-group"><label>日期:</label><input type="date" id="date-daily"></div>
                    <div id="dailyFitnessList"></div>
                </div>
                <div>
                    <h3>+ 手动单次训练</h3>
                    <form id="addManFitForm">
                        <div class="form-group"><input type="text" id="mf-name" placeholder="动作名称" required></div>
                        <div class="form-group"><input type="text" id="mf-det" placeholder="详情"></div>
                        <div class="form-group"><label>到期 (可选):</label><input type="datetime-local" id="mf-rem"></div>
                        <button type="submit" class="btn-submit">添加训练</button>
                    </form>
                </div>
            </div>
        </div>
      </div>

      <!-- 3. Recurring Todos -->
      <div id="recurring-todos" class="content-section">
        <div class="panel">
            <div class="panel-header"><h2>循环待办 (R)</h2></div>
            <div class="panel-body">
                <form id="addRecTodoForm">
                    <div class="form-group"><label>内容:</label>
                        <div style="display:flex;gap:8px;margin-bottom:6px"><select id="todo-library-select"><option value="">从常用库选择...</option></select><button type="button" id="fillTodoLib" class="btn-add-ex">填入</button></div>
                        <input type="text" id="rec-todo-content" required>
                    </div>
                    <div class="exercise-group">
                        <div class="form-group"><label>开始:</label><input type="date" id="rec-todo-start" required></div>
                        <div class="form-group"><label>结束 (可选):</label><input type="date" id="rec-todo-end"></div>
                    </div>
                    <div class="form-group"><label>每日时间 (例 08:00):</label><input type="time" id="rec-todo-time"></div>
                    <div class="form-group">
                        <label>循环方式:</label>
                        <select id="rec-todo-interval">
                            <option value="daily">每日</option><option value="weekly">每周</option><option value="monthly">每月</option>
                        </select>
                    </div>
                    <div class="form-group hidden" id="rec-todo-week-box"><label>星期几:</label><select id="rec-todo-day-w"><option value="1">周一</option><option value="2">周二</option><option value="3">周三</option><option value="4">周四</option><option value="5">周五</option><option value="6">周六</option><option value="7">周日</option></select></div>
                    <div class="form-group hidden" id="rec-todo-month-box"><label>几号:</label><input type="number" id="rec-todo-day-m" min="1" max="31" value="1"></div>
                    <button type="submit" class="btn-submit">保存循环待办</button>
                </form>
                <div class="divider"></div>
                <h3>已保存的循环待办</h3>
                <div id="recTodoList"></div>
                <div class="divider"></div>
                <h3>常用待办库</h3>
                <form id="todoLibraryForm" style="display:flex;gap:10px;"><input type="text" id="lib-todo-content" placeholder="内容"><button type="submit" class="btn-submit">添加</button></form>
                <div id="todoLibraryList" style="margin-top:10px"></div>
            </div>
        </div>
      </div>

      <!-- 4. Recurring Fitness -->
      <div id="recurring-fitness" class="content-section">
        <div class="panel">
            <div class="panel-header"><h2>循环健身计划 (R)</h2></div>
            <div class="panel-body">
                <form id="addRecPlanForm">
                    <div class="form-group"><label>计划名:</label><input type="text" id="rec-plan-name" required></div>
                    <div class="exercise-group">
                        <div class="form-group"><label>开始:</label><input type="date" id="rec-plan-start" required></div>
                        <div class="form-group"><label>结束 (可选):</label><input type="date" id="rec-plan-end"></div>
                    </div>
                    <div class="form-group"><label>每日时间 (例 20:00):</label><input type="time" id="rec-plan-time"></div>
                    <div class="form-group">
                        <label>循环方式:</label>
                        <select id="rec-plan-interval"><option value="daily">每日</option><option value="weekly">每周</option></select>
                    </div>
                    <div class="form-group hidden" id="rec-plan-week-box"><label>星期几:</label><select id="rec-plan-day-w"><option value="1">周一</option><option value="2">周二</option><option value="3">周三</option><option value="4">周四</option><option value="5">周五</option><option value="6">周六</option><option value="7">周日</option></select></div>
                    <label>动作:</label>
                    <div style="display:flex;gap:8px;margin-bottom:10px"><select id="ex-lib-select"><option value="">从训练库...</option></select><button type="button" id="addFromLibBtn" class="btn-add-ex">添加</button></div>
                    <div id="ex-container"><div class="exercise-group"><input type="text" class="ex-n" placeholder="动作"><input type="text" class="ex-d" placeholder="组数"></div></div>
                    <button type="button" id="addExBtn" class="btn-add-ex">+ 增加动作</button>
                    <br><br>
                    <button type="submit" class="btn-submit">保存计划</button>
                </form>
                <div class="divider"></div>
                <h3>已保存的循环计划</h3>
                <div id="recPlanList"></div>
                <div class="divider"></div>
                <h3>训练项目库</h3>
                <form id="exLibForm" style="display:flex;gap:10px;"><input type="text" id="lib-ex-name" placeholder="动作"><input type="text" id="lib-ex-det" placeholder="组数"><button type="submit" class="btn-submit">添加</button></form>
                <div id="exLibraryList" style="margin-top:10px"></div>
            </div>
        </div>
      </div>

      <!-- 5. Stats -->
      <div id="stats" class="content-section">
        <div class="panel">
            <div class="panel-header"><h2>统计与分析</h2></div>
            <div class="panel-body">
                <div class="form-group"><label>区间:</label><div style="display:flex;gap:10px"><input type="date" id="stats-s"><input type="date" id="stats-e"><button class="btn-secondary" onclick="loadStats()">刷新</button></div></div>
                <div class="stats-buttons"><button class="btn-secondary" id="btn-week">最近7天</button><button class="btn-secondary" id="btn-month">最近30天</button></div>
                <div class="stats-layout"><div><canvas id="statsChart"></canvas></div><div><div class="stats-card"><h4>概览</h4><div id="statsTxt">请选择日期刷新</div></div></div></div>
            </div>
        </div>
      </div>

      <!-- 6. Mobile Config -->
      <div id="mobile-config" class="content-section">
        <div class="panel">
            <div class="panel-header"><h2>大屏 & 移动端配置</h2></div>
            <div class="panel-body">
                <div class="profile-box" style="margin-bottom:20px">
                    <h3>大屏字幕滚动速度</h3>
                    <form id="scrollSpeedForm" style="display:flex;gap:10px;align-items:center">
                        <!-- ★★★ MODIFIED: MIN 1 SEC ★★★ -->
                        <input type="number" id="scroll-speed-input" min="1" max="120" style="width:80px"> <span>秒/轮</span>
                        <button type="submit" class="btn-submit">保存</button>
                    </form>
                </div>
                <iframe src="/mobile_config" style="width:100%;height:500px;border:none;border-radius:12px;box-shadow:inset 0 0 10px rgba(0,0,0,0.05)"></iframe>
            </div>
        </div>
      </div>

      <!-- 7. User Mgmt -->
      <div id="user-management" class="content-section">
        <div class="panel"><h2>用户管理</h2>
            <div id="userList"></div>
            <h3>新增用户</h3>
            <form id="addUserForm">
                <div class="form-group"><input type="text" id="nu-name" placeholder="用户名" required></div>
                <div class="form-group"><input type="password" id="nu-pass" placeholder="密码" required></div>
                <label style="font-size:13px;color:#666"><input type="checkbox" id="nu-admin"> 设为管理员</label>
                <br><br><button type="submit" class="btn-submit">创建</button>
            </form>
        </div>
      </div>

      <!-- 8. Profile -->
      <div id="profile" class="content-section">
        <div class="panel"><h2>个人资料</h2>
            <div class="panel-body">
                <form id="profForm">
                    <div class="form-group"><label>新用户名:</label><input type="text" id="pu-name"></div>
                    <div class="form-group"><label>新密码:</label><input type="password" id="pu-pass"></div>
                    <button type="submit" class="btn-submit">更新资料</button>
                </form>
            </div>
        </div>
      </div>

    </main>
  </div>

<script>
const today = new Date();
const todayStr = `${today.getFullYear()}-${String(today.getMonth()+1).padStart(2,'0')}-${String(today.getDate()).padStart(2,'0')}`;
const api = async (u, o) => (await fetch(u, {credentials:'include', ...o})).json();
function msg(t) { const e=document.getElementById('globalAlert'); e.textContent=t; e.style.display='block'; setTimeout(()=>e.style.display='none',3000); }

document.querySelectorAll('.nav-button').forEach(b => b.addEventListener('click', () => {
    if(b.classList.contains('btn-display-link')) return;
    document.querySelectorAll('.nav-button').forEach(x=>x.classList.remove('active'));
    document.querySelectorAll('.content-section').forEach(x=>x.classList.remove('active'));
    b.classList.add('active');
    document.getElementById(b.dataset.tab).classList.add('active');
    refreshAll();
}));

document.addEventListener('DOMContentLoaded', ()=>{
    ['date-todo','date-daily','rec-todo-start','rec-plan-start','stats-s','stats-e'].forEach(i=>{ const el=document.getElementById(i); if(el) el.value=todayStr; });
    document.getElementById('rec-todo-interval').dispatchEvent(new Event('change'));
    document.getElementById('rec-plan-interval').dispatchEvent(new Event('change'));
    refreshAll();
    loadScrollSpeed();
});

function refreshAll() {
    loadTodos(document.getElementById('date-todo').value);
    loadFitness(document.getElementById('date-daily').value);
    loadRecTodos(); loadRecPlans(); loadUsers();
    loadExLib(); loadTodoLib();
}

// 1. Todos
async function loadTodos(d) {
    const l = await api(`/todos?date=${d}`);
    document.getElementById('todoList').innerHTML = l.map(t => `
        <div class="list-item">
            <div>[${t.is_completed?'OK':'..'}] ${t.content} ${t.reminder_at ? `<span class="tag-time">⏰ ${t.reminder_at.slice(11,16)}</span>` : ''} ${t.recurring_todo_id ? '<small style="color:#999">(R)</small>' : ''}</div>
            <button class="btn-danger" onclick="del('delete_todo',${t.id})">删</button>
        </div>
    `).join('') || '<div class="small-hint">无待办</div>';
}
document.getElementById('date-todo').onchange=(e)=>loadTodos(e.target.value);
document.getElementById('addTodoForm').onsubmit=async(e)=>{
    e.preventDefault();
    let rem = document.getElementById('todo-reminder').value; if(rem) rem=rem.replace('T',' ');
    await api('/add_todo', {method:'POST',headers:{'Content-Type':'application/json'},body:JSON.stringify({date:document.getElementById('date-todo').value, content:document.getElementById('todo-content').value, reminder_at:rem})});
    e.target.reset(); loadTodos(document.getElementById('date-todo').value); msg('添加成功');
};

// 2. Fitness
async function loadFitness(d) {
    const l = await api(`/fitness_plan/${d}`);
    document.getElementById('dailyFitnessList').innerHTML = l.map(p => `
        <div class="list-item">
            <div>[${p.is_completed?'OK':'..'}] ${p.exercise_name} <small>${p.sets_reps_duration}</small> ${p.reminder_at ? `<span class="tag-time">⏰ ${p.reminder_at.slice(11,16)}</span>` : ''}</div>
            <button class="btn-danger" onclick="del('delete_fitness_plan_exercise',${p.id})">删</button>
        </div>
    `).join('') || '<div class="small-hint">无计划</div>';
}
document.getElementById('date-daily').onchange=(e)=>loadFitness(e.target.value);
document.getElementById('addManFitForm').onsubmit=async(e)=>{
    e.preventDefault();
    let rem = document.getElementById('mf-rem').value; if(rem) rem=rem.replace('T',' ');
    await api('/add_fitness_manual', {method:'POST',headers:{'Content-Type':'application/json'},body:JSON.stringify({date:document.getElementById('date-daily').value, exercise_name:document.getElementById('mf-name').value, sets_reps_duration:document.getElementById('mf-det').value, reminder_at:rem})});
    e.target.reset(); loadFitness(document.getElementById('date-daily').value); msg('添加成功');
};

// 3. Rec Todos
async function loadRecTodos(){
    const l = await api('/get_recurring_todos');
    document.getElementById('recTodoList').innerHTML = l.map(t => `
        <div class="list-item">
            <div>${t.content} [${t.repeat_interval}] <span style="font-size:11px;color:#999">(${t.start_date} ~ ${t.end_date||'∞'})</span> ${t.reminder_time ? `<span class="tag-time">⏰ ${t.reminder_time}</span>` : ''}</div>
            <button class="btn-danger" onclick="del('delete_recurring_todo',${t.id})">删</button>
        </div>
    `).join('');
}
async function loadTodoLib(){
    const l = await api('/api/todo_library');
    const s = document.getElementById('todo-library-select'); s.innerHTML='<option value="">从常用库选择...</option>';
    l.forEach(i=>{ const o=document.createElement('option'); o.text=i.content; o.value=i.content; s.add(o); });
    document.getElementById('todoLibraryList').innerHTML = l.map(i=>`<div class="list-item"><span>${i.content}</span><button class="btn-danger" onclick="del('api/todo_library',${i.id})">删</button></div>`).join('');
}
document.getElementById('todoLibraryForm').onsubmit=async(e)=>{e.preventDefault(); await api('/api/todo_library',{method:'POST',headers:{'Content-Type':'application/json'},body:JSON.stringify({content:document.getElementById('lib-todo-content').value})}); e.target.reset(); loadTodoLib();}
document.getElementById('fillTodoLib').onclick=()=>{ document.getElementById('rec-todo-content').value = document.getElementById('todo-library-select').value; };
document.getElementById('rec-todo-interval').onchange=(e)=>{
    document.getElementById('rec-todo-week-box').classList.toggle('hidden', e.target.value!=='weekly');
    document.getElementById('rec-todo-month-box').classList.toggle('hidden', e.target.value!=='monthly');
};
document.getElementById('addRecTodoForm').onsubmit=async(e)=>{
    e.preventDefault();
    await api('/add_recurring_todo',{method:'POST',headers:{'Content-Type':'application/json'},body:JSON.stringify({
        content:document.getElementById('rec-todo-content').value, start_date:document.getElementById('rec-todo-start').value,
        end_date:document.getElementById('rec-todo-end').value || null,
        repeat_interval:document.getElementById('rec-todo-interval').value, repeat_day:document.getElementById('rec-todo-interval').value==='weekly'?document.getElementById('rec-todo-day-w').value:document.getElementById('rec-todo-day-m').value,
        reminder_time:document.getElementById('rec-todo-time').value
    })});
    e.target.reset(); loadRecTodos(); loadTodos(todayStr); msg('保存成功');
};

// 4. Rec Fitness
async function loadRecPlans(){
    const l = await api('/get_recurring_plans');
    document.getElementById('recPlanList').innerHTML = l.map(p => `
        <div class="list-item">
            <div>${p.plan_name} [${p.repeat_interval}] <span style="font-size:11px;color:#999">(${p.start_date} ~ ${p.end_date||'∞'})</span> ${p.reminder_time ? `<span class="tag-time">⏰ ${p.reminder_time}</span>` : ''}</div>
            <button class="btn-danger" onclick="del('delete_recurring_plan',${p.id})">删</button>
        </div>
    `).join('');
}
async function loadExLib(){
    const l = await api('/api/exercises');
    const s = document.getElementById('ex-lib-select'); s.innerHTML='<option value="">从训练库选择...</option>';
    l.forEach(i=>{ const o=document.createElement('option'); o.text=`${i.name} (${i.details||''})`; o.dataset.name=i.name; o.dataset.det=i.details; s.add(o); });
    document.getElementById('exLibraryList').innerHTML = l.map(i=>`<div class="list-item"><span>${i.name} ${i.details}</span><button class="btn-danger" onclick="del('api/exercises',${i.id})">删</button></div>`).join('');
}
document.getElementById('exLibForm').onsubmit=async(e)=>{e.preventDefault(); await api('/api/exercises',{method:'POST',headers:{'Content-Type':'application/json'},body:JSON.stringify({name:document.getElementById('lib-ex-name').value,details:document.getElementById('lib-ex-det').value})}); e.target.reset(); loadExLib();}
document.getElementById('addFromLibBtn').onclick=()=>{
    const op = document.getElementById('ex-lib-select').selectedOptions[0];
    if(op.value) document.getElementById('ex-container').insertAdjacentHTML('beforeend',`<div class="exercise-group"><input type="text" class="ex-n" value="${op.dataset.name}"><input type="text" class="ex-d" value="${op.dataset.det||''}"></div>`);
};
document.getElementById('rec-plan-interval').onchange=(e)=>{ document.getElementById('rec-plan-week-box').classList.toggle('hidden', e.target.value!=='weekly'); };
document.getElementById('addExBtn').onclick=()=>{ document.getElementById('ex-container').insertAdjacentHTML('beforeend','<div class="exercise-group"><input type="text" class="ex-n" placeholder="动作"><input type="text" class="ex-d" placeholder="组数"></div>'); };
document.getElementById('addRecPlanForm').onsubmit=async(e)=>{
    e.preventDefault();
    const exs = [...document.querySelectorAll('.ex-n')].map((el,i)=>({name:el.value, details:document.querySelectorAll('.ex-d')[i].value})).filter(x=>x.name);
    await api('/add_recurring_plan',{method:'POST',headers:{'Content-Type':'application/json'},body:JSON.stringify({
        plan_name:document.getElementById('rec-plan-name').value, start_date:document.getElementById('rec-plan-start').value,
        end_date:document.getElementById('rec-plan-end').value || null,
        repeat_interval:document.getElementById('rec-plan-interval').value, repeat_day:document.getElementById('rec-plan-day-w').value,
        reminder_time:document.getElementById('rec-plan-time').value, exercises:exs
    })});
    e.target.reset(); document.getElementById('ex-container').innerHTML='<div class="exercise-group"><input type="text" class="ex-n" placeholder="动作"><input type="text" class="ex-d" placeholder="组数"></div>';
    loadRecPlans(); loadFitness(todayStr); msg('保存成功');
};

// 5. Config
async function loadScrollSpeed(){ const d=await api('/api/settings/scroll_speed'); document.getElementById('scroll-speed-input').value = d.scroll_speed; }
document.getElementById('scrollSpeedForm').onsubmit=async(e)=>{e.preventDefault(); await api('/api/settings/scroll_speed',{method:'POST',headers:{'Content-Type':'application/json'},body:JSON.stringify({scroll_speed:document.getElementById('scroll-speed-input').value})}); msg('已保存');}
async function loadUsers(){ try{const u=await api('/api/users'); document.getElementById('userList').innerHTML=u.map(x=>`<div class="list-item"><span>${x.username} ${x.is_admin?'(管)':''}</span><button class="btn-danger" onclick="delUser(${x.id})">删</button></div>`).join('');}catch(e){document.getElementById('userMgmtBtn').style.display='none';} }
document.getElementById('addUserForm').onsubmit=async(e)=>{e.preventDefault();await api('/api/users',{method:'POST',headers:{'Content-Type':'application/json'},body:JSON.stringify({username:document.getElementById('nu-name').value,password:document.getElementById('nu-pass').value,is_admin:document.getElementById('nu-admin').checked?1:0})});e.target.reset();loadUsers();msg('用户已创建');}
async function delUser(id){if(confirm('删用户?')){await api(`/api/users/${id}`,{method:'DELETE'});loadUsers();}}
document.getElementById('profForm').onsubmit=async(e)=>{e.preventDefault();await api('/api/profile',{method:'POST',headers:{'Content-Type':'application/json'},body:JSON.stringify({username:document.getElementById('pu-name').value,password:document.getElementById('pu-pass').value})});msg('资料已更新');}
let chart;
async function loadStats(){
    const d=await api(`/api/stats/daily_completion?start_date=${document.getElementById('stats-s').value}&end_date=${document.getElementById('stats-e').value}`);
    const l=Object.keys(d.todo_data).sort();
    if(chart)chart.destroy();
    chart=new Chart(document.getElementById('statsChart'),{type:'line',data:{labels:l,datasets:[{label:'待办',data:l.map(k=>d.todo_data[k].completed),borderColor:'#22c55e'},{label:'健身',data:l.map(k=>d.fitness_data[k].completed),borderColor:'#0ea5e9'}]}});
    document.getElementById('statsTxt').textContent = '统计已更新';
}
function getDateNDaysAgo(n){const d=new Date();d.setDate(d.getDate()-n);return `${d.getFullYear()}-${String(d.getMonth()+1).padStart(2,'0')}-${String(d.getDate()).padStart(2,'0')}`;}
document.getElementById('btn-week').onclick=()=>{document.getElementById('stats-s').value=getDateNDaysAgo(6);loadStats();}
document.getElementById('btn-month').onclick=()=>{document.getElementById('stats-s').value=getDateNDaysAgo(29);loadStats();}
async function del(ep,id){ if(confirm('删除?')) { await api(`/${ep}/${id}`,{method:'POST'}); refreshAll(); } }
</script>
</body>
</html>
'''

# ==============================================================================
# 3. frontend/templates/display.html (v3.0 - Sync Logic)
# ==============================================================================
display_html_content = r'''<!DOCTYPE html>
<html lang="zh-CN">
<head>
  <meta charset="utf-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no" />
  <title>智能健身看板</title>
  <style>
    :root { --bg: #0f1724; --panel: #1e293b; --accent: #4CAF50; --muted: #94a3b8; --card-radius: 18px; --font-base: 56px; --alert-color: #ef4444; }
    html, body { height: 100%; margin: 0; padding: 0; background: var(--bg); color: #fff; font-family: sans-serif; overflow: hidden; user-select: none; }
    
    .app { height: 100vh; display: flex; flex-direction: column; }
    .header { display: flex; justify-content: space-between; align-items: center; padding: 28px 48px; background: linear-gradient(90deg, #071029, #0b2540); box-shadow: 0 8px 20px rgba(0,0,0,0.6); flex-shrink: 0; z-index: 10; cursor: pointer; }
    .header:active { background: #1e3a8a; }
    
    .brand { font-size: calc(var(--font-base)*0.6); color: var(--accent); font-weight: 700; }
    .datetime { text-align: right; font-size: calc(var(--font-base)*0.38); color: #dbeafe; }
    .datetime .time { font-size: calc(var(--font-base)*0.9); font-weight: 700; }
    
    .main { padding: 28px; display: flex; gap: 28px; flex: 1; box-sizing: border-box; overflow: hidden; min-height: 0; }
    .panel { background: var(--panel); border-radius: var(--card-radius); padding: 28px; box-shadow: 0 12px 30px rgba(0,0,0,0.55); display: flex; flex-direction: column; flex: 1; overflow: hidden; min-height: 0; }
    h2 { margin: 0 0 18px 0; color: var(--accent); font-size: calc(var(--font-base)*0.48); border-bottom: 3px solid rgba(255,255,255,0.03); padding-bottom: 12px; flex-shrink: 0; }
    
    .scroll-container { position: relative; overflow: hidden; flex: 1; min-height: 0; }
    .scroll-content { position: absolute; top: 0; left: 0; width: 100%; will-change: transform; transform: translate3d(0,0,0); }
    
    .item { display: flex; justify-content: space-between; align-items: center; padding: 24px 10px; border-bottom: 1px solid rgba(255,255,255,0.05); gap: 15px; transition: background 0.2s; }
    .item:active { background: rgba(255,255,255,0.1); }
    .item .title { font-size: calc(var(--font-base)*0.65); font-weight: 800; line-height: 1.2; }
    .item .meta { font-size: calc(var(--font-base)*0.4); color: var(--muted); }
    .item.completed { background: rgba(76,175,80,0.08); }
    .item.completed .title { text-decoration: line-through; color: #a7f3d0; }
    .item.alert { background: rgba(239, 68, 68, 0.2); border-left: 10px solid var(--alert-color); animation: pulse 1.5s infinite; }
    .item.alert .tag-time { background: #ef4444; color: #fff; }
    @keyframes pulse { 0% { opacity: 1; } 50% { opacity: 0.6; } 100% { opacity: 1; } }
    .tag-time { font-size: 0.6em; background: #334155; padding: 4px 8px; border-radius: 6px; color: #e2e8f0; margin-left: 10px; vertical-align: middle; white-space: nowrap; }
    
    .btn { background: #6a1b9a; color: white; border: none; padding: 10px 20px; border-radius: 12px; cursor: pointer; font-weight: 700; font-size: calc(var(--font-base)*0.4); white-space: nowrap; pointer-events: auto; }
    .placeholder { color: #94a3b8; font-style: italic; padding: 28px; font-size: calc(var(--font-base)*0.33); text-align: center; }
    
    @media (max-width: 1024px) { :root { --font-base: 32px; } .header { padding: 15px 20px; } .main { padding: 15px; flex-direction: column; } .panel { padding: 15px; min-height: 40vh; } .item { padding: 15px 5px; } }
  </style>
</head>
<body>
  <div class="app">
    <div class="header" id="headerArea" title="Click to Fullscreen/Unlock Sound">
      <div class="brand">智能健身看板</div>
      <div class="datetime">
        <div id="currentDate">...</div>
        <div class="time" id="currentTime">...</div>
      </div>
    </div>
    <div class="main">
      <div class="panel"><h2>今日健身</h2><div id="fitContainer" class="scroll-container"><div id="fitContent" class="scroll-content"></div></div></div>
      <div class="panel"><h2>今日待办</h2><div id="todoContainer" class="scroll-container"><div id="todoContent" class="scroll-content"></div></div></div>
    </div>
  </div>

<script>
const API = location.origin.includes('localhost') ? 'http://localhost:8080' : location.origin;
const TODAY = () => new Date().toLocaleDateString('zh-CN', {year:'numeric',month:'2-digit',day:'2-digit'}).replace(/\//g, '-');

// --- 1. Interaction ---
let clickCount = 0, clickTimer = null;
document.getElementById('headerArea').addEventListener('click', () => {
    if (audioCtx.state === 'suspended') audioCtx.resume().then(()=>beep(true));
    if (!document.fullscreenElement) {
        const el = document.documentElement;
        (el.requestFullscreen || el.webkitRequestFullscreen).call(el);
    }
    clickCount++;
    if (clickCount === 1) clickTimer = setTimeout(() => { clickCount = 0; }, 500);
    if (clickCount >= 3) { clearTimeout(clickTimer); clickCount = 0; window.location.href = '/manage'; }
});

// --- 2. Delegation ---
function setupDelegation(id, isTodo) {
    const el = document.getElementById(id);
    el.addEventListener('click', (e) => {
        const btn = e.target.closest('.btn');
        if (!btn) return; 
        e.stopPropagation();
        isPaused = true; setTimeout(() => isPaused = false, 2000);
        const item = btn.closest('.item');
        toggle(item.dataset.id, isTodo, item.dataset.completed === 'true');
    });
}
setupDelegation('fitContent', false);
setupDelegation('todoContent', true);

async function toggle(id, isTodo, state) {
    const ep = isTodo ? 'mark_todo_completed' : 'mark_exercise_completed_mobile';
    const body = isTodo ? {todo_id:id, is_completed:!state} : {plan_id:id, is_completed:!state};
    await fetch(`${API}/${ep}`, {method:'POST', headers:{'Content-Type':'application/json'}, body:JSON.stringify(body), credentials:'include'});
    loadData();
}

// --- 3. Data Loading ---
let todos = [], fits = [];
let SCROLL_SPEED = 25; // Default
let loadTimer = null;

async function loadData() {
    try {
        const [r1, r2, r3] = await Promise.all([
            fetch(`${API}/fitness_plan/${TODAY()}`, {credentials:'include'}),
            fetch(`${API}/todos?date=${TODAY()}`, {credentials:'include'}),
            fetch(`${API}/api/settings/scroll_speed`)
        ]);
        if(r1.status===401) return window.location.href='/';
        
        const newFits = await r1.json();
        const newTodos = await r2.json();
        const s = await r3.json(); if(s.scroll_speed) SCROLL_SPEED = s.scroll_speed;

        smartRender('fitContent', newFits, false);
        smartRender('todoContent', newTodos, true);
        
        fits = newFits; todos = newTodos;
    } catch(e) { console.error(e); }
    finally {
        clearTimeout(loadTimer);
        loadTimer = setTimeout(loadData, 3000);
    }
}

function smartRender(id, newList, isTodo) {
    const el = document.getElementById(id);
    const now = new Date().getTime();
    if (el.children.length !== newList.length || newList.length === 0) {
        el.innerHTML = newList.map(i => buildItemHTML(i, isTodo, now)).join('');
        return;
    }
    let mismatch = false;
    Array.from(el.children).forEach((node, idx) => {
        if (mismatch) return;
        const data = newList[idx];
        if (parseInt(node.dataset.id) !== data.id) { mismatch = true; return; }
        const isAlert = !data.is_completed && data.reminder_at && now >= new Date(data.reminder_at).getTime();
        const newClass = `item ${data.is_completed ? 'completed' : ''} ${isAlert ? 'alert' : ''}`;
        if (node.className !== newClass) node.className = newClass;
        node.dataset.completed = data.is_completed;
        const btn = node.querySelector('.btn');
        const txt = data.is_completed ? '已完成' : '标记';
        if (btn.innerText !== txt) btn.innerText = txt;
    });
    if (mismatch) el.innerHTML = newList.map(i => buildItemHTML(i, isTodo, now)).join('');
}

function buildItemHTML(i, isTodo, now) {
    let alertClass = '';
    if(!i.is_completed && i.reminder_at && now >= new Date(i.reminder_at).getTime()) alertClass = 'alert';
    return `
      <div class="item ${i.is_completed ? 'completed' : ''} ${alertClass}" data-id="${i.id}" data-completed="${i.is_completed}">
        <div style="flex:1; min-width:0;">
          <div class="title">${isTodo ? i.content : i.exercise_name} ${i.reminder_at ? `<span class="tag-time">⏰ ${i.reminder_at.slice(11,16)}</span>` : ''}</div>
          <div class="meta">${isTodo ? '' : i.sets_reps_duration}</div>
        </div>
        <button class="btn">${i.is_completed ? '已完成' : '标记'}</button>
      </div>`;
}

// --- 4. Scroll Engine (SYNC FIX) ---
let isPaused = false;
const anims = {};
function restartScroll(conId, contentId) {
    if(anims[contentId]) cancelAnimationFrame(anims[contentId]);
    const con = document.getElementById(conId);
    const content = document.getElementById(contentId);
    content.style.transform = 'translate3d(0,0,0)';
    
    let y = 0, dir = 1, wait = 0;
    let lastT = performance.now();

    function step(t) {
        const dt = (t - lastT) / 1000;
        lastT = t;
        
        if (isPaused) { anims[contentId] = requestAnimationFrame(step); return; }
        
        const contentH = content.scrollHeight;
        const boxH = con.clientHeight;
        if (contentH <= boxH) {
            content.style.transform = 'translate3d(0,0,0)';
            anims[contentId] = requestAnimationFrame(step);
            return;
        }

        if (wait > 0) wait -= dt;
        else {
            const dist = contentH - boxH;
            // ★★★ FIX: Calculate speed EVERY FRAME based on GLOBAL SCROLL_SPEED ★★★
            const vel = dist / SCROLL_SPEED; 
            y += dir * vel * dt;
            if (y >= dist) { y = dist; dir = -1; wait = 2.0; }
            else if (y <= 0) { y = 0; dir = 1; wait = 2.0; }
        }
        content.style.transform = `translate3d(0, -${y}px, 0)`;
        anims[contentId] = requestAnimationFrame(step);
    }
    anims[contentId] = requestAnimationFrame(step);
}

// --- 5. Audio & Clock ---
const audioCtx = new (window.AudioContext || window.webkitAudioContext)();
function beep(quiet) {
    if(audioCtx.state === 'suspended') return;
    const o = audioCtx.createOscillator(); const g = audioCtx.createGain();
    o.connect(g); g.connect(audioCtx.destination);
    o.frequency.value = 880; g.gain.value = quiet?0.05:0.2;
    o.start(); setTimeout(()=>o.stop(),300);
}
let lastBeep = 0;
function checkAlerts() {
    const now = new Date().getTime();
    const alert = [...todos, ...fits].some(i => !i.is_completed && i.reminder_at && now >= new Date(i.reminder_at).getTime());
    if(alert) {
        if(now - lastBeep > 15000) { if(audioCtx.state === 'running') beep(); lastBeep = now; }
        smartRender('fitContent', fits, false); smartRender('todoContent', todos, true);
    }
}
setInterval(() => {
    const n = new Date();
    document.getElementById('currentDate').innerText = n.toLocaleDateString('zh-CN',{weekday:'long', month:'long', day:'numeric'});
    document.getElementById('currentTime').innerText = n.toLocaleTimeString('zh-CN',{hour12:false});
    checkAlerts();
}, 1000);

// Init
loadData();
// Start scroll loops
setTimeout(() => { restartScroll('fitContainer', 'fitContent'); restartScroll('todoContainer', 'todoContent'); }, 500);
</script>
</body>
</html>
'''

def write_file(filename, content, folder="."):
    filepath = os.path.join(folder, filename)
    try:
        with open(filepath, 'w', encoding='utf-8') as f: f.write(content)
        print(f"[OK] Updated: {filepath}")
    except Exception as e: print(f"[ERR] Failed {filepath}: {e}")

def main():
    print("=== V3.0 Upgrade: Sync Scroll Speed & Unlock Limits ===")
    backend_dir = "backend" if os.path.exists("backend") else "."
    write_file('app.py', app_py_content, backend_dir)
    
    td = "templates" if os.path.exists("templates") else ("frontend/templates" if os.path.exists("frontend/templates") else ".")
    write_file('manage.html', manage_html_content, td)
    write_file('display.html', display_html_content, td)
    print("\nDone. Please restart python backend/app.py")

if __name__ == '__main__': main()
