import os

# ==============================================================================
# 1. backend/app.py (v2.5 - Port 8080 & Stability) - 保持不变，确保后端稳定
# ==============================================================================
app_py_content = r'''# backend/app.py (v2.5 - Stable)
import sqlite3
from datetime import datetime, timedelta, timezone
from flask import Flask, render_template, request, jsonify, g, redirect, url_for
from flask_login import LoginManager, UserMixin, login_user, logout_user, login_required, current_user
import os, json, uuid
from flask_cors import CORS
from functools import wraps
import sys 
from collections import defaultdict

CN_TZ = timezone(timedelta(hours=8))

is_frozen = getattr(sys, 'frozen', False)
is_in_onefile_temp = 'onefile_' in os.path.abspath(__file__) 
if is_frozen or is_in_onefile_temp:
    base_dir = os.path.dirname(os.path.abspath(sys.executable))
    template_dir = os.path.join(base_dir, 'templates')
    static_dir = os.path.join(base_dir, 'static')
    DATABASE = os.path.join(os.path.dirname(os.path.realpath(sys.argv[0])), 'fitness_manager.db')
else:
    base_dir = os.path.dirname(os.path.abspath(__file__))
    project_root = os.path.dirname(base_dir)
    if os.path.exists(os.path.join(project_root, 'frontend', 'templates')):
        template_dir = os.path.join(project_root, 'frontend', 'templates')
        static_dir = os.path.join(project_root, 'frontend', 'static')
    else:
        template_dir = os.path.join(base_dir, 'templates')
        static_dir = os.path.join(base_dir, 'static')
    DATABASE = os.path.join(base_dir, 'fitness_manager.db')

if not os.path.exists(template_dir): os.makedirs(template_dir, exist_ok=True)

app = Flask(__name__, template_folder=template_dir, static_folder=static_dir)
CORS(app)
app.secret_key = str(uuid.uuid4())

login_manager = LoginManager()
login_manager.init_app(app)
login_manager.login_view = 'index' 

class User(UserMixin):
    def __init__(self, id, username, is_admin, manager_id):
        self.id = id; self.username = username; self.is_admin = is_admin; self.manager_id = manager_id 
    @property
    def is_super_admin(self): return self.is_admin and (self.manager_id is None or self.manager_id == 0)

@login_manager.user_loader
def load_user(uid):
    row = get_db().execute("SELECT * FROM users WHERE id=?", (uid,)).fetchone()
    return User(row['id'], row['username'], row['is_admin'], row['manager_id']) if row else None

def get_db():
    db = getattr(g, '_database', None)
    if db is None: db = g._database = sqlite3.connect(DATABASE, check_same_thread=False); db.row_factory = sqlite3.Row
    return db

def get_setting(key, default=None):
    row = get_db().execute("SELECT value FROM settings WHERE key = ?", (key,)).fetchone()
    return default if row is None else row["value"]

def set_setting(key, value):
    db = get_db()
    db.execute("INSERT INTO settings (key, value) VALUES (?, ?) ON CONFLICT(key) DO UPDATE SET value = excluded.value", (key, str(value)))
    db.commit()

@app.teardown_appcontext
def close_conn(e):
    db = getattr(g, '_database', None)
    if db is not None: db.close()

def init_db():
    db = get_db()
    cur = db.cursor()
    cur.execute("CREATE TABLE IF NOT EXISTS todos (id INTEGER PRIMARY KEY AUTOINCREMENT, date TEXT NOT NULL, content TEXT NOT NULL, is_completed INTEGER DEFAULT 0, user_id INTEGER, recurring_todo_id INTEGER, reminder_at TEXT)")
    cur.execute("CREATE TABLE IF NOT EXISTS fitness_plans (id INTEGER PRIMARY KEY AUTOINCREMENT, date TEXT NOT NULL, exercise_name TEXT NOT NULL, sets_reps_duration TEXT, is_completed INTEGER DEFAULT 0, user_id INTEGER, recurring_plan_id INTEGER, reminder_at TEXT)")
    cur.execute("CREATE TABLE IF NOT EXISTS recurring_plans (id INTEGER PRIMARY KEY AUTOINCREMENT, user_id INTEGER, plan_name TEXT NOT NULL, exercises_json TEXT NOT NULL, start_date TEXT NOT NULL, end_date TEXT, repeat_interval TEXT NOT NULL, repeat_day INTEGER, reminder_time TEXT)")
    cur.execute("CREATE TABLE IF NOT EXISTS recurring_todos (id INTEGER PRIMARY KEY AUTOINCREMENT, user_id INTEGER, content TEXT NOT NULL, start_date TEXT NOT NULL, end_date TEXT, repeat_interval TEXT NOT NULL, repeat_day INTEGER, reminder_time TEXT)")
    cur.execute("CREATE TABLE IF NOT EXISTS exercise_library (id INTEGER PRIMARY KEY AUTOINCREMENT, user_id INTEGER, name TEXT, details TEXT)")
    cur.execute("CREATE TABLE IF NOT EXISTS todo_library (id INTEGER PRIMARY KEY AUTOINCREMENT, user_id INTEGER, content TEXT)")
    cur.execute("CREATE TABLE IF NOT EXISTS settings (key TEXT PRIMARY KEY, value TEXT)")
    cur.execute("CREATE TABLE IF NOT EXISTS users (id INTEGER PRIMARY KEY AUTOINCREMENT, username TEXT UNIQUE, password TEXT, is_admin INTEGER DEFAULT 0, manager_id INTEGER)")
    
    try: cur.execute("SELECT reminder_at FROM todos LIMIT 1"); 
    except: cur.execute("ALTER TABLE todos ADD COLUMN reminder_at TEXT")
    try: cur.execute("SELECT reminder_at FROM fitness_plans LIMIT 1"); 
    except: cur.execute("ALTER TABLE fitness_plans ADD COLUMN reminder_at TEXT")
    try: cur.execute("SELECT reminder_time FROM recurring_plans LIMIT 1"); 
    except: cur.execute("ALTER TABLE recurring_plans ADD COLUMN reminder_time TEXT")
    try: cur.execute("SELECT reminder_time FROM recurring_todos LIMIT 1"); 
    except: cur.execute("ALTER TABLE recurring_todos ADD COLUMN reminder_time TEXT")
    
    cur.execute("INSERT OR IGNORE INTO settings (key, value) VALUES ('scroll_speed', '25')")
    if cur.execute("SELECT COUNT(*) FROM users WHERE username='superuser'").fetchone()[0]==0:
        cur.execute("INSERT INTO users (username,password,is_admin,manager_id) VALUES (?,?,?,?)",('superuser','adminpassword',1,None))
    su_id = cur.execute("SELECT id FROM users WHERE username='superuser'").fetchone()['id']
    if cur.execute("SELECT COUNT(*) FROM users WHERE username='user1'").fetchone()[0]==0:
        cur.execute("INSERT INTO users (username,password,is_admin,manager_id) VALUES (?,?,?,?)",('user1','password',0,su_id))
    db.commit()

with app.app_context(): init_db()

@app.route('/login', methods=['POST'])
def login():
    d = request.get_json(silent=True) or request.form
    u = get_db().execute("SELECT * FROM users WHERE username=?",(d.get('username'),)).fetchone()
    if u and u['password']==d.get('password'):
        login_user(User(u['id'],u['username'],u['is_admin'],u['manager_id']), remember=True)
        return jsonify({'status':'success'})
    return jsonify({'status':'error'}), 401

@app.route('/logout', methods=['POST'])
@login_required
def logout(): logout_user(); return jsonify({'status':'success'})

@app.route('/')
def index():
    if not current_user.is_authenticated: return render_template('login.html')
    if 'Mobile' in request.headers.get('User-Agent','') and request.args.get('view')!='display': return redirect(url_for('mobile_config'))
    today = datetime.now(CN_TZ).strftime('%Y-%m-%d')
    _gen_fit(current_user.id, today)
    _gen_todo(current_user.id, today)
    return render_template('display.html', username=current_user.username)

@app.route('/manage')
@login_required
def manage(): return render_template('manage.html', username=current_user.username, is_admin=current_user.is_admin)

@app.route('/mobile_config')
@login_required
def mobile_config(): return render_template('mobile_config.html', username=current_user.username)

@app.route('/todos', methods=['GET'])
@login_required
def get_todos():
    d = request.args.get('date') or datetime.now(CN_TZ).strftime('%Y-%m-%d')
    return jsonify([dict(r) for r in get_db().execute("SELECT * FROM todos WHERE date=? AND user_id=? ORDER BY is_completed ASC, reminder_at ASC, id DESC", (d, current_user.id)).fetchall()])

@app.route('/add_todo', methods=['POST'])
@login_required
def add_todo():
    d = request.get_json(silent=True) or request.form
    rem = d.get('reminder_at')
    if rem and len(rem)==5: rem = f"{d.get('date')} {rem}"
    get_db().execute("INSERT INTO todos (date,content,is_completed,user_id,reminder_at) VALUES (?,?,0,?,?)", (d.get('date'),d.get('content'),current_user.id,rem)).connection.commit()
    return jsonify(status='success')

@app.route('/add_recurring_todo', methods=['POST'])
@login_required
def add_rec_todo():
    d = request.get_json(silent=True) or request.form
    end_d = d.get('end_date')
    if not end_d: end_d = None
    db = get_db()
    cur = db.cursor()
    cur.execute("INSERT INTO recurring_todos (user_id,content,start_date,end_date,repeat_interval,repeat_day,reminder_time) VALUES (?,?,?,?,?,?,?)",
                (current_user.id, d.get('content'), d.get('start_date'), end_d, d.get('repeat_interval'), d.get('repeat_day'), d.get('reminder_time')))
    new_id = cur.lastrowid
    db.commit()
    _gen_single_rec_todo(current_user.id, datetime.now(CN_TZ).strftime('%Y-%m-%d'), new_id)
    return jsonify(status='success')

@app.route('/fitness_plan/<date>', methods=['GET'])
@login_required
def get_fit(date):
    _gen_fit(current_user.id, date)
    return jsonify([dict(r) for r in get_db().execute("SELECT * FROM fitness_plans WHERE date=? AND user_id=? ORDER BY is_completed ASC, reminder_at ASC, id DESC", (date, current_user.id)).fetchall()])

@app.route('/add_recurring_plan', methods=['POST'])
@login_required
def add_rec_plan():
    d = request.get_json(silent=True) or request.form
    end_d = d.get('end_date')
    if not end_d: end_d = None
    db = get_db()
    cur = db.cursor()
    cur.execute("INSERT INTO recurring_plans (user_id,plan_name,exercises_json,start_date,end_date,repeat_interval,repeat_day,reminder_time) VALUES (?,?,?,?,?,?,?,?)",
                (current_user.id, d.get('plan_name'), json.dumps(d.get('exercises')), d.get('start_date'), end_d, d.get('repeat_interval'), d.get('repeat_day'), d.get('reminder_time')))
    new_id = cur.lastrowid
    db.commit()
    _gen_single_rec_plan(current_user.id, datetime.now(CN_TZ).strftime('%Y-%m-%d'), new_id)
    return jsonify(status='success')

@app.route('/delete_todo/<int:id>', methods=['POST'])
@login_required
def del_todo(id): get_db().execute("DELETE FROM todos WHERE id=? AND user_id=?",(id,current_user.id)).connection.commit(); return jsonify(status='success')
@app.route('/mark_todo_completed', methods=['POST'])
@login_required
def mark_todo(): d=request.get_json(silent=True) or request.form; get_db().execute("UPDATE todos SET is_completed=? WHERE id=? AND user_id=?",(d.get('is_completed'),d.get('todo_id'),current_user.id)).connection.commit(); return jsonify(status='success')
@app.route('/delete_fitness_plan_exercise/<int:id>', methods=['POST'])
@login_required
def del_fit(id): get_db().execute("DELETE FROM fitness_plans WHERE id=? AND user_id=?",(id,current_user.id)).connection.commit(); return jsonify(status='success')
@app.route('/mark_exercise_completed_mobile', methods=['POST'])
@login_required
def mark_fit(): d=request.get_json(silent=True) or request.form; get_db().execute("UPDATE fitness_plans SET is_completed=? WHERE id=? AND user_id=?",(d.get('is_completed'),d.get('plan_id'),current_user.id)).connection.commit(); return jsonify(status='success')
@app.route('/get_recurring_todos', methods=['GET'])
@login_required
def get_rec_todos_list(): return jsonify([dict(r) for r in get_db().execute("SELECT * FROM recurring_todos WHERE user_id=?",(current_user.id,)).fetchall()])
@app.route('/delete_recurring_todo/<int:id>', methods=['POST'])
@login_required
def del_rec_todo(id): get_db().execute("DELETE FROM recurring_todos WHERE id=? AND user_id=?",(id,current_user.id)).connection.commit(); return jsonify(status='success')
@app.route('/get_recurring_plans', methods=['GET'])
@login_required
def get_rec_plans_list(): return jsonify([dict(r) for r in get_db().execute("SELECT * FROM recurring_plans WHERE user_id=?",(current_user.id,)).fetchall()])
@app.route('/delete_recurring_plan/<int:id>', methods=['POST'])
@login_required
def del_rec_plan(id): get_db().execute("DELETE FROM recurring_plans WHERE id=? AND user_id=?",(id,current_user.id)).connection.commit(); return jsonify(status='success')
@app.route('/add_fitness_manual', methods=['POST'])
@login_required
def add_fit_man():
    d=request.get_json(silent=True) or request.form
    rem = d.get('reminder_at')
    if rem and len(rem)==5: rem = f"{d.get('date')} {rem}"
    get_db().execute("INSERT INTO fitness_plans (date,exercise_name,sets_reps_duration,user_id,reminder_at) VALUES (?,?,?,?,?)",(d.get('date'),d.get('exercise_name'),d.get('sets_reps_duration'),current_user.id,rem)).connection.commit()
    return jsonify(status='success')

@app.route('/api/settings/scroll_speed', methods=['GET','POST'])
def scroll_speed():
    if request.method=='POST': 
        d=request.get_json(silent=True) or request.form
        try: v=int(d.get('scroll_speed'))
        except: v=25
        set_setting('scroll_speed', v); return jsonify(status='success', scroll_speed=v)
    try: val = int(get_setting('scroll_speed', 25))
    except: val = 25
    return jsonify(scroll_speed=val)

@app.route('/api/exercises', methods=['GET','POST'])
@app.route('/api/exercises/<int:id>', methods=['DELETE'])
@login_required
def ex_api(id=None):
    if request.method=='GET': return jsonify([dict(r) for r in get_db().execute("SELECT * FROM exercise_library WHERE user_id=?",(current_user.id,)).fetchall()])
    if request.method=='POST': d=request.get_json(silent=True) or request.form; get_db().execute("INSERT INTO exercise_library (user_id,name,details) VALUES (?,?,?)",(current_user.id,d.get('name'),d.get('details'))).connection.commit(); return jsonify(status='success')
    if request.method=='DELETE': get_db().execute("DELETE FROM exercise_library WHERE id=?",(id,)).connection.commit(); return jsonify(status='success')

@app.route('/api/todo_library', methods=['GET','POST'])
@app.route('/api/todo_library/<int:id>', methods=['DELETE'])
@login_required
def td_api(id=None):
    if request.method=='GET': return jsonify([dict(r) for r in get_db().execute("SELECT * FROM todo_library WHERE user_id=?",(current_user.id,)).fetchall()])
    if request.method=='POST': d=request.get_json(silent=True) or request.form; get_db().execute("INSERT INTO todo_library (user_id,content) VALUES (?,?)",(current_user.id,d.get('content'))).connection.commit(); return jsonify(status='success')
    if request.method=='DELETE': get_db().execute("DELETE FROM todo_library WHERE id=?",(id,)).connection.commit(); return jsonify(status='success')

def admin_required(f):
    @wraps(f)
    def dec(*args, **kwargs):
        if not current_user.is_authenticated or not current_user.is_admin: return jsonify(status='error'), 403
        return f(*args, **kwargs)
    return dec

@app.route('/api/users', methods=['GET','POST'])
@app.route('/api/users/<int:id>', methods=['DELETE'])
@login_required
@admin_required
def users_api(id=None):
    if request.method=='GET': return jsonify([dict(u) for u in get_db().execute("SELECT id,username,is_admin FROM users WHERE manager_id=?",(current_user.id,)).fetchall()])
    if request.method=='POST':
        d=request.get_json(silent=True) or request.form
        try: get_db().execute("INSERT INTO users (username,password,is_admin,manager_id) VALUES (?,?,?,?)",(d.get('username'),d.get('password'),1 if d.get('is_admin') else 0, current_user.id)).connection.commit(); return jsonify(status='success')
        except: return jsonify(status='error')
    if request.method=='DELETE':
        if id==current_user.id: return jsonify(status='error')
        get_db().execute("DELETE FROM users WHERE id=?",(id,)).connection.commit(); return jsonify(status='success')

@app.route('/api/profile', methods=['POST'])
@login_required
def profile():
    d=request.get_json(silent=True) or request.form
    if d.get('password'): get_db().execute("UPDATE users SET password=? WHERE id=?",(d.get('password'),current_user.id))
    if d.get('username'): 
        try: get_db().execute("UPDATE users SET username=? WHERE id=?",(d.get('username'),current_user.id))
        except: pass
    get_db().commit(); return jsonify(status='success')

@app.route('/api/stats/daily_completion')
@login_required
def stats():
    s, e = request.args.get('start_date'), request.args.get('end_date')
    db = get_db()
    f = db.execute("SELECT date, is_completed FROM fitness_plans WHERE user_id=? AND date BETWEEN ? AND ?", (current_user.id, s, e)).fetchall()
    t = db.execute("SELECT date, is_completed FROM todos WHERE user_id=? AND date BETWEEN ? AND ?", (current_user.id, s, e)).fetchall()
    fd = defaultdict(lambda: {"total":0,"completed":0}); td = defaultdict(lambda: {"total":0,"completed":0})
    for r in f: fd[r['date']]['total']+=1; 
    for r in f: 
        if r['is_completed']: fd[r['date']]['completed']+=1
    for r in t: td[r['date']]['total']+=1; 
    for r in t:
        if r['is_completed']: td[r['date']]['completed']+=1
    return jsonify(fitness_data=fd, todo_data=td)

def _gen_fit(uid, date_str):
    db = get_db()
    if db.execute("SELECT COUNT(*) FROM fitness_plans WHERE date=? AND user_id=?",(date_str,uid)).fetchone()[0] > 0: return
    plans = db.execute("SELECT * FROM recurring_plans WHERE user_id=? AND start_date<=? AND (end_date IS NULL OR end_date>=?)", (uid,date_str,date_str)).fetchall()
    dt = datetime.strptime(date_str, '%Y-%m-%d').date()
    for p in plans:
        if _is_due(p, dt): _insert_fit(db, uid, date_str, p)
    db.commit()

def _gen_todo(uid, date_str):
    db = get_db()
    if db.execute("SELECT COUNT(*) FROM todos WHERE date=? AND user_id=?",(date_str,uid)).fetchone()[0] > 0: return
    recs = db.execute("SELECT * FROM recurring_todos WHERE user_id=? AND start_date<=? AND (end_date IS NULL OR end_date>=?)", (uid,date_str,date_str)).fetchall()
    dt = datetime.strptime(date_str, '%Y-%m-%d').date()
    for r in recs:
        if _is_due(r, dt): _insert_todo(db, uid, date_str, r)
    db.commit()

def _gen_single_rec_todo(uid, date_str, rec_id):
    db = get_db()
    r = db.execute("SELECT * FROM recurring_todos WHERE id=?",(rec_id,)).fetchone()
    if not r: return
    dt = datetime.strptime(date_str, '%Y-%m-%d').date()
    if r['start_date'] <= date_str and (not r['end_date'] or r['end_date'] >= date_str):
        if _is_due(r, dt): _insert_todo(db, uid, date_str, r)
    db.commit()

def _gen_single_rec_plan(uid, date_str, rec_id):
    db = get_db()
    p = db.execute("SELECT * FROM recurring_plans WHERE id=?",(rec_id,)).fetchone()
    if not p: return
    dt = datetime.strptime(date_str, '%Y-%m-%d').date()
    if p['start_date'] <= date_str and (not p['end_date'] or p['end_date'] >= date_str):
        if _is_due(p, dt): _insert_fit(db, uid, date_str, p)
    db.commit()

def _is_due(item, dt):
    if item['repeat_interval'] == 'daily': return True
    if item['repeat_interval'] == 'weekly' and item['repeat_day'] == dt.isoweekday(): return True
    if item['repeat_interval'] == 'monthly' and item['repeat_day'] == dt.day: return True
    return False

def _insert_todo(db, uid, date_str, r):
    if db.execute("SELECT COUNT(*) FROM todos WHERE date=? AND recurring_todo_id=?",(date_str, r['id'])).fetchone()[0] > 0: return
    rem = f"{date_str} {r['reminder_time']}" if r['reminder_time'] else None
    db.execute("INSERT INTO todos (date, content, is_completed, user_id, recurring_todo_id, reminder_at) VALUES (?,?,?,?,?,?)",
               (date_str, r['content'], 0, uid, r['id'], rem))

def _insert_fit(db, uid, date_str, p):
    if db.execute("SELECT COUNT(*) FROM fitness_plans WHERE date=? AND recurring_plan_id=?",(date_str, p['id'])).fetchone()[0] > 0: return
    rem = f"{date_str} {p['reminder_time']}" if p['reminder_time'] else None
    for ex in json.loads(p['exercises_json']):
        db.execute("INSERT INTO fitness_plans (date, exercise_name, sets_reps_duration, is_completed, user_id, recurring_plan_id, reminder_at) VALUES (?,?,?,0,?,?,?)",
                   (date_str, ex['name'], ex['details'], uid, p['id'], rem))

@app.after_request
def add_headers(r):
    r.headers['Cache-Control'] = 'no-cache, no-store, must-revalidate'
    return r

@app.route('/health', methods=['GET'])
def health(): return jsonify(status='ok', time=datetime.now(CN_TZ).isoformat())

if __name__ == '__main__':
    app.run(host='0.0.0.0', port=8080, debug=True)
'''

# ==============================================================================
# 2. frontend/templates/display.html (v3.0 - Fullscreen Overlay + Mobile Fixes)
# ==============================================================================
display_html_content = r'''<!DOCTYPE html>
<html lang="zh-CN">
<head>
  <meta charset="utf-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no" />
  <title>智能健身看板</title>
  <style>
    :root {
      --bg: #0f1724; --panel: #1e293b; --accent: #4CAF50; --muted: #94a3b8;
      --card-radius: 18px; --font-base: 56px; --alert-color: #ef4444;
    }
    html, body { height: 100%; margin: 0; padding: 0; background: var(--bg); color: #fff; font-family: sans-serif; overflow: hidden; user-select: none; }
    
    /* Layout */
    .app { height: 100vh; display: flex; flex-direction: column; }
    .header { display: flex; justify-content: space-between; align-items: center; padding: 28px 48px; background: linear-gradient(90deg, #071029, #0b2540); box-shadow: 0 8px 20px rgba(0,0,0,0.6); flex-shrink: 0; z-index: 10; }
    .brand { font-size: calc(var(--font-base)*0.6); color: var(--accent); font-weight: 700; }
    .datetime { text-align: right; font-size: calc(var(--font-base)*0.38); color: #dbeafe; cursor: pointer; }
    .datetime .time { font-size: calc(var(--font-base)*0.9); font-weight: 700; }
    
    /* Flex Fix for Mobile: Use min-height:0 */
    .main { padding: 28px; display: flex; gap: 28px; flex: 1; box-sizing: border-box; overflow: hidden; min-height: 0; }
    .panel { background: var(--panel); border-radius: var(--card-radius); padding: 28px; box-shadow: 0 12px 30px rgba(0,0,0,0.55); display: flex; flex-direction: column; flex: 1; overflow: hidden; min-height: 0; }
    
    h2 { margin: 0 0 18px 0; color: var(--accent); font-size: calc(var(--font-base)*0.48); border-bottom: 3px solid rgba(255,255,255,0.03); padding-bottom: 12px; flex-shrink: 0; }
    
    /* Scrolling Area */
    .scroll-container { position: relative; overflow: hidden; flex: 1; min-height: 0; }
    .scroll-content { position: absolute; top: 0; left: 0; width: 100%; will-change: transform; }
    
    /* Items */
    .item { display: flex; justify-content: space-between; align-items: center; padding: 24px 10px; border-bottom: 1px solid rgba(255,255,255,0.05); gap: 15px; }
    .item .title { font-size: calc(var(--font-base)*0.65); font-weight: 800; line-height: 1.2; }
    .item .meta { font-size: calc(var(--font-base)*0.4); color: var(--muted); }
    .item.completed { background: rgba(76,175,80,0.08); }
    .item.completed .title { text-decoration: line-through; color: #a7f3d0; }
    
    /* Alert */
    .item.alert { background: rgba(239, 68, 68, 0.2); border-left: 10px solid var(--alert-color); animation: pulse 1.5s infinite; }
    .item.alert .tag-time { background: #ef4444; color: #fff; }
    @keyframes pulse { 0% { opacity: 1; } 50% { opacity: 0.6; } 100% { opacity: 1; } }
    .tag-time { font-size: 0.6em; background: #334155; padding: 4px 8px; border-radius: 6px; color: #e2e8f0; margin-left: 10px; vertical-align: middle; white-space: nowrap; }

    .btn { background: #6a1b9a; color: white; border: none; padding: 10px 20px; border-radius: 12px; cursor: pointer; font-weight: 700; font-size: calc(var(--font-base)*0.4); white-space: nowrap; }
    .placeholder { color: #94a3b8; font-style: italic; padding: 28px; font-size: calc(var(--font-base)*0.33); text-align: center; }

    /* Overlay for Start */
    #startOverlay { position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.85); z-index: 9999; display: flex; justify-content: center; align-items: center; flex-direction: column; color: #fff; cursor: pointer; }
    #startOverlay h1 { font-size: 40px; margin-bottom: 20px; }
    #startOverlay p { font-size: 20px; color: #ccc; }

    /* Mobile/Tablet Optimization */
    @media (max-width: 1024px) {
        :root { --font-base: 32px; } /* Smaller font base */
        .header { padding: 15px 20px; }
        .main { padding: 15px; flex-direction: column; } /* Vertical stack */
        .panel { padding: 15px; min-height: 40vh; } /* Ensure height */
        .item { padding: 15px 5px; }
    }
  </style>
</head>
<body>
  <!-- Overlay to force interaction for Audio/Fullscreen -->
  <div id="startOverlay">
    <h1>点击屏幕启动</h1>
    <p>Click anywhere to Start Fullscreen & Sound</p>
  </div>

  <div class="app">
    <div class="header">
      <div class="brand">智能健身看板</div>
      <div class="datetime" id="clockArea">
        <div id="currentDate">...</div>
        <div class="time" id="currentTime">...</div>
      </div>
    </div>
    <div class="main">
      <div class="panel">
        <h2>今日健身</h2>
        <div id="fitContainer" class="scroll-container"><div id="fitContent" class="scroll-content"></div></div>
      </div>
      <div class="panel">
        <h2>今日待办</h2>
        <div id="todoContainer" class="scroll-container"><div id="todoContent" class="scroll-content"></div></div>
      </div>
    </div>
  </div>

<script>
const API = location.origin.includes('localhost') ? 'http://localhost:8080' : location.origin;
const TODAY = () => new Date().toLocaleDateString('zh-CN', {year:'numeric',month:'2-digit',day:'2-digit'}).replace(/\//g, '-');

// --- 1. Interaction Start (Fullscreen + Audio) ---
const startOverlay = document.getElementById('startOverlay');
const audioCtx = new (window.AudioContext || window.webkitAudioContext)();

startOverlay.addEventListener('click', () => {
    // 1. Fullscreen
    const docEl = document.documentElement;
    if (docEl.requestFullscreen) docEl.requestFullscreen();
    else if (docEl.webkitRequestFullscreen) docEl.webkitRequestFullscreen();
    
    // 2. Unlock Audio
    audioCtx.resume().then(() => {
        console.log("Audio Context Resumed");
        beep(true); // Test beep
    });

    // 3. Hide Overlay
    startOverlay.style.display = 'none';
    loadData(); // Force reload
});

function beep(quiet=false) {
    if(audioCtx.state === 'suspended') audioCtx.resume();
    const osc = audioCtx.createOscillator();
    const gain = audioCtx.createGain();
    osc.connect(gain);
    gain.connect(audioCtx.destination);
    osc.frequency.value = 880; 
    gain.gain.value = quiet ? 0.01 : 0.15;
    osc.start();
    setTimeout(() => osc.stop(), 300);
}

// --- 2. Clock & Triple Click ---
let clicks = 0;
document.getElementById('clockArea').addEventListener('click', (e) => {
    e.stopPropagation(); // Prevent bubbling
    clicks++;
    if(clicks === 1) setTimeout(() => clicks = 0, 600); 
    if(clicks >= 3) window.location.href = '/manage';
});

setInterval(() => {
    const now = new Date();
    document.getElementById('currentDate').innerText = now.toLocaleDateString('zh-CN',{weekday:'long', month:'long', day:'numeric'});
    document.getElementById('currentTime').innerText = now.toLocaleTimeString('zh-CN',{hour12:false});
    checkAlerts();
}, 1000);

// --- 3. Data & Render ---
let todos = [], fits = [];
let SCROLL_SPEED = 25;

async function loadData() {
    try {
        const [r1, r2, r3] = await Promise.all([
            fetch(`${API}/fitness_plan/${TODAY()}`, {credentials:'include'}),
            fetch(`${API}/todos?date=${TODAY()}`, {credentials:'include'}),
            fetch(`${API}/api/settings/scroll_speed`)
        ]);
        if(r1.status===401) return window.location.href='/';
        
        fits = await r1.json();
        todos = await r2.json();
        const s = await r3.json(); if(s.scroll_speed) SCROLL_SPEED = s.scroll_speed;

        render('fitContent', fits, false);
        render('todoContent', todos, true);
        
        // Wait for DOM render then measure
        setTimeout(() => {
            restartScroll('fitContainer', 'fitContent');
            restartScroll('todoContainer', 'todoContent');
        }, 100);
    } catch(e) { console.error(e); }
}

function render(id, list, isTodo) {
    const el = document.getElementById(id);
    if(!list.length) { el.innerHTML='<div class="placeholder">暂无内容</div>'; return; }
    const now = new Date().getTime();
    
    el.innerHTML = list.map(i => {
        let alertClass = '';
        if(!i.is_completed && i.reminder_at && now >= new Date(i.reminder_at).getTime()) {
            alertClass = 'alert';
        }
        return `
      <div class="item ${i.is_completed ? 'completed' : ''} ${alertClass}">
        <div style="flex:1; min-width:0;">
          <div class="title">
            ${isTodo ? i.content : i.exercise_name}
            ${i.reminder_at ? `<span class="tag-time">⏰ ${i.reminder_at.slice(11,16)}</span>` : ''}
          </div>
          <div class="meta">${isTodo ? '' : i.sets_reps_duration}</div>
        </div>
        <button class="btn" onclick="toggle(event, ${i.id}, ${isTodo}, ${i.is_completed})">
            ${i.is_completed ? '已完成' : '标记'}
        </button>
      </div>`;
    }).join('');
}

async function toggle(e, id, isTodo, state) {
    e.stopPropagation();
    const ep = isTodo ? 'mark_todo_completed' : 'mark_exercise_completed_mobile';
    const body = isTodo ? {todo_id:id, is_completed:!state} : {plan_id:id, is_completed:!state};
    await fetch(`${API}/${ep}`, {method:'POST', headers:{'Content-Type':'application/json'}, body:JSON.stringify(body), credentials:'include'});
    loadData();
}

// --- 4. Alert Logic ---
let lastBeep = 0;
function checkAlerts() {
    const now = new Date().getTime();
    const hasT = todos.some(t => !t.is_completed && t.reminder_at && now >= new Date(t.reminder_at).getTime());
    const hasF = fits.some(f => !f.is_completed && f.reminder_at && now >= new Date(f.reminder_at).getTime());

    if(hasT || hasF) {
        if(now - lastBeep > 15000) { beep(); lastBeep = now; }
        if(!document.querySelector('.alert')) {
             render('fitContent', fits, false);
             render('todoContent', todos, true);
        }
    }
}

// --- 5. Robust Scrolling ---
const anims = {};
function restartScroll(conId, contentId) {
    if(anims[contentId]) cancelAnimationFrame(anims[contentId]);
    const con = document.getElementById(conId);
    const content = document.getElementById(contentId);
    
    content.style.transform = 'translateY(0px)';
    // Only scroll if content is taller than container
    if(content.scrollHeight <= con.clientHeight) return; 
    
    let pos = 0;
    let dir = 1;
    // Calculate speed: Total distance / (Seconds * 60fps)
    // If scroll speed is 25s, it should take 25s to scroll bottom-to-top (or one cycle)
    const dist = content.scrollHeight - con.clientHeight;
    const speed = dist / (SCROLL_SPEED * 60); 
    
    function step() {
        pos += dir * speed;
        
        if(pos > dist) { 
            pos = dist; 
            dir = -1; 
            setTimeout(() => { dir = 1; pos = 0; }, 2000); // Pause at bottom
        } else if (pos < 0) {
            pos = 0;
            dir = 1;
        }
        
        content.style.transform = `translateY(-${pos}px)`;
        anims[contentId] = requestAnimationFrame(step);
    }
    anims[contentId] = requestAnimationFrame(step);
}

// Auto reload data every 10s
setInterval(loadData, 10000);
// Initial load done after click overlay
</script>
</body>
</html>
'''

# ==============================================================================
# Write Utility
# ==============================================================================
def write_file(filename, content, folder="."):
    filepath = os.path.join(folder, filename)
    try:
        with open(filepath, 'w', encoding='utf-8') as f: f.write(content)
        print(f"[成功] 更新文件: {filepath}")
    except Exception as e: print(f"[错误] {filepath}: {e}")

def main():
    print("=== V3.0 升级：解决全屏/声音/小屏显示问题 ===")
    
    # 1. update app.py (Port 8080)
    backend_dir = "backend" if os.path.exists("backend") else "."
    write_file('app.py', app_py_content, backend_dir)
    
    # 2. update templates (manage.html unchanged from v2.4, display.html NEW)
    td = "templates" if os.path.exists("templates") else ("frontend/templates" if os.path.exists("frontend/templates") else ".")
    
    # Note: manage.html is not included in this specific snippet as it was fine in v2.4
    # If you need manage.html again, use v2.4 version. This update focuses on DISPLAY & APP issues.
    write_file('display.html', display_html_content, td)
    
    print("\n升级完成。")
    print("1. 重启程序: python backend/app.py")
    print("2. 在平板上打开页面，点击【点击屏幕启动】黑色遮罩层。")
    print("3. 系统将自动全屏并解锁声音。")

if __name__ == '__main__': main()
