# backend/app.py (v6.4 - Config Driven)
import sqlite3
from datetime import datetime, timedelta, timezone
from flask import Flask, render_template, request, jsonify, g, redirect, url_for
from flask_login import LoginManager, UserMixin, login_user, logout_user, login_required, current_user
import os, json, uuid, random, time
from flask_cors import CORS
from functools import wraps
import sys 
from collections import defaultdict
import threading
import paho.mqtt.client as mqtt

CN_TZ = timezone(timedelta(hours=8))

# --- Path Logic & Config ---
is_frozen = getattr(sys, 'frozen', False)
is_in_onefile_temp = 'onefile_' in os.path.abspath(__file__) 

if is_frozen or is_in_onefile_temp:
    base_run_dir = os.path.dirname(os.path.realpath(sys.argv[0]))
    base_res_dir = os.path.dirname(os.path.abspath(sys.executable))
    template_dir = os.path.join(base_res_dir, 'templates')
    static_dir = os.path.join(base_res_dir, 'static')
else:
    base_run_dir = os.path.dirname(os.path.abspath(__file__))
    # Adjust for standard structure
    project_root = os.path.dirname(base_run_dir) 
    if os.path.exists(os.path.join(project_root, 'frontend', 'templates')):
        template_dir = os.path.join(project_root, 'frontend', 'templates')
        static_dir = os.path.join(project_root, 'frontend', 'static')
    else:
        template_dir = os.path.join(base_run_dir, 'templates')
        static_dir = os.path.join(base_run_dir, 'static')

if not os.path.exists(template_dir): os.makedirs(template_dir, exist_ok=True)

DATABASE = os.path.join(base_run_dir, 'fitness_manager.db')
CONFIG_FILE = os.path.join(base_run_dir, 'config.json')

# Load Config with defaults
DEFAULT_CONFIG = { "port": 8080, "mqtt_broker": "127.0.0.1", "mqtt_port": 1883 }
def load_config():
    cfg = DEFAULT_CONFIG.copy()
    if os.path.exists(CONFIG_FILE):
        try:
            with open(CONFIG_FILE, 'r', encoding='utf-8') as f:
                cfg.update(json.load(f))
                print(f"--- Loaded Config: {cfg} ---")
        except Exception as e: print(f"--- Config Load Error: {e} ---")
    else:
        print(f"--- No config.json found at {CONFIG_FILE}, using defaults ---")
    return cfg

APP_CONFIG = load_config()

app = Flask(__name__, template_folder=template_dir, static_folder=static_dir)
CORS(app)
app.secret_key = str(uuid.uuid4())

login_manager = LoginManager()
login_manager.init_app(app)
login_manager.login_view = 'index_route' 

class User(UserMixin):
    def __init__(self, id, username, is_admin, manager_id, age=None, gender=None, weight=None, height=None):
        self.id = id; self.username = username; self.is_admin = is_admin; self.manager_id = manager_id
        self.age = age; self.gender = gender; self.weight = weight; self.height = height
    @property
    def is_super_admin(self): return self.is_admin and (self.manager_id is None or self.manager_id == 0)

@login_manager.user_loader
def load_user(uid):
    row = get_db().execute("SELECT * FROM users WHERE id=?", (uid,)).fetchone()
    if row:
        d = dict(row)
        return User(d['id'], d['username'], d['is_admin'], d['manager_id'], 
                    d.get('age'), d.get('gender'), d.get('weight'), d.get('height'))
    return None

def get_db():
    db = getattr(g, '_database', None)
    if db is None: db = g._database = sqlite3.connect(DATABASE, check_same_thread=False); db.row_factory = sqlite3.Row
    return db

def set_setting(key, value):
    db = get_db()
    db.execute("INSERT INTO settings (key, value) VALUES (?, ?) ON CONFLICT(key) DO UPDATE SET value = excluded.value", (key, str(value)))
    db.commit()

def get_setting(key, default=None):
    row = get_db().execute("SELECT value FROM settings WHERE key = ?", (key,)).fetchone()
    return default if row is None else row["value"]

@app.teardown_appcontext
def close_conn(e):
    db = getattr(g, '_database', None)
    if db is not None: db.close()

def init_db():
    db = get_db()
    cur = db.cursor()
    cur.execute("CREATE TABLE IF NOT EXISTS todos (id INTEGER PRIMARY KEY AUTOINCREMENT, date TEXT NOT NULL, content TEXT NOT NULL, is_completed INTEGER DEFAULT 0, user_id INTEGER, recurring_todo_id INTEGER, reminder_at TEXT)")
    cur.execute("CREATE TABLE IF NOT EXISTS fitness_plans (id INTEGER PRIMARY KEY AUTOINCREMENT, date TEXT NOT NULL, exercise_name TEXT NOT NULL, sets_reps_duration TEXT, is_completed INTEGER DEFAULT 0, user_id INTEGER, recurring_plan_id INTEGER, reminder_at TEXT)")
    cur.execute("CREATE TABLE IF NOT EXISTS recurring_plans (id INTEGER PRIMARY KEY AUTOINCREMENT, user_id INTEGER, plan_name TEXT NOT NULL, exercises_json TEXT NOT NULL, start_date TEXT NOT NULL, end_date TEXT, repeat_interval TEXT NOT NULL, repeat_day INTEGER, reminder_time TEXT)")
    cur.execute("CREATE TABLE IF NOT EXISTS recurring_todos (id INTEGER PRIMARY KEY AUTOINCREMENT, user_id INTEGER, content TEXT NOT NULL, start_date TEXT NOT NULL, end_date TEXT, repeat_interval TEXT NOT NULL, repeat_day INTEGER, reminder_time TEXT)")
    cur.execute("CREATE TABLE IF NOT EXISTS exercise_library (id INTEGER PRIMARY KEY AUTOINCREMENT, user_id INTEGER, name TEXT, details TEXT)")
    cur.execute("CREATE TABLE IF NOT EXISTS todo_library (id INTEGER PRIMARY KEY AUTOINCREMENT, user_id INTEGER, content TEXT)")
    cur.execute("CREATE TABLE IF NOT EXISTS settings (key TEXT PRIMARY KEY, value TEXT)")
    cur.execute("CREATE TABLE IF NOT EXISTS users (id INTEGER PRIMARY KEY AUTOINCREMENT, username TEXT UNIQUE, password TEXT, is_admin INTEGER DEFAULT 0, manager_id INTEGER, age INTEGER, gender TEXT, weight REAL, height REAL)")
    cur.execute("CREATE TABLE IF NOT EXISTS health_logs (id INTEGER PRIMARY KEY AUTOINCREMENT, user_id INTEGER, metric_type TEXT, value REAL, unit TEXT, analysis TEXT, recorded_at TEXT)")
    cur.execute("CREATE TABLE IF NOT EXISTS device_bindings (id INTEGER PRIMARY KEY AUTOINCREMENT, device_id TEXT UNIQUE NOT NULL, device_name TEXT, user_id INTEGER, created_at TEXT)")

    try: cur.execute("SELECT reminder_at FROM todos LIMIT 1"); 
    except: cur.execute("ALTER TABLE todos ADD COLUMN reminder_at TEXT")
    try: cur.execute("SELECT reminder_at FROM fitness_plans LIMIT 1"); 
    except: cur.execute("ALTER TABLE fitness_plans ADD COLUMN reminder_at TEXT")
    try: cur.execute("SELECT reminder_time FROM recurring_plans LIMIT 1"); 
    except: cur.execute("ALTER TABLE recurring_plans ADD COLUMN reminder_time TEXT")
    try: cur.execute("SELECT reminder_time FROM recurring_todos LIMIT 1"); 
    except: cur.execute("ALTER TABLE recurring_todos ADD COLUMN reminder_time TEXT")
    try: cur.execute("SELECT age FROM users LIMIT 1"); 
    except: 
        cur.execute("ALTER TABLE users ADD COLUMN age INTEGER"); cur.execute("ALTER TABLE users ADD COLUMN gender TEXT")
        cur.execute("ALTER TABLE users ADD COLUMN weight REAL"); cur.execute("ALTER TABLE users ADD COLUMN height REAL")

    cur.execute("INSERT OR IGNORE INTO settings (key, value) VALUES ('scroll_speed', '25')")
    if cur.execute("SELECT COUNT(*) FROM users WHERE username='superuser'").fetchone()[0]==0:
        cur.execute("INSERT INTO users (username,password,is_admin,manager_id) VALUES (?,?,?,?)",('superuser','adminpassword',1,None))
    su_id = cur.execute("SELECT id FROM users WHERE username='superuser'").fetchone()['id']
    if cur.execute("SELECT COUNT(*) FROM users WHERE username='user1'").fetchone()[0]==0:
        cur.execute("INSERT INTO users (username,password,is_admin,manager_id) VALUES (?,?,?,?)",('user1','password',0,su_id))
    db.commit()

with app.app_context(): init_db()

def analyze_health_data(metric, value, user_id):
    status = "正常"
    try:
        with sqlite3.connect(DATABASE) as conn:
            conn.row_factory = sqlite3.Row
            user = conn.execute("SELECT age, gender, weight FROM users WHERE id=?", (user_id,)).fetchone()
            age = user['age'] if user and user['age'] else 30
    except: age = 30 
    
    if metric == 'bp_sys':
        if value < 90: status = "偏低 (低血压)"
        elif value > 140: status = "过高 (高血压)" if age < 60 else "偏高"
        elif value > 120: status = "偏高"
    elif metric == 'bp_dia':
        if value < 60: status = "偏低"
        elif value > 90: status = "过高 (高血压)"
        elif value > 80: status = "偏高"
    elif metric == 'spo2':
        if value < 90: status = "极低 (危险)"
        elif value < 95: status = "偏低"
    elif metric == 'glucose':
        if value < 3.9: status = "过低 (低血糖)"
        elif value > 11.1: status = "过高"
        elif value > 7.0: status = "偏高"
    elif metric == 'hr':
        if value < 50: status = "过缓"
        elif value > 100: status = "过速"
    elif metric == 'temp':
        if value > 37.5: status = "发烧"
        elif value < 35.0: status = "体温过低"
    return status

def get_metric_cn_name(m):
    map_name = {'bp_sys': '收缩压', 'bp_dia': '舒张压', 'hr': '心率', 'spo2': '血氧', 'temp': '体温', 'glucose': '血糖', 'weight': '体重', 'water': '饮水'}
    return map_name.get(m, m)

# --- MQTT ---
def on_connect(client, userdata, flags, rc, properties=None):
    print(f"Connected to MQTT Broker with result code {rc}")
    client.subscribe("health/#")
    client.subscribe("devices/#")

def on_message(client, userdata, msg):
    try:
        topic_parts = msg.topic.split('/')
        root = topic_parts[0]
        metric = topic_parts[-1]
        payload = msg.payload.decode()
        
        try:
            data = json.loads(payload)
            val = float(data.get('value', 0))
            unit = data.get('unit', '')
        except:
            try: val = float(payload)
            except: return
            unit = ""
        target_user_id = None
        if root == 'devices':
            device_id = topic_parts[1]
            with sqlite3.connect(DATABASE) as conn:
                row = conn.execute("SELECT user_id FROM device_bindings WHERE device_id = ?", (device_id,)).fetchone()
                if row: target_user_id = row[0]
        elif root == 'health':
            try: target_user_id = int(topic_parts[1])
            except: return

        if target_user_id:
            analysis = analyze_health_data(metric, val, target_user_id)
            with sqlite3.connect(DATABASE) as conn:
                now_str = datetime.now(CN_TZ).strftime('%Y-%m-%d %H:%M:%S')
                conn.execute("INSERT INTO health_logs (user_id, metric_type, value, unit, analysis, recorded_at) VALUES (?,?,?,?,?,?)",
                             (target_user_id, metric, val, unit, analysis, now_str))
                if analysis != "正常":
                    cn_name = get_metric_cn_name(metric)
                    content = f"⚠️ 健康警报: {cn_name} {val}{unit} ({analysis})"
                    today = datetime.now(CN_TZ).strftime('%Y-%m-%d')
                    conn.execute("INSERT INTO todos (date, content, is_completed, user_id, reminder_at) VALUES (?,?,0,?,?)",
                                 (today, content, target_user_id, now_str))
            print(f"[MQTT] Saved {metric}: {val} -> {analysis}")
    except Exception as e:
        print(f"[MQTT Error] {e}")

mqtt_started = False
def start_mqtt():
    global mqtt_started
    if mqtt_started: return
    try:
        try: client = mqtt.Client(mqtt.CallbackAPIVersion.VERSION1)
        except AttributeError: client = mqtt.Client()
        client.on_connect = on_connect
        client.on_message = on_message
        
        broker = APP_CONFIG.get('mqtt_broker', '127.0.0.1')
        port = int(APP_CONFIG.get('mqtt_port', 1883))
        
        print(f"Connecting MQTT: {broker}:{port}...")
        client.connect(broker, port, 60)
        client.loop_start()
        mqtt_started = True
        print("MQTT Client Started")
    except Exception as e:
        print(f"WARNING: MQTT Connection Failed ({e})")

if os.environ.get('WERKZEUG_RUN_MAIN') == 'true':
    start_mqtt()

# --- Routes ---
@app.route('/login', methods=['POST'])
def login_route():
    d = request.get_json(silent=True) or request.form
    u = get_db().execute("SELECT * FROM users WHERE username=?",(d.get('username'),)).fetchone()
    if u and u['password']==d.get('password'):
        d_row = dict(u)
        login_user(User(d_row['id'], d_row['username'], d_row['is_admin'], d_row['manager_id'], 
                       d_row.get('age'), d_row.get('gender'), d_row.get('weight'), d_row.get('height')), remember=True)
        return jsonify({'status':'success'})
    return jsonify({'status':'error'}), 401

@app.route('/logout', methods=['POST'])
@login_required
def logout_route(): logout_user(); return jsonify({'status':'success'})

@app.route('/')
def index_route():
    if not current_user.is_authenticated: return render_template('login.html')
    if 'Mobile' in request.headers.get('User-Agent','') and request.args.get('view')!='display': 
        return redirect(url_for('mobile_config_route')) 
    today = datetime.now(CN_TZ).strftime('%Y-%m-%d')
    _gen_fit(current_user.id, today); _gen_todo(current_user.id, today)
    return render_template('display.html', username=current_user.username)

@app.route('/manage')
@login_required
def manage_route(): 
    return render_template('manage.html', username=current_user.username, is_admin=current_user.is_admin, manager_id=current_user.manager_id,
                           user_info={'age':current_user.age, 'gender':current_user.gender, 'weight':current_user.weight, 'height':current_user.height})

@app.route('/mobile_config')
@login_required
def mobile_config_route(): return render_template('mobile_config.html', username=current_user.username)

@app.route('/todos', methods=['GET'])
@login_required
def get_todos():
    d = request.args.get('date') or datetime.now(CN_TZ).strftime('%Y-%m-%d')
    return jsonify([dict(r) for r in get_db().execute("SELECT * FROM todos WHERE date=? AND user_id=? ORDER BY reminder_at ASC, id ASC", (d, current_user.id)).fetchall()])

@app.route('/add_todo', methods=['POST'])
@login_required
def add_todo():
    d = request.get_json(silent=True) or request.form
    rem = d.get('reminder_at')
    if rem and len(rem)==5: rem = f"{d.get('date')} {rem}"
    get_db().execute("INSERT INTO todos (date,content,is_completed,user_id,reminder_at) VALUES (?,?,0,?,?)", (d.get('date'),d.get('content'),current_user.id,rem)).connection.commit()
    return jsonify(status='success')

@app.route('/add_recurring_todo', methods=['POST'])
@login_required
def add_rec_todo():
    d = request.get_json(silent=True) or request.form
    end_d = d.get('end_date'); 
    if not end_d: end_d=None
    cur = get_db().cursor()
    cur.execute("INSERT INTO recurring_todos (user_id,content,start_date,end_date,repeat_interval,repeat_day,reminder_time) VALUES (?,?,?,?,?,?,?)",
                (current_user.id, d.get('content'), d.get('start_date'), end_d, d.get('repeat_interval'), d.get('repeat_day'), d.get('reminder_time')))
    get_db().commit()
    _gen_single_rec_todo(current_user.id, datetime.now(CN_TZ).strftime('%Y-%m-%d'), cur.lastrowid)
    return jsonify(status='success')

@app.route('/fitness_plan/<date>', methods=['GET'])
@login_required
def get_fit(date):
    _gen_fit(current_user.id, date)
    return jsonify([dict(r) for r in get_db().execute("SELECT * FROM fitness_plans WHERE date=? AND user_id=? ORDER BY reminder_at ASC, id ASC", (date, current_user.id)).fetchall()])

@app.route('/add_recurring_plan', methods=['POST'])
@login_required
def add_rec_plan():
    d = request.get_json(silent=True) or request.form
    end_d = d.get('end_date'); 
    if not end_d: end_d=None
    cur = get_db().cursor()
    cur.execute("INSERT INTO recurring_plans (user_id,plan_name,exercises_json,start_date,end_date,repeat_interval,repeat_day,reminder_time) VALUES (?,?,?,?,?,?,?,?)",
                (current_user.id, d.get('plan_name'), json.dumps(d.get('exercises')), d.get('start_date'), end_d, d.get('repeat_interval'), d.get('repeat_day'), d.get('reminder_time')))
    get_db().commit()
    _gen_single_rec_plan(current_user.id, datetime.now(CN_TZ).strftime('%Y-%m-%d'), cur.lastrowid)
    return jsonify(status='success')

@app.route('/delete_todo/<int:id>', methods=['POST', 'DELETE'])
@login_required
def del_todo(id): get_db().execute("DELETE FROM todos WHERE id=? AND user_id=?",(id,current_user.id)).connection.commit(); return jsonify(status='success')

@app.route('/mark_todo_completed', methods=['POST'])
@login_required
def mark_todo(): 
    d=request.get_json(silent=True) or request.form
    val = 1 if (d.get('is_completed') == True or d.get('is_completed') == 'true' or d.get('is_completed') == 1) else 0
    get_db().execute("UPDATE todos SET is_completed=? WHERE id=? AND user_id=?",(val,d.get('todo_id'),current_user.id)).connection.commit(); return jsonify(status='success')

@app.route('/delete_fitness_plan_exercise/<int:id>', methods=['POST', 'DELETE'])
@login_required
def del_fit(id): get_db().execute("DELETE FROM fitness_plans WHERE id=? AND user_id=?",(id,current_user.id)).connection.commit(); return jsonify(status='success')

@app.route('/mark_exercise_completed_mobile', methods=['POST'])
@login_required
def mark_fit(): 
    d=request.get_json(silent=True) or request.form
    val = 1 if (d.get('is_completed') == True or d.get('is_completed') == 'true' or d.get('is_completed') == 1) else 0
    get_db().execute("UPDATE fitness_plans SET is_completed=? WHERE id=? AND user_id=?",(val,d.get('plan_id'),current_user.id)).connection.commit(); return jsonify(status='success')

@app.route('/get_recurring_todos', methods=['GET'])
@login_required
def get_rec_todos_list(): return jsonify([dict(r) for r in get_db().execute("SELECT * FROM recurring_todos WHERE user_id=?",(current_user.id,)).fetchall()])

@app.route('/delete_recurring_todo/<int:id>', methods=['POST', 'DELETE'])
@login_required
def del_rec_todo(id): get_db().execute("DELETE FROM recurring_todos WHERE id=? AND user_id=?",(id,current_user.id)).connection.commit(); return jsonify(status='success')

@app.route('/get_recurring_plans', methods=['GET'])
@login_required
def get_rec_plans_list(): return jsonify([dict(r) for r in get_db().execute("SELECT * FROM recurring_plans WHERE user_id=?",(current_user.id,)).fetchall()])

@app.route('/delete_recurring_plan/<int:id>', methods=['POST', 'DELETE'])
@login_required
def del_rec_plan(id): get_db().execute("DELETE FROM recurring_plans WHERE id=? AND user_id=?",(id,current_user.id)).connection.commit(); return jsonify(status='success')

@app.route('/add_fitness_manual', methods=['POST'])
@login_required
def add_fit_man():
    d=request.get_json(silent=True) or request.form
    rem = d.get('reminder_at')
    if rem and len(rem)==5: rem = f"{d.get('date')} {rem}"
    get_db().execute("INSERT INTO fitness_plans (date,exercise_name,sets_reps_duration,user_id,reminder_at) VALUES (?,?,?,?,?)",(d.get('date'),d.get('exercise_name'),d.get('sets_reps_duration'),current_user.id,rem)).connection.commit()
    return jsonify(status='success')
@app.route('/api/settings/scroll_speed', methods=['GET','POST'])
def scroll_speed():
    if request.method=='POST': 
        d=request.get_json(silent=True) or request.form
        try: v=int(d.get('scroll_speed'))
        except: v=25
        if v<1: v=1
        if v>120: v=120
        set_setting('scroll_speed', v); return jsonify(status='success', scroll_speed=v)
    try: val = int(get_setting('scroll_speed', 25))
    except: val = 25
    return jsonify(scroll_speed=val)
@app.route('/api/exercises', methods=['GET','POST'])
@app.route('/api/exercises/<int:id>', methods=['DELETE'])
@login_required
def ex_api(id=None):
    if request.method=='GET': return jsonify([dict(r) for r in get_db().execute("SELECT * FROM exercise_library WHERE user_id=?",(current_user.id,)).fetchall()])
    if request.method=='POST': d=request.get_json(silent=True) or request.form; get_db().execute("INSERT INTO exercise_library (user_id,name,details) VALUES (?,?,?)",(current_user.id,d.get('name'),d.get('details'))).connection.commit(); return jsonify(status='success')
    if request.method=='DELETE': get_db().execute("DELETE FROM exercise_library WHERE id=?",(id,)).connection.commit(); return jsonify(status='success')
@app.route('/api/todo_library', methods=['GET','POST'])
@app.route('/api/todo_library/<int:id>', methods=['DELETE'])
@login_required
def td_api(id=None):
    if request.method=='GET': return jsonify([dict(r) for r in get_db().execute("SELECT * FROM todo_library WHERE user_id=?",(current_user.id,)).fetchall()])
    if request.method=='POST': d=request.get_json(silent=True) or request.form; get_db().execute("INSERT INTO todo_library (user_id,content) VALUES (?,?)",(current_user.id,d.get('content'))).connection.commit(); return jsonify(status='success')
    if request.method=='DELETE': get_db().execute("DELETE FROM todo_library WHERE id=?",(id,)).connection.commit(); return jsonify(status='success')

def admin_required(f):
    @wraps(f)
    def dec(*args, **kwargs):
        if not current_user.is_authenticated or not current_user.is_admin: return jsonify(status='error'), 403
        return f(*args, **kwargs)
    return dec

@app.route('/api/users', methods=['GET','POST'])
@app.route('/api/users/<int:id>', methods=['DELETE'])
@login_required
@admin_required
def users_api(id=None):
    if request.method=='GET': return jsonify([dict(u) for u in get_db().execute("SELECT id,username,is_admin FROM users WHERE manager_id=?",(current_user.id,)).fetchall()])
    if request.method=='POST':
        d=request.get_json(silent=True) or request.form
        try: get_db().execute("INSERT INTO users (username,password,is_admin,manager_id) VALUES (?,?,?,?)",(d.get('username'),d.get('password'),1 if d.get('is_admin') else 0, current_user.id)).connection.commit(); return jsonify(status='success')
        except: return jsonify(status='error')
    if request.method=='DELETE':
        if id==current_user.id: return jsonify(status='error')
        get_db().execute("DELETE FROM users WHERE id=?",(id,)).connection.commit(); return jsonify(status='success')

@app.route('/api/devices', methods=['GET'])
@login_required
def get_devices():
    rows = get_db().execute("SELECT * FROM device_bindings WHERE user_id=?", (current_user.id,)).fetchall()
    return jsonify([dict(r) for r in rows])

@app.route('/api/devices', methods=['POST'])
@login_required
def add_device():
    d = request.get_json(silent=True) or request.form
    dev_id = d.get('device_id')
    dev_name = d.get('device_name')
    if not dev_id: return jsonify(status='error', message='Missing ID'), 400
    try:
        get_db().execute("INSERT INTO device_bindings (device_id, device_name, user_id, created_at) VALUES (?,?,?,?)",
                         (dev_id, dev_name, current_user.id, datetime.now(CN_TZ).strftime('%Y-%m-%d')))
        get_db().commit()
        return jsonify(status='success')
    except sqlite3.IntegrityError:
        return jsonify(status='error', message='Device ID already bound'), 409

@app.route('/api/devices/<int:id>', methods=['DELETE', 'POST'])
@login_required
def del_device(id):
    get_db().execute("DELETE FROM device_bindings WHERE id=? AND user_id=?", (id, current_user.id)).connection.commit()
    return jsonify(status='success')

@app.route('/api/health_logs', methods=['GET'])
@login_required
def get_health_logs():
    limit = request.args.get('limit', 50)
    rows = get_db().execute("SELECT * FROM health_logs WHERE user_id=? ORDER BY recorded_at DESC LIMIT ?", (current_user.id, limit)).fetchall()
    return jsonify([dict(r) for r in rows])

@app.route('/api/profile/bio', methods=['POST'])
@login_required
def update_bio():
    d = request.get_json(silent=True) or request.form
    get_db().execute("UPDATE users SET age=?, gender=?, weight=?, height=? WHERE id=?", 
                     (d.get('age'), d.get('gender'), d.get('weight'), d.get('height'), current_user.id))
    get_db().commit()
    return jsonify(status='success')

@app.route('/api/profile', methods=['POST'])
@login_required
def profile():
    d=request.get_json(silent=True) or request.form
    if d.get('password'): get_db().execute("UPDATE users SET password=? WHERE id=?",(d.get('password'),current_user.id))
    if d.get('username'): 
        try: get_db().execute("UPDATE users SET username=? WHERE id=?",(d.get('username'),current_user.id))
        except: pass
    get_db().commit(); return jsonify(status='success')

@app.route('/api/stats/daily_completion')
@login_required
def stats():
    s, e = request.args.get('start_date'), request.args.get('end_date')
    db = get_db()
    f = db.execute("SELECT date, is_completed FROM fitness_plans WHERE user_id=? AND date BETWEEN ? AND ?", (current_user.id, s, e)).fetchall()
    t = db.execute("SELECT date, is_completed FROM todos WHERE user_id=? AND date BETWEEN ? AND ?", (current_user.id, s, e)).fetchall()
    fd = defaultdict(lambda: {"total":0,"completed":0}); td = defaultdict(lambda: {"total":0,"completed":0})
    for r in f: fd[r['date']]['total']+=1; 
    for r in f: 
        if r['is_completed']: fd[r['date']]['completed']+=1
    for r in t: td[r['date']]['total']+=1; 
    for r in t:
        if r['is_completed']: td[r['date']]['completed']+=1
    return jsonify(fitness_data=fd, todo_data=td)

def _gen_fit(uid, date_str):
    db = get_db()
    if db.execute("SELECT COUNT(*) FROM fitness_plans WHERE date=? AND user_id=?",(date_str,uid)).fetchone()[0] > 0: return
    plans = db.execute("SELECT * FROM recurring_plans WHERE user_id=? AND start_date<=? AND (end_date IS NULL OR end_date>=?)", (uid,date_str,date_str)).fetchall()
    dt = datetime.strptime(date_str, '%Y-%m-%d').date()
    for p in plans:
        if _is_due(p, dt): _insert_fit(db, uid, date_str, p)
    db.commit()
def _gen_todo(uid, date_str):
    db = get_db()
    if db.execute("SELECT COUNT(*) FROM todos WHERE date=? AND user_id=?",(date_str,uid)).fetchone()[0] > 0: return
    recs = db.execute("SELECT * FROM recurring_todos WHERE user_id=? AND start_date<=? AND (end_date IS NULL OR end_date>=?)", (uid,date_str,date_str)).fetchall()
    dt = datetime.strptime(date_str, '%Y-%m-%d').date()
    for r in recs:
        if _is_due(r, dt): _insert_todo(db, uid, date_str, r)
    db.commit()
def _gen_single_rec_todo(uid, date_str, rec_id):
    db = get_db()
    r = db.execute("SELECT * FROM recurring_todos WHERE id=?",(rec_id,)).fetchone()
    if not r: return
    dt = datetime.strptime(date_str, '%Y-%m-%d').date()
    if r['start_date'] <= date_str and (not r['end_date'] or r['end_date'] >= date_str):
        if _is_due(r, dt): _insert_todo(db, uid, date_str, r)
    db.commit()
def _gen_single_rec_plan(uid, date_str, rec_id):
    db = get_db()
    p = db.execute("SELECT * FROM recurring_plans WHERE id=?",(rec_id,)).fetchone()
    if not p: return
    dt = datetime.strptime(date_str, '%Y-%m-%d').date()
    if p['start_date'] <= date_str and (not p['end_date'] or p['end_date'] >= date_str):
        if _is_due(p, dt): _insert_fit(db, uid, date_str, p)
    db.commit()
def _is_due(item, dt):
    if item['repeat_interval'] == 'daily': return True
    if item['repeat_interval'] == 'weekly' and item['repeat_day'] == dt.isoweekday(): return True
    if item['repeat_interval'] == 'monthly' and item['repeat_day'] == dt.day: return True
    return False
def _insert_todo(db, uid, date_str, r):
    if db.execute("SELECT COUNT(*) FROM todos WHERE date=? AND recurring_todo_id=?",(date_str, r['id'])).fetchone()[0] > 0: return
    rem = f"{date_str} {r['reminder_time']}" if r['reminder_time'] else None
    db.execute("INSERT INTO todos (date, content, is_completed, user_id, recurring_todo_id, reminder_at) VALUES (?,?,?,?,?,?)",
               (date_str, r['content'], 0, uid, r['id'], rem))
def _insert_fit(db, uid, date_str, p):
    if db.execute("SELECT COUNT(*) FROM fitness_plans WHERE date=? AND recurring_plan_id=?",(date_str, p['id'])).fetchone()[0] > 0: return
    rem = f"{date_str} {p['reminder_time']}" if p['reminder_time'] else None
    for ex in json.loads(p['exercises_json']):
        db.execute("INSERT INTO fitness_plans (date, exercise_name, sets_reps_duration, is_completed, user_id, recurring_plan_id, reminder_at) VALUES (?,?,?,0,?,?,?)",
                   (date_str, ex['name'], ex['details'], uid, p['id'], rem))

@app.after_request
def add_headers(r):
    r.headers['Cache-Control'] = 'no-cache, no-store, must-revalidate'
    return r

@app.route('/health', methods=['GET'])
def health(): return jsonify(status='ok', time=datetime.now(CN_TZ).isoformat())

if __name__ == '__main__':
    # ★★★ CONFIGURABLE PORT ★★★
    port = int(APP_CONFIG.get('port', 8080))
    
    if is_frozen:
        print(f"--- Nuitka Mode: Starting MQTT (Broker: {APP_CONFIG['mqtt_broker']}) on Port {port} ---")
        start_mqtt()
        app.run(host='0.0.0.0', port=port, debug=False, use_reloader=False)
    else:
        if os.environ.get('WERKZEUG_RUN_MAIN') == 'true':
            print(f"--- Dev Mode: Starting MQTT (Broker: {APP_CONFIG['mqtt_broker']}) on Port {port} ---")
            start_mqtt()
        app.run(host='0.0.0.0', port=port, debug=True)
