import sqlite3
from datetime import datetime, timedelta
from flask import Flask, render_template, request, jsonify, g
import os
import json # 引入 json 模块
from flask_cors import CORS
# 如果需要报警功能，请解除以下两行的注释
# import pygame
# import threading
# import time

# --- 路径计算 (健壮方式) ---
app_file_path = os.path.abspath(__file__)
backend_dir = os.path.dirname(app_file_path)
project_root_dir = os.path.dirname(backend_dir)

template_dir = os.path.join(project_root_dir, 'frontend', 'templates')
static_dir = os.path.join(project_root_dir, 'frontend', 'static')

DATABASE = os.path.join(backend_dir, 'fitness_manager.db')

print(f"Project Root: {project_root_dir}")
print(f"Backend Dir: {backend_dir}")
print(f"Template folder path: {template_dir}")
print(f"Static folder path: {static_dir}")
print(f"Database path: {DATABASE}")

app = Flask(__name__,
            template_folder=template_dir,
            static_folder=static_dir)
CORS(app)
# 如果需要报警功能，请解除以下注释并确保alarm.wav存在
# ALARM_SOUND = os.path.join(backend_dir, 'alarm.wav')
# alarm_on = False
# alarm_thread = None
# try:
#     pygame.mixer.init(frequency=44100, size=-16, channels=1, buffer=8192)
#     alarm_sound_obj = pygame.mixer.Sound(ALARM_SOUND)
#     print(f"报警音 '{ALARM_SOUND}' 加载成功。")
# except pygame.error as e:
#     print(f"Pygame 混音器或报警音加载失败: {e}. 报警功能将不可用。")
#     alarm_sound_obj = None

def get_db():
    db = getattr(g, '_database', None)
    if db is None:
        db = g._database = sqlite3.connect(DATABASE)
        db.row_factory = sqlite3.Row
    return db

@app.teardown_appcontext
def close_connection(exception):
    db = getattr(g, '_database', None)
    if db is not None:
        db.close()

def init_db():
    with app.app_context():
        db = get_db()
        cursor = db.cursor()
        # 创建日常健身计划表
        cursor.execute('''
            CREATE TABLE IF NOT EXISTS fitness_plans (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                date TEXT NOT NULL,
                exercise_name TEXT NOT NULL,
                sets_reps_duration TEXT NOT NULL,
                is_completed INTEGER DEFAULT 0,
                created_by TEXT DEFAULT 'system' -- 新增字段，标记计划来源
            )
        ''')
        # 创建周期性健身计划模板表
        cursor.execute('''
            CREATE TABLE IF NOT EXISTS recurring_fitness_plans (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                plan_name TEXT NOT NULL,
                exercises_json TEXT NOT NULL, -- 存储JSON格式的动作列表
                start_date TEXT NOT NULL,
                end_date TEXT, -- NULL表示长期有效
                repeat_interval TEXT NOT NULL, -- 'daily', 'weekly'
                repeat_day_of_week INTEGER -- 0=周一, 6=周日, 仅当 repeat_interval='weekly' 时有效
            )
        ''')
        # 创建待办事项表
        cursor.execute('''
            CREATE TABLE IF NOT EXISTS todos (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                date TEXT NOT NULL,
                content TEXT NOT NULL
            )
        ''')
        db.commit()
    print("数据库初始化完成。")

def _generate_daily_fitness_plan_if_not_exists(date_str):
    with app.app_context():
        db = get_db()
        cursor = db.cursor()
        
        # 检查该日期是否已经有任何健身计划（无论是系统还是用户添加的）
        cursor.execute("SELECT COUNT(*) FROM fitness_plans WHERE date = ?", (date_str,))
        count = cursor.fetchone()[0]
        
        if count == 0: # 如果该日期还没有任何计划
            print(f"尝试根据周期计划生成 {date_str} 的健身计划...")
            today = datetime.strptime(date_str, '%Y-%m-%d')
            
            # 查询所有当前活跃的周期性计划
            cursor.execute("""
                SELECT * FROM recurring_fitness_plans
                WHERE start_date <= ? AND (end_date IS NULL OR end_date >= ?)
            """, (date_str, date_str))
            
            recurring_plans = cursor.fetchall()
            
            # 为该日期生成计划
            exercises_to_add = []
            if recurring_plans:
                for plan in recurring_plans:
                    add_plan = False
                    if plan['repeat_interval'] == 'daily':
                        add_plan = True
                    elif plan['repeat_interval'] == 'weekly' and plan['repeat_day_of_week'] == today.weekday(): # weekday() 0=周一, 6=周日
                        add_plan = True
                    
                    if add_plan:
                        plan_exercises = json.loads(plan['exercises_json'])
                        for ex in plan_exercises:
                            exercises_to_add.append((date_str, ex['name'], ex['details'], 0, 'system')) # 标记为系统生成

            if not exercises_to_add: # 如果没有匹配的周期性计划，则生成默认示例计划
                print(f"没有匹配的周期计划，生成 {date_str} 的默认示例健身计划。")
                exercises_to_add = [
                    (date_str, "热身：慢跑", "10分钟", 0, 'system'),
                    (date_str, "力量：俯卧撑", "3组x10次", 0, 'system'),
                    (date_str, "力量：深蹲", "3组x12次", 0, 'system'),
                    (date_str, "核心：平板支撑", "3组x45秒", 0, 'system'),
                    (date_str, "有氧：跳绳", "15分钟", 0, 'system'),
                    (date_str, "拉伸", "10分钟", 0, 'system')
                ]
            
            cursor.executemany("INSERT INTO fitness_plans (date, exercise_name, sets_reps_duration, is_completed, created_by) VALUES (?, ?, ?, ?, ?)", exercises_to_add)
            db.commit()
            print(f"健身计划已生成完成，日期为 {date_str}")
        else:
            print(f"{date_str} 已存在健身计划，跳过自动生成。")


def _generate_daily_todos_if_not_exists(date_str):
    with app.app_context():
        db = get_db()
        cursor = db.cursor()
        cursor.execute("SELECT COUNT(*) FROM todos WHERE date = ?", (date_str,))
        count = cursor.fetchone()[0]
        if count == 0:
            print(f"尝试生成 {date_str} 的待办事项 (示例)")
            todos_today = [
                (date_str, "购买蛋白粉"),
                (date_str, "预约健身私教课"),
                (date_str, "研究新的健身食谱")
            ]
            cursor.executemany("INSERT INTO todos (date, content) VALUES (?, ?)", todos_today)
            db.commit()
            print(f"待办事项已生成完成，日期为 {date_str}")

# --- Flask 路由 ---

@app.route('/')
def index():
    # 每次访问主页时，确保生成当天的计划
    today_str = datetime.now().strftime('%Y-%m-%d')
    _generate_daily_fitness_plan_if_not_exists(today_str)
    _generate_daily_todos_if_not_exists(today_str)
    return render_template('display.html')

@app.route('/mobile_config')
def mobile_config():
    return render_template('mobile_config.html')

@app.route('/fitness_plan/<date_str>', methods=['GET'])
def get_fitness_plan(date_str):
    _generate_daily_fitness_plan_if_not_exists(date_str) # 确保计划存在
    db = get_db()
    cursor = db.cursor()
    cursor.execute("SELECT * FROM fitness_plans WHERE date = ? ORDER BY id", (date_str,))
    plans = cursor.fetchall()
    return jsonify([dict(row) for row in plans])

# 标记健身动作完成状态 (手机端调用)
@app.route('/mark_exercise_completed_mobile', methods=['POST'])
def mark_exercise_completed_mobile():
    data = request.get_json()
    plan_id = data.get('plan_id')
    is_completed = data.get('is_completed') # 0 或 1

    if plan_id is None or is_completed is None:
        return jsonify(status='error', message='Missing plan_id or is_completed data.'), 400

    db = get_db()
    cursor = db.cursor()
    cursor.execute("UPDATE fitness_plans SET is_completed = ? WHERE id = ?", (is_completed, plan_id))
    db.commit()
    if cursor.rowcount > 0:
        return jsonify(status='success', message='Exercise completion status updated.')
    else:
        return jsonify(status='error', message='Exercise not found.'), 404

# 添加单个健身计划项到特定日期
@app.route('/add_fitness_plan_exercise', methods=['POST'])
def add_fitness_plan_exercise():
    data = request.get_json()
    date_str = data.get('date')
    exercise_name = data.get('exercise_name')
    sets_reps_duration = data.get('sets_reps_duration')

    if not all([date_str, exercise_name, sets_reps_duration]):
        return jsonify(status='error', message='Missing data.'), 400

    db = get_db()
    cursor = db.cursor()
    cursor.execute(
        "INSERT INTO fitness_plans (date, exercise_name, sets_reps_duration, created_by) VALUES (?, ?, ?, 'user')",
        (date_str, exercise_name, sets_reps_duration)
    )
    db.commit()
    return jsonify(status='success', message='Exercise added successfully.', id=cursor.lastrowid)

# 更新单个健身计划项
@app.route('/update_fitness_plan_exercise/<int:plan_id>', methods=['POST'])
def update_fitness_plan_exercise(plan_id):
    data = request.get_json()
    exercise_name = data.get('exercise_name')
    sets_reps_duration = data.get('sets_reps_duration')

    if not all([exercise_name, sets_reps_duration]):
        return jsonify(status='error', message='Missing data.'), 400

    db = get_db()
    cursor = db.cursor()
    cursor.execute(
        "UPDATE fitness_plans SET exercise_name = ?, sets_reps_duration = ?, created_by = 'user' WHERE id = ?",
        (exercise_name, sets_reps_duration, plan_id)
    )
    db.commit()
    if cursor.rowcount > 0:
        return jsonify(status='success', message='Exercise updated successfully.')
    else:
        return jsonify(status='error', message='Exercise not found.'), 404

# 删除单个健身计划项
@app.route('/delete_fitness_plan_exercise/<int:plan_id>', methods=['POST'])
def delete_fitness_plan_exercise(plan_id):
    db = get_db()
    cursor = db.cursor()
    cursor.execute("DELETE FROM fitness_plans WHERE id = ?", (plan_id,))
    db.commit()
    if cursor.rowcount > 0:
        return jsonify(status='success', message='Exercise deleted successfully.')
    else:
        return jsonify(status='error', message='Exercise not found.'), 404

# --- 周期性健身计划 API ---
@app.route('/recurring_plans', methods=['GET'])
def get_recurring_plans():
    db = get_db()
    cursor = db.cursor()
    cursor.execute("SELECT * FROM recurring_fitness_plans ORDER BY start_date DESC, plan_name")
    plans = cursor.fetchall()
    return jsonify([dict(row) for row in plans])

@app.route('/add_recurring_plan', methods=['POST'])
def add_recurring_plan():
    data = request.get_json()
    plan_name = data.get('plan_name')
    exercises = data.get('exercises') # 预期是列表的JSON字符串
    start_date = data.get('start_date')
    end_date = data.get('end_date')
    repeat_interval = data.get('repeat_interval')
    repeat_day_of_week = data.get('repeat_day_of_week')

    if not all([plan_name, exercises, start_date, repeat_interval]):
        return jsonify(status='error', message='Missing required recurring plan data.'), 400

    db = get_db()
    cursor = db.cursor()
    try:
        cursor.execute(
            "INSERT INTO recurring_fitness_plans (plan_name, exercises_json, start_date, end_date, repeat_interval, repeat_day_of_week) VALUES (?, ?, ?, ?, ?, ?)",
            (plan_name, json.dumps(exercises), start_date, end_date, repeat_interval, repeat_day_of_week)
        )
        db.commit()
        return jsonify(status='success', message='Recurring plan added successfully.', id=cursor.lastrowid)
    except sqlite3.Error as e:
        return jsonify(status='error', message=f'Database error: {e}'), 500

@app.route('/delete_recurring_plan/<int:plan_id>', methods=['POST'])
def delete_recurring_plan(plan_id):
    db = get_db()
    cursor = db.cursor()
    cursor.execute("DELETE FROM recurring_fitness_plans WHERE id = ?", (plan_id,))
    db.commit()
    if cursor.rowcount > 0:
        return jsonify(status='success', message='Recurring plan deleted successfully.')
    else:
        return jsonify(status='error', message='Recurring plan not found.'), 404

@app.route('/update_recurring_plan/<int:plan_id>', methods=['POST'])
def update_recurring_plan(plan_id):
    data = request.get_json()
    plan_name = data.get('plan_name')
    exercises = data.get('exercises')
    start_date = data.get('start_date')
    end_date = data.get('end_date')
    repeat_interval = data.get('repeat_interval')
    repeat_day_of_week = data.get('repeat_day_of_week')

    if not all([plan_name, exercises, start_date, repeat_interval]):
        return jsonify(status='error', message='Missing required recurring plan data.'), 400

    db = get_db()
    cursor = db.cursor()
    try:
        cursor.execute(
            "UPDATE recurring_fitness_plans SET plan_name = ?, exercises_json = ?, start_date = ?, end_date = ?, repeat_interval = ?, repeat_day_of_week = ? WHERE id = ?",
            (plan_name, json.dumps(exercises), start_date, end_date, repeat_interval, repeat_day_of_week, plan_id)
        )
        db.commit()
        if cursor.rowcount > 0:
            return jsonify(status='success', message='Recurring plan updated successfully.')
        else:
            return jsonify(status='error', message='Recurring plan not found.'), 404
    except sqlite3.Error as e:
        return jsonify(status='error', message=f'Database error: {e}'), 500

# 应用周期性计划到指定日期范围（用于批量生成）
@app.route('/apply_recurring_plan', methods=['POST'])
def apply_recurring_plan():
    data = request.get_json()
    recurring_plan_id = data.get('recurring_plan_id')
    apply_start_date_str = data.get('apply_start_date')
    apply_end_date_str = data.get('apply_end_date')

    if not all([recurring_plan_id, apply_start_date_str, apply_end_date_str]):
        return jsonify(status='error', message='Missing data.'), 400

    db = get_db()
    cursor = db.cursor()
    
    cursor.execute("SELECT * FROM recurring_fitness_plans WHERE id = ?", (recurring_plan_id,))
    recurring_plan = cursor.fetchone()

    if not recurring_plan:
        return jsonify(status='error', message='Recurring plan not found.'), 404

    exercises_to_apply = json.loads(recurring_plan['exercises_json'])
    
    current_date = datetime.strptime(apply_start_date_str, '%Y-%m-%d')
    end_date = datetime.strptime(apply_end_date_str, '%Y-%m-%d')

    rows_inserted = 0
    while current_date <= end_date:
        date_str = current_date.strftime('%Y-%m-%d')
        
        # 检查该日期是否已有用户添加的计划，如果有则不覆盖
        cursor.execute("SELECT COUNT(*) FROM fitness_plans WHERE date = ? AND created_by = 'user'", (date_str,))
        user_created_count = cursor.fetchone()[0]

        if user_created_count == 0: # 只有当该日期没有用户自定义计划时才自动应用
            # 清除该日期所有由系统生成的旧计划
            cursor.execute("DELETE FROM fitness_plans WHERE date = ? AND created_by = 'system'", (date_str,))

            # 根据重复间隔判断是否应用
            add_plan_for_date = False
            if recurring_plan['repeat_interval'] == 'daily':
                add_plan_for_date = True
            elif recurring_plan['repeat_interval'] == 'weekly' and recurring_plan['repeat_day_of_week'] == current_date.weekday():
                add_plan_for_date = True

            if add_plan_for_date:
                for ex in exercises_to_apply:
                    cursor.execute(
                        "INSERT INTO fitness_plans (date, exercise_name, sets_reps_duration, is_completed, created_by) VALUES (?, ?, ?, ?, ?)",
                        (date_str, ex['name'], ex['details'], 0, 'system')
                    )
                    rows_inserted += 1
        else:
            print(f"Skipping apply for {date_str} due to user-created plans.")

        current_date += timedelta(days=1)
    
    db.commit()
    return jsonify(status='success', message=f'Recurring plan applied to dates. {rows_inserted} exercises inserted.')

# --- 待办事项 API ---
@app.route('/todos', methods=['GET'])
def get_todos():
    date_str = request.args.get('date') # 从查询参数获取日期
    if not date_str:
        return jsonify(status='error', message='Missing date parameter.'), 400

    _generate_daily_todos_if_not_exists(date_str) # 确保待办事项存在
    db = get_db()
    cursor = db.cursor()
    cursor.execute("SELECT * FROM todos WHERE date = ? ORDER BY id", (date_str,))
    todos = cursor.fetchall()
    return jsonify([dict(row) for row in todos])

@app.route('/add_todo', methods=['POST'])
def add_todo():
    data = request.get_json()
    date_str = data.get('date')
    content = data.get('content')

    if not all([date_str, content]):
        return jsonify(status='error', message='Missing date or content data.'), 400

    db = get_db()
    cursor = db.cursor()
    try:
        cursor.execute("INSERT INTO todos (date, content) VALUES (?, ?)", (date_str, content))
        db.commit()
        return jsonify(status='success', message='Todo added successfully.', id=cursor.lastrowid)
    except sqlite3.Error as e:
        return jsonify(status='error', message=f'Database error: {e}'), 500


# 如果需要报警功能，可以把服药系统中的报警相关路由复制过来，并修改其含义
# @app.route('/alarm_status', methods=['GET'])
# def get_alarm_status():
#     # 对于健身系统，可以返回一些简单的训练状态，例如：
#     # 计算已完成的训练项目数量与总项目数量，来表示进度
#     today_str = datetime.now().strftime('%Y-%m-%d')
#     db = get_db()
#     cursor = db.cursor()
#     cursor.execute("SELECT COUNT(*) FROM fitness_plans WHERE date = ?", (today_str,))
#     total_exercises = cursor.fetchone()[0]
#     cursor.execute("SELECT COUNT(*) FROM fitness_plans WHERE date = ? AND is_completed = 1", (today_str,))
#     completed_exercises = cursor.fetchone()[0]

#     if total_exercises == 0:
#         status_text = "今日无计划"
#         status_type = "status-ready"
#     elif completed_exercises == total_exercises:
#         status_text = "全部完成！"
#         status_type = "status-completed"
#     else:
#         status_text = f"进行中 ({completed_exercises}/{total_exercises})"
#         status_type = "status-inprogress"
#     return jsonify(text=status_text, type=status_type)
@app.route('/update_todo/<int:todo_id>', methods=['POST'])
def update_todo(todo_id):
    data = request.get_json()
    content = data.get('content')

    if not content:
        return jsonify(status='error', message='Missing content data.'), 400

    db = get_db()
    cursor = db.cursor()
    try:
        cursor.execute("UPDATE todos SET content = ? WHERE id = ?", (content, todo_id))
        db.commit()
        if cursor.rowcount > 0:
            return jsonify(status='success', message='Todo updated successfully.')
        else:
            return jsonify(status='error', message='Todo not found.'), 404
    except sqlite3.Error as e:
        return jsonify(status='error', message=f'Database error: {e}'), 500

@app.route('/delete_todo/<int:todo_id>', methods=['POST'])
def delete_todo(todo_id):
    db = get_db()
    cursor = db.cursor()
    try:
        cursor.execute("DELETE FROM todos WHERE id = ?", (todo_id,))
        db.commit()
        if cursor.rowcount > 0:
            return jsonify(status='success', message='Todo deleted successfully.')
        else:
            return jsonify(status='error', message='Todo not found.'), 404
    except sqlite3.Error as e:
        return jsonify(status='error', message=f'Database error: {e}'), 500

# --- 后端兼容性补丁 (建议加入) ---
from flask import make_response

@app.after_request
def add_common_headers(response):
    # 明确字符集与缓存策略，帮助某些WebView正确渲染
    response.headers['Content-Type'] = response.headers.get('Content-Type', 'text/html; charset=utf-8')
    # 允许跨域（CORS 已启用），这里补强下常见 header
    response.headers.setdefault('X-Content-Type-Options', 'nosniff')
    response.headers.setdefault('X-Frame-Options', 'SAMEORIGIN')
    # 给静态资源短时间缓存，API 响应不缓存
    if request.path.startswith('/static/') or request.path.endswith(('.css', '.js', '.png', '.jpg', '.webp', '.svg')):
        response.headers['Cache-Control'] = 'public, max-age=3600'
    else:
        response.headers['Cache-Control'] = 'no-cache, no-store, must-revalidate'
    return response

@app.route('/health', methods=['GET'])
def health():
    return jsonify(status='ok', time=datetime.now().isoformat())



if __name__ == '__main__':
    init_db() # 确保在应用启动时初始化数据库
    app.run(host='0.0.0.0', port=8000, debug=True)

