# backend/app.py (Final version with Schema Migration and Default User Insertion fix)
import sqlite3
from datetime import datetime, timedelta
from flask import Flask, render_template, request, jsonify, g, session, redirect, url_for
import os, json, uuid
from flask_cors import CORS
from functools import wraps

# Paths — assume repository layout: backend/, frontend/templates/, frontend/static/
app_file_path = os.path.abspath(__file__)
backend_dir = os.path.dirname(app_file_path)
project_root_dir = os.path.dirname(backend_dir)
template_dir = os.path.join(project_root_dir, 'frontend', 'templates')
static_dir = os.path.join(project_root_dir, 'frontend', 'static')
DATABASE = os.path.join(backend_dir, 'fitness_manager.db')

print("Template folder:", template_dir)
print("Static folder:", static_dir)
print("DB path:", DATABASE)

app = Flask(__name__, template_folder=template_dir, static_folder=static_dir)
CORS(app)

# --- Session Configuration ---
app.secret_key = str(uuid.uuid4())
# -----------------------------

# -------------------- DB --------------------
def get_db():
    db = getattr(g, '_database', None)
    if db is None:
        db = g._database = sqlite3.connect(DATABASE)
        db.row_factory = sqlite3.Row
    return db

@app.teardown_appcontext
def close_connection(exception):
    db = getattr(g, '_database', None)
    if db is not None:
        db.close()

def init_db():
    db = get_db()
    cur = db.cursor()
    
    # 1. 创建 todos 表 (如果不存在)
    cur.execute("""
        CREATE TABLE IF NOT EXISTS todos (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            date TEXT NOT NULL,
            content TEXT NOT NULL,
            is_completed INTEGER DEFAULT 0,
            user_id INTEGER
        )
    """)
    
    # 2. 创建 fitness_plans 表 (如果不存在)
    cur.execute("""
        CREATE TABLE IF NOT EXISTS fitness_plans (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            date TEXT NOT NULL,
            exercise_name TEXT NOT NULL,
            sets_reps_duration TEXT,
            is_completed INTEGER DEFAULT 0,
            user_id INTEGER
        )
    """)
    
    # 3. 创建 recurring_plans 表 (如果不存在)
    cur.execute("""
        CREATE TABLE IF NOT EXISTS recurring_plans (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            start_date TEXT NOT NULL,
            end_date TEXT,
            exercises_json TEXT NOT NULL,
            user_id INTEGER
        )
    """)

# --- 用户表和 user_id 字段保证函数 (FIXED) ---
def ensure_user_schema():
    db = get_db()
    cur = db.cursor()
    
    # 1. 创建 users 表 (新增 is_admin)
    cur.execute("""
        CREATE TABLE IF NOT EXISTS users (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            username TEXT UNIQUE NOT NULL,
            password TEXT NOT NULL,
            is_admin INTEGER DEFAULT 0  -- 0: 普通用户, 1: 超级用户
        )
    """)

    # 2. 自动添加 is_admin 字段 (如果它不存在)
    try:
        cur.execute("SELECT is_admin FROM users LIMIT 1")
    except sqlite3.OperationalError:
        print("Migrating users table: Adding is_admin column.")
        cur.execute("ALTER TABLE users ADD COLUMN is_admin INTEGER DEFAULT 0")
    
    # 3. 确保默认超级用户和普通用户存在 (无论表是否为空)
    # 插入超级管理员
    cur.execute("SELECT COUNT(*) FROM users WHERE username = 'superuser'")
    if cur.fetchone()[0] == 0:
        print("Inserting default superuser: superuser/adminpassword (Admin)")
        cur.execute("INSERT INTO users (username, password, is_admin) VALUES (?, ?, ?)", ('superuser', 'adminpassword', 1))
        
    # 插入一个普通用户
    cur.execute("SELECT COUNT(*) FROM users WHERE username = 'user1'")
    if cur.fetchone()[0] == 0:
        print("Inserting default regular user: user1/password (Regular)")
        cur.execute("INSERT INTO users (username, password, is_admin) VALUES (?, ?, ?)", ('user1', 'password', 0))

    # 4. 为所有数据表添加 user_id 字段 (如果尚未添加)
    for table in ['fitness_plans', 'todos', 'recurring_plans']:
        try:
            cur.execute(f"SELECT user_id FROM {table} LIMIT 1") 
        except sqlite3.OperationalError:
            cur.execute(f"ALTER TABLE {table} ADD COLUMN user_id INTEGER")
        
        # 将所有现有数据分配给 ID=2 的普通用户 (如果数据库为空，则ID=2是user1)
        cur.execute(f"UPDATE {table} SET user_id = 2 WHERE user_id IS NULL")
    
    db.commit()
# -----------------------------------------

# 保持兼容性检查函数
def ensure_todos_is_completed_column():
    db = get_db()
    cur = db.cursor()
    try:
        cur.execute("SELECT is_completed FROM todos LIMIT 1")
    except sqlite3.OperationalError:
        cur.execute("ALTER TABLE todos ADD COLUMN is_completed INTEGER DEFAULT 0")
        db.commit()

def ensure_fitness_plans_is_completed_column():
    db = get_db()
    cur = db.cursor()
    try:
        cur.execute("SELECT is_completed FROM fitness_plans LIMIT 1")
    except sqlite3.OperationalError:
        cur.execute("ALTER TABLE fitness_plans ADD COLUMN is_completed INTEGER DEFAULT 0")
        db.commit()


# 在应用启动时执行所有 Schema 检查
with app.app_context():
    init_db() 
    ensure_todos_is_completed_column()
    ensure_fitness_plans_is_completed_column()
    ensure_user_schema() # 这个函数现在是修复问题的关键


# -------------------- Decorator & Auth --------------------

def login_required(f):
    @wraps(f)
    def decorated_function(*args, **kwargs):
        user_id = session.get('user_id')
        if user_id is None:
            if request.path.startswith('/api/') or request.path.startswith('/fitness_plan/'):
                 return jsonify(status='error', message='Unauthorized'), 401
            return redirect(url_for('index', next=request.url))
        
        kwargs['user_id'] = user_id
        return f(*args, **kwargs)
    return decorated_function

# NEW: Admin Decorator
def admin_required(f):
    @wraps(f)
    def decorated_function(*args, **kwargs):
        user_id = session.get('user_id')
        is_admin = session.get('is_admin', 0)
        
        if user_id is None:
            return jsonify(status='error', message='Unauthorized'), 401
            
        if is_admin != 1:
            return jsonify(status='error', message='Admin access required'), 403
        
        kwargs['user_id'] = user_id
        return f(*args, **kwargs)
    return decorated_function


@app.route('/login', methods=['POST'])
def login():
    username = request.form.get('username') or request.json.get('username')
    password = request.form.get('password') or request.json.get('password')
    
    db = get_db()
    # NEW: Fetch is_admin
    user = db.execute("SELECT id, username, password, is_admin FROM users WHERE username = ?", (username,)).fetchone()
    
    if user and user['password'] == password: 
        session.clear()
        session['user_id'] = user['id']
        session['username'] = user['username']
        session['is_admin'] = user['is_admin'] # Store admin status
        return jsonify({'status': 'success', 'username': user['username'], 'is_admin': user['is_admin']})
    
    return jsonify({'status': 'error', 'message': 'Invalid Credentials'}), 401

@app.route('/logout', methods=['POST'])
def logout():
    session.pop('user_id', None)
    session.pop('username', None)
    session.pop('is_admin', None) # Clear admin status
    return jsonify({'status': 'success'})

@app.route('/get_current_user', methods=['GET'])
def get_current_user():
    user_id = session.get('user_id')
    username = session.get('username')
    is_admin = session.get('is_admin', 0) # Return admin status
    if user_id:
        return jsonify({'status': 'logged_in', 'user_id': user_id, 'username': username, 'is_admin': is_admin})
    return jsonify({'status': 'logged_out'}), 401

# -------------------- UI Routes --------------------

@app.route('/')
def index():
    if 'user_id' not in session:
        return render_template('login.html') 
        
    today_str = datetime.now().strftime('%Y-%m-%d')
    _generate_daily_fitness_plan_if_not_exists(session['user_id'], today_str)
    _generate_daily_todos_if_not_exists(session['user_id'], today_str)
    
    # 传递 username 和 is_admin 到 display.html
    return render_template('display.html', 
                           username=session['username'],
                           is_admin=session.get('is_admin', 0)) 

@app.route('/mobile_config')
@login_required
def mobile_config(user_id):
    return render_template('mobile_config.html', username=session['username'])

@app.route('/manage')
@login_required
def manage(user_id):
    # 传递 is_admin 状态到 manage.html
    return render_template('manage.html', 
                           username=session['username'],
                           is_admin=session.get('is_admin', 0))

# -------------------- NEW: User Management API (Admin Only) --------------------

@app.route('/api/users', methods=['GET'])
@admin_required
def list_users(user_id):
    db = get_db()
    # 不返回密码
    users = db.execute("SELECT id, username, is_admin FROM users ORDER BY is_admin DESC, username").fetchall()
    return jsonify([dict(u) for u in users])

@app.route('/api/users', methods=['POST'])
@admin_required
def create_user(user_id):
    data = request.get_json()
    username = data.get('username')
    password = data.get('password')
    is_admin = data.get('is_admin', 0)
    
    if not username or not password:
        return jsonify(status='error', message='Missing username or password'), 400
    
    try:
        db = get_db()
        # NOTE: In a real app, use password hashing!
        db.execute(
            "INSERT INTO users (username, password, is_admin) VALUES (?, ?, ?)",
            (username, password, is_admin)
        )
        db.commit()
        return jsonify(status='success', message='User created')
    except sqlite3.IntegrityError:
        return jsonify(status='error', message='Username already exists'), 409
    except sqlite3.Error as e:
        return jsonify(status='error', message=f'Database error: {e}'), 500

@app.route('/api/users/<int:target_user_id>', methods=['DELETE'])
@admin_required
def delete_user(user_id, target_user_id):
    # 禁止超级管理员删除自己
    if user_id == target_user_id:
        return jsonify(status='error', message='Cannot delete self.'), 400
        
    db = get_db()
    cur = db.cursor()
    
    # 安全检查：防止删除其他管理员 (可选)
    target_user = cur.execute("SELECT is_admin FROM users WHERE id = ?", (target_user_id,)).fetchone()
    if target_user and target_user['is_admin'] == 1:
        return jsonify(status='error', message='Cannot delete another admin.'), 403

    # 删除用户及其所有数据 (健身计划、待办事项、循环计划)
    cur.execute("DELETE FROM users WHERE id = ?", (target_user_id,))
    cur.execute("DELETE FROM todos WHERE user_id = ?", (target_user_id,))
    cur.execute("DELETE FROM fitness_plans WHERE user_id = ?", (target_user_id,))
    cur.execute("DELETE FROM recurring_plans WHERE user_id = ?", (target_user_id,))
    
    db.commit()
    if cur.rowcount > 0:
        return jsonify(status='success', message='User and all data deleted.')
    return jsonify(status='error', message='User not found.'), 404

# -------------------- NEW: Profile Update API (Any User) --------------------

@app.route('/api/profile', methods=['POST'])
@login_required
def update_profile(user_id):
    data = request.get_json()
    new_username = data.get('username')
    new_password = data.get('password')
    
    db = get_db()
    cur = db.cursor()
    
    update_count = 0
    
    # 1. Update Password
    if new_password:
        cur.execute("UPDATE users SET password = ? WHERE id = ?", (new_password, user_id))
        update_count += 1
        
    # 2. Update Username
    if new_username:
        current_user = cur.execute("SELECT username FROM users WHERE id = ?", (user_id,)).fetchone()
        if current_user and current_user['username'] != new_username:
            try:
                cur.execute("UPDATE users SET username = ? WHERE id = ?", (new_username, user_id))
                session['username'] = new_username # Update session immediately
                update_count += 1
            except sqlite3.IntegrityError:
                db.rollback()
                return jsonify(status='error', message='Username already exists'), 409
    
    db.commit()
    if update_count > 0:
        return jsonify(status='success', message='Profile updated successfully. Please note: if username changed, you need to re-login next time.')
    
    return jsonify(status='error', message='No fields to update.'), 400


# -------------------- Helpers (Data Isolation) - (No changes needed here as they already use user_id) --------------------
def _get_daily_fitness_plans(user_id, date_str):
    db = get_db()
    plans = db.execute(
        "SELECT * FROM fitness_plans WHERE date = ? AND user_id = ? ORDER BY id", 
        (date_str, user_id)
    ).fetchall()
    return [dict(plan) for plan in plans]

def _generate_daily_fitness_plan_if_not_exists(user_id, date_str):
    db = get_db()
    cur = db.cursor()
    cur.execute("SELECT COUNT(*) FROM fitness_plans WHERE date = ? AND user_id = ?", (date_str, user_id))
    if cur.fetchone()[0] > 0: return

    recurring_plans = db.execute(
        "SELECT * FROM recurring_plans WHERE user_id = ? AND (start_date <= ? AND (end_date IS NULL OR end_date >= ?))", 
        (user_id, date_str, date_str)
    ).fetchall()
    
    new_plans = []
    for plan in recurring_plans:
        exercises = json.loads(plan['exercises_json'])
        for ex in exercises:
            new_plans.append({'exercise_name': ex['name'], 'sets_reps_duration': ex['details']})

    for plan in new_plans:
        cur.execute(
            "INSERT INTO fitness_plans (date, exercise_name, sets_reps_duration, is_completed, user_id) VALUES (?, ?, ?, ?, ?)",
            (date_str, plan['exercise_name'], plan['sets_reps_duration'], 0, user_id)
        )
    db.commit()

def _generate_daily_todos_if_not_exists(user_id, date_str):
    db = get_db()
    cur = db.cursor()
    cur.execute("SELECT COUNT(*) FROM todos WHERE date = ? AND user_id = ?", (date_str, user_id))
    if cur.fetchone()[0] > 0: return

    cur.execute(
        "INSERT INTO todos (date, content, is_completed, user_id) VALUES (?, ?, ?, ?)",
        (date_str, '完成今日健身计划', 0, user_id)
    )
    db.commit()


# -------------------- Existing API endpoints --------------------

@app.route('/fitness_plan/<date_str>', methods=['GET'])
@login_required
def get_fitness_plan(date_str, user_id):
    _generate_daily_fitness_plan_if_not_exists(user_id, date_str)
    db = get_db()
    plans = db.execute(
        "SELECT * FROM fitness_plans WHERE date = ? AND user_id = ? ORDER BY id", 
        (date_str, user_id)
    ).fetchall()
    return jsonify([dict(plan) for plan in plans])

@app.route('/mark_exercise_completed_mobile', methods=['POST'])
@login_required
def mark_exercise_completed_mobile(user_id):
    data = request.get_json()
    plan_id = data.get('plan_id')
    is_completed = data.get('is_completed')
    
    if plan_id is None or is_completed is None: return jsonify(status='error', message='Missing plan_id or is_completed'), 400
        
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute(
            "UPDATE fitness_plans SET is_completed = ? WHERE id = ? AND user_id = ?",
            (is_completed, plan_id, user_id)
        )
        db.commit()
        if cur.rowcount > 0: return jsonify(status='success')
        else: return jsonify(status='error', message='Plan not found or not owned by user.'), 404
    except sqlite3.Error as e: return jsonify(status='error', message=f'Database error: {e}'), 500

@app.route('/todos', methods=['GET'])
@login_required
def get_todos(user_id):
    date_str = request.args.get('date') or datetime.now().strftime('%Y-%m-%d')
    _generate_daily_todos_if_not_exists(user_id, date_str)
    
    db = get_db()
    todos = db.execute(
        "SELECT * FROM todos WHERE date = ? AND user_id = ? ORDER BY id", 
        (date_str, user_id)
    ).fetchall()
    return jsonify([dict(todo) for todo in todos])

@app.route('/add_todo', methods=['POST'])
@login_required
def add_todo(user_id):
    data = request.get_json()
    date_str = data.get('date')
    content = data.get('content')
    
    if not date_str or not content: return jsonify(status='error', message='Missing date or content'), 400
        
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute(
            "INSERT INTO todos (date, content, is_completed, user_id) VALUES (?, ?, 0, ?)",
            (date_str, content, user_id)
        )
        db.commit()
        return jsonify(status='success', todo_id=cur.lastrowid)
    except sqlite3.Error as e: return jsonify(status='error', message=f'Database error: {e}'), 500

@app.route('/mark_todo_completed', methods=['POST'])
@login_required
def mark_todo_completed(user_id):
    data = request.get_json()
    todo_id = data.get('todo_id')
    is_completed = data.get('is_completed')
    
    if todo_id is None or is_completed is None: return jsonify(status='error', message='Missing todo_id or is_completed'), 400
        
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute(
            "UPDATE todos SET is_completed = ? WHERE id = ? AND user_id = ?",
            (is_completed, todo_id, user_id)
        )
        db.commit()
        if cur.rowcount > 0: return jsonify(status='success')
        else: return jsonify(status='error', message='Todo not found or not owned by user.'), 404
    except sqlite3.Error as e: return jsonify(status='error', message=f'Database error: {e}'), 500

@app.route('/delete_todo/<int:todo_id>', methods=['POST'])
@login_required
def delete_todo(todo_id, user_id):
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute("DELETE FROM todos WHERE id = ? AND user_id = ?", (todo_id, user_id))
        db.commit()
        if cur.rowcount > 0: return jsonify(status='success', message='Todo deleted successfully.')
        else: return jsonify(status='error', message='Todo not found or not owned by user.'), 404
    except sqlite3.Error as e: return jsonify(status='error', message=f'Database error: {e}'), 500

@app.route('/get_recurring_plans', methods=['GET'])
@login_required
def get_recurring_plans(user_id):
    db = get_db()
    plans = db.execute(
        "SELECT * FROM recurring_plans WHERE user_id = ? ORDER BY start_date DESC",
        (user_id,)
    ).fetchall()
    return jsonify([dict(plan) for plan in plans])

@app.route('/add_recurring_plan', methods=['POST'])
@login_required
def add_recurring_plan(user_id):
    data = request.get_json()
    start_date = data.get('start_date')
    exercises_json = json.dumps(data.get('exercises'))
    end_date = data.get('end_date')
    
    if not start_date or not exercises_json: return jsonify(status='error', message='Missing start_date or exercises'), 400

    try:
        db = get_db()
        cur = db.cursor()
        cur.execute(
            "INSERT INTO recurring_plans (start_date, exercises_json, end_date, user_id) VALUES (?, ?, ?, ?)",
            (start_date, exercises_json, end_date, user_id)
        )
        db.commit()
        return jsonify(status='success', plan_id=cur.lastrowid)
    except sqlite3.Error as e: return jsonify(status='error', message=f'Database error: {e}'), 500

@app.route('/delete_recurring_plan/<int:plan_id>', methods=['POST'])
@login_required
def delete_recurring_plan(plan_id, user_id):
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute("DELETE FROM recurring_plans WHERE id = ? AND user_id = ?", (plan_id, user_id))
        db.commit()
        if cur.rowcount > 0: return jsonify(status='success', message='Recurring plan deleted successfully.')
        else: return jsonify(status='error', message='Recurring plan not found or not owned by user.'), 404
    except sqlite3.Error as e: return jsonify(status='error', message=f'Database error: {e}'), 500

@app.route('/delete_fitness_plan_exercise/<int:plan_id>', methods=['POST'])
@login_required
def delete_fitness_plan_exercise(plan_id, user_id):
    try:
        db = get_db()
        cur = db.cursor()
        cur.execute("DELETE FROM fitness_plans WHERE id = ? AND user_id = ?", (plan_id, user_id))
        db.commit()
        if cur.rowcount > 0: return jsonify(status='success', message='Daily exercise deleted successfully.')
        else: return jsonify(status='error', message='Daily exercise not found or not owned by user.'), 404
    except sqlite3.Error as e: return jsonify(status='error', message=f'Database error: {e}'), 500


# -------------------- Compatibility headers & health --------------------
from flask import make_response
@app.after_request
def add_common_headers(response):
    if 'Content-Type' not in response.headers:
        response.headers['Content-Type'] = 'text/html; charset=utf-8'
    else:
        c = response.headers['Content-Type']
        if c.startswith('text/') and 'charset' not in c:
            response.headers['Content-Type'] = c + '; charset=utf-8'
    response.headers.setdefault('X-Content-Type-Options', 'nosniff')
    response.headers.setdefault('X-Frame-Options', 'SAMEORIGIN')
    path = request.path or ''
    if path.startswith('/static/') or path.endswith(('.css', '.js', '.png', '.jpg', '.svg', '.webp')):
        response.headers['Cache-Control'] = 'public, max-age=3600'
    else:
        response.headers['Cache-Control'] = 'no-cache, no-store, must-revalidate'
    return response

@app.route('/health', methods=['GET'])
def health():
    return jsonify(status='ok', time=datetime.now().isoformat())

# -------------------- Run --------------------
if __name__ == '__main__':
    app.run(host='0.0.0.0', port=8000, debug=True)
