# backend/app.py
import sqlite3
from datetime import datetime, timedelta
from flask import Flask, render_template, request, jsonify, g
import os, json
from flask_cors import CORS

# Paths — assume repository layout: backend/, frontend/templates/, frontend/static/
app_file_path = os.path.abspath(__file__)
backend_dir = os.path.dirname(app_file_path)
project_root_dir = os.path.dirname(backend_dir)
template_dir = os.path.join(project_root_dir, 'frontend', 'templates')
static_dir = os.path.join(project_root_dir, 'frontend', 'static')
DATABASE = os.path.join(backend_dir, 'fitness_manager.db')

print("Template folder:", template_dir)
print("Static folder:", static_dir)
print("DB path:", DATABASE)

app = Flask(__name__, template_folder=template_dir, static_folder=static_dir)
CORS(app)

# -------------------- DB --------------------
def get_db():
    db = getattr(g, '_database', None)
    if db is None:
        db = g._database = sqlite3.connect(DATABASE)
        db.row_factory = sqlite3.Row
    return db

@app.teardown_appcontext
def close_connection(exception):
    db = getattr(g, '_database', None)
    if db is not None:
        db.close()

def ensure_todos_is_completed_column():
    db = get_db()
    cur = db.cursor()
    cur.execute("PRAGMA table_info(todos)")
    cols = [r['name'] for r in cur.fetchall()]
    if 'is_completed' not in cols:
        try:
            cur.execute("ALTER TABLE todos ADD COLUMN is_completed INTEGER DEFAULT 0")
            db.commit()
            print("Added todos.is_completed column.")
        except Exception as e:
            print("Failed to add todos.is_completed:", e)

def init_db():
    with app.app_context():
        db = get_db()
        cursor = db.cursor()
        cursor.execute('''
            CREATE TABLE IF NOT EXISTS fitness_plans (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                date TEXT NOT NULL,
                exercise_name TEXT NOT NULL,
                sets_reps_duration TEXT NOT NULL,
                is_completed INTEGER DEFAULT 0,
                created_by TEXT DEFAULT 'system'
            )
        ''')
        cursor.execute('''
            CREATE TABLE IF NOT EXISTS recurring_fitness_plans (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                plan_name TEXT NOT NULL,
                exercises_json TEXT NOT NULL,
                start_date TEXT NOT NULL,
                end_date TEXT,
                repeat_interval TEXT NOT NULL,
                repeat_day_of_week INTEGER
            )
        ''')
        cursor.execute('''
            CREATE TABLE IF NOT EXISTS todos (
                id INTEGER PRIMARY KEY AUTOINCREMENT,
                date TEXT NOT NULL,
                content TEXT NOT NULL
            )
        ''')
        db.commit()

        # migration: add todos.is_completed if missing
        ensure_todos_is_completed_column()

    print("Database initialized.")

# -------------------- Helpers --------------------
def _generate_daily_fitness_plan_if_not_exists(date_str):
    with app.app_context():
        db = get_db()
        cursor = db.cursor()
        cursor.execute("SELECT COUNT(*) FROM fitness_plans WHERE date = ?", (date_str,))
        count = cursor.fetchone()[0]
        if count == 0:
            today = datetime.strptime(date_str, '%Y-%m-%d')
            cursor.execute("""
                SELECT * FROM recurring_fitness_plans
                WHERE start_date <= ? AND (end_date IS NULL OR end_date >= ?)
            """, (date_str, date_str))
            recurring_plans = cursor.fetchall()

            exercises_to_add = []
            if recurring_plans:
                for plan in recurring_plans:
                    add_plan = False
                    if plan['repeat_interval'] == 'daily':
                        add_plan = True
                    elif plan['repeat_interval'] == 'weekly' and plan['repeat_day_of_week'] == today.weekday():
                        add_plan = True
                    if add_plan:
                        plan_exercises = json.loads(plan['exercises_json'])
                        for ex in plan_exercises:
                            exercises_to_add.append((date_str, ex['name'], ex['details'], 0, 'system'))

            if not exercises_to_add:
                exercises_to_add = [
                    (date_str, "热身：慢跑", "10分钟", 0, 'system'),
                    (date_str, "力量：俯卧撑", "3组x10次", 0, 'system'),
                    (date_str, "力量：深蹲", "3组x12次", 0, 'system'),
                    (date_str, "核心：平板支撑", "3组x45秒", 0, 'system'),
                    (date_str, "有氧：跳绳", "15分钟", 0, 'system'),
                    (date_str, "拉伸", "10分钟", 0, 'system')
                ]
            cursor.executemany(
                "INSERT INTO fitness_plans (date, exercise_name, sets_reps_duration, is_completed, created_by) VALUES (?, ?, ?, ?, ?)",
                exercises_to_add
            )
            db.commit()

def _generate_daily_todos_if_not_exists(date_str):
    with app.app_context():
        db = get_db()
        cursor = db.cursor()
        cursor.execute("SELECT COUNT(*) FROM todos WHERE date = ?", (date_str,))
        count = cursor.fetchone()[0]
        if count == 0:
            # insert with is_completed if column exists
            cursor.execute("PRAGMA table_info(todos)")
            cols = [r['name'] for r in cursor.fetchall()]
            if 'is_completed' in cols:
                todos_today = [
                    (date_str, "购买蛋白粉", 0),
                    (date_str, "预约健身私教课", 0),
                    (date_str, "研究新的健身食谱", 0)
                ]
                cursor.executemany("INSERT INTO todos (date, content, is_completed) VALUES (?, ?, ?)", todos_today)
            else:
                todos_today = [
                    (date_str, "购买蛋白粉"),
                    (date_str, "预约健身私教课"),
                    (date_str, "研究新的健身食谱")
                ]
                cursor.executemany("INSERT INTO todos (date, content) VALUES (?, ?)", todos_today)
            db.commit()

# -------------------- Routes (public UI) --------------------
@app.route('/')
def index():
    today_str = datetime.now().strftime('%Y-%m-%d')
    _generate_daily_fitness_plan_if_not_exists(today_str)
    _generate_daily_todos_if_not_exists(today_str)
    return render_template('display.html')

@app.route('/mobile_config')
def mobile_config():
    return render_template('mobile_config.html')

# -------------------- API endpoints used by frontends --------------------
@app.route('/fitness_plan/<date_str>', methods=['GET'])
def get_fitness_plan(date_str):
    _generate_daily_fitness_plan_if_not_exists(date_str)
    db = get_db()
    cursor = db.cursor()
    cursor.execute("SELECT id, date, exercise_name, sets_reps_duration, is_completed, created_by FROM fitness_plans WHERE date = ? ORDER BY id", (date_str,))
    plans = cursor.fetchall()
    return jsonify([dict(row) for row in plans])

@app.route('/todos', methods=['GET'])
def get_todos():
    date_str = request.args.get('date')
    if not date_str:
        return jsonify(status='error', message='Missing date parameter.'), 400
    _generate_daily_todos_if_not_exists(date_str)
    db = get_db()
    cursor = db.cursor()
    cursor.execute("PRAGMA table_info(todos)")
    cols = [r['name'] for r in cursor.fetchall()]
    if 'is_completed' in cols:
        cursor.execute("SELECT id, date, content, is_completed FROM todos WHERE date = ? ORDER BY id", (date_str,))
    else:
        cursor.execute("SELECT id, date, content, 0 as is_completed FROM todos WHERE date = ? ORDER BY id", (date_str,))
    todos = cursor.fetchall()
    return jsonify([dict(row) for row in todos])

# mark fitness completed
@app.route('/mark_exercise_completed_mobile', methods=['POST'])
def mark_exercise_completed_mobile():
    data = request.get_json()
    plan_id = data.get('plan_id')
    is_completed = data.get('is_completed')
    if plan_id is None or is_completed is None:
        return jsonify(status='error', message='Missing plan_id or is_completed data.'), 400
    db = get_db()
    cursor = db.cursor()
    cursor.execute("UPDATE fitness_plans SET is_completed = ? WHERE id = ?", (is_completed, plan_id))
    db.commit()
    if cursor.rowcount > 0:
        return jsonify(status='success', message='Exercise completion status updated.')
    else:
        return jsonify(status='error', message='Exercise not found.'), 404

# mark todo completed
@app.route('/mark_todo_completed', methods=['POST'])
def mark_todo_completed():
    data = request.get_json()
    todo_id = data.get('todo_id')
    is_completed = data.get('is_completed')
    if todo_id is None or is_completed is None:
        return jsonify(status='error', message='Missing todo_id or is_completed'), 400
    db = get_db()
    cursor = db.cursor()
    cursor.execute("UPDATE todos SET is_completed = ? WHERE id = ?", (is_completed, todo_id))
    db.commit()
    if cursor.rowcount > 0:
        return jsonify(status='success', message='Todo status updated')
    else:
        return jsonify(status='error', message='Todo not found'), 404

# -------------------- Management API (add/update/delete) --------------------
# Fitness plan CRUD (user-managed per-day)
@app.route('/add_fitness_plan_exercise', methods=['POST'])
def add_fitness_plan_exercise():
    data = request.get_json()
    date_str = data.get('date')
    exercise_name = data.get('exercise_name')
    sets_reps_duration = data.get('sets_reps_duration')
    if not all([date_str, exercise_name, sets_reps_duration]):
        return jsonify(status='error', message='Missing data.'), 400
    db = get_db()
    cursor = db.cursor()
    cursor.execute("INSERT INTO fitness_plans (date, exercise_name, sets_reps_duration, created_by) VALUES (?, ?, ?, 'user')", (date_str, exercise_name, sets_reps_duration))
    db.commit()
    return jsonify(status='success', message='Exercise added successfully.', id=cursor.lastrowid)

@app.route('/update_fitness_plan_exercise/<int:plan_id>', methods=['POST'])
def update_fitness_plan_exercise(plan_id):
    data = request.get_json()
    exercise_name = data.get('exercise_name')
    sets_reps_duration = data.get('sets_reps_duration')
    if not all([exercise_name, sets_reps_duration]):
        return jsonify(status='error', message='Missing data.'), 400
    db = get_db()
    cursor = db.cursor()
    cursor.execute("UPDATE fitness_plans SET exercise_name = ?, sets_reps_duration = ?, created_by = 'user' WHERE id = ?", (exercise_name, sets_reps_duration, plan_id))
    db.commit()
    if cursor.rowcount > 0:
        return jsonify(status='success', message='Exercise updated successfully.')
    else:
        return jsonify(status='error', message='Exercise not found.'), 404

@app.route('/delete_fitness_plan_exercise/<int:plan_id>', methods=['POST'])
def delete_fitness_plan_exercise(plan_id):
    db = get_db()
    cursor = db.cursor()
    cursor.execute("DELETE FROM fitness_plans WHERE id = ?", (plan_id,))
    db.commit()
    if cursor.rowcount > 0:
        return jsonify(status='success', message='Exercise deleted successfully.')
    else:
        return jsonify(status='error', message='Exercise not found.'), 404

# Recurring plans CRUD (user-managed)
@app.route('/recurring_plans', methods=['GET'])
def get_recurring_plans():
    db = get_db()
    cursor = db.cursor()
    cursor.execute("SELECT * FROM recurring_fitness_plans ORDER BY start_date DESC, plan_name")
    plans = cursor.fetchall()
    return jsonify([dict(row) for row in plans])

@app.route('/add_recurring_plan', methods=['POST'])
def add_recurring_plan():
    data = request.get_json()
    plan_name = data.get('plan_name')
    exercises = data.get('exercises')
    start_date = data.get('start_date')
    end_date = data.get('end_date')
    repeat_interval = data.get('repeat_interval')
    repeat_day_of_week = data.get('repeat_day_of_week')
    if not all([plan_name, exercises, start_date, repeat_interval]):
        return jsonify(status='error', message='Missing required recurring plan data.'), 400
    db = get_db()
    cursor = db.cursor()
    try:
        cursor.execute("INSERT INTO recurring_fitness_plans (plan_name, exercises_json, start_date, end_date, repeat_interval, repeat_day_of_week) VALUES (?, ?, ?, ?, ?, ?)", (plan_name, json.dumps(exercises), start_date, end_date, repeat_interval, repeat_day_of_week))
        db.commit()
        return jsonify(status='success', message='Recurring plan added successfully.', id=cursor.lastrowid)
    except sqlite3.Error as e:
        return jsonify(status='error', message=f'Database error: {e}'), 500

@app.route('/update_recurring_plan/<int:plan_id>', methods=['POST'])
def update_recurring_plan(plan_id):
    data = request.get_json()
    plan_name = data.get('plan_name')
    exercises = data.get('exercises')
    start_date = data.get('start_date')
    end_date = data.get('end_date')
    repeat_interval = data.get('repeat_interval')
    repeat_day_of_week = data.get('repeat_day_of_week')
    if not all([plan_name, exercises, start_date, repeat_interval]):
        return jsonify(status='error', message='Missing required recurring plan data.'), 400
    db = get_db()
    cursor = db.cursor()
    try:
        cursor.execute("UPDATE recurring_fitness_plans SET plan_name = ?, exercises_json = ?, start_date = ?, end_date = ?, repeat_interval = ?, repeat_day_of_week = ? WHERE id = ?", (plan_name, json.dumps(exercises), start_date, end_date, repeat_interval, repeat_day_of_week, plan_id))
        db.commit()
        if cursor.rowcount > 0:
            return jsonify(status='success', message='Recurring plan updated successfully.')
        else:
            return jsonify(status='error', message='Recurring plan not found.'), 404
    except sqlite3.Error as e:
        return jsonify(status='error', message=f'Database error: {e}'), 500

@app.route('/delete_recurring_plan/<int:plan_id>', methods=['POST'])
def delete_recurring_plan(plan_id):
    db = get_db()
    cursor = db.cursor()
    cursor.execute("DELETE FROM recurring_fitness_plans WHERE id = ?", (plan_id,))
    db.commit()
    if cursor.rowcount > 0:
        return jsonify(status='success', message='Recurring plan deleted successfully.')
    else:
        return jsonify(status='error', message='Recurring plan not found.'), 404

# Todos CRUD
@app.route('/add_todo', methods=['POST'])
def add_todo():
    data = request.get_json()
    date_str = data.get('date')
    content = data.get('content')
    if not all([date_str, content]):
        return jsonify(status='error', message='Missing date or content data.'), 400
    db = get_db()
    cursor = db.cursor()
    cursor.execute("PRAGMA table_info(todos)")
    cols = [r['name'] for r in cursor.fetchall()]
    try:
        if 'is_completed' in cols:
            cursor.execute("INSERT INTO todos (date, content, is_completed) VALUES (?, ?, 0)", (date_str, content))
        else:
            cursor.execute("INSERT INTO todos (date, content) VALUES (?, ?)", (date_str, content))
        db.commit()
        return jsonify(status='success', message='Todo added successfully.', id=cursor.lastrowid)
    except sqlite3.Error as e:
        return jsonify(status='error', message=f'Database error: {e}'), 500

@app.route('/update_todo/<int:todo_id>', methods=['POST'])
def update_todo(todo_id):
    data = request.get_json()
    content = data.get('content')
    if not content:
        return jsonify(status='error', message='Missing content data.'), 400
    db = get_db()
    cursor = db.cursor()
    try:
        cursor.execute("UPDATE todos SET content = ? WHERE id = ?", (content, todo_id))
        db.commit()
        if cursor.rowcount > 0:
            return jsonify(status='success', message='Todo updated successfully.')
        else:
            return jsonify(status='error', message='Todo not found.'), 404
    except sqlite3.Error as e:
        return jsonify(status='error', message=f'Database error: {e}'), 500

@app.route('/delete_todo/<int:todo_id>', methods=['POST'])
def delete_todo(todo_id):
    db = get_db()
    cursor = db.cursor()
    try:
        cursor.execute("DELETE FROM todos WHERE id = ?", (todo_id,))
        db.commit()
        if cursor.rowcount > 0:
            return jsonify(status='success', message='Todo deleted successfully.')
        else:
            return jsonify(status='error', message='Todo not found.'), 404
    except sqlite3.Error as e:
        return jsonify(status='error', message=f'Database error: {e}'), 500

# -------------------- Compatibility headers & health --------------------
from flask import make_response
@app.after_request
def add_common_headers(response):
    if 'Content-Type' not in response.headers:
        response.headers['Content-Type'] = 'text/html; charset=utf-8'
    else:
        c = response.headers['Content-Type']
        if c.startswith('text/') and 'charset' not in c:
            response.headers['Content-Type'] = c + '; charset=utf-8'
    response.headers.setdefault('X-Content-Type-Options', 'nosniff')
    response.headers.setdefault('X-Frame-Options', 'SAMEORIGIN')
    path = request.path or ''
    if path.startswith('/static/') or path.endswith(('.css', '.js', '.png', '.jpg', '.svg', '.webp')):
        response.headers['Cache-Control'] = 'public, max-age=3600'
    else:
        response.headers['Cache-Control'] = 'no-cache, no-store, must-revalidate'
    return response

@app.route('/health', methods=['GET'])
def health():
    return jsonify(status='ok', time=datetime.now().isoformat())

# -------------------- Run --------------------
if __name__ == '__main__':
    init_db()
    print("Starting Flask on 0.0.0.0:8000")
    app.run(host='0.0.0.0', port=8000, debug=True)
