#!/usr/bin/python
# -*- coding:utf-8 -*-
import serial
import RPi.GPIO as GPIO
import time

# 定义发送使能引脚
TXDEN_1 = 27  # 通道1发送使能
TXDEN_2 = 22  # 通道2发送使能

# 初始化串口
ser1 = serial.Serial(
    port='/dev/ttySC0',
    baudrate=2400,
    bytesize=serial.EIGHTBITS,
    parity=serial.PARITY_NONE,
    stopbits=serial.STOPBITS_ONE,
    timeout=1
)

def calculate_checksum(data):
    """计算Pelco-D协议的校验和"""
    checksum = 0
    for byte in data:
        checksum += byte
    return checksum % 256

def send_pelco_d_command(address, command1, command2, data1, data2):
    """
    发送Pelco-D协议命令
    """
    sync_byte = 0xFF
    checksum = calculate_checksum([address, command1, command2, data1, data2])
    
    command_frame = bytes([sync_byte, address, command1, command2, data1, data2, checksum])
    
    print(f"发送命令: {[f'{x:02X}' for x in command_frame]}")
    
    GPIO.output(TXDEN_1, GPIO.LOW)
    ser1.write(command_frame)
    time.sleep(0.1)
    GPIO.output(TXDEN_1, GPIO.HIGH)

def stop_all_movements(address=1):
    """停止所有运动"""
    print("🛑 停止所有运动")
    send_pelco_d_command(address, 0x00, 0x00, 0x00, 0x00)

def pan_clockwise(address=1, pan_speed=0x20, duration=3.0):
    """
    水平顺时针旋转
    :param pan_speed: 水平转动速度 (0x00-0x3F)
    :param duration: 转动持续时间（秒）
    """
    # Pelco-D命令: 命令1=0x00, 命令2=0x02 (右/顺时针)
    command1 = 0x00
    command2 = 0x02  # 右/顺时针
    data1 = pan_speed   # 水平速度
    data2 = 0x00       # 垂直速度=0
    
    print(f"🔄 开始顺时针旋转，速度: 0x{pan_speed:02X}，持续时间: {duration}秒")
    send_pelco_d_command(address, command1, command2, data1, data2)
    
    # 等待转动完成
    time.sleep(duration)
    
    # 停止转动
    stop_all_movements(address)
    print("✅ 顺时针旋转完成")

def pan_counter_clockwise(address=1, pan_speed=0x20, duration=4.5):
    """
    水平逆时针旋转
    :param pan_speed: 水平转动速度 (0x00-0x3F)
    :param duration: 转动持续时间（秒）
    """
    # Pelco-D命令: 命令1=0x00, 命令2=0x04 (左/逆时针)
    command1 = 0x00
    command2 = 0x04  # 左/逆时针
    data1 = pan_speed   # 水平速度
    data2 = 0x00       # 垂直速度=0
    
    print(f"🔄 开始逆时针旋转，速度: 0x{pan_speed:02X}，持续时间: {duration}秒")
    send_pelco_d_command(address, command1, command2, data1, data2)
    
    # 等待转动完成
    time.sleep(duration)
    
    # 停止转动
    stop_all_movements(address)
    print("✅ 逆时针旋转完成")

def calculate_duration_for_angle(angle_degrees, speed_level=0x20):
    """
    根据角度计算需要的转动时间
    :param angle_degrees: 旋转角度（度）
    :param speed_level: 速度等级 (0x00-0x3F)
    :return: 需要的转动时间（秒）
    """
    # 这些参数需要根据实际云台性能调整
    base_speed = 30.0  # 度/秒（在速度0x20时）
    speed_factor = speed_level / 0x20  # 速度系数
    
    actual_speed = base_speed * speed_factor
    duration = angle_degrees / actual_speed
    
    print(f"角度: {angle_degrees}°, 速度等级: 0x{speed_level:02X}, 预计时间: {duration:.2f}秒")
    return duration

# 主程序
try:
    # 初始化GPIO
    GPIO.setmode(GPIO.BCM)
    GPIO.setup(TXDEN_1, GPIO.OUT)
    GPIO.setup(TXDEN_2, GPIO.OUT)
    
    # 初始化为接收状态
    GPIO.output(TXDEN_1, GPIO.HIGH)
    GPIO.output(TXDEN_2, GPIO.HIGH)
    
    print("=" * 50)
    print("Pelco-D云台角度控制程序")
    print(f"波特率: 2400, 设备地址: 1")
    print("=" * 50)
    
    # 设置转动速度 (0x00-0x3F, 0x20是中等速度)
    pan_speed = 0x20
    
    # 计算转动30度和45度需要的时间
    duration_30deg = calculate_duration_for_angle(60, pan_speed)
    duration_45deg = calculate_duration_for_angle(180, pan_speed)
    
    print("\n开始执行旋转序列...")
    
    # 第一步：顺时针旋转30度
    print(f"\n1. 顺时针旋转30度")
    pan_clockwise(address=1, pan_speed=pan_speed, duration=duration_30deg)
    time.sleep(1)  # 停顿1秒
    
    # 第二步：逆时针旋转45度
    print(f"\n2. 逆时针旋转45度")
    pan_counter_clockwise(address=1, pan_speed=pan_speed, duration=duration_45deg)
    
    print("\n🎉 所有旋转操作完成！")
    print("最终位置: 从起点逆时针15度")

except KeyboardInterrupt:
    print("\n⚠️ 程序被用户中断")
    stop_all_movements(address=1)

except Exception as e:
    print(f"❌ 发生错误: {e}")
    import traceback
    traceback.print_exc()
    stop_all_movements(address=1)

finally:
    # 清理资源
    GPIO.cleanup()
    ser1.close()
    print("🔚 程序结束，资源已清理")
